"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.testConstructTags = exports.resourceTagMatcher = void 0;
const assert_1 = require("@aws-cdk/assert");
/**
 * The name of the tag that RFDK is expected to use
 */
const RFDK_TAG_NAME = 'aws-rfdk';
/**
 * The current version of the RFDK package
 */
const RFDK_VERSION = require('../../../package.json').version; // eslint-disable-line @typescript-eslint/no-require-imports
/**
 * Returns the resource properties for an expected RFDK tag for a given resource
 * type
 *
 * @param resourceType The L1 CloudFormation resource type (e.g. "AWS::EC2::Instance")
 * @param constructName The name of the L2 (or higher) RFDK construct class
 */
function getExpectedRfdkTagProperties(resourceType, constructName) {
    const expectedValue = `${RFDK_VERSION}:${constructName}`;
    return resourceTagMatcher(resourceType, RFDK_TAG_NAME, expectedValue);
}
/**
 * Returns a CDK matcher for an expected tag key/value pair for a given Cfn resource type.
 * This is known to support the following resource types:
 *
 * * `AWS::AutoScaling::AutoScalingGroup`
 * * `AWS::EC2::SecurityGroup`
 * * `AWS::IAM::Role`
 * * `AWS::SSM::Parameter`
 *
 * All other resources are assumed to allow passing the following tag properties:
 *
 * ```js
 * {
 *   Tags: [
 *     {
 *       Key: 'key',
 *       Value: 'value',
 *     },
 *     // ...
 *   ]
 * }
 * ```
 */
/* eslint-disable-next-line jest/no-export */
function resourceTagMatcher(resourceType, tagName, tagValue) {
    if (resourceType === 'AWS::SSM::Parameter') {
        return {
            Tags: {
                [tagName]: tagValue,
            },
        };
    }
    else if (resourceType === 'AWS::AutoScaling::AutoScalingGroup') {
        return {
            Tags: assert_1.arrayWith({
                Key: tagName,
                PropagateAtLaunch: true,
                Value: tagValue,
            }),
        };
    }
    else {
        return {
            Tags: assert_1.arrayWith({
                Key: tagName,
                Value: tagValue,
            }),
        };
    }
}
exports.resourceTagMatcher = resourceTagMatcher;
/**
 * Creates tests for the specified resources created by a L2 (or higher construct) to ensure that the resources it
 * creates are created with the RFDK tagging convention.
 *
 * The convention is to create a tag named "aws-rfdk" whose value follows:
 *
 * <VERSION>:<CONSTRUCT_NAME>
 *
 * @param args Arguments to configure the creation of construct tagging tests
 */
/* eslint-disable-next-line jest/no-export */
function testConstructTags(args) {
    const { constructName, createConstruct, resourceTypeCounts } = args;
    const entries = Object.entries(resourceTypeCounts);
    test.each(entries)('tags %s x%d', (resourceType, expectedCount) => {
        // GIVEN
        const expectedProps = getExpectedRfdkTagProperties(resourceType, constructName);
        // WHEN
        const stack = createConstruct();
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike(resourceType, expectedCount, expectedProps));
    });
}
exports.testConstructTags = testConstructTags;
//# sourceMappingURL=data:application/json;base64,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