"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const core_1 = require("@aws-cdk/core");
const pad_efs_storage_1 = require("../lib/pad-efs-storage");
describe('Test PadEfsStorage', () => {
    let app;
    let stack;
    let vpc;
    let efsFS;
    let accessPoint;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        efsFS = new aws_efs_1.FileSystem(stack, 'EFS', { vpc });
        accessPoint = new aws_efs_1.AccessPoint(stack, 'AccessPoint', {
            fileSystem: efsFS,
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            path: '/SpaceFillingFiles',
            posixUser: {
                uid: '1000',
                gid: '1000',
            },
        });
    });
    test('Create with defaults', () => {
        // WHEN
        const pad = new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: core_1.Size.gibibytes(20),
        });
        const sg = pad.node.findChild('LambdaSecurityGroup');
        const diskUsage = pad.node.findChild('DiskUsage');
        const padFilesystem = pad.node.findChild('PadFilesystem');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Properties: {
                FileSystemConfigs: [
                    {
                        Arn: stack.resolve(accessPoint.accessPointArn),
                        LocalMountPath: '/mnt/efs',
                    },
                ],
                Handler: 'pad-efs-storage.getDiskUsage',
                Runtime: 'nodejs16.x',
                Timeout: 300,
                VpcConfig: {
                    SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
                    SubnetIds: [
                        {
                            Ref: 'VpcPrivateSubnet1Subnet536B997A',
                        },
                        {
                            Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                        },
                    ],
                },
            },
            DependsOn: assert_1.arrayWith(stack.getLogicalId(accessPoint.node.defaultChild)),
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Properties: {
                FileSystemConfigs: [
                    {
                        Arn: stack.resolve(accessPoint.accessPointArn),
                        LocalMountPath: '/mnt/efs',
                    },
                ],
                Handler: 'pad-efs-storage.padFilesystem',
                Runtime: 'nodejs16.x',
                Timeout: 900,
                VpcConfig: {
                    SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
                    SubnetIds: [
                        {
                            Ref: 'VpcPrivateSubnet1Subnet536B997A',
                        },
                        {
                            Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                        },
                    ],
                },
            },
            DependsOn: assert_1.arrayWith(stack.getLogicalId(accessPoint.node.defaultChild)),
        }, assert_1.ResourcePart.CompleteDefinition));
        const lambdaRetryCatch = {
            Retry: [
                {
                    ErrorEquals: [
                        'Lambda.ServiceException',
                        'Lambda.AWSLambdaException',
                        'Lambda.SdkClientException',
                    ],
                    IntervalSeconds: 2,
                    MaxAttempts: 6,
                    BackoffRate: 2,
                },
            ],
            Catch: [
                {
                    ErrorEquals: ['States.ALL'],
                    Next: 'Fail',
                },
            ],
        };
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::StepFunctions::StateMachine', {
            DefinitionString: stack.resolve(JSON.stringify({
                StartAt: 'QueryDiskUsage',
                States: {
                    QueryDiskUsage: {
                        Next: 'BranchOnDiskUsage',
                        ...lambdaRetryCatch,
                        Type: 'Task',
                        Comment: 'Determine the number of GB currently stored in the EFS access point',
                        TimeoutSeconds: 300,
                        ResultPath: '$.diskUsage',
                        Resource: `arn:${stack.partition}:states:::lambda:invoke`,
                        Parameters: {
                            FunctionName: `${diskUsage.functionArn}`,
                            Payload: {
                                'desiredPadding.$': '$.desiredPadding',
                                'mountPoint': '/mnt/efs',
                            },
                        },
                    },
                    GrowTask: {
                        Next: 'QueryDiskUsage',
                        ...lambdaRetryCatch,
                        Type: 'Task',
                        Comment: 'Add up to 20 numbered 1GB files to the EFS access point',
                        TimeoutSeconds: 900,
                        ResultPath: '$.null',
                        Resource: `arn:${stack.partition}:states:::lambda:invoke`,
                        Parameters: {
                            FunctionName: `${padFilesystem.functionArn}`,
                            Payload: {
                                'desiredPadding.$': '$.desiredPadding',
                                'mountPoint': '/mnt/efs',
                            },
                        },
                    },
                    BranchOnDiskUsage: {
                        Type: 'Choice',
                        Choices: [
                            {
                                Variable: '$.diskUsage.Payload',
                                NumericLessThanPath: '$.desiredPadding',
                                Next: 'GrowTask',
                            },
                            {
                                Variable: '$.diskUsage.Payload',
                                NumericGreaterThanPath: '$.desiredPadding',
                                Next: 'ShrinkTask',
                            },
                        ],
                        Default: 'Succeed',
                    },
                    Succeed: {
                        Type: 'Succeed',
                    },
                    ShrinkTask: {
                        Next: 'Succeed',
                        ...lambdaRetryCatch,
                        Type: 'Task',
                        Comment: 'Remove 1GB numbered files from the EFS access point to shrink the padding',
                        TimeoutSeconds: 900,
                        ResultPath: '$.null',
                        Resource: `arn:${stack.partition}:states:::lambda:invoke`,
                        Parameters: {
                            FunctionName: `${padFilesystem.functionArn}`,
                            Payload: {
                                'desiredPadding.$': '$.desiredPadding',
                                'mountPoint': '/mnt/efs',
                            },
                        },
                    },
                    Fail: {
                        Type: 'Fail',
                    },
                },
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::AWS', {
            Create: {
                'Fn::Join': [
                    '',
                    [
                        '{"action":"startExecution","service":"StepFunctions","apiVersion":"2016-11-23","region":"',
                        {
                            Ref: 'AWS::Region',
                        },
                        '","physicalResourceId":{"responsePath":"executionArn"},"parameters":{"stateMachineArn":"',
                        {
                            Ref: 'PadEfsStateMachineDA538E87',
                        },
                        '","input":"{\\"desiredPadding\\":20}"}}',
                    ],
                ],
            },
            Update: {
                'Fn::Join': [
                    '',
                    [
                        '{"action":"startExecution","service":"StepFunctions","apiVersion":"2016-11-23","region":"',
                        {
                            Ref: 'AWS::Region',
                        },
                        '","physicalResourceId":{"responsePath":"executionArn"},"parameters":{"stateMachineArn":"',
                        {
                            Ref: 'PadEfsStateMachineDA538E87',
                        },
                        '","input":"{\\"desiredPadding\\":20}"}}',
                    ],
                ],
            },
        }));
    });
    test('Set desiredPadding', () => {
        // WHEN
        const desiredPadding = 200;
        new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: core_1.Size.gibibytes(desiredPadding),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::AWS', {
            Create: {
                'Fn::Join': [
                    '',
                    assert_1.arrayWith(`","input":"{\\"desiredPadding\\":${desiredPadding}}"}}`),
                ],
            },
            Update: {
                'Fn::Join': [
                    '',
                    assert_1.arrayWith(`","input":"{\\"desiredPadding\\":${desiredPadding}}"}}`),
                ],
            },
        }));
    });
    test('Throws on bad desiredPadding', () => {
        // GIVEN
        const pad = new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: core_1.Size.mebibytes(100),
        });
        // THEN
        expect(pad.node.metadataEntry).toEqual(expect.arrayContaining([
            expect.objectContaining({
                type: 'aws:cdk:error',
                data: 'Failed to round desiredSize to an integer number of GiB. The size must be in GiB.',
            }),
        ]));
    });
    test('Provide SecurityGroup', () => {
        // GIVEN
        const sg = new aws_ec2_1.SecurityGroup(stack, 'TestSG', {
            vpc,
        });
        // WHEN
        new pad_efs_storage_1.PadEfsStorage(stack, 'PadEfs', {
            vpc,
            accessPoint,
            desiredPadding: core_1.Size.gibibytes(20),
            securityGroup: sg,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'pad-efs-storage.getDiskUsage',
            VpcConfig: {
                SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'pad-efs-storage.padFilesystem',
            VpcConfig: {
                SecurityGroupIds: [stack.resolve(sg.securityGroupId)],
            },
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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