"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const fsx = require("@aws-cdk/aws-fsx");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('MountableFsxLustre', () => {
    let app;
    let stack;
    let vpc;
    let fs;
    let fsSecurityGroup;
    let instance;
    let instanceSecurityGroup;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        fsSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'FSxLSecurityGroup', {
            vpc,
        });
        fs = new fsx.LustreFileSystem(stack, 'FSxL', {
            vpc,
            vpcSubnet: vpc.privateSubnets[0],
            lustreConfiguration: {
                deploymentType: fsx.LustreDeploymentType.SCRATCH_1,
            },
            storageCapacityGiB: 1200,
            securityGroup: fsSecurityGroup,
        });
        instanceSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'InstanceSecurityGroup', {
            vpc,
        });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            securityGroup: instanceSecurityGroup,
        });
    });
    test('mounts with defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the FSxL's security group
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 988,
            ToPort: 1023,
            SourceSecurityGroupId: stack.resolve(instanceSecurityGroup.securityGroupId),
            GroupId: stack.resolve(fsSecurityGroup.securityGroupId),
        }));
        // Make sure we download the mountFsxLustre script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('unzip /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
        // Make sure we install the Lustre client
        expect(userData).toMatch('bash ./installLustreClient.sh');
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/fsx/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} r')));
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} rw,option1,option2')));
    });
    test('asset is singleton', () => {
        // GIVEN
        const mount1 = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        const mount2 = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = userData.match(regex) ?? [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    test('applies Lustre fileset', () => {
        // GIVEN
        const fileset = 'fileset';
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
            fileset,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`bash ./mountFsxLustre.sh \${Token[TOKEN.\\d+]} /mnt/fsx/fs1 \${Token[TOKEN.\\d+]}/${fileset} rw`)));
    });
    test('.usesUserPosixPermissions() is true', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        const usesUserPosixPermissions = mount.usesUserPosixPermissions();
        // THEN
        expect(usesUserPosixPermissions).toEqual(true);
    });
});
//# sourceMappingURL=data:application/json;base64,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