"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const mount_permissions_helper_1 = require("../lib/mount-permissions-helper");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MountableEFS', () => {
    let app;
    let stack;
    let vpc;
    let efsFS;
    let instance;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        efsFS = new efs.FileSystem(stack, 'EFS', { vpc });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the EFS's security group
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 2049,
            ToPort: 2049,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'InstanceInstanceSecurityGroupF0E2D5BE',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'EFSEfsSecurityGroup56F189CE',
                    'GroupId',
                ],
            },
        }));
        // Make sure we download the mountEfs script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('unzip /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/efs/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false r')));
    });
    describe.each([
        [undefined],
        [lib_1.MountPermissions.READONLY],
        [lib_1.MountPermissions.READWRITE],
    ])('access point with %s access permissions', (mountPermission) => {
        describe.each([
            [
                'unspecified POSIX user',
                {
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved non-root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '1000' },
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '0' },
                    expectedClientRootAccess: true,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '0', gid: '1000' },
                    expectedClientRootAccess: true,
                },
            ],
        ])('%s', (_name, testCase) => {
            // GIVEN
            const { posixUser, expectedClientRootAccess } = testCase;
            const expectedActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mountPermission);
            if (expectedClientRootAccess) {
                expectedActions.push('elasticfilesystem:ClientRootAccess');
            }
            const mountPath = '/mnt/efs/fs1';
            let userData;
            let accessPoint;
            let expectedMountMode;
            beforeEach(() => {
                // GIVEN
                accessPoint = new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser,
                });
                const mount = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                });
                expectedMountMode = (mountPermission === lib_1.MountPermissions.READONLY) ? 'ro' : 'rw';
                // WHEN
                mount.mountToLinuxInstance(instance, {
                    location: mountPath,
                    permissions: mountPermission,
                });
                userData = stack.resolve(instance.userData.render());
            });
            test('userdata specifies access point when mounting', () => {
                // THEN
                expect(userData).toEqual({
                    'Fn::Join': [
                        '',
                        expect.arrayContaining([
                            expect.stringMatching(new RegExp('(\\n|^)bash \\./mountEfs.sh $')),
                            stack.resolve(efsFS.fileSystemId),
                            ` ${mountPath} false ${expectedMountMode},iam,accesspoint=`,
                            stack.resolve(accessPoint.accessPointId),
                            expect.stringMatching(/^\n/),
                        ]),
                    ],
                });
            });
            test('grants IAM access point permissions', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: assert_1.objectLike({
                        Statement: assert_1.arrayWith({
                            Action: expectedActions.length === 1 ? expectedActions[0] : expectedActions,
                            Condition: {
                                StringEquals: {
                                    'elasticfilesystem:AccessPointArn': stack.resolve(accessPoint.accessPointArn),
                                },
                            },
                            Effect: 'Allow',
                            Resource: stack.resolve(efsFS.node.defaultChild.attrArn),
                        }),
                        Version: '2012-10-17',
                    }),
                    Roles: assert_1.arrayWith(
                    // The Policy construct micro-optimizes the reference to a role in the same stack using its logical ID
                    stack.resolve(instance.role.node.defaultChild.ref)),
                }));
            });
        });
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw,option1,option2')));
    });
    test('asset is singleton', () => {
        // GIVEN
        const mount1 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        const mount2 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = userData.match(regex) ?? [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    describe('resolves mount target using API', () => {
        describe.each([
            ['with access point', () => {
                    return new efs.AccessPoint(stack, 'AccessPoint', {
                        fileSystem: efsFS,
                        posixUser: {
                            gid: '1',
                            uid: '1',
                        },
                    });
                }],
            ['without access point', () => undefined],
        ])('%s', (_, getAccessPoint) => {
            let accessPoint;
            beforeEach(() => {
                // GIVEN
                accessPoint = getAccessPoint();
                const mountable = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                    resolveMountTargetDnsWithApi: true,
                });
                // WHEN
                mountable.mountToLinuxInstance(instance, {
                    location: '/mnt/efs',
                });
            });
            test('grants DescribeMountTargets permission', () => {
                const expectedResources = [
                    stack.resolve(efsFS.node.defaultChild.attrArn),
                ];
                if (accessPoint) {
                    expectedResources.push(stack.resolve(accessPoint?.accessPointArn));
                }
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: assert_1.objectLike({
                        Statement: assert_1.arrayWith({
                            Action: 'elasticfilesystem:DescribeMountTargets',
                            Effect: 'Allow',
                            Resource: expectedResources.length == 1 ? expectedResources[0] : expectedResources,
                        }),
                    }),
                    Roles: assert_1.arrayWith(stack.resolve(instance.role.node.defaultChild.ref)),
                }));
            });
        });
    });
    describe('.usesUserPosixPermissions()', () => {
        test('access point with POSIX user returns false', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint: new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser: {
                        uid: '1000',
                        gid: '1000',
                    },
                }),
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(false);
        });
        test('access point without POSIX user returns true', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint: new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                }),
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(true);
        });
        test.each([
            [
                'AccessPoint.fromAccessPointId(...)',
                (inputStack) => efs.AccessPoint.fromAccessPointId(inputStack, 'AccessPoint', 'accessPointId'),
            ],
            [
                'AccessPoint.fromAccessPointAttributes(...)',
                (inputStack) => {
                    return efs.AccessPoint.fromAccessPointAttributes(inputStack, 'AccessPoint', {
                        accessPointArn: core_1.Arn.format({
                            resource: 'AccessPoint',
                            service: 'efs',
                            resourceName: 'accessPointName',
                        }, inputStack),
                        fileSystem: efsFS,
                    });
                },
            ],
        ])('%s throws error', (_label, accessPointProvider) => {
            // GIVEN
            const accessPoint = accessPointProvider(stack);
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint,
            });
            // WHEN
            function when() {
                mount.usesUserPosixPermissions();
            }
            // THEN
            expect(when).toThrow(/^MountableEfs.usesUserPosixPermissions\(\) only supports efs.AccessPoint instances, got ".*"$/);
        });
        test('no access point returns true', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(true);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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