"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
describe('Endpoint', () => {
    test('accepts tokens for the port value', () => {
        // GIVEN
        const token = core_1.Lazy.number({ produce: () => 123 });
        // WHEN
        const endpoint = new lib_1.Endpoint({
            address: '127.0.0.1',
            port: token,
        });
        // THEN
        expect(endpoint.port.toRuleJson()).toEqual(expect.objectContaining({
            fromPort: token,
            toPort: token,
            ipProtocol: 'tcp',
        }));
        expect(endpoint.socketAddress).toEqual(expect.stringMatching(new RegExp(escapeTokenRegex('127.0.0.1:${Token[TOKEN.\\d+]}'))));
    });
    test('accepts valid port string numbers', () => {
        // GIVEN
        for (const port of [1, 50, 65535]) {
            // WHEN
            const endpoint = new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
            // THEN
            expect(endpoint.port.toRuleJson()).toEqual(expect.objectContaining({
                fromPort: port,
                toPort: port,
                ipProtocol: 'tcp',
            }));
            expect(endpoint.socketAddress).toBe(`127.0.0.1:${port}`);
        }
    });
    test('throws an exception for port numbers below the minimum', () => {
        // GIVEN
        const port = 0;
        // WHEN
        function createInvalidEnpoint() {
            new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
        }
        // THEN
        expect(createInvalidEnpoint).toThrow();
    });
    test('throws an exception for port numbers above the maximum', () => {
        // GIVEN
        const port = 65536;
        // WHEN
        function createInvalidEnpoint() {
            new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
        }
        // THEN
        expect(createInvalidEnpoint).toThrow();
    });
    test('throws an exception for floating-point port numbers', () => {
        // GIVEN
        const port = 1.5;
        // WHEN
        function createInvalidEnpoint() {
            new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
        }
        // THEN
        expect(createInvalidEnpoint).toThrow();
    });
    describe('.portAsString()', () => {
        test('converts port tokens to string tokens', () => {
            // GIVEN
            const port = core_1.Lazy.number({ produce: () => 123 });
            const endpoint = new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
            // WHEN
            const result = endpoint.portAsString();
            // THEN
            // Should return a string token
            expect(core_1.Token.isUnresolved(result)).toBeTruthy();
            // It should not just be the string representation of the numeric token
            expect(result).not.toBe(port.toString());
        });
        test('converts resolved port numbers to string representation', () => {
            // GIVEN
            const port = 1500;
            const endpoint = new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
            // WHEN
            const result = endpoint.portAsString();
            // THEN
            expect(result).toBe(port.toString());
        });
    });
});
describe('ApplicationEndpoint', () => {
    test('uses TCP transport', () => {
        // WHEN
        const endpoint = new lib_1.ApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
        });
        // THEN
        expect(endpoint.protocol).toBe(aws_ec2_1.Protocol.TCP);
    });
    test('defaults to https', () => {
        // WHEN
        const endpoint = new lib_1.ApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
        });
        // THEN
        expect(endpoint.applicationProtocol).toBe(aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS);
    });
    test.each([
        [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP],
        [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS],
    ])('sets protocol to %p', (protocol) => {
        // WHEN
        const endpoint = new lib_1.ApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
            protocol,
        });
        // THEN
        expect(endpoint.applicationProtocol).toBe(protocol);
    });
});
describe('ConnectableApplicationEndpoint', () => {
    test('Is connectable', () => {
        // WHEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const sg1 = new aws_ec2_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new aws_ec2_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        const endpoint = new lib_1.ConnectableApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
            connections: new aws_ec2_1.Connections({ securityGroups: [securityGroup] }),
        });
        // WHEN
        somethingConnectable.connections.allowTo(endpoint, aws_ec2_1.Port.tcp(80), 'Connecting to endpoint');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress'));
    });
});
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
//# sourceMappingURL=data:application/json;base64,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