"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            encryption: aws_dynamodb_1.TableEncryption.AWS_MANAGED,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${core_1.Names.uniqueId(this)}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${core_1.Names.uniqueId(this)} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: core_1.Duration.seconds(90),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        props.encryptionKey?.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(core_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new core_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into
 * Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated.
 * When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is
 * comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key
 * that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePem.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * - Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *   to only the principal(s) that require access.
 */
class X509CertificatePem extends X509CertificateBase {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        if ((props.validFor ?? 1) < 1 && !core_1.Token.isUnresolved(props.validFor)) {
            core_1.Annotations.of(this).addError('Certificates must be valid for at least one day.');
        }
        props.signingCertificate?.cert.grantRead(this.lambdaFunc);
        props.signingCertificate?.key.grantRead(this.lambdaFunc);
        props.signingCertificate?.passphrase.grantRead(this.lambdaFunc);
        props.signingCertificate?.certChain?.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: props.subject.o ?? 'AWS',
                OU: props.subject.ou ?? 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: props.encryptionKey?.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
            CertificateValidFor: props.validFor?.toString(),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretCompleteArn: core_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretCompleteArn: core_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate
     */
    grantCertRead(grantee) {
        const result = this.cert.grantRead(grantee);
        this.certChain?.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase
     */
    grantFullRead(grantee) {
        const result = this.cert.grantRead(grantee);
        this.certChain?.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
_a = JSII_RTTI_SYMBOL_1;
X509CertificatePem[_a] = { fqn: "aws-rfdk.X509CertificatePem", version: "0.42.0" };
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be
 * provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and
 * the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly
 * generated and stored in a Secret.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePkcs12.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the CustomResource.
 * - Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * - Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *   to only the principal(s) that require access.
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: props.encryptionKey?.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
_b = JSII_RTTI_SYMBOL_1;
X509CertificatePkcs12[_b] = { fqn: "aws-rfdk.X509CertificatePkcs12", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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