"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticPrivateIpServer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that
 * has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Elastic Network Interface (ENI).
 * - Security Group for the ASG.
 * - Instance Role and corresponding IAM Policy.
 * - SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack.
 * - Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda that is deployed through this construct has broad IAM permissions to attach any Elastic
 *   Network Interface (ENI) to any instance. You should not grant any additional actors/principals the ability
 *   to modify or execute this Lambda.
 * - The SNS Topic that is deployed through this construct controls the execution of the Lambda discussed above.
 *   Principals that can publish messages to this SNS Topic will be able to trigger the Lambda to run. You should
 *   not allow any additional principals to publish messages to this SNS Topic.
 */
class StaticPrivateIpServer extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        if (props.resourceSignalTimeout && props.resourceSignalTimeout.toSeconds() > (12 * 60 * 60)) {
            throw new Error('Resource signal timeout cannot exceed 12 hours.');
        }
        this.autoscalingGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Asg', {
            minCapacity: 1,
            maxCapacity: 1,
            vpc: props.vpc,
            instanceType: props.instanceType,
            machineImage: props.machineImage,
            vpcSubnets: { subnets: [subnet] },
            blockDevices: props.blockDevices,
            keyName: props.keyName,
            resourceSignalCount: props.resourceSignalTimeout ? 1 : undefined,
            resourceSignalTimeout: props.resourceSignalTimeout,
            role: props.role,
            securityGroup: props.securityGroup,
            userData: props.userData,
        });
        this.connections = this.autoscalingGroup.connections;
        this.grantPrincipal = this.autoscalingGroup.grantPrincipal;
        this.osType = this.autoscalingGroup.osType;
        this.role = this.autoscalingGroup.role;
        this.userData = this.autoscalingGroup.userData;
        const scopePath = this.node.scopes.map(construct => construct.node.id).slice(1); // Slice to remove the unnamed <root> scope.
        const eni = new aws_ec2_1.CfnNetworkInterface(this, 'Eni', {
            subnetId: subnet.subnetId,
            description: `Static ENI for ${scopePath.join('/')}`,
            groupSet: core_1.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            privateIpAddress: props.privateIpAddress,
        });
        this.privateIpAddress = eni.attrPrimaryPrivateIpAddress;
        // We need to be sure that the ENI is created before the instance would be brought up; otherwise, we cannot attach it.
        this.autoscalingGroup.node.defaultChild.addDependsOn(eni);
        this.attachEniLifecyleTarget(eni);
        this.node.defaultChild = this.autoscalingGroup.node.defaultChild;
    }
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance
     * in this construct's AutoScalingGroup when a new instance is launched.
     */
    attachEniLifecyleTarget(eni) {
        // Note: The design of AutoScalingGroup life cycle notifications in CDK v1.49.1 is such that
        // using the provided AutoScalingGroup.addLifecycleHook() will result in a setup that misses
        // launch notifications for instances created when the ASG is created. This is because
        // it uses the separate CfnLifecycleHook resource to do it, and that resource references the
        // ASG ARN; i.e. it must be created after the ASG has an ARN... thus it can miss instance launches
        // when the ASG is first created.
        //
        // We work around this by using an escape-hatch to the L1 ASG to create our own notification from scratch.
        const eventHandler = this.setupLifecycleEventHandlerFunction();
        const { topic, role } = this.setupLifecycleNotificationTopic(eventHandler);
        // Ensure no race conditions that might prevent the lambda from being able to perform its required functions by making
        // the ASG depend on the creation of the SNS Subscription.
        // Note: The topic subscriptions are children of the lambda, and are given an id equal to the Topic's id.
        this.autoscalingGroup.node.defaultChild.node.addDependency(eventHandler.node.findChild(topic.node.id));
        this.autoscalingGroup.node.defaultChild.lifecycleHookSpecificationList = [
            {
                defaultResult: aws_autoscaling_1.DefaultResult.ABANDON,
                heartbeatTimeout: 120,
                lifecycleHookName: 'NewStaticPrivateIpServer',
                lifecycleTransition: aws_autoscaling_1.LifecycleTransition.INSTANCE_LAUNCHING,
                notificationTargetArn: topic.topicArn,
                roleArn: role.roleArn,
                notificationMetadata: JSON.stringify({ eniId: eni.ref }),
            },
        ];
    }
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     */
    setupLifecycleEventHandlerFunction() {
        const stack = core_1.Stack.of(this);
        // The SingletonFunction does not tell us when it's newly created vs. finding a pre-existing
        // one. So, we do our own singleton Function so that we know when it's the first creation, and, thus,
        // we must attach one-time permissions.
        const functionUniqueId = 'AttachEniToInstance' + this.removeHyphens('83a5dca5-db54-4aa4-85d2-8d419cdf85ce');
        let singletonPreExists = true;
        let eventHandler = stack.node.tryFindChild(functionUniqueId);
        if (!eventHandler) {
            const handlerCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'asg-attach-eni'), {
                exclude: ['**/*', '!index*'],
            });
            eventHandler = new aws_lambda_1.Function(stack, functionUniqueId, {
                code: handlerCode,
                handler: 'index.handler',
                runtime: aws_lambda_1.Runtime.NODEJS_16_X,
                description: `Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack '${stack.stackName}'. This lambda attaches an ENI to newly launched instances.`,
                logRetention: aws_logs_1.RetentionDays.THREE_DAYS,
            });
            singletonPreExists = false;
        }
        // Note: We **cannot** reference the ASG's ARN in the lambda's policy. It would create a deadlock at deployment:
        //  Lambda policy waiting on ASG completion to get ARN
        //  -> lambda waiting on policy to be created
        //  -> ASG waiting on lambda to signal lifecycle continue for instance start
        //  -> back to the start of the cycle.
        // Instead we use resourcetags condition to limit the scope of the lambda.
        const tagKey = 'RfdkStaticPrivateIpServerGrantConditionKey';
        const tagValue = core_1.Names.uniqueId(eventHandler);
        const grantCondition = {};
        grantCondition[`autoscaling:ResourceTag/${tagKey}`] = tagValue;
        core_1.Tags.of(this.autoscalingGroup).add(tagKey, tagValue);
        // Allow the lambda to complete the lifecycle action for only tagged ASGs.
        const iamCompleteLifecycle = new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'autoscaling:CompleteLifecycleAction',
            ],
            resources: [
                `arn:${stack.partition}:autoscaling:${stack.region}:${stack.account}:autoScalingGroup:*:autoScalingGroupName/*`,
            ],
            conditions: {
                'ForAnyValue:StringEquals': grantCondition,
            },
        });
        eventHandler.role.addToPrincipalPolicy(iamCompleteLifecycle);
        if (!singletonPreExists) {
            // Allow the lambda to attach the ENI to the instance that was created.
            // Referencing: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            // Last-Accessed: July 2020
            // The ec2:DescribeNetworkInterfaces, and ec2:AttachNetworkInterface operations
            // do not support conditions, and do not support resource restriction.
            // So, we only attach the policy to the lambda function once; when we first create it.
            const iamEniAttach = new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AttachNetworkInterface',
                ],
                resources: ['*'],
            });
            eventHandler.role.addToPrincipalPolicy(iamEniAttach);
        }
        return eventHandler;
    }
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events. Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns { topic: Topic, role: Role }
     */
    setupLifecycleNotificationTopic(lambdaHandler) {
        const stack = core_1.Stack.of(this);
        // We only need to have a single SNS topic & subscription set up to handle lifecycle events for *all* instances of this class.
        // We have to be careful, however, to ensure that our initial setup only happens once when we first add the topic and such
        // to this stack; otherwise, we will not be able to deploy more than one of these constructs in a stack.
        const notificationRoleUniqueId = 'AttachEniNotificationRole' + this.removeHyphens('a0376ff8-248e-4534-bf42-58c6ffa4d5b4');
        const notificationTopicUniqueId = 'AttachEniNotificationTopic' + this.removeHyphens('c8b1e9a6-783c-4954-b191-204dd5e3b9e0');
        let notificationTopic = stack.node.tryFindChild(notificationTopicUniqueId);
        let notificationRole;
        if (!notificationTopic) {
            // First time creating the singleton Topic in this stack. Set it all up...
            notificationRole = new aws_iam_1.Role(stack, notificationRoleUniqueId, {
                assumedBy: new aws_iam_1.ServicePrincipal('autoscaling.amazonaws.com'),
            });
            notificationTopic = new aws_sns_1.Topic(stack, notificationTopicUniqueId, {
                displayName: `For RFDK instance-launch notifications for stack '${stack.stackName}'`,
            });
            notificationTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(lambdaHandler));
            notificationTopic.grantPublish(notificationRole);
        }
        else {
            notificationRole = stack.node.findChild(notificationRoleUniqueId);
        }
        return {
            topic: notificationTopic,
            role: notificationRole,
        };
    }
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    removeHyphens(x) {
        return x.replace(/[-]/g, '');
    }
}
exports.StaticPrivateIpServer = StaticPrivateIpServer;
_a = JSII_RTTI_SYMBOL_1;
StaticPrivateIpServer[_a] = { fqn: "aws-rfdk.StaticPrivateIpServer", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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