"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PadEfsStorage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a mechanism that adds 1GB-sized files containing only zero-bytes
 * to an Amazon EFS filesystem through a given Access Point to that filesystem. This is being
 * provided to give you a way to increase the baseline throughput of an Amazon EFS filesystem
 * that has been deployed in bursting throughput mode (see: https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes).
 * This is most useful for your Amazon EFS filesystems that contain a very small amount of data and
 * have a baseline throughput that exceeds the throughput provided by the size of the filesystem.
 *
 * When deployed in bursting throughput mode, an Amazon EFS filesystem provides you with a baseline
 * throughput that is proportional to the amount of data stored in that filesystem. However, usage
 * of that filesystem is allowed to burst above that throughput; doing so consumes burst credits that
 * are associated with the filesystem. When all burst credits have been expended, then your filesystem
 * is no longer allowed to burst throughput and you will be limited in throughput to the greater of 1MiB/s
 * or the throughput dictated by the amount of data stored in your filesystem; the filesystem will be able
 * to burst again if it is able to accrue burst credits by staying below its baseline throughput for a time.
 *
 * Customers that deploy the Deadline Repository Filesystem on an Amazon EFS filesystem may find that
 * the filesystem does not contain sufficient data to meet the throughput needs of Deadline; evidenced by
 * a downward trend in EFS bursting credits over time. When bursting credits are expended, then the render
 * farm may begin to exhibit failure mode behaviors such as the RenderQueue dropping or refusing connections,
 * or becoming unresponsive.
 *
 * If you find that your Amazon EFS is depleting its burst credits and would like to increase the
 * amount of padding that has been added to it then you can either:
 * - Modify the value of the desired padding property of this construct and redeploy your infrastructure
 *   to force an update; or
 * - Manually invoke the AWS Step Function that has been created by this construct by finding it
 *   in your AWS Console (its name will be prefixed with "<id of this construct>StateMachine"), and
 *   then start an execution of the state machine with the following JSON document as input:
 *   { "desiredPadding": <number of GiB you want to store> }
 *
 * Warning: The implementation of this construct creates and starts an AWS Step Function to add the files
 * to the filesystem. The execution of this Step Function occurs asynchronously from your deployment. We recommend
 * verifying that the step function completed successfully via your Step Functions console.
 *
 * Resources Deployed
 * --------------------------
 * - Two AWS Lambda Functions, with roles, with full access to the given EFS Access Point.
 * - An Elastic Network Interface (ENI) for each Lambda Function in each of the selected VPC Subnets, so
 *   that the Lambda Functions can connect to the given EFS Access Point.
 * - An AWS Step Function to coordinate execution of the two Lambda Functions.
 * - Security Groups for each AWS Lambda Function.
 * - A CloudFormation custom resource that executes StepFunctions.startExecution on the Step Function
 *   whenever the stack containing this construct is created or updated.
 *
 * Security Considerations
 * ---------------------------
 * - The AWS Lambdas that are deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these Lambdas.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - By default, the network interfaces created by this construct's AWS Lambda Functions have Security Groups
 *   that restrict egress access from the Lambda Function into your VPC such that the Lambda Functions can
 *   access only the given EFS Access Point.
 */
class PadEfsStorage extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /*
        Implementation:
         This is implemented as an AWS Step Function that implements the following
         algorithm:
         try {
          du = diskUsage(<efs access point directory>)
          while (du != desiredPadding) {
            if (du < desiredPadding) {
              <grow padding by adding up to 20 1GB numbered files to the filesystem.>
            } else if (du > desiredPadding) {
              <delete 1GB numbered files from the filesystem to reduce the padding to the desired amount>
              // Note: We break here to prevent two separate invocations of the step function (e.g. accidental manual
              // invocations) from looping indefinitely. Without a break, one invocation trying to grow while another
              // tries to shrink will infinitely loop both -- the diskUsage will never settle on the value that either
              // invocation wants.
              break;
            }
            du = diskUsage(<efs access point directory>)
          }
          return success
        } catch (error) {
          return failure
        }
         */
        const diskUsageTimeout = core_1.Duration.minutes(5);
        const paddingTimeout = core_1.Duration.minutes(15);
        // Location in the lambda environment where the EFS will be mounted.
        const efsMountPoint = '/mnt/efs';
        let desiredSize;
        try {
            desiredSize = props.desiredPadding.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
        }
        catch (err) {
            core_1.Annotations.of(this).addError('Failed to round desiredSize to an integer number of GiB. The size must be in GiB.');
        }
        const securityGroup = props.securityGroup ?? new aws_ec2_1.SecurityGroup(this, 'LambdaSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: false,
        });
        const lambdaProps = {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            // Required for access point...
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
            securityGroups: [securityGroup],
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(props.accessPoint, efsMountPoint),
        };
        const diskUsage = new aws_lambda_1.Function(this, 'DiskUsage', {
            description: 'Used by RFDK PadEfsStorage to calculate disk usage of an EFS access point',
            handler: 'pad-efs-storage.getDiskUsage',
            timeout: diskUsageTimeout,
            memorySize: 128,
            ...lambdaProps,
        });
        // Implicit reference should have been fine, but the lambda is unable to mount the filesystem if
        // executed before the filesystem has been fully formed. We shouldn't have the lambda created until
        // after the EFS is created.
        diskUsage.node.addDependency(props.accessPoint);
        const doPadding = new aws_lambda_1.Function(this, 'PadFilesystem', {
            description: 'Used by RFDK PadEfsStorage to add or remove numbered 1GB files in an EFS access point',
            handler: 'pad-efs-storage.padFilesystem',
            timeout: paddingTimeout,
            // Execution requires about 70MB for just the lambda, but the filesystem driver will use every available byte.
            // Larger sizes do not seem to make a difference on filesystem write performance.
            // Set to 256MB just to give a buffer.
            memorySize: 256,
            ...lambdaProps,
        });
        // Implicit reference should have been fine, but the lambda is unable to mount the filesystem if
        // executed before the filesystem has been fully formed. We shouldn't have the lambda created until
        // after the EFS is created.
        doPadding.node.addDependency(props.accessPoint);
        // Build the step function's state machine.
        const fail = new aws_stepfunctions_1.Fail(this, 'Fail');
        const succeed = new aws_stepfunctions_1.Succeed(this, 'Succeed');
        const diskUsageTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'QueryDiskUsage', {
            lambdaFunction: diskUsage,
            comment: 'Determine the number of GB currently stored in the EFS access point',
            timeout: diskUsageTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.diskUsage',
        });
        const growTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'GrowTask', {
            lambdaFunction: doPadding,
            comment: 'Add up to 20 numbered 1GB files to the EFS access point',
            timeout: paddingTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.null',
        });
        const shrinkTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'ShrinkTask', {
            lambdaFunction: doPadding,
            comment: 'Remove 1GB numbered files from the EFS access point to shrink the padding',
            timeout: paddingTimeout,
            payload: {
                type: aws_stepfunctions_1.InputType.OBJECT,
                value: {
                    'desiredPadding.$': '$.desiredPadding',
                    'mountPoint': efsMountPoint,
                },
            },
            resultPath: '$.null',
        });
        const choice = new aws_stepfunctions_1.Choice(this, 'BranchOnDiskUsage')
            .when(aws_stepfunctions_1.Condition.numberLessThanJsonPath('$.diskUsage.Payload', '$.desiredPadding'), growTask)
            .when(aws_stepfunctions_1.Condition.numberGreaterThanJsonPath('$.diskUsage.Payload', '$.desiredPadding'), shrinkTask)
            .otherwise(succeed);
        diskUsageTask.next(choice);
        diskUsageTask.addCatch(fail, {
            // See: https://docs.aws.amazon.com/step-functions/latest/dg/concepts-error-handling.html
            errors: ['States.ALL'],
        });
        growTask.next(diskUsageTask);
        growTask.addCatch(fail, {
            errors: ['States.ALL'],
        });
        shrinkTask.next(succeed);
        shrinkTask.addCatch(fail, {
            errors: ['States.ALL'],
        });
        const statemachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: diskUsageTask,
        });
        // ==========
        // Invoke the step function on stack create & update.
        const invokeCall = {
            action: 'startExecution',
            service: 'StepFunctions',
            apiVersion: '2016-11-23',
            region: core_1.Stack.of(this).region,
            physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('executionArn'),
            parameters: {
                stateMachineArn: statemachine.stateMachineArn,
                input: JSON.stringify({
                    desiredPadding: desiredSize,
                }),
            },
        };
        const resource = new custom_resources_1.AwsCustomResource(this, 'Default', {
            installLatestAwsSdk: true,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            onCreate: invokeCall,
            onUpdate: invokeCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [statemachine.stateMachineArn],
            }),
        });
        resource.node.addDependency(statemachine);
        // Add RFDK tags to the construct tree.
        runtime_info_1.tagConstruct(this);
    }
}
exports.PadEfsStorage = PadEfsStorage;
_a = JSII_RTTI_SYMBOL_1;
PadEfsStorage[_a] = { fqn: "aws-rfdk.PadEfsStorage", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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