"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableFsxLustre = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon FSx for Lustre File System onto
 * an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
class MountableFsxLustre {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const mountName = this.props.fileset ? path.posix.join(this.props.filesystem.mountName, this.props.fileset) : this.props.filesystem.mountName;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, 'bash ./installLustreClient.sh', `bash ./mountFsxLustre.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${mountName} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * @inheritdoc
     */
    usesUserPosixPermissions() {
        return true;
    }
    /**
     * Fetch the Asset singleton for the FSx for Lustre mounting scripts, or generate it if needed.
     */
    mountAssetSingleton(scope) {
        const stack = core_1.Stack.of(scope);
        const uuid = '0db888da-5901-4948-aaa5-e71c541c8060';
        const uniqueId = 'MountableFsxLustreAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountFsxLustre.sh', '!installLustreClient.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableFsxLustre = MountableFsxLustre;
_a = JSII_RTTI_SYMBOL_1;
MountableFsxLustre[_a] = { fqn: "aws-rfdk.MountableFsxLustre", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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