/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { LustreFileSystem } from '@aws-cdk/aws-fsx';
import { Asset } from '@aws-cdk/aws-s3-assets';
import { Construct, IConstruct } from '@aws-cdk/core';
import { IMountableLinuxFilesystem, IMountingInstance, LinuxMountPointProps } from './mountable-filesystem';
/**
 * Properties that are required to create a {@link MountableFsxLustre}.
 */
export interface MountableFsxLustreProps {
    /**
     * The {@link https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-fsx.LustreFileSystem.html|FSx for Lustre}
     * filesystem that will be mounted by the object.
     */
    readonly filesystem: LustreFileSystem;
    /**
     * The fileset to mount.
     * @see https://docs.aws.amazon.com/fsx/latest/LustreGuide/mounting-from-fileset.html
     * @default Mounts the root of the filesystem.
     */
    readonly fileset?: string;
    /**
     * Extra Lustre mount options that will be added to /etc/fstab for the file system.
     * See: {@link http://manpages.ubuntu.com/manpages/precise/man8/mount.lustre.8.html}
     *
     * The given values will be joined together into a single string by commas.
     * ex: ['soft', 'rsize=4096'] will become 'soft,rsize=4096'
     *
     * @default No extra options.
     */
    readonly extraMountOptions?: string[];
}
/**
 * This class encapsulates scripting that can be used to mount an Amazon FSx for Lustre File System onto
 * an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
export declare class MountableFsxLustre implements IMountableLinuxFilesystem {
    protected readonly scope: Construct;
    protected readonly props: MountableFsxLustreProps;
    constructor(scope: Construct, props: MountableFsxLustreProps);
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target: IMountingInstance, mount: LinuxMountPointProps): void;
    /**
     * @inheritdoc
     */
    usesUserPosixPermissions(): boolean;
    /**
     * Fetch the Asset singleton for the FSx for Lustre mounting scripts, or generate it if needed.
     */
    protected mountAssetSingleton(scope: IConstruct): Asset;
}
