"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableEfs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto
 * an instance.
 *
 * An optional EFS access point can be specified for mounting the EFS file-system. For more information on using EFS
 * Access Points, see https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html. For this to work properly, the
 * EFS mount helper is required. The EFS Mount helper comes pre-installed on Amazon Linux 2. For other Linux
 * distributions, the host machine must have the Amazon EFS client installed. We advise installing the Amazon EFS Client
 * when building your AMI. For instructions on installing the Amazon EFS client for other distributions, see
 * https://docs.aws.amazon.com/efs/latest/ug/installing-amazon-efs-utils.html#installing-other-distro.
 *
 * NOTE: Without an EFS access point, the file-system is writeable only by the root user.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
class MountableEfs {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        this.fileSystem = props.filesystem;
        this.accessPoint = props.accessPoint;
    }
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        if (core_1.Construct.isConstruct(target)) {
            target.node.addDependency(this.props.filesystem.mountTargetsAvailable);
        }
        if (this.props.accessPoint) {
            const grantActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mount?.permissions);
            if (this.accessPointRequiresClientRootAccess(this.props.accessPoint)) {
                grantActions.push('elasticfilesystem:ClientRootAccess');
            }
            target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                resources: [
                    this.props.filesystem.node.defaultChild.attrArn,
                ],
                actions: grantActions,
                conditions: {
                    StringEquals: {
                        'elasticfilesystem:AccessPointArn': this.props.accessPoint.accessPointArn,
                    },
                },
            }));
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.accessPoint) {
            mountOptions.push('iam', `accesspoint=${this.props.accessPoint.accessPointId}`);
        }
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const resolveMountTargetDnsWithApi = this.props.resolveMountTargetDnsWithApi ?? false;
        if (resolveMountTargetDnsWithApi) {
            const describeMountTargetResources = [
                this.props.filesystem.node.defaultChild.attrArn,
            ];
            if (this.props.accessPoint) {
                describeMountTargetResources.push(this.props.accessPoint.accessPointArn);
            }
            target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                resources: describeMountTargetResources,
                actions: ['elasticfilesystem:DescribeMountTargets'],
            }));
        }
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, `bash ./mountEfs.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${resolveMountTargetDnsWithApi} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * @inheritdoc
     */
    usesUserPosixPermissions() {
        if (this.accessPoint) {
            // We cannot determine if the access point forces a POSIX user in the import case
            if (!(this.accessPoint instanceof efs.AccessPoint)) {
                throw new Error(`MountableEfs.usesUserPosixPermissions() only supports efs.AccessPoint instances, got "${this.accessPoint.constructor.name}"`);
            }
            const accessPointResource = this.accessPoint.node.defaultChild;
            return !accessPointResource.posixUser;
        }
        return true;
    }
    /**
     * Uses a CDK escape-hatch to fetch the UID/GID of the access point POSIX user.
     *
     * @param accessPoint The access point to obtain the POSIX user for
     */
    getAccessPointPosixUser(accessPoint) {
        const accessPointResource = accessPoint.node.defaultChild;
        return accessPointResource.posixUser;
    }
    /**
     * Uses a synthesis-time check to determine whether an access point is setting its UID/GID to 0 (root). Mounting such
     * an access point requires the `ClientRootAccess` IAM permission.
     *
     * If this introspection is possible and the access point is determined to require root access, the method returns
     * true.
     *
     * If there is no information at synthesis-time, the method returns false as a secure default.
     *
     * @param accessPoint The access point to introspect
     */
    accessPointRequiresClientRootAccess(accessPoint) {
        if (accessPoint instanceof efs.AccessPoint) {
            const posixUser = this.getAccessPointPosixUser(accessPoint);
            // The following code path is cannot be triggered using the L2 construct for EFS Access Points. It currently
            // accepts a PosixUser struct. We will skip coverage for the time-being.
            /* istanbul ignore next */
            if (core_1.isResolvableObject(posixUser)) {
                // We can't know at synthesis time whether this POSIX user is root. Use secure defaults.
                return false;
            }
            if (!posixUser) {
                // No POSIX user specified we will not grant ClientRootAccess permission to opt on the side of secure defaults.
                return false;
            }
            // We have synthesis-time values for the UID/GID being set in the access point. Return true if either is 0 (root).
            return Number(posixUser.uid) === 0 || Number(posixUser.gid) === 0;
        }
        else {
            // This code path is for imported or custom-implementations of efs.AccessPoint
            // We cannot introspect the access point, so we will impose secure defaults and not grant ClientRootAccess.
            return false;
        }
    }
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     */
    mountAssetSingleton(scope) {
        const stack = core_1.Stack.of(scope);
        const uuid = '2b31c419-5b0b-4bb8-99ad-5b2575b2c06b';
        const uniqueId = 'MountableEfsAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEfs.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableEfs = MountableEfs;
_a = JSII_RTTI_SYMBOL_1;
MountableEfs[_a] = { fqn: "aws-rfdk.MountableEfs", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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