/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import * as efs from '@aws-cdk/aws-efs';
import { Asset } from '@aws-cdk/aws-s3-assets';
import { Construct, IConstruct } from '@aws-cdk/core';
import { IMountableLinuxFilesystem, IMountingInstance, LinuxMountPointProps } from './mountable-filesystem';
/**
 * Properties that are required to create a {@link MountableEfs}.
 */
export interface MountableEfsProps {
    /**
     * The {@link https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-efs.FileSystem.html|EFS}
     * filesystem that will be mounted by the object.
     */
    readonly filesystem: efs.IFileSystem;
    /**
     * An optional access point to use for mounting the file-system
     *
     * NOTE: Access points are only supported when using the EFS mount helper. The EFS Mount helper comes pre-installed on
     * Amazon Linux 2. For other Linux distributions, you must have the Amazon EFS client installed on your AMI for this
     * to work properly. For instructions on installing the Amazon EFS client for other distributions, see:
     *
     * https://docs.aws.amazon.com/efs/latest/ug/installing-amazon-efs-utils.html#installing-other-distro
     *
     * @default no access point is used
     */
    readonly accessPoint?: efs.IAccessPoint;
    /**
     * Extra NFSv4 mount options that will be added to /etc/fstab for the file system.
     * See: {@link https://www.man7.org/linux/man-pages//man5/nfs.5.html}
     *
     * The given values will be joined together into a single string by commas.
     * ex: ['soft', 'rsize=4096'] will become 'soft,rsize=4096'
     *
     * @default No extra options.
     */
    readonly extraMountOptions?: string[];
    /**
     * If enabled, RFDK will add user-data to the instances mounting this EFS file-system that obtains the mount target
     * IP address using AWS APIs and writes them to the system's `/etc/hosts` file to not require DNS lookups.
     *
     * If mounting EFS from instances in a VPC configured to not use the Amazon-provided DNS Route 53 Resolver server,
     * then the EFS mount targets will not be resolvable using DNS (see
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-dns.html) and enabling this will work around that issue.
     *
     * @default false
     */
    readonly resolveMountTargetDnsWithApi?: boolean;
}
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto
 * an instance.
 *
 * An optional EFS access point can be specified for mounting the EFS file-system. For more information on using EFS
 * Access Points, see https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html. For this to work properly, the
 * EFS mount helper is required. The EFS Mount helper comes pre-installed on Amazon Linux 2. For other Linux
 * distributions, the host machine must have the Amazon EFS client installed. We advise installing the Amazon EFS Client
 * when building your AMI. For instructions on installing the Amazon EFS client for other distributions, see
 * https://docs.aws.amazon.com/efs/latest/ug/installing-amazon-efs-utils.html#installing-other-distro.
 *
 * NOTE: Without an EFS access point, the file-system is writeable only by the root user.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
export declare class MountableEfs implements IMountableLinuxFilesystem {
    protected readonly scope: Construct;
    protected readonly props: MountableEfsProps;
    /**
     * The underlying EFS filesystem that is mounted
     */
    readonly fileSystem: efs.IFileSystem;
    /**
     * The optional access point used to mount the EFS file-system
     */
    readonly accessPoint?: efs.IAccessPoint;
    constructor(scope: Construct, props: MountableEfsProps);
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target: IMountingInstance, mount: LinuxMountPointProps): void;
    /**
     * @inheritdoc
     */
    usesUserPosixPermissions(): boolean;
    /**
     * Uses a CDK escape-hatch to fetch the UID/GID of the access point POSIX user.
     *
     * @param accessPoint The access point to obtain the POSIX user for
     */
    private getAccessPointPosixUser;
    /**
     * Uses a synthesis-time check to determine whether an access point is setting its UID/GID to 0 (root). Mounting such
     * an access point requires the `ClientRootAccess` IAM permission.
     *
     * If this introspection is possible and the access point is determined to require root access, the method returns
     * true.
     *
     * If there is no information at synthesis-time, the method returns false as a secure default.
     *
     * @param accessPoint The access point to introspect
     */
    private accessPointRequiresClientRootAccess;
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     */
    protected mountAssetSingleton(scope: IConstruct): Asset;
}
