"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbPostInstallSetup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and
 * executing commands against it. To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda function that is created by this resource has access to both the MongoDB administrator credentials,
 *   and the MongoDB application port. An attacker that can find a way to modify and execute this lambda could use it to
 *   modify or read any data in the database. You should not grant any additional actors/principals the ability to modify
 *   or execute this Lambda.
 */
class MongoDbPostInstallSetup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        props.users.x509AuthUsers?.forEach(user => {
            try {
                JSON.parse(user.roles);
            }
            catch (e) {
                throw new Error(`MongoDbPostInstallSetup: Could not parse JSON role for x509 user: ${user.roles}`);
            }
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lamdbaFunc = new aws_lambda_1.Function(this, 'Lambda', {
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? { subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT },
            description: `Used by a MongoDbPostInstallSetup ${core_1.Names.uniqueId(this)} to perform post-installation setup on a MongoDB`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {
            // Exclude commented out, for now, as a work-around for a CDK bug with at least CDK v1.49.1.
            // If we exclude files, then the asset hash is not calculated correctly and can result in updates to these
            // files not being picked up by the live system.
            // exclude: [
            //   '**/*',
            //   '!mongodb', '!mongodb/*',
            //   '!lib',
            //   '!lib/custom-resource', '!lib/custom-resource/*',
            //   '!lib/aws-lambda', '!lib/aws-lambda/*',
            //   '!lib/secrets-manager', '!lib/secrets-manager/*',
            //   '**/test',
            // ],
            }),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'mongodb.configureMongo',
            layers: [openSslLayer],
            timeout: core_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        lamdbaFunc.connections.allowTo(props.mongoDb, aws_ec2_1.Port.tcp(props.mongoDb.port));
        props.mongoDb.certificateChain.grantRead(lamdbaFunc.grantPrincipal);
        props.mongoDb.adminUser.grantRead(lamdbaFunc.grantPrincipal);
        props.users.passwordAuthUsers?.forEach(secret => secret.grantRead(lamdbaFunc));
        props.users.x509AuthUsers?.forEach(user => user.certificate.grantRead(lamdbaFunc));
        const properties = {
            Connection: {
                Hostname: props.mongoDb.fullHostname,
                Port: props.mongoDb.port.toString(),
                CaCertificate: props.mongoDb.certificateChain.secretArn,
                Credentials: props.mongoDb.adminUser.secretArn,
            },
            PasswordAuthUsers: props.users.passwordAuthUsers?.map(secret => secret.secretArn),
            X509AuthUsers: props.users.x509AuthUsers?.map(user => ({ Certificate: user.certificate.secretArn, Roles: user.roles })),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lamdbaFunc.functionArn,
            resourceType: 'Custom::RFDK_MongoDbPostInstallSetup',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lamdbaFunc.role);
        /* istanbul ignore next */
        if (props.mongoDb.node.defaultChild) {
            // Add a dependency on the ASG within the StaticPrivateIpServer to ensure that
            // mongo is running before we try to login to it.
            resource.node.addDependency(props.mongoDb.node.defaultChild.node.defaultChild);
        }
        this.node.defaultChild = resource;
    }
}
exports.MongoDbPostInstallSetup = MongoDbPostInstallSetup;
_a = JSII_RTTI_SYMBOL_1;
MongoDbPostInstallSetup[_a] = { fqn: "aws-rfdk.MongoDbPostInstallSetup", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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