"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstance = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const _1 = require("./");
const runtime_info_1 = require("./runtime-info");
/**
 * This construct provides a {@link StaticPrivateIpServer} that is hosting MongoDB. The data for this MongoDB database
 * is stored in an Amazon Elastic Block Storage (EBS) Volume that is automatically attached to the instance when it is
 * launched, and is separate from the instance's root volume; it is recommended that you set up a backup schedule for
 * this volume.
 *
 * When this instance is first launched, or relaunched after an instance replacement, it will:
 * 1. Attach an EBS volume to /var/lib/mongo upon which the MongoDB data is stored;
 * 2. Automatically install the specified version of MongoDB, from the official Mongo Inc. sources;
 * 3. Create an admin user in that database if one has not yet been created -- the credentials for this user
 * can be provided by you, or randomly generated;
 * 4. Configure MongoDB to require authentication, and only allow encrypted connections over TLS.
 *
 * The instance's launch logs and MongoDB logs will be automatically stored in Amazon CloudWatch logs; the
 * default log group name is: /renderfarm/<this construct ID>
 *
 * Resources Deployed
 * ------------------------
 * - {@link StaticPrivateIpServer} that hosts MongoDB.
 * - An A-Record in the provided PrivateHostedZone to create a DNS entry for this server's static private IP.
 * - A Secret in AWS SecretsManager that contains the administrator credentials for MongoDB.
 * - An encrypted Amazon Elastic Block Store (EBS) Volume on which the MongoDB data is stored.
 * - Amazon CloudWatch log group that contains instance-launch and MongoDB application logs.
 *
 * Security Considerations
 * ------------------------
 * - The administrator credentials for MongoDB are stored in a Secret within AWS SecretsManager. You must strictly limit
 *   access to this secret to only entities that require it.
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The EBS Volume that is created by, or provided to, this construct is used to store the contents of your MongoDB data. To
 *   protect the sensitive data in your database, you should not grant access to this EBS Volume to any principal or instance
 *   other than the instance created by this construct. Furthermore, we recommend that you ensure that the volume that is
 *   used for this purpose is encrypted at rest.
 * - This construct uses this package's {@link StaticPrivateIpServer}, {@link MongoDbInstaller}, {@link CloudWatchAgent},
 *   {@link ExportingLogGroup}, and {@link MountableBlockVolume}. Security considerations that are outlined by the documentation
 *   for those constructs should also be taken into account.
 */
class MongoDbInstance extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.version = props.mongoDb.version;
        // Select the subnet for this instance.
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        this.server = new _1.StaticPrivateIpServer(this, 'Server', {
            vpc: props.vpc,
            vpcSubnets: { subnets: [subnet] },
            instanceType: props.instanceType ?? new aws_ec2_1.InstanceType('r5.large'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            blockDevices: [
                {
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(MongoDbInstance.ROOT_DEVICE_SIZE.toGibibytes(), { encrypted: true }),
                },
            ],
            keyName: props.keyName,
            resourceSignalTimeout: core_1.Duration.minutes(5),
            role: props.role,
            securityGroup: props.securityGroup,
        });
        new aws_route53_1.ARecord(this, 'ARecord', {
            target: aws_route53_1.RecordTarget.fromIpAddresses(this.server.privateIpAddress),
            zone: props.mongoDb.dnsZone,
            recordName: props.mongoDb.hostname,
        });
        this.adminUser = props.mongoDb.adminUser ?? new aws_secretsmanager_1.Secret(this, 'AdminUser', {
            description: `Admin credentials for the MongoDB database ${core_1.Names.uniqueId(this)}`,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        this.mongoDataVolume = props.mongoDb.mongoDataVolume?.volume ?? new aws_ec2_1.Volume(this, 'MongoDbData', {
            size: MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE,
            ...props.mongoDb.mongoDataVolume?.volumeProps,
            availabilityZone: subnet.availabilityZone,
            encrypted: true,
        });
        const volumeMount = new _1.MountableBlockVolume(this, {
            blockVolume: this.mongoDataVolume,
            volumeFormat: _1.BlockVolumeFormat.XFS,
        });
        const mongoInstaller = new _1.MongoDbInstaller(this, {
            version: props.mongoDb.version,
            userSsplAcceptance: props.mongoDb.userSsplAcceptance,
        });
        // Set up the server's UserData.
        this.server.userData.addCommands('set -xefuo pipefail');
        this.server.userData.addSignalOnExitCommand(this.server.autoscalingGroup);
        this.configureCloudWatchLogStreams(this.server, id, props.logGroupProps); // MUST BE FIRST
        volumeMount.mountToLinuxInstance(this.server, {
            location: MongoDbInstance.MONGO_DEVICE_MOUNT_POINT,
        });
        mongoInstaller.installOnLinuxInstance(this.server);
        this.configureMongoDb(this.server, props.mongoDb);
        this.certificateChain = props.mongoDb.serverCertificate.certChain;
        this.connections = this.server.connections;
        this.grantPrincipal = this.server.grantPrincipal;
        this.port = 27017;
        this.role = this.server.role;
        this.userData = this.server.userData;
        this.fullHostname = `${props.mongoDb.hostname}.${props.mongoDb.dnsZone.zoneName}`;
        this.node.defaultChild = this.server;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        securityGroups?.forEach(securityGroup => this.server.autoscalingGroup.addSecurityGroup(securityGroup));
    }
    /**
     * Adds UserData commands to install & configure the CloudWatch Agent onto the instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *     - The cloud-init log
     *     - The MongoDB application log.
     *
     * @param host The instance/host to setup the CloudWatchAgent upon.
     * @param groupName Name to append to the log group prefix when forming the log group name.
     * @param logGroupProps Properties for the log group
     */
    configureCloudWatchLogStreams(host, groupName, logGroupProps) {
        const prefix = logGroupProps?.logGroupPrefix ?? MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'MongoDbInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(host.grantPrincipal);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'MongoDB', '/var/log/mongodb/mongod.log');
        new _1.CloudWatchAgent(this, 'MongoDbInstanceLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host,
        });
    }
    /**
     * Adds commands to the userData of the instance to install MongoDB, create an admin user if one does not exist, and
     * to to start mongod running.
     */
    configureMongoDb(instance, settings) {
        const scriptsAsset = new aws_s3_assets_1.Asset(this, 'MongoSetup', {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', settings.version),
        });
        scriptsAsset.grantRead(instance.grantPrincipal);
        const scriptZipfile = instance.userData.addS3DownloadCommand({
            bucket: scriptsAsset.bucket,
            bucketKey: scriptsAsset.s3ObjectKey,
        });
        instance.userData.addCommands(
        // Ensure mongod is installed and stopped before we go any further
        'which mongod && test -f /etc/mongod.conf', 'sudo service mongod stop', 
        // We're going to make a temporary RAM filesystem for the mongo setup files.
        // This will let us write sensitive data to "disk" without worrying about it
        // being persisted in any physical disk, even temporarily.
        'MONGO_SETUP_DIR=$(mktemp -d)', 'mkdir -p "${MONGO_SETUP_DIR}"', 'sudo mount -t tmpfs -o size=50M tmpfs "${MONGO_SETUP_DIR}"', 'pushd "${MONGO_SETUP_DIR}"', `unzip ${scriptZipfile}`, 
        // Backup mongod.conf for now
        'cp /etc/mongod.conf .');
        const cert = settings.serverCertificate;
        instance.userData.addCommands(`bash serverCertFromSecrets.sh "${cert.cert.secretArn}" "${cert.certChain.secretArn}" "${cert.key.secretArn}" "${cert.passphrase.secretArn}"`);
        cert.cert.grantRead(instance.grantPrincipal);
        cert.certChain.grantRead(instance.grantPrincipal);
        cert.key.grantRead(instance.grantPrincipal);
        cert.passphrase.grantRead(instance.grantPrincipal);
        const certsDirectory = '/etc/mongod_certs';
        instance.userData.addCommands(
        // Move the certificates into place
        `sudo mkdir -p ${certsDirectory}`, `sudo mv ./ca.crt ./key.pem ${certsDirectory}`, 'sudo chown root.mongod -R /etc/mongod_certs/', // Something weird about shell interpretation. Can't use '*' on this or next line.
        'sudo chmod 640 -R /etc/mongod_certs/', 'sudo chmod 750 /etc/mongod_certs/', // Directory needs to be executable.
        // mongod user id might, potentially change on reboot. Make sure we own all mongo data
        `sudo chown mongod.mongod -R ${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}`, 
        // Configure mongod
        'bash ./setMongoLimits.sh', `bash ./setStoragePath.sh "${MongoDbInstance.MONGO_DEVICE_MOUNT_POINT}"`, 'bash ./setMongoNoAuth.sh', 'sudo service mongod start', `bash ./setAdminCredentials.sh "${this.adminUser.secretArn}"`);
        this.adminUser.grantRead(instance.grantPrincipal);
        instance.userData.addCommands(
        // Setup for live deployment, and start mongod
        'sudo service mongod stop', 'bash ./setLiveConfiguration.sh', 'sudo systemctl enable mongod', // Enable restart on reboot
        'sudo service mongod start', 'popd');
        instance.userData.addOnExitCommands(
        // Clean up the temporary RAM filesystem
        'test "${MONGO_SETUP_DIR} != "" && sudo umount "${MONGO_SETUP_DIR}');
    }
}
exports.MongoDbInstance = MongoDbInstance;
_a = JSII_RTTI_SYMBOL_1;
MongoDbInstance[_a] = { fqn: "aws-rfdk.MongoDbInstance", version: "0.42.0" };
// How often Cloudwatch logs will be flushed.
MongoDbInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
// Default prefix for a LogGroup if one isn't provided in the props.
MongoDbInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
// Size of the EBS volume for MongoDB data, if we create one.
MongoDbInstance.DEFAULT_MONGO_DEVICE_SIZE = core_1.Size.gibibytes(20);
// Mount point for the MongoDB data volume.
MongoDbInstance.MONGO_DEVICE_MOUNT_POINT = '/var/lib/mongo';
// Size of the root device volume on the instance.
MongoDbInstance.ROOT_DEVICE_SIZE = core_1.Size.gibibytes(10);
//# sourceMappingURL=data:application/json;base64,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