"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstaller = exports.MongoDbSsplLicenseAcceptance = exports.MongoDbVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
/**
 * Versions of MongoDB Community Edition that the {@link MongoDbInstaller} is
 * able to install.
 */
var MongoDbVersion;
(function (MongoDbVersion) {
    /**
     * MongoDB 3.6 Community Edition.
     * See: https://docs.mongodb.com/v3.6/introduction/
     */
    MongoDbVersion["COMMUNITY_3_6"] = "3.6";
})(MongoDbVersion = exports.MongoDbVersion || (exports.MongoDbVersion = {}));
/**
 * Choices for signifying the user's stance on the terms of the SSPL.
 * See: https://www.mongodb.com/licensing/server-side-public-license
 */
var MongoDbSsplLicenseAcceptance;
(function (MongoDbSsplLicenseAcceptance) {
    /**
     * The user signifies their explicit rejection of the tems of the SSPL.
     */
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_REJECTS_SSPL"] = 0] = "USER_REJECTS_SSPL";
    /**
     * The user signifies their explicit acceptance of the terms of the SSPL.
     */
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_ACCEPTS_SSPL"] = 1] = "USER_ACCEPTS_SSPL";
})(MongoDbSsplLicenseAcceptance = exports.MongoDbSsplLicenseAcceptance || (exports.MongoDbSsplLicenseAcceptance = {}));
/**
 * This class provides a mechanism to install a version of MongoDB Community Edition during the
 * initial launch of an instance. MongoDB is installed from the official sources using the system
 * package manger (yum). It installs the mongodb-org metapackage which will install the following packages:
 * 1) mongodb-org-mongos;
 * 2) mongodb-org-server;
 * 3) mongodb-org-shell; and
 * 4) mongodb-org-tools.
 *
 * Successful installation of MongoDB with this class requires:
 * 1) Explicit acceptance of the terms of the SSPL license, under which MongoDB is distributed; and
 * 2) The instance on which the installation is being performed is in a subnet that can access
 * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org
 *
 * Resources Deployed
 * ------------------------
 * - A CDK Asset package containing the installation scripts is deployed to your CDK staging bucket.
 *
 * Security Considerations
 * ------------------------
 * - Since this class installs MongoDB from official sources dynamically during instance start-up, it is succeptable
 *   to an attacker compromising the official MongoDB Inc. distribution channel for MongoDB. Such a compromise may
 *   result in the installation of unauthorized MongoDB binaries. Executing this attack would require an attacker
 *   compromise both the official installation packages and the MongoDB Inc. gpg key with which they are signed.
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
class MongoDbInstaller {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        // Legal requirement: Users of this class must agree to the terms of the SSPL, without exception.
        // Developer note: It is a legal requirement that the default be USER_REJECTS_SSPL, so this check
        // must throw an error for every value except USER_ACCEPTS_SSPL.
        if (props.userSsplAcceptance !== MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL) {
            throw new Error(MongoDbInstaller.SSPL_ACCEPT_MESSAGE);
        }
    }
    /**
     * Install MongoDB to the given instance at instance startup. This is accomplished by
     * adding scripting to the instance's UserData to install MongoDB.
     *
     * Notes:
     * 1) The instance on which the installation is being performed must be in a subnet that can access
     * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org; and
     * 2) At this time, this method only supports installation onto instances that are running an operating system
     * that is compatible with x86-64 RedHat 7 -- this includes Amazon Linux 2, RedHat 7, and CentOS 7.
     *
     * @param target The target instance onto which to install MongoDB.
     */
    installOnLinuxInstance(target) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        const installerScriptAsset = this.installerAssetSingleton();
        installerScriptAsset.grantRead(target.grantPrincipal);
        const installerScript = target.userData.addS3DownloadCommand({
            bucket: installerScriptAsset.bucket,
            bucketKey: installerScriptAsset.s3ObjectKey,
        });
        target.userData.addCommands(`bash ${installerScript}`);
    }
    /**
     * Fetch the Asset singleton for the installation script, or generate it if needed.
     */
    installerAssetSingleton() {
        const stack = core_1.Stack.of(this.scope);
        const uuid = '5b141ac9-fde5-45d8-9961-b7108fb3b18a';
        const uniqueId = 'MongoDbInstallerAsset' + uuid.replace(/[-]/g, '');
        return stack.node.tryFindChild(uniqueId) ?? new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', this.props.version, 'installMongoDb.sh'),
        });
    }
}
exports.MongoDbInstaller = MongoDbInstaller;
_a = JSII_RTTI_SYMBOL_1;
MongoDbInstaller[_a] = { fqn: "aws-rfdk.MongoDbInstaller", version: "0.42.0" };
/**
 * The SSPL licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the SSPL.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
MongoDbInstaller.SSPL_ACCEPT_MESSAGE = `
The MongoDbInstaller will install MongoDB Community Edition onto one or more EC2 instances.

MongoDB is provided by MongoDB Inc. under the SSPL license. By installing MongoDB, you are agreeing to the terms of this license.
Follow the link below to read the terms of the SSPL license.
https://www.mongodb.com/licensing/server-side-public-license

By using the MongoDbInstaller to install MongoDB you agree to the terms of the SSPL license.

Please set the userSsplAcceptance property to USER_ACCEPTS_SSPL to signify your acceptance of the terms of the SSPL license.
`;
//# sourceMappingURL=data:application/json;base64,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