"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AWSLimitExhaustedError = exports.LoadBalancerFactory = void 0;
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const health_monitor_1 = require("./health-monitor");
/**
 * This class is responsible for managing the statistics for all the
 * load balancers created in this construct. It is also responsible to search
 * for the finding the first Load balancer/Listener which can accomodate the
 * worker-fleet based on its size.
 *
 * A typical load balancer hierarchy looks like following:
 *  |__ Load Balancer 1
 *  |         |____________Listener 1
 *  |         |                 |_______Target Group 1 ------- Target/Fleet
 *  |         |                 |_______Target Group 2 ------- Target/Fleet
 *  |         |
 *  |         |____________Listener 2
 *  |                           |_______Target Group 1 ------- Target/Fleet
 *  |                           |_______Target Group 2 ------- Target/Fleet
 *  |
 *  |__ Load Balancer 2
 *            |____________Listener 1
 *            |                 |_______Target Group 1 ------- Target/Fleet
 *            |                 |_______Target Group 2 ------- Target/Fleet
 *            |
 *            |____________Listener 2
 *                              |_______Target Group 1 ------- Target/Fleet
 *                              |_______Target Group 2 ------- Target/Fleet
 *
 *  Components:
 *  1. LoadBalancerFactory: This is the root node of the tree. It contains the
 *     map of load balancer to its managers. It is responsible for creating a
 *     new load balancer if required. It delegates the registerFleet calls to
 *     downstream and returns parent load balancer, listener and target group
 *     of the registered fleet if the registration was successful
 *
 *  2. LoadBalancerManager: This class manages a single load balancer. It
 *     contains a map of all the listeners->manager. It also contains the component
 *     counts like listener, target group and target count. It delegates the
 *     registration call to downstream listeners and updates the stats when
 *     the registration is successful. It returns the parent listener and
 *     target group on successful registration.
 *
 *  3. ListenerManager: This class managers a single Listener. It contains a map
 *     of all of its target groups to its associated fleet. It also contains the
 *     component counts. It returns the target group on registration.
 */
class LoadBalancerFactory {
    constructor(healthMonitorScope, vpc) {
        this.loadBalancerMap = new Map();
        this.healthMonitorScope = healthMonitorScope;
        this.vpc = vpc;
    }
    static getAccountLimit(limitName, defaultValue, elbAccountLimits) {
        if (!elbAccountLimits) {
            return defaultValue;
        }
        const foundLimit = elbAccountLimits.find(limit => limit.name === limitName);
        if (!foundLimit) {
            return defaultValue;
        }
        return foundLimit.max;
    }
    /**
     * This method scans all the load balancers and its listeners and registers the fleet
     * to the load balancer and/or listener which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the load balancer, listener
     * and target group to which the fleet was registered.
     *
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(fleet, healthCheckConfig, healthMonitorProps) {
        let loadBalancerParent = null;
        let listenerParent = null;
        let targetGroupParent = null;
        // iterate through each load balancer and try registering to each one.
        for (const [loadBalancer, loadBalancerMeta] of this.loadBalancerMap.entries()) {
            try {
                const { listener, targetGroup } = loadBalancerMeta.registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps);
                loadBalancerParent = loadBalancer;
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        // Check if fleet was not registered.
        if (!loadBalancerParent) {
            // If this section is reached, no load balancer was found which could
            // accommodate fleet, create a new one and register
            loadBalancerParent = this.createLoadBalancer(this.healthMonitorScope, this.loadBalancerMap.size, healthMonitorProps);
            const loadBalancerManager = new LoadBalancerManager();
            // Add it to the map
            this.loadBalancerMap.set(loadBalancerParent, loadBalancerManager);
            // try registering the fleet to the new load balancer
            try {
                const { listener, targetGroup } = loadBalancerManager.registerWorkerFleet(loadBalancerParent, fleet, healthCheckConfig, healthMonitorProps);
                listenerParent = listener;
                targetGroupParent = targetGroup;
            }
            catch (e) {
                throw e;
            }
        }
        /* istanbul ignore next */
        if (!loadBalancerParent || !listenerParent || !targetGroupParent) {
            /* istanbul ignore next */
            throw new Error('Fleet registered successfully but a parent was found null');
        }
        return {
            loadBalancer: loadBalancerParent,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new load balancer within the given scope.
     *
     * @param scope
     * @param loadBalancerindex
     */
    createLoadBalancer(scope, loadBalancerindex, healthMonitorProps) {
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(scope, `ALB_${loadBalancerindex}`, {
            vpc: this.vpc,
            internetFacing: false,
            vpcSubnets: healthMonitorProps.vpcSubnets,
            deletionProtection: healthMonitorProps.deletionProtection ?? true,
            securityGroup: healthMonitorProps.securityGroup,
        });
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        return loadBalancer;
    }
}
exports.LoadBalancerFactory = LoadBalancerFactory;
LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER = 50;
LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER = 1000;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER = 5;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER = 100;
/**
 * This class manages the properties of a single load balancer and its statistics.
 * It is also responsible to scan through all the listeners registered under it
 * and register the given fleet.
 */
class LoadBalancerManager {
    constructor() {
        this.listenerMap = new Map();
        this.loadBalancerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accomodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the listener
     * and target group to which the fleet was registered.
     *
     * @param loadBalancer
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps) {
        // this initializes with 0 and keeps the track of all components
        // newly added down the hierarchy.
        const statsDelta = new LoadBalancerComponentStats();
        // Do all the load balancer level service limit checks first
        // check for target limit in load balancer
        const targetPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('targets-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetCount + fleet.targetCapacity) > targetPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "targets-per-application-load-balancer" reached. Limit: ' +
                targetPerLoadBalancerLimit);
        }
        // check for target group limit in load balancer
        const targetGroupsPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetGroupCount + 1) > targetGroupsPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-application-load-balancer" reached. Limit: ' +
                targetGroupsPerLoadBalancerLimit);
        }
        let listenerParent = null;
        let targetGroupParent = null;
        // try registering to each listener.
        for (const [listener, listenerMeta] of this.listenerMap.entries()) {
            try {
                const { componentsAdded, targetGroup } = listenerMeta.registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps);
                statsDelta.add(componentsAdded);
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        /* istanbul ignore next */
        if (!listenerParent) {
            // If this section is reached, no listener was found which could accommodate fleet
            // create new listener and register
            const listenersPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('listeners-per-application-load-balancer', LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
            if ((this.loadBalancerComponentCount.listenerCount + 1) > listenersPerLoadBalancerLimit) {
                throw new AWSLimitExhaustedError('AWS service limit "listeners-per-application-load-balancer" reached. Limit: ' +
                    listenersPerLoadBalancerLimit);
            }
            listenerParent = this.createListener(fleet.targetScope, loadBalancer);
            const listenerManager = new ListenerManager();
            this.listenerMap.set(listenerParent, listenerManager);
            statsDelta.add(new LoadBalancerComponentStats(1, 0, 0));
            try {
                const { componentsAdded, targetGroup } = listenerManager.registerWorkerFleet(loadBalancer, listenerParent, fleet, healthCheckConfig, healthMonitorProps);
                targetGroupParent = targetGroup;
                statsDelta.add(componentsAdded);
            }
            catch (e) {
                throw e;
            }
        }
        // update the current load balancer's stats
        this.loadBalancerComponentCount.add(statsDelta);
        return {
            componentsAdded: statsDelta,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new listener in the fleet's scope and
     * registers it to the given load balancer.
     *
     * @param scope
     * @param loadBalancer
     */
    createListener(scope, loadBalancer) {
        return new aws_elasticloadbalancingv2_1.ApplicationListener(scope, 'Listener', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT + this.listenerMap.size,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            loadBalancer,
            open: false,
        });
    }
}
/**
 * This class manages the properties of a single listener and all the components
 * under its hierarchy.
 * It is also responsible to create a new target group and register the given fleet.
 */
class ListenerManager {
    constructor() {
        this.targetMap = new Map();
        this.listenerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the target group
     * to which the fleet was registered.
     *
     * @param loadBalancer
     * @param listener
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps) {
        const componentsAdded = new LoadBalancerComponentStats();
        // Do all listener level service limit checks
        // check for target limit in listener
        const targetGroupPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-action-on-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.listenerComponentCount.targetGroupCount + 1) > targetGroupPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-action-on-application-load-balancer" reached. Limit: ' +
                targetGroupPerLoadBalancerLimit);
        }
        // latest version of CDK does not support 'forwardConfig' in listener rule yet. This means
        // we cannot add multiple target groups to a single listener. Adding this check till this
        // feature is supported.
        if (this.listenerComponentCount.targetGroupCount > 0) {
            throw new AWSLimitExhaustedError('Unable to add more than 1 Target Group to Listener.');
        }
        // Create a new target group
        const targetGroup = this.createTargetGroup(fleet.targetScope, loadBalancer, listener, fleet, healthCheckConfig);
        this.targetMap.set(targetGroup, fleet);
        // update the listener stats
        componentsAdded.targetGroupCount++;
        componentsAdded.targetCount += fleet.targetCapacity;
        // update the current listener's stats
        this.listenerComponentCount.add(componentsAdded);
        return {
            componentsAdded,
            targetGroup,
        };
    }
    /**
     * Following method creates a new new target group in the fleet's scope and
     * registers it to the given listener.
     *
     * @param scope
     * @param loadBalancer
     * @param listener
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    createTargetGroup(scope, loadBalancer, listener, monitorableFleet, healthCheckConfig) {
        const targetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(scope, 'TargetGroup', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            targets: [monitorableFleet.targetToMonitor],
            healthCheck: {
                port: healthCheckConfig.port ? healthCheckConfig.port.toString() : health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT.toString(),
                interval: healthCheckConfig.interval || health_monitor_1.HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL,
                healthyThresholdCount: healthCheckConfig.instanceHealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD,
                unhealthyThresholdCount: healthCheckConfig.instanceUnhealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD,
                protocol: aws_elasticloadbalancingv2_1.Protocol.HTTP,
            },
            vpc: loadBalancer.vpc,
        });
        listener.addTargetGroups('TargetGroup', {
            targetGroups: [targetGroup],
        });
        return targetGroup;
    }
}
/**
 * This class contains the statistics of all the nested load balancer
 * components like listener count, target group count and target count.
 * This statistics object will be associated with each load balancer
 * and listener for tracking the count of components.
 */
class LoadBalancerComponentStats {
    constructor(listenerCount = 0, targetGroupCount = 0, targetCount = 0) {
        this.listenerCount = listenerCount;
        this.targetGroupCount = targetGroupCount;
        this.targetCount = targetCount;
    }
    add(operand) {
        this.listenerCount += operand.listenerCount;
        this.targetGroupCount += operand.targetGroupCount;
        this.targetCount += operand.targetCount;
    }
}
class AWSLimitExhaustedError extends Error {
    constructor(message) {
        super(message);
    }
}
exports.AWSLimitExhaustedError = AWSLimitExhaustedError;
//# sourceMappingURL=data:application/json;base64,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