"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImportedAcmCertificate = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * A Construct that creates an AWS CloudFormation Custom Resource that models a certificate that is imported into
 * AWS Certificate Manager (ACM). It uses an AWS Lambda Function to extract the certificate from Secrets in AWS SecretsManager
 * and then import it into ACM. The interface is intended to be used with the {@link X509CertificatePem} Construct.
 *
 * ![architecture diagram](/diagrams/core/ImportedAcmCertificate.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - An AWS Lambda Function, with IAM Role - Used to create/update/delete the Custom Resource.
 * - AWS Certificate Manager Certificate - Created by the Custom Resource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda for this construct also has broad IAM permissions to delete any Certificate that is stored
 *   in AWS Certificate Manager. You should not grant any additional actors/principals the ability to modify or
 *   execute this Lambda.
 */
class ImportedAcmCertificate extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            encryption: aws_dynamodb_1.TableEncryption.AWS_MANAGED,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lambda = new aws_lambda_1.SingletonFunction(this, 'AcmImporter', {
            uuid: ImportedAcmCertificate.IMPORTER_UUID,
            code: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            handler: 'x509-certificate.importCert',
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            layers: [openSslLayer],
            retryAttempts: 0,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            timeout: core_1.Duration.minutes(5),
        });
        this.database.grantReadWriteData(lambda);
        this.database.grant(lambda, 'dynamodb:DescribeTable');
        props.cert.grantRead(lambda);
        props.key.grantRead(lambda);
        props.passphrase.grantRead(lambda);
        props.certChain?.grantRead(lambda);
        props.encryptionKey?.grantDecrypt(lambda);
        const uniqueValue = crypto.createHash('md5').update(core_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new core_1.Tag(`AcmCertImport-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`aws:RequestTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:AddTagsToCertificate',
                'acm:ImportCertificate',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // GetCertificate and DeleteCertificate don't currently support and conditions, so we have to give a broader policy
        // on them for now.
        // See https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html#awscertificatemanager-aws_TagKeys
        // for the condition keys currently available on ACM actions.
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:DeleteCertificate',
                'acm:DescribeCertificate',
                'acm:GetCertificate',
            ],
            resources: ['*'],
        }));
        const properties = {
            X509CertificatePem: {
                Cert: props.cert.secretArn,
                Key: props.key.secretArn,
                Passphrase: props.passphrase.secretArn,
                CertChain: props.certChain ? props.certChain.secretArn : '',
            },
            Tags: [
                {
                    Key: this.uniqueTag.key,
                    Value: this.uniqueTag.value,
                },
                { Key: 'Name',
                    Value: this.uniqueTag.value,
                },
            ],
        };
        this.resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lambda.functionArn,
            properties,
            resourceType: 'Custom::RFDK_AcmImportedCertificate',
        });
        this.certificateArn = core_1.Token.asString(this.resource.getAtt('CertificateArn'));
    }
    /**
     * Apply a removal policy to the custom resource that represents the certificate imported into ACM
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    /**
     * @inheritdoc
     */
    metricDaysToExpiry(props) {
        const certLookupNode = this.node.tryFindChild(ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID);
        let certLookup;
        /* istanbul ignore next */
        if (certLookupNode) {
            certLookup = certLookupNode;
        }
        else {
            certLookup = aws_certificatemanager_1.Certificate.fromCertificateArn(this, ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID, this.certificateArn);
        }
        return certLookup.metricDaysToExpiry(props);
    }
}
exports.ImportedAcmCertificate = ImportedAcmCertificate;
_a = JSII_RTTI_SYMBOL_1;
ImportedAcmCertificate[_a] = { fqn: "aws-rfdk.ImportedAcmCertificate", version: "0.42.0" };
ImportedAcmCertificate.IMPORTER_UUID = '2d20d8f2-7b84-444e-b738-c75b499a9eaa';
ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID = 'CertificateLookup';
//# sourceMappingURL=data:application/json;base64,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