"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthMonitor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_cloudwatch_actions_1 = require("@aws-cdk/aws-cloudwatch-actions");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
const load_balancer_manager_1 = require("./load-balancer-manager");
const runtime_info_1 = require("./runtime-info");
/**
 *  A new or imported Health Monitor.
 */
class HealthMonitorBase extends core_1.Construct {
}
/**
 * This construct is responsible for the deep health checks of compute instances.
 * It also replaces unhealthy instances and suspends unhealthy fleets.
 * Although, using this constructs adds up additional costs for monitoring,
 * it is highly recommended using this construct to help avoid / minimize runaway costs for compute instances.
 *
 * An instance is considered to be unhealthy when:
 *   1) Deadline client is not installed on it;
 *   2) Deadline client is installed but not running on it;
 *   3) RCS is not configured correctly for Deadline client;
 *   4) it is unable to connect to RCS due to any infrastructure issues;
 *   5) the health monitor is unable to reach it because of some infrastructure issues.
 *
 * A fleet is considered to be unhealthy when:
 *   1) at least 1 instance is unhealthy for the configured grace period;
 *   2) a percentage of unhealthy instances in the fleet is above a threshold at any given point of time.
 *
 * This internally creates an array of application load balancers and attaches
 * the worker-fleet (which internally is implemented as an Auto Scaling Group) to its listeners.
 * There is no load-balancing traffic on the load balancers,
 * it is only used for health checks.
 * Intention is to use the default properties of laod balancer health
 * checks which does HTTP pings at frequent intervals to all the
 * instances in the fleet and determines its health. If any of the
 * instance is found unhealthy, it is replaced. The target group
 * also publishes the unhealthy target count metric which is used
 * to identify the unhealthy fleet.
 *
 * Other than the default instance level protection, it also creates a lambda
 * which is responsible to set the fleet size to 0 in the event of a fleet
 * being sufficiently unhealthy to warrant termination.
 * This lambda is triggered by CloudWatch alarms via SNS (Simple Notification Service).
 *
 * ![architecture diagram](/diagrams/core/HealthMonitor.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Application Load Balancer(s) doing frequent pings to the workers.
 * - An Amazon Simple Notification Service (SNS) topic for all unhealthy fleet notifications.
 * - An AWS Key Management Service (KMS) Key to encrypt SNS messages - If no encryption key is provided.
 * - An Amazon CloudWatch Alarm that triggers if a worker fleet is unhealthy for a long period.
 * - Another CloudWatch Alarm that triggers if the healthy host percentage of a worker fleet is lower than allowed.
 * - A single AWS Lambda function that sets fleet size to 0 when triggered in response to messages on the SNS Topic.
 * - Execution logs of the AWS Lambda function are published to a log group in Amazon CloudWatch.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 * - The AWS Lambda that is created by this construct to terminate unhealthy worker fleets has permission to
 *   UpdateAutoScalingGroup ( https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_UpdateAutoScalingGroup.html )
 *   on all of the fleets that this construct is monitoring. You should not grant any additional actors/principals the
 *   ability to modify or execute this Lambda.
 * - Execution of the AWS Lambda for terminating unhealthy workers is triggered by messages to the Amazon Simple
 *   Notification Service (SNS) Topic that is created by this construct. Any principal that is able to publish notification
 *   to this SNS Topic can cause the Lambda to execute and reduce one of your worker fleets to zero instances. You should
 *   not grant any additional principals permissions to publish to this SNS Topic.
 */
class HealthMonitor extends HealthMonitorBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.lbFactory = new load_balancer_manager_1.LoadBalancerFactory(this, props.vpc);
        const topicEncryptKey = props.encryptionKey || new aws_kms_1.Key(this, 'SNSEncryptionKey', {
            description: `This key is used to encrypt SNS messages for ${core_1.Names.uniqueId(this)}.`,
            enableKeyRotation: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            trustAccountIdentities: true,
        });
        // allow cloudwatch service to send encrypted messages
        topicEncryptKey.grant(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'), 'kms:Decrypt', 'kms:GenerateDataKey');
        this.unhealthyFleetActionTopic = new aws_sns_1.Topic(this, 'UnhealthyFleetTopic', {
            masterKey: topicEncryptKey,
        });
        this.unhealthyFleetActionTopic.grantPublish(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'));
        this.alarmTopicAction = new aws_cloudwatch_actions_1.SnsAction(this.unhealthyFleetActionTopic);
        this.unhealthyFleetActionLambda = new aws_lambda_1.SingletonFunction(this, 'UnhealthyFleetAction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'unhealthyFleetAction')),
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            lambdaPurpose: 'unhealthyFleetTermination',
            timeout: core_1.Duration.seconds(300),
            uuid: '28bccf6a-aa76-478c-9239-e2f5bcc0254c',
        });
        this.unhealthyFleetActionTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.unhealthyFleetActionLambda));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Attaches the load-balancing target to the ELB for instance-level
     * monitoring. The ELB does frequent pings to the workers and determines
     * if a worker node is unhealthy. If so, it replaces the instance.
     *
     * It also creates an Alarm for healthy host percent and suspends the
     * fleet if the given alarm is breaching. It sets the maxCapacity
     * property of the auto-scaling group to 0. This should be
     * reset manually after fixing the issue.
     *
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    registerFleet(monitorableFleet, healthCheckConfig) {
        const { loadBalancer, targetGroup } = this.lbFactory.registerWorkerFleet(monitorableFleet, healthCheckConfig, this.props);
        this.createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup);
    }
    createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup) {
        monitorableFleet.connections.allowFrom(loadBalancer, aws_ec2_1.Port.tcp(healthCheckConfig.port || HealthMonitor.LOAD_BALANCER_LISTENING_PORT));
        const percentMetric = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD,
        });
        // When unhealthy fleet is more than healthyFleetThresholdPercent or 35% at any given period of 5 minutes
        const immediateTerminationAlarm = percentMetric.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetTermination', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: 100 - (healthCheckConfig.healthyFleetThresholdPercent || HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            actionsEnabled: true,
        });
        immediateTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        // When at least one node is unhealthy over a period of 2 hours
        const percentMetricGracePeriod = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE,
        });
        const gracePeriodTerminationAlarm = percentMetricGracePeriod.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetGracePeriod', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            actionsEnabled: true,
        });
        gracePeriodTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        monitorableFleet.targetUpdatePolicy.attachToRole(this.unhealthyFleetActionLambda.role);
    }
}
exports.HealthMonitor = HealthMonitor;
_a = JSII_RTTI_SYMBOL_1;
HealthMonitor[_a] = { fqn: "aws-rfdk.HealthMonitor", version: "0.42.0" };
/**
 * Default health check listening port
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_PORT = 63415;
/**
 * Resource Tracker in Deadline currently publish health status every 5 min, hence keeping this same
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(5);
/**
 * Resource Tracker in Deadline currently determines host unhealthy in 15 min, hence keeping this count
 */
HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD = 3;
/**
 * This is the minimum possible value of ALB health-check config, we want to mark worker healthy ASAP
 */
HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD = 2;
/**
 * Since we are not doing any load balancing, this port is just an arbitrary port.
 */
HealthMonitor.LOAD_BALANCER_LISTENING_PORT = 8081;
/**
 * This number is taken from Resource Tracker implementation. If a fleet's healthy percent
 * is less than this threshold at any given point of time, it is suspended.
 */
HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD = 65;
/**
 * This number is taken from Resource Tracker implementation. If a fleet has at least 1
 * unhealthy host for a period of 2 hours, it is suspended.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE = 0;
/**
 * This number is taken from Resource Tracker implementation. We monitor unhealthy fleet for immediate
 * termination for a period fo 5 minutes.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD = core_1.Duration.minutes(5);
/**
 * In Resource Tracker, we evaluate the fleet's health for determining the grace period over a period
 * of 5 minutes. For the first unhealthy signal, a instance can take upto 10min (max), hence we are
 * setting this period to be 15.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE = core_1.Duration.minutes(15);
/**
 * This number is taken from Resource Tracker implementation. Fleet is terminated immediately if it
 * has unhealthy host percent above the hard limit.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD = 1;
/**
 * This number is taken from Resource Tracker implementation. The grace period duration is 2 hours,
 * since the grace period is 15 minutes, we need continuous 8 data points crossing the threshold.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE = 8;
//# sourceMappingURL=data:application/json;base64,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