"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExportingLogGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
/**
 * This construct takes the name of a CloudWatch LogGroup and will either create it if it doesn't already exist,
 * or reuse the existing one. It also creates a regularly scheduled lambda that will export LogEvents to S3
 * before they expire in CloudWatch.
 *
 * It's used for cost-reduction, as it is more economical to archive logs in S3 than CloudWatch when
 * retaining them for more than a week.
 * Note, that it isn't economical to export logs to S3 if you plan on storing them for less than
 * 7 days total (CloudWatch and S3 combined).
 *
 * Resources Deployed
 * ------------------------
 * - The Lambda SingletonFunction that checks for the existence of the LogGroup.
 * - The CloudWatch LogGroup (if it didn't exist already).
 * - The CloudWatch Alarm watching log exportation failures.
 * - The CloudWatch Event Rule to schedule log exportation.
 * - The Lambda SingletonFunction, with role, to export log groups to S3 by schedule.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *   that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *   your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *   We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *   or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 */
class ExportingLogGroup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * UUID needed to identify the SingletonFunction for the log exporter.
         */
        this.LOG_EXPORTER_UUID = '6382448c-e4b2-42e9-b14f-a0a9ccdb198e';
        /**
         * Duration of time between export task Lambda runs.
         */
        this.EXPORT_TASK_FREQUENCY = core_1.Duration.hours(1);
        /**
         * Default value for the number of days to retain logs in CloudWatch for.
         */
        this.RETENTION_DEFAULT = aws_logs_1.RetentionDays.THREE_DAYS;
        const retentionInDays = props.retention ? props.retention : this.RETENTION_DEFAULT;
        const exportLogsFunction = this.setupLogsExporter();
        this.exportErrorAlarm = exportLogsFunction.metricErrors().createAlarm(this, 'LogExporterFailureAlarm', {
            evaluationPeriods: 1,
            threshold: 1,
        });
        const logRetention = new aws_logs_1.LogRetention(this, 'LogRetention', {
            logGroupName: props.logGroupName,
            retention: retentionInDays,
        });
        this.logGroup = aws_logs_1.LogGroup.fromLogGroupArn(scope, `${props.logGroupName}LogGroup`, logRetention.logGroupArn);
        this.logGroup.grant(exportLogsFunction, 'logs:CreateExportTask');
        const scheduledLogExportation = new aws_events_1.Rule(this, 'LogExporterRule', {
            schedule: aws_events_1.Schedule.rate(this.EXPORT_TASK_FREQUENCY),
        });
        scheduledLogExportation.addTarget(new aws_events_targets_1.LambdaFunction(exportLogsFunction, {
            event: aws_events_1.RuleTargetInput.fromObject({
                BucketName: props.bucketName,
                ExportFrequencyInHours: this.EXPORT_TASK_FREQUENCY.toHours(),
                LogGroupName: props.logGroupName,
                RetentionInHours: retentionInDays.valueOf() * 24,
            }),
        }));
    }
    setupLogsExporter() {
        const exportLogsFunction = new aws_lambda_1.SingletonFunction(this, 'LogExporterFunction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'export-logs')),
            handler: 'index.handler',
            lambdaPurpose: 'LogGroupExporter',
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            uuid: this.LOG_EXPORTER_UUID,
        });
        exportLogsFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['logs:DescribeExportTasks'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: ['*'],
        }));
        return exportLogsFunction;
    }
}
exports.ExportingLogGroup = ExportingLogGroup;
_a = JSII_RTTI_SYMBOL_1;
ExportingLogGroup[_a] = { fqn: "aws-rfdk.ExportingLogGroup", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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