"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectableApplicationEndpoint = exports.ApplicationEndpoint = exports.Endpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
/**
 * Connection endpoint
 *
 * Consists of a combination of hostname, port, and transport protocol.
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param props The properties for the endpoint
     */
    constructor(props) {
        const { address, port } = props;
        const protocol = props.protocol ?? aws_ec2_1.Protocol.TCP;
        if (!core_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.portNumber = port;
        this.protocol = protocol;
        this.port = new aws_ec2_1.Port({
            protocol: this.protocol,
            fromPort: port,
            toPort: port,
            stringRepresentation: this.renderPort(port),
        });
        this.socketAddress = `${address}:${this.portAsString()}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns {string} An (un)resolved string representation of the endpoint's port number
     */
    portAsString() {
        if (core_1.Token.isUnresolved(this.portNumber)) {
            return core_1.Token.asString(this.portNumber);
        }
        else {
            return this.portNumber.toString();
        }
    }
    renderPort(port) {
        return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
    }
}
exports.Endpoint = Endpoint;
_a = JSII_RTTI_SYMBOL_1;
Endpoint[_a] = { fqn: "aws-rfdk.Endpoint", version: "0.42.0" };
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
/**
 * An endpoint serving http or https for an application.
 */
class ApplicationEndpoint extends Endpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint
     */
    constructor(props) {
        super({
            address: props.address,
            port: props.port,
            protocol: aws_ec2_1.Protocol.TCP,
        });
        this.applicationProtocol = props.protocol ?? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
    }
}
exports.ApplicationEndpoint = ApplicationEndpoint;
_b = JSII_RTTI_SYMBOL_1;
ApplicationEndpoint[_b] = { fqn: "aws-rfdk.ApplicationEndpoint", version: "0.42.0" };
/**
 * An endpoint serving http or https for an application.
 */
class ConnectableApplicationEndpoint extends ApplicationEndpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint
     */
    constructor(props) {
        super(props);
        this.connections = props.connections;
    }
}
exports.ConnectableApplicationEndpoint = ConnectableApplicationEndpoint;
_c = JSII_RTTI_SYMBOL_1;
ConnectableApplicationEndpoint[_c] = { fqn: "aws-rfdk.ConnectableApplicationEndpoint", version: "0.42.0" };
//# sourceMappingURL=data:application/json;base64,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