"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentInstance = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const runtime_info_1 = require("./runtime-info");
/**
 * Deploys an instance that runs its user data on deployment, waits for that user data to succeed, and optionally
 * terminates itself afterwards.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with max capacity of 1 instance.
 * - IAM instance profile, IAM role, and IAM policy
 * - An Amazon CloudWatch log group that contains the instance cloud-init logs
 * - A Lambda Function to fetch and existing Log Group or create a new one
 * - IAM role and policy for the Lambda Function
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 */
class DeploymentInstance extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.asg = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: props.instanceType ?? aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.SMALL),
            keyName: props.keyName,
            machineImage: props.machineImage ?? aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            minCapacity: 1,
            maxCapacity: 1,
            securityGroup: props.securityGroup,
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: props.executionTimeout ?? DeploymentInstance.DEFAULT_EXECUTION_TIMEOUT,
            }),
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets ?? {
                subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
            },
        });
        this.node.defaultChild = this.asg;
        this.connections = this.asg.connections;
        const logGroupName = props.logGroupName ?? id;
        this.configureCloudWatchAgent(this.asg, logGroupName, props.logGroupProps);
        if (props.selfTerminate ?? true) {
            this.configureSelfTermination();
        }
        this.asg.userData.addSignalOnExitCommand(this.asg);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Make the execution of the instance dependent upon another construct
     *
     * @param dependency The construct that should be dependended upon
     */
    addExecutionDependency(dependency) {
        if (core_1.Construct.isConstruct(dependency)) {
            this.asg.node.defaultChild.node.addDependency(dependency);
        }
    }
    /**
     * @inheritdoc
     */
    get osType() {
        return this.asg.osType;
    }
    /**
     * @inheritdoc
     */
    get userData() {
        return this.asg.userData;
    }
    /**
     * @inheritdoc
     */
    get grantPrincipal() {
        return this.asg.grantPrincipal;
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the deployment instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *
     * @param asg The auto-scaling group
     * @param groupName The name of the Log Group, or suffix of the Log Group if `logGroupProps.logGroupPrefix` is
     *                  specified
     * @param logGroupProps The properties for LogGroupFactory to create or fetch the log group
     */
    configureCloudWatchAgent(asg, groupName, logGroupProps) {
        const prefix = logGroupProps?.logGroupPrefix ?? DeploymentInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'DeploymentInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(asg);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        new _1.CloudWatchAgent(this, 'CloudWatchAgent', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: asg,
        });
    }
    configureSelfTermination() {
        // Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources as it will cause
        // cyclic dependency. Hence, using Condition Keys
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${DeploymentInstance.ASG_TAG_KEY}`] = core_1.Names.uniqueId(this);
        core_1.Tags.of(this.asg).add(DeploymentInstance.ASG_TAG_KEY, core_1.Names.uniqueId(this));
        this.asg.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.asg.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.asg.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and ASG name and then setting its capacity to 0
        this.asg.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)', 'INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"', 'ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"', 'aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
}
exports.DeploymentInstance = DeploymentInstance;
/**
 * The tag key name used as an IAM condition to restrict autoscaling API grants
 */
DeploymentInstance.ASG_TAG_KEY = 'resourceLogicalId';
/**
 * How often the CloudWatch agent will flush its log files to CloudWatch
 */
DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The default timeout to wait for CloudFormation success signals before failing the resource create/update
 */
DeploymentInstance.DEFAULT_EXECUTION_TIMEOUT = core_1.Duration.minutes(15);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
DeploymentInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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