"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchAgent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const script_assets_1 = require("./script-assets");
/**
 * This construct is a thin wrapper that provides the ability to install and configure the CloudWatchAgent
 * ( https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html )
 * on one or more EC2 instances during instance startup.
 *
 * It accomplishes this by downloading and executing the configuration script on the instance.
 * The script will download the CloudWatch Agent installer,
 * optionally verify the installer, and finally install the CloudWatch Agent.
 * The installer is downloaded via the Amazon S3 API, thus, this construct can be used
 * on instances that have no access to the internet as long as the VPC contains
 * an VPC Gateway Endpoint for S3 ( https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-s3.html ).
 *
 * {@link CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR} - Context variable to skip validation
 * of the downloaded CloudWatch Agent installer if set to 'TRUE'.
 * WARNING: Only use this if your deployments are failing due to a validation failure,
 * but you have verified that the failure is benign.
 *
 * Resources Deployed
 * ------------------------
 * - String SSM Parameter in Systems Manager Parameter Store to store the cloudwatch agent configuration;
 * - A script Asset which is uploaded to S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *   from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *   bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *   you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *   account to assist in post-incident analysis of compromised production environments.
 */
class CloudWatchAgent extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const shouldInstallAgent = props.shouldInstallAgent ?? true;
        // Create the asset for the configuration script
        this.configurationScript = script_assets_1.ScriptAsset.fromPathConvention(scope, 'CloudWatchConfigurationScriptAsset', {
            osType: props.host.osType,
            baseName: 'configureCloudWatchAgent',
            rootDir: path.join(__dirname, '../scripts/'),
        });
        // Create a new SSM Parameter holding the json configuration
        this.ssmParameterForConfig = new aws_ssm_1.StringParameter(scope, 'StringParameter', {
            description: 'config file for Repository logs config',
            stringValue: props.cloudWatchConfig,
        });
        this.grantRead(props.host);
        this.configure(props.host, shouldInstallAgent, this.node.tryGetContext(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR) === 'TRUE');
    }
    /**
     * Grants read permissions to the principal on the assets bucket and parameter store.
     */
    grantRead(grantee) {
        this.configurationScript.grantRead(grantee);
        this.ssmParameterForConfig.grantRead(grantee);
    }
    /**
     * Configures the CloudWatch Agent on the target host.
     *
     * This is done by adding UserData commands to the target host.
     *
     * @param host The host to configure the CloudWatch agent on
     * @param shouldInstallAgent Attempts to install the CloudWatch agent on the instance if set to true.
     * @param skipValidation Skips the validation of the CloudWatch agent installer if set to true.
     */
    configure(host, shouldInstallAgent, skipValidation) {
        const region = core_1.Stack.of(this).region;
        if (shouldInstallAgent) {
            // Grant access to the required CloudWatch Agent and GPG installer files.
            const cloudWatchAgentBucket = aws_s3_1.Bucket.fromBucketArn(this, 'CloudWatchAgentBucket', `arn:aws:s3:::amazoncloudwatch-agent-${region}`);
            cloudWatchAgentBucket.grantRead(host);
            const gpgBucket = aws_s3_1.Bucket.fromBucketArn(this, 'GpgBucket', `arn:aws:s3:::rfdk-external-dependencies-${region}`);
            gpgBucket.grantRead(host);
        }
        const scriptArgs = [];
        // Flags must be set before positional arguments for some scripts
        if (shouldInstallAgent) {
            scriptArgs.push(CloudWatchAgent.INSTALL_CWAGENT_FLAG);
        }
        if (skipValidation) {
            scriptArgs.push(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG);
        }
        // This assumes that the CloudWatch agent construct is always put in the same region as the instance or ASG
        // using it, which should always hold true.
        scriptArgs.push(region);
        scriptArgs.push(this.ssmParameterForConfig.parameterName);
        this.configurationScript.executeOn({
            host,
            args: scriptArgs,
        });
    }
}
exports.CloudWatchAgent = CloudWatchAgent;
_a = JSII_RTTI_SYMBOL_1;
CloudWatchAgent[_a] = { fqn: "aws-rfdk.CloudWatchAgent", version: "0.42.0" };
/**
 * The context variable to indicate that CloudWatch agent installer validation should be skipped.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR = 'RFDK_SKIP_CWAGENT_VALIDATION';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG = '-s';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.INSTALL_CWAGENT_FLAG = '-i';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWR3YXRjaC1hZ2VudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNsb3Vkd2F0Y2gtYWdlbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0dBR0c7QUFFSCw2QkFBNkI7QUFHN0IsNENBQXlDO0FBQ3pDLDhDQUFtRDtBQUNuRCx3Q0FBaUQ7QUFFakQsbURBQTJEO0FBMEIzRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0E2Qkc7QUFDSCxNQUFhLGVBQWdCLFNBQVEsZ0JBQVM7SUEyQjVDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7UUFDbkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxJQUFJLENBQUM7UUFFNUQsZ0RBQWdEO1FBQ2hELElBQUksQ0FBQyxtQkFBbUIsR0FBRywyQkFBVyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxvQ0FBb0MsRUFBRTtZQUNyRyxNQUFNLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxNQUFNO1lBQ3pCLFFBQVEsRUFBRSwwQkFBMEI7WUFDcEMsT0FBTyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGFBQWEsQ0FBQztTQUM3QyxDQUFDLENBQUM7UUFFSCw0REFBNEQ7UUFDNUQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLElBQUkseUJBQWUsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7WUFDekUsV0FBVyxFQUFFLHdDQUF3QztZQUNyRCxXQUFXLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtTQUNwQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMzQixJQUFJLENBQUMsU0FBUyxDQUNaLEtBQUssQ0FBQyxJQUFJLEVBQ1Ysa0JBQWtCLEVBQ2xCLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQywrQkFBK0IsQ0FBQyxLQUFLLE1BQU0sQ0FDcEYsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNLLFNBQVMsQ0FBQyxPQUFtQjtRQUNuQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzVDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ssU0FBUyxDQUNmLElBQWlCLEVBQ2pCLGtCQUEyQixFQUMzQixjQUF1QjtRQUV2QixNQUFNLE1BQU0sR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUNyQyxJQUFJLGtCQUFrQixFQUFFO1lBQ3RCLHlFQUF5RTtZQUN6RSxNQUFNLHFCQUFxQixHQUFHLGVBQU0sQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFLHVDQUF1QyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1lBQ25JLHFCQUFxQixDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUN0QyxNQUFNLFNBQVMsR0FBRyxlQUFNLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUUsMkNBQTJDLE1BQU0sRUFBRSxDQUFDLENBQUM7WUFDL0csU0FBUyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUMzQjtRQUVELE1BQU0sVUFBVSxHQUFHLEVBQUUsQ0FBQztRQUV0QixpRUFBaUU7UUFDakUsSUFBSSxrQkFBa0IsRUFBRTtZQUN0QixVQUFVLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1NBQ3ZEO1FBQ0QsSUFBSSxjQUFjLEVBQUU7WUFDbEIsVUFBVSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsNEJBQTRCLENBQUMsQ0FBQztTQUMvRDtRQUVELDJHQUEyRztRQUMzRywyQ0FBMkM7UUFDM0MsVUFBVSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUV4QixVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUUxRCxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDO1lBQ2pDLElBQUk7WUFDSixJQUFJLEVBQUUsVUFBVTtTQUNqQixDQUFDLENBQUM7SUFDTCxDQUFDOztBQXhHSCwwQ0F5R0M7OztBQXZHQzs7R0FFRztBQUNvQiwrQ0FBK0IsR0FBRyw4QkFBOEIsQ0FBQztBQUV4Rjs7R0FFRztBQUNxQiw0Q0FBNEIsR0FBRyxJQUFJLENBQUM7QUFFNUQ7O0dBRUc7QUFDcUIsb0NBQW9CLEdBQUcsSUFBSSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcblxuaW1wb3J0IHsgSUdyYW50YWJsZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0IHsgQnVja2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IFN0cmluZ1BhcmFtZXRlciB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zc20nO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQgeyBJU2NyaXB0SG9zdCwgU2NyaXB0QXNzZXQgfSBmcm9tICcuL3NjcmlwdC1hc3NldHMnO1xuXG4vKipcbiAqICBQcm9wZXJ0aWVzIGZvciBjcmVhdGluZyB0aGUgcmVzb3VyY2VzIG5lZWRlZCBmb3IgQ2xvdWRXYXRjaCBBZ2VudCBjb25maWd1cmF0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENsb3VkV2F0Y2hBZ2VudFByb3BzIHtcbiAgLyoqXG4gICAqIENsb3VkV2F0Y2ggYWdlbnQgY29uZmlndXJhdGlvbiBzdHJpbmcgaW4ganNvbiBmb3JtYXQuXG4gICAqXG4gICAqIEBzZWUgLSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uQ2xvdWRXYXRjaC9sYXRlc3QvbW9uaXRvcmluZy9DbG91ZFdhdGNoLUFnZW50LUNvbmZpZ3VyYXRpb24tRmlsZS1EZXRhaWxzLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IGNsb3VkV2F0Y2hDb25maWc6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGhvc3QgaW5zdGFuY2UvQVNHL2ZsZWV0IHdpdGggYSBDbG91ZFdhdGNoIEFnZW50IHRvIGJlIGNvbmZpZ3VyZWQuXG4gICAqL1xuICByZWFkb25seSBob3N0OiBJU2NyaXB0SG9zdDtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3Qgd2Ugc2hvdWxkIGF0dGVtcHQgdG8gaW5zdGFsbCB0aGUgQ2xvdWRXYXRjaCBhZ2VudFxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBzaG91bGRJbnN0YWxsQWdlbnQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFRoaXMgY29uc3RydWN0IGlzIGEgdGhpbiB3cmFwcGVyIHRoYXQgcHJvdmlkZXMgdGhlIGFiaWxpdHkgdG8gaW5zdGFsbCBhbmQgY29uZmlndXJlIHRoZSBDbG91ZFdhdGNoQWdlbnRcbiAqICggaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3JpbmcvSW5zdGFsbC1DbG91ZFdhdGNoLUFnZW50Lmh0bWwgKVxuICogb24gb25lIG9yIG1vcmUgRUMyIGluc3RhbmNlcyBkdXJpbmcgaW5zdGFuY2Ugc3RhcnR1cC5cbiAqXG4gKiBJdCBhY2NvbXBsaXNoZXMgdGhpcyBieSBkb3dubG9hZGluZyBhbmQgZXhlY3V0aW5nIHRoZSBjb25maWd1cmF0aW9uIHNjcmlwdCBvbiB0aGUgaW5zdGFuY2UuXG4gKiBUaGUgc2NyaXB0IHdpbGwgZG93bmxvYWQgdGhlIENsb3VkV2F0Y2ggQWdlbnQgaW5zdGFsbGVyLFxuICogb3B0aW9uYWxseSB2ZXJpZnkgdGhlIGluc3RhbGxlciwgYW5kIGZpbmFsbHkgaW5zdGFsbCB0aGUgQ2xvdWRXYXRjaCBBZ2VudC5cbiAqIFRoZSBpbnN0YWxsZXIgaXMgZG93bmxvYWRlZCB2aWEgdGhlIEFtYXpvbiBTMyBBUEksIHRodXMsIHRoaXMgY29uc3RydWN0IGNhbiBiZSB1c2VkXG4gKiBvbiBpbnN0YW5jZXMgdGhhdCBoYXZlIG5vIGFjY2VzcyB0byB0aGUgaW50ZXJuZXQgYXMgbG9uZyBhcyB0aGUgVlBDIGNvbnRhaW5zXG4gKiBhbiBWUEMgR2F0ZXdheSBFbmRwb2ludCBmb3IgUzMgKCBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdnBjL2xhdGVzdC91c2VyZ3VpZGUvdnBjLWVuZHBvaW50cy1zMy5odG1sICkuXG4gKlxuICoge0BsaW5rIENsb3VkV2F0Y2hBZ2VudC5TS0lQX0NXQUdFTlRfVkFMSURBVElPTl9DVFhfVkFSfSAtIENvbnRleHQgdmFyaWFibGUgdG8gc2tpcCB2YWxpZGF0aW9uXG4gKiBvZiB0aGUgZG93bmxvYWRlZCBDbG91ZFdhdGNoIEFnZW50IGluc3RhbGxlciBpZiBzZXQgdG8gJ1RSVUUnLlxuICogV0FSTklORzogT25seSB1c2UgdGhpcyBpZiB5b3VyIGRlcGxveW1lbnRzIGFyZSBmYWlsaW5nIGR1ZSB0byBhIHZhbGlkYXRpb24gZmFpbHVyZSxcbiAqIGJ1dCB5b3UgaGF2ZSB2ZXJpZmllZCB0aGF0IHRoZSBmYWlsdXJlIGlzIGJlbmlnbi5cbiAqXG4gKiBSZXNvdXJjZXMgRGVwbG95ZWRcbiAqIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICogLSBTdHJpbmcgU1NNIFBhcmFtZXRlciBpbiBTeXN0ZW1zIE1hbmFnZXIgUGFyYW1ldGVyIFN0b3JlIHRvIHN0b3JlIHRoZSBjbG91ZHdhdGNoIGFnZW50IGNvbmZpZ3VyYXRpb247XG4gKiAtIEEgc2NyaXB0IEFzc2V0IHdoaWNoIGlzIHVwbG9hZGVkIHRvIFMzIGJ1Y2tldC5cbiAqXG4gKiBTZWN1cml0eSBDb25zaWRlcmF0aW9uc1xuICogLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4gKiAtIFVzaW5nIHRoaXMgY29uc3RydWN0IG9uIGFuIGluc3RhbmNlIHdpbGwgcmVzdWx0IGluIHRoYXQgaW5zdGFuY2UgZHluYW1pY2FsbHkgZG93bmxvYWRpbmcgYW5kIHJ1bm5pbmcgc2NyaXB0c1xuICogICBmcm9tIHlvdXIgQ0RLIGJvb3RzdHJhcCBidWNrZXQgd2hlbiB0aGF0IGluc3RhbmNlIGlzIGxhdW5jaGVkLiBZb3UgbXVzdCBsaW1pdCB3cml0ZSBhY2Nlc3MgdG8geW91ciBDREsgYm9vdHN0cmFwXG4gKiAgIGJ1Y2tldCB0byBwcmV2ZW50IGFuIGF0dGFja2VyIGZyb20gbW9kaWZ5aW5nIHRoZSBhY3Rpb25zIHBlcmZvcm1lZCBieSB0aGVzZSBzY3JpcHRzLiBXZSBzdHJvbmdseSByZWNvbW1lbmQgdGhhdFxuICogICB5b3UgZWl0aGVyIGVuYWJsZSBBbWF6b24gUzMgc2VydmVyIGFjY2VzcyBsb2dnaW5nIG9uIHlvdXIgQ0RLIGJvb3RzdHJhcCBidWNrZXQsIG9yIGVuYWJsZSBBV1MgQ2xvdWRUcmFpbCBvbiB5b3VyXG4gKiAgIGFjY291bnQgdG8gYXNzaXN0IGluIHBvc3QtaW5jaWRlbnQgYW5hbHlzaXMgb2YgY29tcHJvbWlzZWQgcHJvZHVjdGlvbiBlbnZpcm9ubWVudHMuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG91ZFdhdGNoQWdlbnQgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIC8qKlxuICAgKiBUaGUgY29udGV4dCB2YXJpYWJsZSB0byBpbmRpY2F0ZSB0aGF0IENsb3VkV2F0Y2ggYWdlbnQgaW5zdGFsbGVyIHZhbGlkYXRpb24gc2hvdWxkIGJlIHNraXBwZWQuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNLSVBfQ1dBR0VOVF9WQUxJREFUSU9OX0NUWF9WQVIgPSAnUkZES19TS0lQX0NXQUdFTlRfVkFMSURBVElPTic7XG5cbiAgLyoqXG4gICAqIFRoZSBmbGFnIGZvciBjb25maWd1cmVDbG91ZFdhdGNoQWdlbnQgc2NyaXB0IHRvIHNraXAgQ2xvdWRXYXRjaCBhZ2VudCBpbnN0YWxsZXIgdmFsaWRhdGlvbi5cbiAgICovXG4gIHByaXZhdGUgc3RhdGljIHJlYWRvbmx5IFNLSVBfQ1dBR0VOVF9WQUxJREFUSU9OX0ZMQUcgPSAnLXMnO1xuXG4gIC8qKlxuICAgKiBUaGUgZmxhZyBmb3IgY29uZmlndXJlQ2xvdWRXYXRjaEFnZW50IHNjcmlwdCB0byBza2lwIENsb3VkV2F0Y2ggYWdlbnQgaW5zdGFsbGVyIHZhbGlkYXRpb24uXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBJTlNUQUxMX0NXQUdFTlRfRkxBRyA9ICctaSc7XG5cbiAgLyoqXG4gICAqIEFuIFMzIHNjcmlwdCBhc3NldCB0aGF0IGNvbmZpZ3VyZXMgdGhlIENsb3VkV2F0Y2ggYWdlbnQuXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IGNvbmZpZ3VyYXRpb25TY3JpcHQ6IFNjcmlwdEFzc2V0O1xuXG4gIC8qKlxuICAgKiBBbiBBV1MgU3RyaW5nIFBhcmFtZXRlciBjcmVhdGVkIGZvciBzdG9yaW5nIHRoZSBjbG91ZHdhdGNoIGFnZW50IGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IHNzbVBhcmFtZXRlckZvckNvbmZpZzogU3RyaW5nUGFyYW1ldGVyO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDbG91ZFdhdGNoQWdlbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBzaG91bGRJbnN0YWxsQWdlbnQgPSBwcm9wcy5zaG91bGRJbnN0YWxsQWdlbnQgPz8gdHJ1ZTtcblxuICAgIC8vIENyZWF0ZSB0aGUgYXNzZXQgZm9yIHRoZSBjb25maWd1cmF0aW9uIHNjcmlwdFxuICAgIHRoaXMuY29uZmlndXJhdGlvblNjcmlwdCA9IFNjcmlwdEFzc2V0LmZyb21QYXRoQ29udmVudGlvbihzY29wZSwgJ0Nsb3VkV2F0Y2hDb25maWd1cmF0aW9uU2NyaXB0QXNzZXQnLCB7XG4gICAgICBvc1R5cGU6IHByb3BzLmhvc3Qub3NUeXBlLFxuICAgICAgYmFzZU5hbWU6ICdjb25maWd1cmVDbG91ZFdhdGNoQWdlbnQnLFxuICAgICAgcm9vdERpcjogcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL3NjcmlwdHMvJyksXG4gICAgfSk7XG5cbiAgICAvLyBDcmVhdGUgYSBuZXcgU1NNIFBhcmFtZXRlciBob2xkaW5nIHRoZSBqc29uIGNvbmZpZ3VyYXRpb25cbiAgICB0aGlzLnNzbVBhcmFtZXRlckZvckNvbmZpZyA9IG5ldyBTdHJpbmdQYXJhbWV0ZXIoc2NvcGUsICdTdHJpbmdQYXJhbWV0ZXInLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ2NvbmZpZyBmaWxlIGZvciBSZXBvc2l0b3J5IGxvZ3MgY29uZmlnJyxcbiAgICAgIHN0cmluZ1ZhbHVlOiBwcm9wcy5jbG91ZFdhdGNoQ29uZmlnLFxuICAgIH0pO1xuXG4gICAgdGhpcy5ncmFudFJlYWQocHJvcHMuaG9zdCk7XG4gICAgdGhpcy5jb25maWd1cmUoXG4gICAgICBwcm9wcy5ob3N0LFxuICAgICAgc2hvdWxkSW5zdGFsbEFnZW50LFxuICAgICAgdGhpcy5ub2RlLnRyeUdldENvbnRleHQoQ2xvdWRXYXRjaEFnZW50LlNLSVBfQ1dBR0VOVF9WQUxJREFUSU9OX0NUWF9WQVIpID09PSAnVFJVRScsXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcmVhZCBwZXJtaXNzaW9ucyB0byB0aGUgcHJpbmNpcGFsIG9uIHRoZSBhc3NldHMgYnVja2V0IGFuZCBwYXJhbWV0ZXIgc3RvcmUuXG4gICAqL1xuICBwcml2YXRlIGdyYW50UmVhZChncmFudGVlOiBJR3JhbnRhYmxlKTogdm9pZCB7XG4gICAgdGhpcy5jb25maWd1cmF0aW9uU2NyaXB0LmdyYW50UmVhZChncmFudGVlKTtcbiAgICB0aGlzLnNzbVBhcmFtZXRlckZvckNvbmZpZy5ncmFudFJlYWQoZ3JhbnRlZSk7XG4gIH1cblxuICAvKipcbiAgICogQ29uZmlndXJlcyB0aGUgQ2xvdWRXYXRjaCBBZ2VudCBvbiB0aGUgdGFyZ2V0IGhvc3QuXG4gICAqXG4gICAqIFRoaXMgaXMgZG9uZSBieSBhZGRpbmcgVXNlckRhdGEgY29tbWFuZHMgdG8gdGhlIHRhcmdldCBob3N0LlxuICAgKlxuICAgKiBAcGFyYW0gaG9zdCBUaGUgaG9zdCB0byBjb25maWd1cmUgdGhlIENsb3VkV2F0Y2ggYWdlbnQgb25cbiAgICogQHBhcmFtIHNob3VsZEluc3RhbGxBZ2VudCBBdHRlbXB0cyB0byBpbnN0YWxsIHRoZSBDbG91ZFdhdGNoIGFnZW50IG9uIHRoZSBpbnN0YW5jZSBpZiBzZXQgdG8gdHJ1ZS5cbiAgICogQHBhcmFtIHNraXBWYWxpZGF0aW9uIFNraXBzIHRoZSB2YWxpZGF0aW9uIG9mIHRoZSBDbG91ZFdhdGNoIGFnZW50IGluc3RhbGxlciBpZiBzZXQgdG8gdHJ1ZS5cbiAgICovXG4gIHByaXZhdGUgY29uZmlndXJlKFxuICAgIGhvc3Q6IElTY3JpcHRIb3N0LFxuICAgIHNob3VsZEluc3RhbGxBZ2VudDogYm9vbGVhbixcbiAgICBza2lwVmFsaWRhdGlvbjogYm9vbGVhbixcbiAgKSB7XG4gICAgY29uc3QgcmVnaW9uID0gU3RhY2sub2YodGhpcykucmVnaW9uO1xuICAgIGlmIChzaG91bGRJbnN0YWxsQWdlbnQpIHtcbiAgICAgIC8vIEdyYW50IGFjY2VzcyB0byB0aGUgcmVxdWlyZWQgQ2xvdWRXYXRjaCBBZ2VudCBhbmQgR1BHIGluc3RhbGxlciBmaWxlcy5cbiAgICAgIGNvbnN0IGNsb3VkV2F0Y2hBZ2VudEJ1Y2tldCA9IEJ1Y2tldC5mcm9tQnVja2V0QXJuKHRoaXMsICdDbG91ZFdhdGNoQWdlbnRCdWNrZXQnLCBgYXJuOmF3czpzMzo6OmFtYXpvbmNsb3Vkd2F0Y2gtYWdlbnQtJHtyZWdpb259YCk7XG4gICAgICBjbG91ZFdhdGNoQWdlbnRCdWNrZXQuZ3JhbnRSZWFkKGhvc3QpO1xuICAgICAgY29uc3QgZ3BnQnVja2V0ID0gQnVja2V0LmZyb21CdWNrZXRBcm4odGhpcywgJ0dwZ0J1Y2tldCcsIGBhcm46YXdzOnMzOjo6cmZkay1leHRlcm5hbC1kZXBlbmRlbmNpZXMtJHtyZWdpb259YCk7XG4gICAgICBncGdCdWNrZXQuZ3JhbnRSZWFkKGhvc3QpO1xuICAgIH1cblxuICAgIGNvbnN0IHNjcmlwdEFyZ3MgPSBbXTtcblxuICAgIC8vIEZsYWdzIG11c3QgYmUgc2V0IGJlZm9yZSBwb3NpdGlvbmFsIGFyZ3VtZW50cyBmb3Igc29tZSBzY3JpcHRzXG4gICAgaWYgKHNob3VsZEluc3RhbGxBZ2VudCkge1xuICAgICAgc2NyaXB0QXJncy5wdXNoKENsb3VkV2F0Y2hBZ2VudC5JTlNUQUxMX0NXQUdFTlRfRkxBRyk7XG4gICAgfVxuICAgIGlmIChza2lwVmFsaWRhdGlvbikge1xuICAgICAgc2NyaXB0QXJncy5wdXNoKENsb3VkV2F0Y2hBZ2VudC5TS0lQX0NXQUdFTlRfVkFMSURBVElPTl9GTEFHKTtcbiAgICB9XG5cbiAgICAvLyBUaGlzIGFzc3VtZXMgdGhhdCB0aGUgQ2xvdWRXYXRjaCBhZ2VudCBjb25zdHJ1Y3QgaXMgYWx3YXlzIHB1dCBpbiB0aGUgc2FtZSByZWdpb24gYXMgdGhlIGluc3RhbmNlIG9yIEFTR1xuICAgIC8vIHVzaW5nIGl0LCB3aGljaCBzaG91bGQgYWx3YXlzIGhvbGQgdHJ1ZS5cbiAgICBzY3JpcHRBcmdzLnB1c2gocmVnaW9uKTtcblxuICAgIHNjcmlwdEFyZ3MucHVzaCh0aGlzLnNzbVBhcmFtZXRlckZvckNvbmZpZy5wYXJhbWV0ZXJOYW1lKTtcblxuICAgIHRoaXMuY29uZmlndXJhdGlvblNjcmlwdC5leGVjdXRlT24oe1xuICAgICAgaG9zdCxcbiAgICAgIGFyZ3M6IHNjcmlwdEFyZ3MsXG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==