#!/usr/bin/env node
"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const url = require("url");
const util_1 = require("util");
const deadline_1 = require("../lib/deadline");
const version_provider_1 = require("../lib/lambdas/nodejs/lib/version-provider");
const args = process.argv.slice(2);
let deadlineInstallerURI = '';
let dockerRecipesURI = '';
let deadlineReleaseVersion = '';
let outputFolder = './stage';
let verbose = false;
// Parse command parameters
let n = 0;
while (n < args.length) {
    switch (args[n]) {
        case '-d':
        case '--deadlineInstallerURI':
            n++;
            deadlineInstallerURI = args[n];
            break;
        case '-c':
        case '--dockerRecipesURI':
            n++;
            dockerRecipesURI = args[n];
            break;
        case '-o':
        case '--output':
            n++;
            outputFolder = args[n];
            break;
        case '--verbose':
            verbose = true;
            break;
        default:
            if (!deadlineReleaseVersion) {
                deadlineReleaseVersion = args[n];
            }
            else {
                console.error(`Unexpected command parameter ${args[n]}`);
                process.exit(1);
            }
            break;
    }
    n++;
}
if (!fs.existsSync(outputFolder)) {
    fs.mkdirSync(outputFolder);
}
else if (fs.readdirSync(outputFolder).length > 0) {
    console.error('The target directory is not empty.');
    process.exit(1);
}
const handler = new version_provider_1.VersionProvider();
// populate installer URI
if (deadlineInstallerURI === '') {
    handler.getVersionUris({ platform: version_provider_1.Platform.linux, product: version_provider_1.Product.deadline, versionString: deadlineReleaseVersion })
        .then(result => {
        const installerVersion = result.get(version_provider_1.Platform.linux);
        if (installerVersion) {
            validateDeadlineVersion(`${installerVersion.MajorVersion}.${installerVersion.MinorVersion}.${installerVersion.ReleaseVersion}.${installerVersion.PatchVersion}`);
            const installerUrl = installerVersion.Uris.clientInstaller;
            if (installerUrl) {
                getDeadlineInstaller(installerUrl);
            }
        }
        else {
            console.error(`Deadline installer for version ${deadlineReleaseVersion} was not found.`);
            exitAndCleanup(1);
        }
    })
        .catch(error => {
        console.error(error.message);
        exitAndCleanup(error.code);
    });
}
else {
    getDeadlineInstaller(deadlineInstallerURI);
}
// populate docker recipe URI
if (dockerRecipesURI === '') {
    handler.getVersionUris({ platform: version_provider_1.Platform.linux, product: version_provider_1.Product.deadlineDocker, versionString: deadlineReleaseVersion })
        .then(result => {
        const installerVersion = result.get(version_provider_1.Platform.linux);
        if (installerVersion) {
            getDockerRecipe(installerVersion.Uris.bundle);
        }
        else {
            console.error(`Docker recipies for version ${deadlineReleaseVersion} was not found.`);
            exitAndCleanup(1);
        }
    })
        .catch(error => {
        console.error(error.message);
        exitAndCleanup(error.code);
    });
}
else {
    getDockerRecipe(dockerRecipesURI);
}
/**
 * Download Deadline installer
 *
 * @param deadlineInstallerUri - Specifies a URI pointing to the Deadline Linux Client installer. This currently supports S3 URIs.
 */
function getDeadlineInstaller(deadlineInstallerUri) {
    const deadlineInstallerURL = new url.URL(deadlineInstallerUri);
    if (deadlineInstallerURL.protocol !== 's3:') {
        console.error(`ERROR: Invalid URI protocol ("${deadlineInstallerURL.protocol}") for --deadlineInstallerURI. Only "s3:" URIs are supported.`);
        usage(1);
    }
    if (!validateBucketName(deadlineInstallerURL.hostname)) {
        usage(1);
    }
    try {
        // Get Deadline client installer
        const deadlineInstallerPath = getFile({
            uri: deadlineInstallerURL,
            targetFolder: path.join(outputFolder, 'bin'),
            verbose,
        });
        // Make installer executable
        makeExecutable(deadlineInstallerPath);
    }
    catch (e) {
        let errorMsg;
        if (util_1.types.isNativeError(e)) {
            errorMsg = e.message;
        }
        else {
            errorMsg = e.toString();
        }
        console.error(`ERROR: ${errorMsg}`);
        exitAndCleanup(1);
    }
}
/**
 * Download and extract Docker recipe.
 *
 * @param dockerRecipesUri - Specifies a URI pointing to the Deadline Docker recipes. This currently supports S3 URIs.
 */
function getDockerRecipe(dockerRecipesUri) {
    const dockerRecipeURL = new url.URL(dockerRecipesUri);
    if (dockerRecipeURL.protocol !== 's3:') {
        console.error(`ERROR: Invalid URI protocol ("${dockerRecipeURL.protocol}") for --dockerRecipeURL. Only "s3:" URIs are supported.`);
        usage(1);
    }
    if (!validateBucketName(dockerRecipeURL.hostname)) {
        usage(1);
    }
    try {
        // Get Docker recipe
        getAndExtractArchive({
            uri: dockerRecipeURL,
            targetFolder: outputFolder,
            verbose,
            tarOptions: [`-x${verbose ? 'v' : ''}z`],
        });
    }
    catch (e) {
        let errorMsg;
        if (util_1.types.isNativeError(e)) {
            errorMsg = e.message;
        }
        else {
            errorMsg = e.toString();
        }
        console.error(`ERROR: ${errorMsg}`);
        exitAndCleanup(1);
    }
}
/**
 * Attempts to add UNIX executable permission bits to a file.
 *
 * Any errors are caught and:
 *  - more user-friendly error message is displayed
 *
 * @param filePath Path to make executable
 */
function makeExecutable(filePath) {
    try {
        let mode = fs.statSync(filePath).mode;
        // eslint-disable-next-line no-bitwise
        mode = mode | fs.constants.S_IXUSR | fs.constants.S_IXGRP | fs.constants.S_IXOTH;
        fs.chmodSync(filePath, mode);
    }
    catch (e) {
        let errorDetail;
        if (e instanceof Error) {
            errorDetail = e.message;
        }
        else {
            errorDetail = e.toString();
        }
        throw new Error(`Could not add executable permission to Deadline installer: ${errorDetail}`);
    }
}
/**
 * Validate name of S3 bucket
 *
 * @param s3BucketName - name of S3 bucket
 */
function validateBucketName(s3BucketName) {
    const regExpForS3Bucket = new RegExp('^([a-z]|(\\d(?!\\d{0,2}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})))\
([a-z\\d]|(\\.(?!(\\.|-)))|(-(?!\\.))){1,61}[a-z\\d]$');
    if (!regExpForS3Bucket.test(s3BucketName)) {
        console.error(`S3 bucket name '${s3BucketName}' has invalid format.\
    Please follow S3 bucket naming requirements:\
    https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html`);
        return false;
    }
    return true;
}
/**
 * Spawns a shell then executes the command within that shell.
 * Method will not return until the child process has fully closed.
 *
 * @param command - The command to run.
 * @param showCommand - If true shows which command will be spawned.
 */
function spawnCommand(command, showCommand, commandArgs = []) {
    if (showCommand) {
        console.info(`Command: ${command} ${commandArgs.join(' ')}`);
    }
    const result = child_process_1.spawnSync(command, commandArgs, {
        stdio: 'inherit',
    });
    return result.status === 0;
}
/**
 * Show help string for usage and exit with error code.
 *
 * @param errorCode - THe code of error that will be returned.
 */
function usage(errorCode) {
    console.info(`
Stages a Deadline release for building Docker container images with RFDK.
This tool requires that tar and the AWS CLI are installed.
The AWS CLI must be configured to authenticate using your AWS account. This can be done by configuring your default profile or with environment.
See https://docs.aws.amazon.com/cli/latest/userguide/cli-chap-configure.html for documentation on how to configure the AWS CLI.

Usage: stage-deadline [--output <output_dir>] [--verbose]
                      <deadline_release_version>
  OR
       stage-deadline [--output <output_dir>] [--verbose]
                      -d <deadline_installer_uri>
                      -c <deadline_recipes_uri>


Arguments:
    <deadline_release_version>
        Specifies the official release of Deadline that should be staged. This must be of the form "a.b.c.d", "a.b.c", "a.b" or "a".

        Note: The minimum supported deadline version is ${deadline_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}

    -d, --deadlineInstallerURI <deadline_installer_uri>
        Specifies a URI pointing to the Deadline Linux Client installer. This currently supports S3 URIs of the form:

        s3://thinkbox-installers/Deadline/10.1.x.y/Linux/DeadlineClient-10.1.x.y-linux-x64-installer.run

        If this argument is provided <deadline_release_version> will be ignored for Deadline Linux Client.

    -c, --dockerRecipesURI <deadline_recipes_uri>
        Specifies a URI pointing to the Deadline Docker recipes. This currently supports S3 URIs of the form:

        s3://thinkbox-installers/DeadlineDocker/10.1.x.y/DeadlineDocker-10.1.x.y.tar.gz

        If this argument is provided <deadline_release_version> will be ignored for Deadline Docker recipes.

Options:
    -o, --output <output_dir>
        Specifies a path to an output directory where Deadline will be staged. The default is to use a "stage"
        sub-directory of the working directory.

    --verbose
        Increases the verbosity of the output
  `.trimStart());
    exitAndCleanup(errorCode);
}
/**
 * Exit with error code and remove output folder.
 *
 * @param errorCode - THe code of error that will be returned.
 */
function exitAndCleanup(errorCode) {
    fs.rmdirSync(outputFolder, { recursive: true });
    process.exit(errorCode);
}
/**
 * Gets a file from a specified URI.
 *
 * This is currently limited to obtaining objects from S3.
 *
 * @param props Properties for fetching the file
 * @returns The path to the fetched file
 */
function getFile(props) {
    if (!fs.existsSync(props.targetFolder)) {
        fs.mkdirSync(props.targetFolder);
    }
    const cmdArgs = ['s3', 'cp'];
    if (!props.verbose) {
        cmdArgs.push('--quiet');
    }
    cmdArgs.push(props.uri.href);
    cmdArgs.push(props.targetFolder);
    const success = spawnCommand('aws', props.verbose, cmdArgs);
    if (!success) {
        throw new Error(`Could not fetch ${props.uri.href} (Are you authenticated with the AWS CLI?)`);
    }
    return path.join(props.targetFolder, uriFilename(props.uri));
}
/**
 * Return the filename portion of the URL.
 *
 * This is the right-most component (separated by / characters) of a URI's path.
 *
 * @param uri The source URI
 */
function uriFilename(uri) {
    const pathParts = uri.pathname.split('/');
    const fileName = pathParts[pathParts.length - 1];
    return fileName;
}
/**
 * Obtain and extract a tar archive.
 *
 * @param props Properties to specify how to get and extract the archive
 */
function getAndExtractArchive(props) {
    const { targetFolder, tarOptions } = props;
    const tarPath = getFile(props);
    const filesExtracted = spawnCommand('tar', props.verbose, [
        '-C',
        targetFolder,
        ...tarOptions,
        '-f',
        tarPath,
    ]);
    if (fs.existsSync(tarPath)) {
        fs.unlinkSync(tarPath);
    }
    // Exit with error if recipe wasn't extracted.
    if (!filesExtracted) {
        throw new Error(`File ${tarPath} has not been extracted successfully.`);
    }
}
function validateDeadlineVersion(versionString) {
    // Automatically populate the installer & recipe URI using the version, if it is provided.
    try {
        const version = deadline_1.Version.parse(versionString);
        if (version.isLessThan(deadline_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION)) {
            console.error(`ERROR: Unsupported Deadline Version ${version.toString()}. Minimum supported version is ${deadline_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION} \n`);
            usage(1);
        }
    }
    catch (e) {
        console.error(`ERROR: ${e.message} \n`);
        usage(1);
    }
}
//# sourceMappingURL=data:application/json;base64,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