# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['chromato']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'chromato',
    'version': '0.5.0',
    'description': '🍅 Fresh color utilities.',
    'long_description': '<div align="center">\n    <h1>🍅 chromato</h1>\n    <p>\n        <b>Fresh color utilities for Python</b>\n    </p>\n\n<!--[![PyPI version](https://badge.fury.io/py/chromato.svg)](https://pypi.org/project/chromato/) -->\n[![test](https://github.com/vikpe/chromato/actions/workflows/test.yml/badge.svg)](https://github.com/vikpe/chromato/actions/workflows/test.yml) [![codecov](https://codecov.io/gh/vikpe/chromato/branch/main/graph/badge.svg)](https://codecov.io/gh/vikpe/chromato)\n\n<br>\n\n</div>\n\n* [**Color spaces**](#color-spaces): CMYK, HEX, RGB, HSL, HSV\n* [**Color class**](#color-class): Convenience class for color manipulation\n* [**Operations**](#operations): shade, tone, tint, grayscale, invert, complement, add, subtract, multiply\n* [**Conversion**](#conversion): Convert any color space to any color space\n* [Parsing](#parsing)\n* [Validation](#validation)\n* (zero dependecies!)\n\n<br>\n\n# Install\n\n```shell\npip install chromato\n```\n\n<br>\n\n# Example usage\n\n**Color class**\n\n```python\nfrom chromato.spaces import Color\n\nred = Color(255, 0, 0)\n\nred.cmyk  # CMYK(c=0, m=100, y=100, k=0)\nred.hex   # HEX(ff0000)\nred.rgb   # RGB(r=255, g=0, b=0)\nred.hsl   # HSL(h=0, s=1, l=0.5)\nred.hsv   # HSV(h=0, s=1, v=1)\n```\n\n**Operations**\n\n```python\nfrom chromato import operations\nfrom chromato.spaces import RGB\n\nred = RGB(255, 0, 0)\nblue = RGB(0, 0, 255)\n\noperations.add(red, blue)        # RGB(r=255, g=0, b=255)\noperations.blend(red, blue).rgb  # RGB(r=128, g=0, b=128)\noperations.invert(red).rgb       # RGB(r=0, g=255, b=255)\noperations.tint(red, 0.1).rgb    # RGB(r=255, g=26, b=26)\n```\n\n**Conversion**\n\n```python\nfrom chromato import convert\n\nconvert.rgb_to_hex(255, 0, 0)  # HEX(ff0000)\nconvert.hex_to_rgb("ff0000")   # RGB(r=255, g=0, b=0)\nconvert.hex_to_cmyk("f0f")     # CMYK(c=0, m=100, y=0, k=0)\n```\n\n<br>\n\n# API\n\n## Color spaces\n\nName | Properties | Range\n---|---|---\nCMYK | `c`, `m`, `y`, `k` | `0-100`, `0-100`, `0-100`, `0-100`\nHEX | (is a string) | `000000-ffffff`\nHSL | `h`, `s`, `l` |  `0-1`, `0-1`, `0-1`\nHSV | `h`, `s`, `v` |  `0-1`, `0-1`, `0-1`\nRGB | `r`, `g`, `b` | `0-255`, `0-255`, `0-255`\n\n```python\nfrom chromato.spaces import CMYK, HEX, HSL, HSV, RGB\n\nred_cmyk = CMYK(0, 100, 100, 0)\nred_hex  = HEX("ff0000")\nred_hsl  = HSL(0, 1, 0.5)\nred_hsv  = HSV(0, 1, 1)\nred_rgb  = RGB(255, 0, 0)\n```\n\n## Color class\n\n**Properties**\n\n```python\nfrom chromato.spaces import Color\n\nred = Color(255, 0, 0)\n\nred.cmyk  # CMYK(c=0, m=100, y=100, k=0)\nred.hex   # HEX(ff0000)\nred.hsl   # HSL(h=0, s=1, l=0.5)\nred.hsv   # HSV(h=0, s=1, v=1)\nred.rgb   # RGB(r=255, g=0, b=0)\n```\n\n**Construct**\n\n```python\n# examples below are equal\nColor(255, 0, 0)\nColor((255, 0, 0))\nColor([255, 0, 0])\nColor({"r": 255, "g": 0, "b": 0})\nColor("ff0000")\nColor("ff0")\nColor(Color(255, 0, 0))\nColor(RGB(255, 0, 0))\nColor(HEX("ff0"))\nColor(HSV(0, 1, 1))\nColor(HSL(0, 1, 0.5))\nColor(CMYK(0, 100, 100, 0))\n```\n\n## Operations\n\nEach operation take one or several color values and returns a `Color` instance.\n\nOperation | Description\n--- | ---\n**`add`**`(color1, color2)` | Add colors\n**`blend`**`(color1, color2, factor)` | Blend/mix colors\n**`complement`**`(color)` | Complementary color\n**`hsv_mod`**`(color, hue_shift, saturation_shift, value_shift)` | HSV modification\n**`invert`**`(color)` | Invert color\n**`multiply`**`(color1, color2)` | Multiply colors\n**`shade`**`(color, factor)` | Increase darkness (blend with black)\n**`subtract`**`(color1, color2)` | Subtract colors\n**`tint`**`(color, factor)` | Increase lightness (blend with white)\n**`tone`**`(color, factor)` | Reduce colorfullness (blend with gray)\n\n* **Arguments**: `color<any>`, `factor <float> [0-1]`\n* **Returns**: instance of `Color`\n\n**Example**\n\n```python\nfrom chromato.spaces import RGB\nfrom chromato.operations import blend, invert\n\nwhite = RGB(255, 255, 255)\nblack = RGB(0, 0, 0)\n\nblend(white, black).rgb       # RGB(r=128, g=128, b=128)\nblend(white, black, 0.2).rgb  # RGB(r=204, g=204, b=204)\nblend(white, black, 0.8).rgb  # RGB(r=51, g=51, b=51)\n\ninvert(black).rgb             # RGB(r=255, g=255, b=255)\ninvert((255, 0, 0)).cmyk      # CMYK(c=0, m=100, y=100, k=0)\ninvert("ff0000").hex          # HEX(00ffff)\n```\n\n## Conversion\n\nConvert any color space to any color space.\n\n🔀 | RGB | HEX | CMYK | HSL | HSV\n--- | --- | --- | --- | --- | ---\nRGB | <!-- null --> | `hex_to_rgb` | `cmyk_to_rgb` | `hsl_to_rgb` | `hsv_to_rgb`\nHEX | `rgb_to_hex` | <!-- null --> | `cmyk_to_hex` | `hsl_to_hex` | `hsv_to_hex`\nCMYK | `rgb_to_cmyk` | `hex_to_cmyk` | <!-- null --> | `hsl_to_cmyk` | `hsv_to_cmyk`\nHSL | `rgb_to_hsl` | `hex_to_hsl` | `cmyk_to_hsl` | <!-- null --> | `hsv_to_hsl`\nHSV | `rgb_to_hsv` | `hex_to_hsv` | `cmyk_to_hsv` | `hsl_to_hsv` | <!-- null -->\n\n**Example**\n\n```python\nfrom chromato import convert\n\nconvert.rgb_to_cmyk(255, 0, 0)  # CMYK(c=0, m=100, y=100, k=0)\nconvert.rgb_to_hex(255, 0, 0)   # HEX(ff0000)\nconvert.rgb_to_hex(255, 0, 0)   # HEX(ff0000)\nconvert.rgb_to_hsl(255, 0, 0)   # HSL(h=0, s=1, l=0.5)\nconvert.rgb_to_hsv(255, 0, 0)   # HSV(h=0, s=1, v=1)\n```\n\n## Parsing\n\nEach parse function takes any kind of value and tries to parse it.\n\nFunction | Returns | Description\n--- | --- | ---\n**`parse_cmyk`**`(value)` | `tuple(c,m,y,k)` |  Parse value as CMYK\n**`parse_hex`**`(value)` | `str(hex)` |  Parse value as HEX\n**`parse_hsl`**`(value)` | `tuple(h,l,s)` |  Parse value as HSL\n**`parse_hsv`**`(value)` | `tuple(h,s,v)` |  Parse value as HSV\n**`parse_rgb`**`(value)` | `tuple(r,g,b)` |  Parse value as RGB\n\n**Example**\n\n```python\nfrom chromato import parse\nfrom chromato.spaces import Color, HEX, RGB\n\nparse.parse_hex("f")                 # "ffffff"\nparse.parse_hex("f60")               # "ff6600"\nparse.parse_hex("ff6600")            # "ff6600"\nparse.parse_hex(" #ff6600 ")         # "ff6600"\nparse.parse_hex(333)                 # "333333"\nparse.parse_hex(HEX("ff6600"))       # "ff6600"\nparse.parse_hex(RGB(255, 102, 0))    # "ff6600"\nparse.parse_hex(Color(255, 102, 0))  # "ff6600"\n```\n\n## Validation\n\nEach validation function validates type and range. Returns `True`/`False`.\n\n```python\nfrom chromato import validation\n\nvalidation.is_cmyk(c, m, y, k)\nvalidation.is_hex(hex)\nvalidation.is_hsl(h, s, l)\nvalidation.is_hsv(h, s, v)\nvalidation.is_rgb(r, g, b)\n```\n\n<br>\n\n# Development\n\n**Setup**\n\n1. Install [poetry](https://github.com/python-poetry/poetry)\n2. `poetry install`\n\n**Tests**\n\nRun tests on changes in source code or tests.\n\n```shell\npoetry run ptw --clear --runner "poetry run pytest --cov -vv" \n```\n\n**Code formatting (black)**\n\n```shell\npoetry run black . \n```\n',
    'author': 'Viktor Persson',
    'author_email': 'viktor.persson@arcsin.se',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/vikpe/chromato',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
