# ---------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# ---------------------------------------------------------

"""Telemetry logger helper class"""

import logging
import sys
import copy
import platform

try:
    from azureml.telemetry import get_telemetry_log_handler
    from azureml.telemetry.activity import log_activity as _log_activity, ActivityType
    from azureml.telemetry.logging_handler import AppInsightsLoggingHandler
    from azureml.telemetry.contracts import (RequiredFields, StandardFields, ExtensionFields, Event)
    from azureml import telemetry

    telemetry_imported = True
    DEFAULT_ACTIVITY_TYPE = ActivityType.INTERNALCALL
except ImportError:
    telemetry_imported = False
    DEFAULT_ACTIVITY_TYPE = "InternalCall"

try:
    from azureml._base_sdk_common import __version__
except ImportError:
    __version__ = "default"

from .constants import REGION_KEY, SUBSCRIPTION_KEY, TELEMETRY_COMPONENT_NAME, TELEMETRY_EVENT_ID_KEY, WORKSPACE_KEY
from .telemetry_logger_context_formatter import TelemetryLoggerContextFormatter
from .telemetry_logger_context_filter import TelemetryLoggerContextFilter


class NullContextManager(object):
    """A class for null context manager"""

    def __init__(self, dummy_resource=None):
        self.dummy_resource = dummy_resource

    def __enter__(self):
        return self.dummy_resource

    def __exit__(self, *args):
        pass


class TelemetryLogger:
    """A class for telemetry logger"""

    @staticmethod
    def get_telemetry_logger(name, component_name=TELEMETRY_COMPONENT_NAME, stream_handler_verbosity=logging.INFO):
        """
        gets the telemetry logger
        :param name: name of the logger
        :type name: str
        :param component_name: telemetry component name
        :type component_name: str
        :param stream_handler_verbosity: verbosity for stream handler
        :type stream_handler_verbosity: int
        :return: logger
        :rtype: logging.Logger
        """
        logger = logging.getLogger(__name__).getChild(name)
        logger.propagate = False
        logger.setLevel(logging.INFO)

        if telemetry_imported:
            if not TelemetryLogger._found_handler(logger, AppInsightsLoggingHandler):
                logger.addHandler(get_telemetry_log_handler(component_name=component_name))

        if not TelemetryLogger._found_handler(logger, logging.StreamHandler):
            stream_handler = logging.StreamHandler(sys.stdout)
            stream_handler.setFormatter(TelemetryLoggerContextFormatter())
            stream_handler.setLevel(stream_handler_verbosity)
            logger.addHandler(stream_handler)

        context = {'sdk_version': __version__, 'telemetry_component_name': component_name}
        logger.addFilter(TelemetryLoggerContextFilter(context))

        return logger

    @staticmethod
    def log_activity(logger, activity_name, activity_type=DEFAULT_ACTIVITY_TYPE,
                     custom_dimensions=None, telemetry_event_name=None):
        """
        the wrapper of log_activity from azureml-telemetry
        :param logger: the logger object
        :type logger: logging.Logger
        :param activity_name: the name of the activity which should be unique per the wrapped logical code block
        :type activity_name: str
        :param activity_type: the type of the activity
        :type activity_type: str
        :param custom_dimensions: custom properties of the activity
        :type custom_dimensions: dict
        :param telemetry_event_name: name for telemetry events in required format
        :type custom_dimensions: str
        """
        if telemetry_imported:
            # integrate telemetry event logging if telemetry event name is provided.
            if telemetry_event_name:
                # log activity as telemetry event
                event_id = TelemetryLogger.log_telemetry_event(telemetry_event_name, **custom_dimensions)
                # event id is generated by telemetry, store it in activity in case join is needed in future.
                custom_dimensions[TELEMETRY_EVENT_ID_KEY] = event_id

            # log activity
            return _log_activity(logger, activity_name, activity_type, custom_dimensions)
        else:
            return NullContextManager(dummy_resource=logger)

    @staticmethod
    def log_telemetry_event(event_name, **kwargs):
        """
        :param event_name: name of the event
        :type event_name: str
        :param kwargs: a list of the key/value pairs which will be stored in event
        :type kwargs: dict
        """
        req = RequiredFields()
        std = StandardFields()
        dct = copy.deepcopy(kwargs)

        req.client_type = 'SDK'
        req.client_version = __version__

        std.client_os = platform.system()

        if WORKSPACE_KEY in kwargs:
            req.workspace_id = kwargs[WORKSPACE_KEY]
            dct.pop(WORKSPACE_KEY)
        if SUBSCRIPTION_KEY in kwargs:
            req.subscription_id = kwargs[SUBSCRIPTION_KEY]
            dct.pop(SUBSCRIPTION_KEY)
        if REGION_KEY in kwargs:
            std.workspace_region = kwargs[REGION_KEY]
            dct.pop(REGION_KEY)

        ext = ExtensionFields(**dct)
        event = Event(name=event_name, required_fields=req, standard_fields=std, extension_fields=ext)
        logger = telemetry.get_event_logger()
        logger.log_event(event)

        return event.required_fields.event_id

    @staticmethod
    def _found_handler(logger, handle_name):
        """
        checks logger for the given handler and returns the found status
        :param logger: Logger
        :param handle_name: handler name
        :return: boolean: True if found else False
        """
        for log_handler in logger.handlers:
            if isinstance(log_handler, handle_name):
                return True

        return False
