'''
# Amazon CloudFront Construct Library

<!--BEGIN STABILITY BANNER-->---


![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)

![cdk-constructs: Stable](https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge)

---
<!--END STABILITY BANNER-->

Amazon CloudFront is a web service that speeds up distribution of your static and dynamic web content, such as .html, .css, .js, and image files, to
your users. CloudFront delivers your content through a worldwide network of data centers called edge locations. When a user requests content that
you're serving with CloudFront, the user is routed to the edge location that provides the lowest latency, so that content is delivered with the best
possible performance.

## Distribution API

The `Distribution` API is currently being built to replace the existing `CloudFrontWebDistribution` API. The `Distribution` API is optimized for the
most common use cases of CloudFront distributions (e.g., single origin and behavior, few customizations) while still providing the ability for more
advanced use cases. The API focuses on simplicity for the common use cases, and convenience methods for creating the behaviors and origins necessary
for more complex use cases.

### Creating a distribution

CloudFront distributions deliver your content from one or more origins; an origin is the location where you store the original version of your
content. Origins can be created from S3 buckets or a custom origin (HTTP server). Constructs to define origins are in the `@aws-cdk/aws-cloudfront-origins` module.

Each distribution has a default behavior which applies to all requests to that distribution, and routes requests to a primary origin.
Additional behaviors may be specified for an origin with a given URL path pattern. Behaviors allow routing with multiple origins,
controlling which HTTP methods to support, whether to require users to use HTTPS, and what query strings or cookies to forward to your origin,
among other settings.

#### From an S3 Bucket

An S3 bucket can be added as an origin. If the bucket is configured as a website endpoint, the distribution can use S3 redirects and S3 custom error
documents.

```python
# Creates a distribution from an S3 bucket.
my_bucket = s3.Bucket(self, "myBucket")
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=origins.S3Origin(my_bucket))
)
```

The above will treat the bucket differently based on if `IBucket.isWebsite` is set or not. If the bucket is configured as a website, the bucket is
treated as an HTTP origin, and the built-in S3 redirects and error pages can be used. Otherwise, the bucket is handled as a bucket origin and
CloudFront's redirect and error handling will be used. In the latter case, the Origin will create an origin access identity and grant it access to the
underlying bucket. This can be used in conjunction with a bucket that is not public to require that your users access your content using CloudFront
URLs and not S3 URLs directly.

#### ELBv2 Load Balancer

An Elastic Load Balancing (ELB) v2 load balancer may be used as an origin. In order for a load balancer to serve as an origin, it must be publicly
accessible (`internetFacing` is true). Both Application and Network load balancers are supported.

```python
# Creates a distribution from an ELBv2 load balancer
# vpc: ec2.Vpc

# Create an application load balancer in a VPC. 'internetFacing' must be 'true'
# for CloudFront to access the load balancer and use it as an origin.
lb = elbv2.ApplicationLoadBalancer(self, "LB",
    vpc=vpc,
    internet_facing=True
)
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=origins.LoadBalancerV2Origin(lb))
)
```

#### From an HTTP endpoint

Origins can also be created from any other HTTP endpoint, given the domain name, and optionally, other origin properties.

```python
# Creates a distribution from an HTTP endpoint
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=origins.HttpOrigin("www.example.com"))
)
```

### Domain Names and Certificates

When you create a distribution, CloudFront assigns a domain name for the distribution, for example: `d111111abcdef8.cloudfront.net`; this value can
be retrieved from `distribution.distributionDomainName`. CloudFront distributions use a default certificate (`*.cloudfront.net`) to support HTTPS by
default. If you want to use your own domain name, such as `www.example.com`, you must associate a certificate with your distribution that contains
your domain name, and provide one (or more) domain names from the certificate for the distribution.

The certificate must be present in the AWS Certificate Manager (ACM) service in the US East (N. Virginia) region; the certificate
may either be created by ACM, or created elsewhere and imported into ACM. When a certificate is used, the distribution will support HTTPS connections
from SNI only and a minimum protocol version of TLSv1.2_2021 if the `@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021` feature flag is set, and TLSv1.2_2019 otherwise.

```python
# To use your own domain name in a Distribution, you must associate a certificate
import aws_cdk.aws_certificatemanager as acm
import aws_cdk.aws_route53 as route53

# hosted_zone: route53.HostedZone

# my_bucket: s3.Bucket

my_certificate = acm.DnsValidatedCertificate(self, "mySiteCert",
    domain_name="www.example.com",
    hosted_zone=hosted_zone
)
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=origins.S3Origin(my_bucket)),
    domain_names=["www.example.com"],
    certificate=my_certificate
)
```

However, you can customize the minimum protocol version for the certificate while creating the distribution using `minimumProtocolVersion` property.

```python
# Create a Distribution with a custom domain name and a minimum protocol version.
# my_bucket: s3.Bucket

cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=origins.S3Origin(my_bucket)),
    domain_names=["www.example.com"],
    minimum_protocol_version=cloudfront.SecurityPolicyProtocol.TLS_V1_2016,
    ssl_support_method=cloudfront.SSLMethod.SNI
)
```

### Multiple Behaviors & Origins

Each distribution has a default behavior which applies to all requests to that distribution; additional behaviors may be specified for a
given URL path pattern. Behaviors allow routing with multiple origins, controlling which HTTP methods to support, whether to require users to
use HTTPS, and what query strings or cookies to forward to your origin, among others.

The properties of the default behavior can be adjusted as part of the distribution creation. The following example shows configuring the HTTP
methods and viewer protocol policy of the cache.

```python
# Create a Distribution with configured HTTP methods and viewer protocol policy of the cache.
# my_bucket: s3.Bucket

my_web_distribution = cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(my_bucket),
        allowed_methods=cloudfront.AllowedMethods.ALLOW_ALL,
        viewer_protocol_policy=cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
    )
)
```

Additional behaviors can be specified at creation, or added after the initial creation. Each additional behavior is associated with an origin,
and enable customization for a specific set of resources based on a URL path pattern. For example, we can add a behavior to `myWebDistribution` to
override the default viewer protocol policy for all of the images.

```python
# Add a behavior to a Distribution after initial creation.
# my_bucket: s3.Bucket
# my_web_distribution: cloudfront.Distribution

my_web_distribution.add_behavior("/images/*.jpg", origins.S3Origin(my_bucket),
    viewer_protocol_policy=cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
)
```

These behaviors can also be specified at distribution creation time.

```python
# Create a Distribution with additional behaviors at creation time.
# my_bucket: s3.Bucket

bucket_origin = origins.S3Origin(my_bucket)
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(
        origin=bucket_origin,
        allowed_methods=cloudfront.AllowedMethods.ALLOW_ALL,
        viewer_protocol_policy=cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
    ),
    additional_behaviors={
        "/images/*.jpg": cloudfront.BehaviorOptions(
            origin=bucket_origin,
            viewer_protocol_policy=cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
        )
    }
)
```

### Customizing Cache Keys and TTLs with Cache Policies

You can use a cache policy to improve your cache hit ratio by controlling the values (URL query strings, HTTP headers, and cookies)
that are included in the cache key, and/or adjusting how long items remain in the cache via the time-to-live (TTL) settings.
CloudFront provides some predefined cache policies, known as managed policies, for common use cases. You can use these managed policies,
or you can create your own cache policy that’s specific to your needs.
See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html for more details.

```python
# Using an existing cache policy for a Distribution
# bucket_origin: origins.S3Origin

cloudfront.Distribution(self, "myDistManagedPolicy",
    default_behavior=cloudfront.BehaviorOptions(
        origin=bucket_origin,
        cache_policy=cloudfront.CachePolicy.CACHING_OPTIMIZED
    )
)
```

```python
# Creating a custom cache policy for a Distribution -- all parameters optional
# bucket_origin: origins.S3Origin

my_cache_policy = cloudfront.CachePolicy(self, "myCachePolicy",
    cache_policy_name="MyPolicy",
    comment="A default policy",
    default_ttl=Duration.days(2),
    min_ttl=Duration.minutes(1),
    max_ttl=Duration.days(10),
    cookie_behavior=cloudfront.CacheCookieBehavior.all(),
    header_behavior=cloudfront.CacheHeaderBehavior.allow_list("X-CustomHeader"),
    query_string_behavior=cloudfront.CacheQueryStringBehavior.deny_list("username"),
    enable_accept_encoding_gzip=True,
    enable_accept_encoding_brotli=True
)
cloudfront.Distribution(self, "myDistCustomPolicy",
    default_behavior=cloudfront.BehaviorOptions(
        origin=bucket_origin,
        cache_policy=my_cache_policy
    )
)
```

### Customizing Origin Requests with Origin Request Policies

When CloudFront makes a request to an origin, the URL path, request body (if present), and a few standard headers are included.
Other information from the viewer request, such as URL query strings, HTTP headers, and cookies, is not included in the origin request by default.
You can use an origin request policy to control the information that’s included in an origin request.
CloudFront provides some predefined origin request policies, known as managed policies, for common use cases. You can use these managed policies,
or you can create your own origin request policy that’s specific to your needs.
See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html for more details.

```python
# Using an existing origin request policy for a Distribution
# bucket_origin: origins.S3Origin

cloudfront.Distribution(self, "myDistManagedPolicy",
    default_behavior=cloudfront.BehaviorOptions(
        origin=bucket_origin,
        origin_request_policy=cloudfront.OriginRequestPolicy.CORS_S3_ORIGIN
    )
)
```

```python
# Creating a custom origin request policy for a Distribution -- all parameters optional
# bucket_origin: origins.S3Origin

my_origin_request_policy = cloudfront.OriginRequestPolicy(self, "OriginRequestPolicy",
    origin_request_policy_name="MyPolicy",
    comment="A default policy",
    cookie_behavior=cloudfront.OriginRequestCookieBehavior.none(),
    header_behavior=cloudfront.OriginRequestHeaderBehavior.all("CloudFront-Is-Android-Viewer"),
    query_string_behavior=cloudfront.OriginRequestQueryStringBehavior.allow_list("username")
)

cloudfront.Distribution(self, "myDistCustomPolicy",
    default_behavior=cloudfront.BehaviorOptions(
        origin=bucket_origin,
        origin_request_policy=my_origin_request_policy
    )
)
```

### Customizing Response Headers with Response Headers Policies

You can configure CloudFront to add one or more HTTP headers to the responses that it sends to viewers (web browsers or other clients), without making any changes to the origin or writing any code.
To specify the headers that CloudFront adds to HTTP responses, you use a response headers policy. CloudFront adds the headers regardless of whether it serves the object from the cache or has to retrieve the object from the origin. If the origin response includes one or more of the headers that’s in a response headers policy, the policy can specify whether CloudFront uses the header it received from the origin or overwrites it with the one in the policy.
See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/adding-response-headers.html

```python
# Using an existing managed response headers policy
# bucket_origin: origins.S3Origin

cloudfront.Distribution(self, "myDistManagedPolicy",
    default_behavior=cloudfront.BehaviorOptions(
        origin=bucket_origin,
        response_headers_policy=cloudfront.ResponseHeadersPolicy.CORS_ALLOW_ALL_ORIGINS
    )
)

# Creating a custom response headers policy -- all parameters optional
my_response_headers_policy = cloudfront.ResponseHeadersPolicy(self, "ResponseHeadersPolicy",
    response_headers_policy_name="MyPolicy",
    comment="A default policy",
    cors_behavior=cloudfront.ResponseHeadersCorsBehavior(
        access_control_allow_credentials=False,
        access_control_allow_headers=["X-Custom-Header-1", "X-Custom-Header-2"],
        access_control_allow_methods=["GET", "POST"],
        access_control_allow_origins=["*"],
        access_control_expose_headers=["X-Custom-Header-1", "X-Custom-Header-2"],
        access_control_max_age=Duration.seconds(600),
        origin_override=True
    ),
    custom_headers_behavior=cloudfront.ResponseCustomHeadersBehavior(
        custom_headers=[cloudfront.ResponseCustomHeader(header="X-Amz-Date", value="some-value", override=True), cloudfront.ResponseCustomHeader(header="X-Amz-Security-Token", value="some-value", override=False)
        ]
    ),
    security_headers_behavior=cloudfront.ResponseSecurityHeadersBehavior(
        content_security_policy=cloudfront.ResponseHeadersContentSecurityPolicy(content_security_policy="default-src https:;", override=True),
        content_type_options=cloudfront.ResponseHeadersContentTypeOptions(override=True),
        frame_options=cloudfront.ResponseHeadersFrameOptions(frame_option=cloudfront.HeadersFrameOption.DENY, override=True),
        referrer_policy=cloudfront.ResponseHeadersReferrerPolicy(referrer_policy=cloudfront.HeadersReferrerPolicy.NO_REFERRER, override=True),
        strict_transport_security=cloudfront.ResponseHeadersStrictTransportSecurity(access_control_max_age=Duration.seconds(600), include_subdomains=True, override=True),
        xss_protection=cloudfront.ResponseHeadersXSSProtection(protection=True, mode_block=True, report_uri="https://example.com/csp-report", override=True)
    )
)
cloudfront.Distribution(self, "myDistCustomPolicy",
    default_behavior=cloudfront.BehaviorOptions(
        origin=bucket_origin,
        response_headers_policy=my_response_headers_policy
    )
)
```

### Validating signed URLs or signed cookies with Trusted Key Groups

CloudFront Distribution supports validating signed URLs or signed cookies using key groups.
When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed
cookies for all requests that match the cache behavior.

```python
# Validating signed URLs or signed cookies with Trusted Key Groups

# public key in PEM format
# public_key: str

pub_key = cloudfront.PublicKey(self, "MyPubKey",
    encoded_key=public_key
)

key_group = cloudfront.KeyGroup(self, "MyKeyGroup",
    items=[pub_key
    ]
)

cloudfront.Distribution(self, "Dist",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.HttpOrigin("www.example.com"),
        trusted_key_groups=[key_group
        ]
    )
)
```

### Lambda@Edge

Lambda@Edge is an extension of AWS Lambda, a compute service that lets you execute
functions that customize the content that CloudFront delivers. You can author Node.js
or Python functions in the US East (N. Virginia) region, and then execute them in AWS
locations globally that are closer to the viewer, without provisioning or managing servers.
Lambda@Edge functions are associated with a specific behavior and event type. Lambda@Edge
can be used to rewrite URLs, alter responses based on headers or cookies, or authorize
requests based on headers or authorization tokens.

The following shows a Lambda@Edge function added to the default behavior and triggered
on every request:

```python
# my_bucket: s3.Bucket
# A Lambda@Edge function added to default behavior of a Distribution
# and triggered on every request
my_func = cloudfront.experimental.EdgeFunction(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(my_bucket),
        edge_lambdas=[cloudfront.EdgeLambda(
            function_version=my_func.current_version,
            event_type=cloudfront.LambdaEdgeEventType.VIEWER_REQUEST
        )
        ]
    )
)
```

> **Note:** Lambda@Edge functions must be created in the `us-east-1` region, regardless of the region of the CloudFront distribution and stack.
> To make it easier to request functions for Lambda@Edge, the `EdgeFunction` construct can be used.
> The `EdgeFunction` construct will automatically request a function in `us-east-1`, regardless of the region of the current stack.
> `EdgeFunction` has the same interface as `Function` and can be created and used interchangeably.
> Please note that using `EdgeFunction` requires that the `us-east-1` region has been bootstrapped.
> See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html for more about bootstrapping regions.

If the stack is in `us-east-1`, a "normal" `lambda.Function` can be used instead of an `EdgeFunction`.

```python
# Using a lambda Function instead of an EdgeFunction for stacks in `us-east-`.
my_func = lambda_.Function(self, "MyFunction",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
)
```

If the stack is not in `us-east-1`, and you need references from different applications on the same account,
you can also set a specific stack ID for each Lambda@Edge.

```python
# Setting stackIds for EdgeFunctions that can be referenced from different applications
# on the same account.
my_func1 = cloudfront.experimental.EdgeFunction(self, "MyFunction1",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler1")),
    stack_id="edge-lambda-stack-id-1"
)

my_func2 = cloudfront.experimental.EdgeFunction(self, "MyFunction2",
    runtime=lambda_.Runtime.NODEJS_14_X,
    handler="index.handler",
    code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler2")),
    stack_id="edge-lambda-stack-id-2"
)
```

Lambda@Edge functions can also be associated with additional behaviors,
either at or after Distribution creation time.

```python
# Associating a Lambda@Edge function with additional behaviors.

# my_func: cloudfront.experimental.EdgeFunction
# assigning at Distribution creation
# my_bucket: s3.Bucket

my_origin = origins.S3Origin(my_bucket)
my_distribution = cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=my_origin),
    additional_behaviors={
        "images/*": cloudfront.BehaviorOptions(
            origin=my_origin,
            edge_lambdas=[cloudfront.EdgeLambda(
                function_version=my_func.current_version,
                event_type=cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                include_body=True
            )
            ]
        )
    }
)

# assigning after creation
my_distribution.add_behavior("images/*", my_origin,
    edge_lambdas=[cloudfront.EdgeLambda(
        function_version=my_func.current_version,
        event_type=cloudfront.LambdaEdgeEventType.VIEWER_RESPONSE
    )
    ]
)
```

Adding an existing Lambda@Edge function created in a different stack to a CloudFront distribution.

```python
# Adding an existing Lambda@Edge function created in a different stack
# to a CloudFront distribution.
# s3_bucket: s3.Bucket

function_version = lambda_.Version.from_version_arn(self, "Version", "arn:aws:lambda:us-east-1:123456789012:function:functionName:1")

cloudfront.Distribution(self, "distro",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(s3_bucket),
        edge_lambdas=[cloudfront.EdgeLambda(
            function_version=function_version,
            event_type=cloudfront.LambdaEdgeEventType.VIEWER_REQUEST
        )
        ]
    )
)
```

### CloudFront Function

You can also deploy CloudFront functions and add them to a CloudFront distribution.

```python
# s3_bucket: s3.Bucket
# Add a cloudfront Function to a Distribution
cf_function = cloudfront.Function(self, "Function",
    code=cloudfront.FunctionCode.from_inline("function handler(event) { return event.request }")
)
cloudfront.Distribution(self, "distro",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(s3_bucket),
        function_associations=[cloudfront.FunctionAssociation(
            function=cf_function,
            event_type=cloudfront.FunctionEventType.VIEWER_REQUEST
        )]
    )
)
```

It will auto-generate the name of the function and deploy it to the `live` stage.

Additionally, you can load the function's code from a file using the `FunctionCode.fromFile()` method.

### Logging

You can configure CloudFront to create log files that contain detailed information about every user request that CloudFront receives.
The logs can go to either an existing bucket, or a bucket will be created for you.

```python
# Configure logging for Distributions

# Simplest form - creates a new bucket and logs to it.
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=origins.HttpOrigin("www.example.com")),
    enable_logging=True
)

# You can optionally log to a specific bucket, configure whether cookies are logged, and give the log files a prefix.
cloudfront.Distribution(self, "myDist",
    default_behavior=cloudfront.BehaviorOptions(origin=origins.HttpOrigin("www.example.com")),
    enable_logging=True,  # Optional, this is implied if logBucket is specified
    log_bucket=s3.Bucket(self, "LogBucket"),
    log_file_prefix="distribution-access-logs/",
    log_includes_cookies=True
)
```

### Importing Distributions

Existing distributions can be imported as well; note that like most imported constructs, an imported distribution cannot be modified.
However, it can be used as a reference for other higher-level constructs.

```python
# Using a reference to an imported Distribution
distribution = cloudfront.Distribution.from_distribution_attributes(self, "ImportedDist",
    domain_name="d111111abcdef8.cloudfront.net",
    distribution_id="012345ABCDEF"
)
```

## Migrating from the original CloudFrontWebDistribution to the newer Distribution construct

It's possible to migrate a distribution from the original to the modern API.
The changes necessary are the following:

### The Distribution

Replace `new CloudFrontWebDistribution` with `new Distribution`. Some
configuration properties have been changed:

| Old API                        | New API                                                                                        |
|--------------------------------|------------------------------------------------------------------------------------------------|
| `originConfigs`                | `defaultBehavior`; use `additionalBehaviors` if necessary                                      |
| `viewerCertificate`            | `certificate`; use `domainNames` for aliases                                                   |
| `errorConfigurations`          | `errorResponses`                                                                               |
| `loggingConfig`                | `enableLogging`; configure with `logBucket` `logFilePrefix` and `logIncludesCookies`           |
| `viewerProtocolPolicy`         | removed; set on each behavior instead. default changed from `REDIRECT_TO_HTTPS` to `ALLOW_ALL` |

After switching constructs, you need to maintain the same logical ID for the underlying [CfnDistribution](https://docs.aws.amazon.com/cdk/api/v1/docs/@aws-cdk_aws-cloudfront.CfnDistribution.html) if you wish to avoid the deletion and recreation of your distribution.
To do this, use [escape hatches](https://docs.aws.amazon.com/cdk/v2/guide/cfn_layer.html) to override the logical ID created by the new Distribution construct with the logical ID created by the old construct.

Example:

```python
# source_bucket: s3.Bucket


my_distribution = cloudfront.Distribution(self, "MyCfWebDistribution",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(source_bucket)
    )
)
cfn_distribution = my_distribution.node.default_child
cfn_distribution.override_logical_id("MyDistributionCFDistribution3H55TI9Q")
```

### Behaviors

The modern API makes use of the [CloudFront Origins](https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_cloudfront_origins-readme.html) module to easily configure your origin. Replace your origin configuration with the relevant CloudFront Origins class. For example, here's a behavior with an S3 origin:

```python
# source_bucket: s3.Bucket
# oai: cloudfront.OriginAccessIdentity


cloudfront.CloudFrontWebDistribution(self, "MyCfWebDistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=source_bucket,
            origin_access_identity=oai
        ),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )
    ]
)
```

Becomes:

```python
# source_bucket: s3.Bucket


distribution = cloudfront.Distribution(self, "MyCfWebDistribution",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(source_bucket)
    )
)
```

In the original API all behaviors are defined in the `originConfigs` property. The new API is optimized for a single origin and behavior, so the default behavior and additional behaviors will be defined separately.

```python
# source_bucket: s3.Bucket
# oai: cloudfront.OriginAccessIdentity


cloudfront.CloudFrontWebDistribution(self, "MyCfWebDistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=source_bucket,
            origin_access_identity=oai
        ),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    ), cloudfront.SourceConfiguration(
        custom_origin_source=cloudfront.CustomOriginConfig(
            domain_name="MYALIAS"
        ),
        behaviors=[cloudfront.Behavior(path_pattern="/somewhere")]
    )
    ]
)
```

Becomes:

```python
# source_bucket: s3.Bucket


distribution = cloudfront.Distribution(self, "MyCfWebDistribution",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(source_bucket)
    ),
    additional_behaviors={
        "/somewhere": cloudfront.BehaviorOptions(
            origin=origins.HttpOrigin("MYALIAS")
        )
    }
)
```

### Certificates

If you are using an ACM certificate, you can pass the certificate directly to the `certificate` prop.
Any aliases used before in the `ViewerCertificate` class should be passed in to the `domainNames` prop in the modern API.

```python
import aws_cdk.aws_certificatemanager as acm
# certificate: acm.Certificate
# source_bucket: s3.Bucket


viewer_certificate = cloudfront.ViewerCertificate.from_acm_certificate(certificate,
    aliases=["MYALIAS"]
)

cloudfront.CloudFrontWebDistribution(self, "MyCfWebDistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=source_bucket
        ),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )
    ],
    viewer_certificate=viewer_certificate
)
```

Becomes:

```python
import aws_cdk.aws_certificatemanager as acm
# certificate: acm.Certificate
# source_bucket: s3.Bucket


distribution = cloudfront.Distribution(self, "MyCfWebDistribution",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(source_bucket)
    ),
    domain_names=["MYALIAS"],
    certificate=certificate
)
```

IAM certificates aren't directly supported by the new API, but can be easily configured through [escape hatches](https://docs.aws.amazon.com/cdk/v2/guide/cfn_layer.html)

```python
# source_bucket: s3.Bucket

viewer_certificate = cloudfront.ViewerCertificate.from_iam_certificate("MYIAMROLEIDENTIFIER",
    aliases=["MYALIAS"]
)

cloudfront.CloudFrontWebDistribution(self, "MyCfWebDistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=source_bucket
        ),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )
    ],
    viewer_certificate=viewer_certificate
)
```

Becomes:

```python
# source_bucket: s3.Bucket

distribution = cloudfront.Distribution(self, "MyCfWebDistribution",
    default_behavior=cloudfront.BehaviorOptions(
        origin=origins.S3Origin(source_bucket)
    ),
    domain_names=["MYALIAS"]
)

cfn_distribution = distribution.node.default_child

cfn_distribution.add_property_override("ViewerCertificate.IamCertificateId", "MYIAMROLEIDENTIFIER")
cfn_distribution.add_property_override("ViewerCertificate.SslSupportMethod", "sni-only")
```

### Other changes

A number of default settings have changed on the new API when creating a new distribution, behavior, and origin.
After making the major changes needed for the migration, run `cdk diff` to see what settings have changed.
If no changes are desired during migration, you will at the least be able to use [escape hatches](https://docs.aws.amazon.com/cdk/v2/guide/cfn_layer.html) to override what the CDK synthesizes, if you can't change the properties directly.

## CloudFrontWebDistribution API

> The `CloudFrontWebDistribution` construct is the original construct written for working with CloudFront distributions.
> Users are encouraged to use the newer `Distribution` instead, as it has a simpler interface and receives new features faster.

Example usage:

```python
# Using a CloudFrontWebDistribution construct.

# source_bucket: s3.Bucket

distribution = cloudfront.CloudFrontWebDistribution(self, "MyDistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=source_bucket
        ),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )
    ]
)
```

### Viewer certificate

By default, CloudFront Web Distributions will answer HTTPS requests with CloudFront's default certificate,
only containing the distribution `domainName` (e.g. d111111abcdef8.cloudfront.net).
You can customize the viewer certificate property to provide a custom certificate and/or list of domain name aliases to fit your needs.

See [Using Alternate Domain Names and HTTPS](https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-alternate-domain-names.html) in the CloudFront User Guide.

#### Default certificate

You can customize the default certificate aliases. This is intended to be used in combination with CNAME records in your DNS zone.

Example:

```python
s3_bucket_source = s3.Bucket(self, "Bucket")

distribution = cloudfront.CloudFrontWebDistribution(self, "AnAmazingWebsiteProbably",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(s3_bucket_source=s3_bucket_source),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )],
    viewer_certificate=cloudfront.ViewerCertificate.from_cloud_front_default_certificate("www.example.com")
)
```

#### ACM certificate

You can change the default certificate by one stored AWS Certificate Manager, or ACM.
Those certificate can either be generated by AWS, or purchased by another CA imported into ACM.

For more information, see
[the aws-certificatemanager module documentation](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-certificatemanager-readme.html)
or [Importing Certificates into AWS Certificate Manager](https://docs.aws.amazon.com/acm/latest/userguide/import-certificate.html)
in the AWS Certificate Manager User Guide.

Example:

```python
s3_bucket_source = s3.Bucket(self, "Bucket")

certificate = certificatemanager.Certificate(self, "Certificate",
    domain_name="example.com",
    subject_alternative_names=["*.example.com"]
)

distribution = cloudfront.CloudFrontWebDistribution(self, "AnAmazingWebsiteProbably",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(s3_bucket_source=s3_bucket_source),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )],
    viewer_certificate=cloudfront.ViewerCertificate.from_acm_certificate(certificate,
        aliases=["example.com", "www.example.com"],
        security_policy=cloudfront.SecurityPolicyProtocol.TLS_V1,  # default
        ssl_method=cloudfront.SSLMethod.SNI
    )
)
```

#### IAM certificate

You can also import a certificate into the IAM certificate store.

See [Importing an SSL/TLS Certificate](https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cnames-and-https-procedures.html#cnames-and-https-uploading-certificates) in the CloudFront User Guide.

Example:

```python
s3_bucket_source = s3.Bucket(self, "Bucket")

distribution = cloudfront.CloudFrontWebDistribution(self, "AnAmazingWebsiteProbably",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(s3_bucket_source=s3_bucket_source),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )],
    viewer_certificate=cloudfront.ViewerCertificate.from_iam_certificate("certificateId",
        aliases=["example.com"],
        security_policy=cloudfront.SecurityPolicyProtocol.SSL_V3,  # default
        ssl_method=cloudfront.SSLMethod.SNI
    )
)
```

### Trusted Key Groups

CloudFront Web Distributions supports validating signed URLs or signed cookies using key groups.
When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior.

Example:

```python
# Using trusted key groups for Cloudfront Web Distributions.
# source_bucket: s3.Bucket
# public_key: str

pub_key = cloudfront.PublicKey(self, "MyPubKey",
    encoded_key=public_key
)

key_group = cloudfront.KeyGroup(self, "MyKeyGroup",
    items=[pub_key
    ]
)

cloudfront.CloudFrontWebDistribution(self, "AnAmazingWebsiteProbably",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=source_bucket
        ),
        behaviors=[cloudfront.Behavior(
            is_default_behavior=True,
            trusted_key_groups=[key_group
            ]
        )
        ]
    )
    ]
)
```

### Restrictions

CloudFront supports adding restrictions to your distribution.

See [Restricting the Geographic Distribution of Your Content](https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/georestrictions.html) in the CloudFront User Guide.

Example:

```python
# Adding restrictions to a Cloudfront Web Distribution.
# source_bucket: s3.Bucket

cloudfront.CloudFrontWebDistribution(self, "MyDistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=source_bucket
        ),
        behaviors=[cloudfront.Behavior(is_default_behavior=True)]
    )
    ],
    geo_restriction=cloudfront.GeoRestriction.allowlist("US", "GB")
)
```

### Connection behaviors between CloudFront and your origin

CloudFront provides you even more control over the connection behaviors between CloudFront and your origin.
You can now configure the number of connection attempts CloudFront will make to your origin and the origin connection timeout for each attempt.

See [Origin Connection Attempts](https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts)

See [Origin Connection Timeout](https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout)

Example usage:

```python
# Configuring connection behaviors between Cloudfront and your origin
distribution = cloudfront.CloudFrontWebDistribution(self, "MyDistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        connection_attempts=3,
        connection_timeout=Duration.seconds(10),
        behaviors=[cloudfront.Behavior(
            is_default_behavior=True
        )
        ]
    )
    ]
)
```

#### Origin Fallback

In case the origin source is not available and answers with one of the
specified status codes the failover origin source will be used.

```python
# Configuring origin fallback options for the CloudFrontWebDistribution
cloudfront.CloudFrontWebDistribution(self, "ADistribution",
    origin_configs=[cloudfront.SourceConfiguration(
        s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=s3.Bucket.from_bucket_name(self, "aBucket", "myoriginbucket"),
            origin_path="/",
            origin_headers={
                "myHeader": "42"
            },
            origin_shield_region="us-west-2"
        ),
        failover_s3_origin_source=cloudfront.S3OriginConfig(
            s3_bucket_source=s3.Bucket.from_bucket_name(self, "aBucketFallback", "myoriginbucketfallback"),
            origin_path="/somewhere",
            origin_headers={
                "myHeader2": "21"
            },
            origin_shield_region="us-east-1"
        ),
        failover_criteria_status_codes=[cloudfront.FailoverStatusCode.INTERNAL_SERVER_ERROR],
        behaviors=[cloudfront.Behavior(
            is_default_behavior=True
        )
        ]
    )
    ]
)
```

## KeyGroup & PublicKey API

You can create a key group to use with CloudFront signed URLs and signed cookies
You can add public keys to use with CloudFront features such as signed URLs, signed cookies, and field-level encryption.

The following example command uses OpenSSL to generate an RSA key pair with a length of 2048 bits and save to the file named `private_key.pem`.

```bash
openssl genrsa -out private_key.pem 2048
```

The resulting file contains both the public and the private key. The following example command extracts the public key from the file named `private_key.pem` and stores it in `public_key.pem`.

```bash
openssl rsa -pubout -in private_key.pem -out public_key.pem
```

Note: Don't forget to copy/paste the contents of `public_key.pem` file including `-----BEGIN PUBLIC KEY-----` and `-----END PUBLIC KEY-----` lines into `encodedKey` parameter when creating a `PublicKey`.

Example:

```python
# Create a key group to use with CloudFront signed URLs and signed cookies.
cloudfront.KeyGroup(self, "MyKeyGroup",
    items=[
        cloudfront.PublicKey(self, "MyPublicKey",
            encoded_key="..."
        )
    ]
)
```

See:

* https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html
* https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-trusted-signers.html
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from typeguard import check_type

from ._jsii import *

import aws_cdk.aws_certificatemanager as _aws_cdk_aws_certificatemanager_1662be0d
import aws_cdk.aws_iam as _aws_cdk_aws_iam_940a1ce0
import aws_cdk.aws_lambda as _aws_cdk_aws_lambda_5443dbc3
import aws_cdk.aws_s3 as _aws_cdk_aws_s3_55f001a5
import aws_cdk.core as _aws_cdk_core_f4b25747
import constructs as _constructs_77d1e7e8


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.AddBehaviorOptions",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_methods": "allowedMethods",
        "cached_methods": "cachedMethods",
        "cache_policy": "cachePolicy",
        "compress": "compress",
        "edge_lambdas": "edgeLambdas",
        "function_associations": "functionAssociations",
        "origin_request_policy": "originRequestPolicy",
        "response_headers_policy": "responseHeadersPolicy",
        "smooth_streaming": "smoothStreaming",
        "trusted_key_groups": "trustedKeyGroups",
        "viewer_protocol_policy": "viewerProtocolPolicy",
    },
)
class AddBehaviorOptions:
    def __init__(
        self,
        *,
        allowed_methods: typing.Optional["AllowedMethods"] = None,
        cached_methods: typing.Optional["CachedMethods"] = None,
        cache_policy: typing.Optional["ICachePolicy"] = None,
        compress: typing.Optional[builtins.bool] = None,
        edge_lambdas: typing.Optional[typing.Sequence[typing.Union["EdgeLambda", typing.Dict[builtins.str, typing.Any]]]] = None,
        function_associations: typing.Optional[typing.Sequence[typing.Union["FunctionAssociation", typing.Dict[builtins.str, typing.Any]]]] = None,
        origin_request_policy: typing.Optional["IOriginRequestPolicy"] = None,
        response_headers_policy: typing.Optional["IResponseHeadersPolicy"] = None,
        smooth_streaming: typing.Optional[builtins.bool] = None,
        trusted_key_groups: typing.Optional[typing.Sequence["IKeyGroup"]] = None,
        viewer_protocol_policy: typing.Optional["ViewerProtocolPolicy"] = None,
    ) -> None:
        '''Options for adding a new behavior to a Distribution.

        :param allowed_methods: HTTP methods to allow for this behavior. Default: AllowedMethods.ALLOW_GET_HEAD
        :param cached_methods: HTTP methods to cache for this behavior. Default: CachedMethods.CACHE_GET_HEAD
        :param cache_policy: The cache policy for this behavior. The cache policy determines what values are included in the cache key, and the time-to-live (TTL) values for the cache. Default: CachePolicy.CACHING_OPTIMIZED
        :param compress: Whether you want CloudFront to automatically compress certain files for this cache behavior. See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html#compressed-content-cloudfront-file-types for file types CloudFront will compress. Default: true
        :param edge_lambdas: The Lambda@Edge functions to invoke before serving the contents. Default: - no Lambda functions will be invoked
        :param function_associations: The CloudFront functions to invoke before serving the contents. Default: - no functions will be invoked
        :param origin_request_policy: The origin request policy for this behavior. The origin request policy determines which values (e.g., headers, cookies) are included in requests that CloudFront sends to the origin. Default: - none
        :param response_headers_policy: The response headers policy for this behavior. The response headers policy determines which headers are included in responses Default: - none
        :param smooth_streaming: Set this to true to indicate you want to distribute media files in the Microsoft Smooth Streaming format using this behavior. Default: false
        :param trusted_key_groups: A list of Key Groups that CloudFront can use to validate signed URLs or signed cookies. Default: - no KeyGroups are associated with cache behavior
        :param viewer_protocol_policy: The protocol that viewers can use to access the files controlled by this behavior. Default: ViewerProtocolPolicy.ALLOW_ALL

        :exampleMetadata: infused

        Example::

            # Add a behavior to a Distribution after initial creation.
            # my_bucket: s3.Bucket
            # my_web_distribution: cloudfront.Distribution
            
            my_web_distribution.add_behavior("/images/*.jpg", origins.S3Origin(my_bucket),
                viewer_protocol_policy=cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__380add120d42a86cda3ca738d0d8c05d895a86c28bdd53fc8797a595913ea606)
            check_type(argname="argument allowed_methods", value=allowed_methods, expected_type=type_hints["allowed_methods"])
            check_type(argname="argument cached_methods", value=cached_methods, expected_type=type_hints["cached_methods"])
            check_type(argname="argument cache_policy", value=cache_policy, expected_type=type_hints["cache_policy"])
            check_type(argname="argument compress", value=compress, expected_type=type_hints["compress"])
            check_type(argname="argument edge_lambdas", value=edge_lambdas, expected_type=type_hints["edge_lambdas"])
            check_type(argname="argument function_associations", value=function_associations, expected_type=type_hints["function_associations"])
            check_type(argname="argument origin_request_policy", value=origin_request_policy, expected_type=type_hints["origin_request_policy"])
            check_type(argname="argument response_headers_policy", value=response_headers_policy, expected_type=type_hints["response_headers_policy"])
            check_type(argname="argument smooth_streaming", value=smooth_streaming, expected_type=type_hints["smooth_streaming"])
            check_type(argname="argument trusted_key_groups", value=trusted_key_groups, expected_type=type_hints["trusted_key_groups"])
            check_type(argname="argument viewer_protocol_policy", value=viewer_protocol_policy, expected_type=type_hints["viewer_protocol_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allowed_methods is not None:
            self._values["allowed_methods"] = allowed_methods
        if cached_methods is not None:
            self._values["cached_methods"] = cached_methods
        if cache_policy is not None:
            self._values["cache_policy"] = cache_policy
        if compress is not None:
            self._values["compress"] = compress
        if edge_lambdas is not None:
            self._values["edge_lambdas"] = edge_lambdas
        if function_associations is not None:
            self._values["function_associations"] = function_associations
        if origin_request_policy is not None:
            self._values["origin_request_policy"] = origin_request_policy
        if response_headers_policy is not None:
            self._values["response_headers_policy"] = response_headers_policy
        if smooth_streaming is not None:
            self._values["smooth_streaming"] = smooth_streaming
        if trusted_key_groups is not None:
            self._values["trusted_key_groups"] = trusted_key_groups
        if viewer_protocol_policy is not None:
            self._values["viewer_protocol_policy"] = viewer_protocol_policy

    @builtins.property
    def allowed_methods(self) -> typing.Optional["AllowedMethods"]:
        '''HTTP methods to allow for this behavior.

        :default: AllowedMethods.ALLOW_GET_HEAD
        '''
        result = self._values.get("allowed_methods")
        return typing.cast(typing.Optional["AllowedMethods"], result)

    @builtins.property
    def cached_methods(self) -> typing.Optional["CachedMethods"]:
        '''HTTP methods to cache for this behavior.

        :default: CachedMethods.CACHE_GET_HEAD
        '''
        result = self._values.get("cached_methods")
        return typing.cast(typing.Optional["CachedMethods"], result)

    @builtins.property
    def cache_policy(self) -> typing.Optional["ICachePolicy"]:
        '''The cache policy for this behavior.

        The cache policy determines what values are included in the cache key,
        and the time-to-live (TTL) values for the cache.

        :default: CachePolicy.CACHING_OPTIMIZED

        :see: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html.
        '''
        result = self._values.get("cache_policy")
        return typing.cast(typing.Optional["ICachePolicy"], result)

    @builtins.property
    def compress(self) -> typing.Optional[builtins.bool]:
        '''Whether you want CloudFront to automatically compress certain files for this cache behavior.

        See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html#compressed-content-cloudfront-file-types
        for file types CloudFront will compress.

        :default: true
        '''
        result = self._values.get("compress")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def edge_lambdas(self) -> typing.Optional[typing.List["EdgeLambda"]]:
        '''The Lambda@Edge functions to invoke before serving the contents.

        :default: - no Lambda functions will be invoked

        :see: https://aws.amazon.com/lambda/edge
        '''
        result = self._values.get("edge_lambdas")
        return typing.cast(typing.Optional[typing.List["EdgeLambda"]], result)

    @builtins.property
    def function_associations(
        self,
    ) -> typing.Optional[typing.List["FunctionAssociation"]]:
        '''The CloudFront functions to invoke before serving the contents.

        :default: - no functions will be invoked
        '''
        result = self._values.get("function_associations")
        return typing.cast(typing.Optional[typing.List["FunctionAssociation"]], result)

    @builtins.property
    def origin_request_policy(self) -> typing.Optional["IOriginRequestPolicy"]:
        '''The origin request policy for this behavior.

        The origin request policy determines which values (e.g., headers, cookies)
        are included in requests that CloudFront sends to the origin.

        :default: - none
        '''
        result = self._values.get("origin_request_policy")
        return typing.cast(typing.Optional["IOriginRequestPolicy"], result)

    @builtins.property
    def response_headers_policy(self) -> typing.Optional["IResponseHeadersPolicy"]:
        '''The response headers policy for this behavior.

        The response headers policy determines which headers are included in responses

        :default: - none
        '''
        result = self._values.get("response_headers_policy")
        return typing.cast(typing.Optional["IResponseHeadersPolicy"], result)

    @builtins.property
    def smooth_streaming(self) -> typing.Optional[builtins.bool]:
        '''Set this to true to indicate you want to distribute media files in the Microsoft Smooth Streaming format using this behavior.

        :default: false
        '''
        result = self._values.get("smooth_streaming")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def trusted_key_groups(self) -> typing.Optional[typing.List["IKeyGroup"]]:
        '''A list of Key Groups that CloudFront can use to validate signed URLs or signed cookies.

        :default: - no KeyGroups are associated with cache behavior

        :see: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html
        '''
        result = self._values.get("trusted_key_groups")
        return typing.cast(typing.Optional[typing.List["IKeyGroup"]], result)

    @builtins.property
    def viewer_protocol_policy(self) -> typing.Optional["ViewerProtocolPolicy"]:
        '''The protocol that viewers can use to access the files controlled by this behavior.

        :default: ViewerProtocolPolicy.ALLOW_ALL
        '''
        result = self._values.get("viewer_protocol_policy")
        return typing.cast(typing.Optional["ViewerProtocolPolicy"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddBehaviorOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.AliasConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "acm_cert_ref": "acmCertRef",
        "names": "names",
        "security_policy": "securityPolicy",
        "ssl_method": "sslMethod",
    },
)
class AliasConfiguration:
    def __init__(
        self,
        *,
        acm_cert_ref: builtins.str,
        names: typing.Sequence[builtins.str],
        security_policy: typing.Optional["SecurityPolicyProtocol"] = None,
        ssl_method: typing.Optional["SSLMethod"] = None,
    ) -> None:
        '''(deprecated) Configuration for custom domain names.

        CloudFront can use a custom domain that you provide instead of a
        "cloudfront.net" domain. To use this feature you must provide the list of
        additional domains, and the ACM Certificate that CloudFront should use for
        these additional domains.

        :param acm_cert_ref: (deprecated) ARN of an AWS Certificate Manager (ACM) certificate.
        :param names: (deprecated) Domain names on the certificate. Both main domain name and Subject Alternative Names.
        :param security_policy: (deprecated) The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections. CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify. Default: - SSLv3 if sslMethod VIP, TLSv1 if sslMethod SNI
        :param ssl_method: (deprecated) How CloudFront should serve HTTPS requests. See the notes on SSLMethod if you wish to use other SSL termination types. Default: SSLMethod.SNI

        :deprecated: see {@link CloudFrontWebDistributionProps#viewerCertificate} with {@link ViewerCertificate#acmCertificate}

        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            alias_configuration = cloudfront.AliasConfiguration(
                acm_cert_ref="acmCertRef",
                names=["names"],
            
                # the properties below are optional
                security_policy=cloudfront.SecurityPolicyProtocol.SSL_V3,
                ssl_method=cloudfront.SSLMethod.SNI
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f677a05c5a2badd61beb80043764e5ede03e21068c46a16d54af97b7e23952ed)
            check_type(argname="argument acm_cert_ref", value=acm_cert_ref, expected_type=type_hints["acm_cert_ref"])
            check_type(argname="argument names", value=names, expected_type=type_hints["names"])
            check_type(argname="argument security_policy", value=security_policy, expected_type=type_hints["security_policy"])
            check_type(argname="argument ssl_method", value=ssl_method, expected_type=type_hints["ssl_method"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "acm_cert_ref": acm_cert_ref,
            "names": names,
        }
        if security_policy is not None:
            self._values["security_policy"] = security_policy
        if ssl_method is not None:
            self._values["ssl_method"] = ssl_method

    @builtins.property
    def acm_cert_ref(self) -> builtins.str:
        '''(deprecated) ARN of an AWS Certificate Manager (ACM) certificate.

        :stability: deprecated
        '''
        result = self._values.get("acm_cert_ref")
        assert result is not None, "Required property 'acm_cert_ref' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def names(self) -> typing.List[builtins.str]:
        '''(deprecated) Domain names on the certificate.

        Both main domain name and Subject Alternative Names.

        :stability: deprecated
        '''
        result = self._values.get("names")
        assert result is not None, "Required property 'names' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def security_policy(self) -> typing.Optional["SecurityPolicyProtocol"]:
        '''(deprecated) The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.

        CloudFront serves your objects only to browsers or devices that support at
        least the SSL version that you specify.

        :default: - SSLv3 if sslMethod VIP, TLSv1 if sslMethod SNI

        :stability: deprecated
        '''
        result = self._values.get("security_policy")
        return typing.cast(typing.Optional["SecurityPolicyProtocol"], result)

    @builtins.property
    def ssl_method(self) -> typing.Optional["SSLMethod"]:
        '''(deprecated) How CloudFront should serve HTTPS requests.

        See the notes on SSLMethod if you wish to use other SSL termination types.

        :default: SSLMethod.SNI

        :see: https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html
        :stability: deprecated
        '''
        result = self._values.get("ssl_method")
        return typing.cast(typing.Optional["SSLMethod"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AliasConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AllowedMethods(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.AllowedMethods",
):
    '''The HTTP methods that the Behavior will accept requests on.

    :exampleMetadata: infused

    Example::

        # Create a Distribution with configured HTTP methods and viewer protocol policy of the cache.
        # my_bucket: s3.Bucket
        
        my_web_distribution = cloudfront.Distribution(self, "myDist",
            default_behavior=cloudfront.BehaviorOptions(
                origin=origins.S3Origin(my_bucket),
                allowed_methods=cloudfront.AllowedMethods.ALLOW_ALL,
                viewer_protocol_policy=cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS
            )
        )
    '''

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALLOW_ALL")
    def ALLOW_ALL(cls) -> "AllowedMethods":
        '''All supported HTTP methods.'''
        return typing.cast("AllowedMethods", jsii.sget(cls, "ALLOW_ALL"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALLOW_GET_HEAD")
    def ALLOW_GET_HEAD(cls) -> "AllowedMethods":
        '''HEAD and GET.'''
        return typing.cast("AllowedMethods", jsii.sget(cls, "ALLOW_GET_HEAD"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="ALLOW_GET_HEAD_OPTIONS")
    def ALLOW_GET_HEAD_OPTIONS(cls) -> "AllowedMethods":
        '''HEAD, GET, and OPTIONS.'''
        return typing.cast("AllowedMethods", jsii.sget(cls, "ALLOW_GET_HEAD_OPTIONS"))

    @builtins.property
    @jsii.member(jsii_name="methods")
    def methods(self) -> typing.List[builtins.str]:
        '''HTTP methods supported.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "methods"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.Behavior",
    jsii_struct_bases=[],
    name_mapping={
        "allowed_methods": "allowedMethods",
        "cached_methods": "cachedMethods",
        "compress": "compress",
        "default_ttl": "defaultTtl",
        "forwarded_values": "forwardedValues",
        "function_associations": "functionAssociations",
        "is_default_behavior": "isDefaultBehavior",
        "lambda_function_associations": "lambdaFunctionAssociations",
        "max_ttl": "maxTtl",
        "min_ttl": "minTtl",
        "path_pattern": "pathPattern",
        "trusted_key_groups": "trustedKeyGroups",
        "trusted_signers": "trustedSigners",
        "viewer_protocol_policy": "viewerProtocolPolicy",
    },
)
class Behavior:
    def __init__(
        self,
        *,
        allowed_methods: typing.Optional["CloudFrontAllowedMethods"] = None,
        cached_methods: typing.Optional["CloudFrontAllowedCachedMethods"] = None,
        compress: typing.Optional[builtins.bool] = None,
        default_ttl: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        forwarded_values: typing.Optional[typing.Union["CfnDistribution.ForwardedValuesProperty", typing.Dict[builtins.str, typing.Any]]] = None,
        function_associations: typing.Optional[typing.Sequence[typing.Union["FunctionAssociation", typing.Dict[builtins.str, typing.Any]]]] = None,
        is_default_behavior: typing.Optional[builtins.bool] = None,
        lambda_function_associations: typing.Optional[typing.Sequence[typing.Union["LambdaFunctionAssociation", typing.Dict[builtins.str, typing.Any]]]] = None,
        max_ttl: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        min_ttl: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        path_pattern: typing.Optional[builtins.str] = None,
        trusted_key_groups: typing.Optional[typing.Sequence["IKeyGroup"]] = None,
        trusted_signers: typing.Optional[typing.Sequence[builtins.str]] = None,
        viewer_protocol_policy: typing.Optional["ViewerProtocolPolicy"] = None,
    ) -> None:
        '''A CloudFront behavior wrapper.

        :param allowed_methods: The method this CloudFront distribution responds do. Default: GET_HEAD
        :param cached_methods: Which methods are cached by CloudFront by default. Default: GET_HEAD
        :param compress: If CloudFront should automatically compress some content types. Default: true
        :param default_ttl: The default amount of time CloudFront will cache an object. This value applies only when your custom origin does not add HTTP headers, such as Cache-Control max-age, Cache-Control s-maxage, and Expires to objects. Default: 86400 (1 day)
        :param forwarded_values: The values CloudFront will forward to the origin when making a request. Default: none (no cookies - no headers)
        :param function_associations: The CloudFront functions to invoke before serving the contents. Default: - no functions will be invoked
        :param is_default_behavior: If this behavior is the default behavior for the distribution. You must specify exactly one default distribution per CloudFront distribution. The default behavior is allowed to omit the "path" property.
        :param lambda_function_associations: Declares associated lambda@edge functions for this distribution behaviour. Default: No lambda function associated
        :param max_ttl: The max amount of time you want objects to stay in the cache before CloudFront queries your origin. Default: Duration.seconds(31536000) (one year)
        :param min_ttl: The minimum amount of time that you want objects to stay in the cache before CloudFront queries your origin.
        :param path_pattern: The path this behavior responds to. Required for all non-default behaviors. (The default behavior implicitly has "*" as the path pattern. )
        :param trusted_key_groups: A list of Key Groups that CloudFront can use to validate signed URLs or signed cookies. Default: - no KeyGroups are associated with cache behavior
        :param trusted_signers: (deprecated) Trusted signers is how CloudFront allows you to serve private content. The signers are the account IDs that are allowed to sign cookies/presigned URLs for this distribution. If you pass a non empty value, all requests for this behavior must be signed (no public access will be allowed)
        :param viewer_protocol_policy: The viewer policy for this behavior. Default: - the distribution wide viewer protocol policy will be used

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            import aws_cdk.aws_lambda as lambda_
            import aws_cdk.core as cdk
            
            # function_: cloudfront.Function
            # key_group: cloudfront.KeyGroup
            # version: lambda.Version
            
            behavior = cloudfront.Behavior(
                allowed_methods=cloudfront.CloudFrontAllowedMethods.GET_HEAD,
                cached_methods=cloudfront.CloudFrontAllowedCachedMethods.GET_HEAD,
                compress=False,
                default_ttl=cdk.Duration.minutes(30),
                forwarded_values=cloudfront.CfnDistribution.ForwardedValuesProperty(
                    query_string=False,
            
                    # the properties below are optional
                    cookies=cloudfront.CfnDistribution.CookiesProperty(
                        forward="forward",
            
                        # the properties below are optional
                        whitelisted_names=["whitelistedNames"]
                    ),
                    headers=["headers"],
                    query_string_cache_keys=["queryStringCacheKeys"]
                ),
                function_associations=[cloudfront.FunctionAssociation(
                    event_type=cloudfront.FunctionEventType.VIEWER_REQUEST,
                    function=function_
                )],
                is_default_behavior=False,
                lambda_function_associations=[cloudfront.LambdaFunctionAssociation(
                    event_type=cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                    lambda_function=version,
            
                    # the properties below are optional
                    include_body=False
                )],
                max_ttl=cdk.Duration.minutes(30),
                min_ttl=cdk.Duration.minutes(30),
                path_pattern="pathPattern",
                trusted_key_groups=[key_group],
                trusted_signers=["trustedSigners"],
                viewer_protocol_policy=cloudfront.ViewerProtocolPolicy.HTTPS_ONLY
            )
        '''
        if isinstance(forwarded_values, dict):
            forwarded_values = CfnDistribution.ForwardedValuesProperty(**forwarded_values)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__628ec93ba21bbe7ba458bb113dc5c50b2198ce98a784af6757b883b4525a538d)
            check_type(argname="argument allowed_methods", value=allowed_methods, expected_type=type_hints["allowed_methods"])
            check_type(argname="argument cached_methods", value=cached_methods, expected_type=type_hints["cached_methods"])
            check_type(argname="argument compress", value=compress, expected_type=type_hints["compress"])
            check_type(argname="argument default_ttl", value=default_ttl, expected_type=type_hints["default_ttl"])
            check_type(argname="argument forwarded_values", value=forwarded_values, expected_type=type_hints["forwarded_values"])
            check_type(argname="argument function_associations", value=function_associations, expected_type=type_hints["function_associations"])
            check_type(argname="argument is_default_behavior", value=is_default_behavior, expected_type=type_hints["is_default_behavior"])
            check_type(argname="argument lambda_function_associations", value=lambda_function_associations, expected_type=type_hints["lambda_function_associations"])
            check_type(argname="argument max_ttl", value=max_ttl, expected_type=type_hints["max_ttl"])
            check_type(argname="argument min_ttl", value=min_ttl, expected_type=type_hints["min_ttl"])
            check_type(argname="argument path_pattern", value=path_pattern, expected_type=type_hints["path_pattern"])
            check_type(argname="argument trusted_key_groups", value=trusted_key_groups, expected_type=type_hints["trusted_key_groups"])
            check_type(argname="argument trusted_signers", value=trusted_signers, expected_type=type_hints["trusted_signers"])
            check_type(argname="argument viewer_protocol_policy", value=viewer_protocol_policy, expected_type=type_hints["viewer_protocol_policy"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if allowed_methods is not None:
            self._values["allowed_methods"] = allowed_methods
        if cached_methods is not None:
            self._values["cached_methods"] = cached_methods
        if compress is not None:
            self._values["compress"] = compress
        if default_ttl is not None:
            self._values["default_ttl"] = default_ttl
        if forwarded_values is not None:
            self._values["forwarded_values"] = forwarded_values
        if function_associations is not None:
            self._values["function_associations"] = function_associations
        if is_default_behavior is not None:
            self._values["is_default_behavior"] = is_default_behavior
        if lambda_function_associations is not None:
            self._values["lambda_function_associations"] = lambda_function_associations
        if max_ttl is not None:
            self._values["max_ttl"] = max_ttl
        if min_ttl is not None:
            self._values["min_ttl"] = min_ttl
        if path_pattern is not None:
            self._values["path_pattern"] = path_pattern
        if trusted_key_groups is not None:
            self._values["trusted_key_groups"] = trusted_key_groups
        if trusted_signers is not None:
            self._values["trusted_signers"] = trusted_signers
        if viewer_protocol_policy is not None:
            self._values["viewer_protocol_policy"] = viewer_protocol_policy

    @builtins.property
    def allowed_methods(self) -> typing.Optional["CloudFrontAllowedMethods"]:
        '''The method this CloudFront distribution responds do.

        :default: GET_HEAD
        '''
        result = self._values.get("allowed_methods")
        return typing.cast(typing.Optional["CloudFrontAllowedMethods"], result)

    @builtins.property
    def cached_methods(self) -> typing.Optional["CloudFrontAllowedCachedMethods"]:
        '''Which methods are cached by CloudFront by default.

        :default: GET_HEAD
        '''
        result = self._values.get("cached_methods")
        return typing.cast(typing.Optional["CloudFrontAllowedCachedMethods"], result)

    @builtins.property
    def compress(self) -> typing.Optional[builtins.bool]:
        '''If CloudFront should automatically compress some content types.

        :default: true
        '''
        result = self._values.get("compress")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def default_ttl(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The default amount of time CloudFront will cache an object.

        This value applies only when your custom origin does not add HTTP headers,
        such as Cache-Control max-age, Cache-Control s-maxage, and Expires to objects.

        :default: 86400 (1 day)
        '''
        result = self._values.get("default_ttl")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def forwarded_values(
        self,
    ) -> typing.Optional["CfnDistribution.ForwardedValuesProperty"]:
        '''The values CloudFront will forward to the origin when making a request.

        :default: none (no cookies - no headers)
        '''
        result = self._values.get("forwarded_values")
        return typing.cast(typing.Optional["CfnDistribution.ForwardedValuesProperty"], result)

    @builtins.property
    def function_associations(
        self,
    ) -> typing.Optional[typing.List["FunctionAssociation"]]:
        '''The CloudFront functions to invoke before serving the contents.

        :default: - no functions will be invoked
        '''
        result = self._values.get("function_associations")
        return typing.cast(typing.Optional[typing.List["FunctionAssociation"]], result)

    @builtins.property
    def is_default_behavior(self) -> typing.Optional[builtins.bool]:
        '''If this behavior is the default behavior for the distribution.

        You must specify exactly one default distribution per CloudFront distribution.
        The default behavior is allowed to omit the "path" property.
        '''
        result = self._values.get("is_default_behavior")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def lambda_function_associations(
        self,
    ) -> typing.Optional[typing.List["LambdaFunctionAssociation"]]:
        '''Declares associated lambda@edge functions for this distribution behaviour.

        :default: No lambda function associated
        '''
        result = self._values.get("lambda_function_associations")
        return typing.cast(typing.Optional[typing.List["LambdaFunctionAssociation"]], result)

    @builtins.property
    def max_ttl(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The max amount of time you want objects to stay in the cache before CloudFront queries your origin.

        :default: Duration.seconds(31536000) (one year)
        '''
        result = self._values.get("max_ttl")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def min_ttl(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The minimum amount of time that you want objects to stay in the cache before CloudFront queries your origin.'''
        result = self._values.get("min_ttl")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def path_pattern(self) -> typing.Optional[builtins.str]:
        '''The path this behavior responds to.

        Required for all non-default behaviors. (The default behavior implicitly has "*" as the path pattern. )
        '''
        result = self._values.get("path_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def trusted_key_groups(self) -> typing.Optional[typing.List["IKeyGroup"]]:
        '''A list of Key Groups that CloudFront can use to validate signed URLs or signed cookies.

        :default: - no KeyGroups are associated with cache behavior

        :see: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html
        '''
        result = self._values.get("trusted_key_groups")
        return typing.cast(typing.Optional[typing.List["IKeyGroup"]], result)

    @builtins.property
    def trusted_signers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(deprecated) Trusted signers is how CloudFront allows you to serve private content.

        The signers are the account IDs that are allowed to sign cookies/presigned URLs for this distribution.

        If you pass a non empty value, all requests for this behavior must be signed (no public access will be allowed)

        :deprecated: - We recommend using trustedKeyGroups instead of trustedSigners.

        :stability: deprecated
        '''
        result = self._values.get("trusted_signers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def viewer_protocol_policy(self) -> typing.Optional["ViewerProtocolPolicy"]:
        '''The viewer policy for this behavior.

        :default: - the distribution wide viewer protocol policy will be used
        '''
        result = self._values.get("viewer_protocol_policy")
        return typing.cast(typing.Optional["ViewerProtocolPolicy"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Behavior(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.BehaviorOptions",
    jsii_struct_bases=[AddBehaviorOptions],
    name_mapping={
        "allowed_methods": "allowedMethods",
        "cached_methods": "cachedMethods",
        "cache_policy": "cachePolicy",
        "compress": "compress",
        "edge_lambdas": "edgeLambdas",
        "function_associations": "functionAssociations",
        "origin_request_policy": "originRequestPolicy",
        "response_headers_policy": "responseHeadersPolicy",
        "smooth_streaming": "smoothStreaming",
        "trusted_key_groups": "trustedKeyGroups",
        "viewer_protocol_policy": "viewerProtocolPolicy",
        "origin": "origin",
    },
)
class BehaviorOptions(AddBehaviorOptions):
    def __init__(
        self,
        *,
        allowed_methods: typing.Optional[AllowedMethods] = None,
        cached_methods: typing.Optional["CachedMethods"] = None,
        cache_policy: typing.Optional["ICachePolicy"] = None,
        compress: typing.Optional[builtins.bool] = None,
        edge_lambdas: typing.Optional[typing.Sequence[typing.Union["EdgeLambda", typing.Dict[builtins.str, typing.Any]]]] = None,
        function_associations: typing.Optional[typing.Sequence[typing.Union["FunctionAssociation", typing.Dict[builtins.str, typing.Any]]]] = None,
        origin_request_policy: typing.Optional["IOriginRequestPolicy"] = None,
        response_headers_policy: typing.Optional["IResponseHeadersPolicy"] = None,
        smooth_streaming: typing.Optional[builtins.bool] = None,
        trusted_key_groups: typing.Optional[typing.Sequence["IKeyGroup"]] = None,
        viewer_protocol_policy: typing.Optional["ViewerProtocolPolicy"] = None,
        origin: "IOrigin",
    ) -> None:
        '''Options for creating a new behavior.

        :param allowed_methods: HTTP methods to allow for this behavior. Default: AllowedMethods.ALLOW_GET_HEAD
        :param cached_methods: HTTP methods to cache for this behavior. Default: CachedMethods.CACHE_GET_HEAD
        :param cache_policy: The cache policy for this behavior. The cache policy determines what values are included in the cache key, and the time-to-live (TTL) values for the cache. Default: CachePolicy.CACHING_OPTIMIZED
        :param compress: Whether you want CloudFront to automatically compress certain files for this cache behavior. See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html#compressed-content-cloudfront-file-types for file types CloudFront will compress. Default: true
        :param edge_lambdas: The Lambda@Edge functions to invoke before serving the contents. Default: - no Lambda functions will be invoked
        :param function_associations: The CloudFront functions to invoke before serving the contents. Default: - no functions will be invoked
        :param origin_request_policy: The origin request policy for this behavior. The origin request policy determines which values (e.g., headers, cookies) are included in requests that CloudFront sends to the origin. Default: - none
        :param response_headers_policy: The response headers policy for this behavior. The response headers policy determines which headers are included in responses Default: - none
        :param smooth_streaming: Set this to true to indicate you want to distribute media files in the Microsoft Smooth Streaming format using this behavior. Default: false
        :param trusted_key_groups: A list of Key Groups that CloudFront can use to validate signed URLs or signed cookies. Default: - no KeyGroups are associated with cache behavior
        :param viewer_protocol_policy: The protocol that viewers can use to access the files controlled by this behavior. Default: ViewerProtocolPolicy.ALLOW_ALL
        :param origin: The origin that you want CloudFront to route requests to when they match this behavior.

        :exampleMetadata: infused

        Example::

            # Adding an existing Lambda@Edge function created in a different stack
            # to a CloudFront distribution.
            # s3_bucket: s3.Bucket
            
            function_version = lambda_.Version.from_version_arn(self, "Version", "arn:aws:lambda:us-east-1:123456789012:function:functionName:1")
            
            cloudfront.Distribution(self, "distro",
                default_behavior=cloudfront.BehaviorOptions(
                    origin=origins.S3Origin(s3_bucket),
                    edge_lambdas=[cloudfront.EdgeLambda(
                        function_version=function_version,
                        event_type=cloudfront.LambdaEdgeEventType.VIEWER_REQUEST
                    )
                    ]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ef98b534214803fbc603fd59fd5484763fe7972bccd091719ba5ee0e9f49fe86)
            check_type(argname="argument allowed_methods", value=allowed_methods, expected_type=type_hints["allowed_methods"])
            check_type(argname="argument cached_methods", value=cached_methods, expected_type=type_hints["cached_methods"])
            check_type(argname="argument cache_policy", value=cache_policy, expected_type=type_hints["cache_policy"])
            check_type(argname="argument compress", value=compress, expected_type=type_hints["compress"])
            check_type(argname="argument edge_lambdas", value=edge_lambdas, expected_type=type_hints["edge_lambdas"])
            check_type(argname="argument function_associations", value=function_associations, expected_type=type_hints["function_associations"])
            check_type(argname="argument origin_request_policy", value=origin_request_policy, expected_type=type_hints["origin_request_policy"])
            check_type(argname="argument response_headers_policy", value=response_headers_policy, expected_type=type_hints["response_headers_policy"])
            check_type(argname="argument smooth_streaming", value=smooth_streaming, expected_type=type_hints["smooth_streaming"])
            check_type(argname="argument trusted_key_groups", value=trusted_key_groups, expected_type=type_hints["trusted_key_groups"])
            check_type(argname="argument viewer_protocol_policy", value=viewer_protocol_policy, expected_type=type_hints["viewer_protocol_policy"])
            check_type(argname="argument origin", value=origin, expected_type=type_hints["origin"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "origin": origin,
        }
        if allowed_methods is not None:
            self._values["allowed_methods"] = allowed_methods
        if cached_methods is not None:
            self._values["cached_methods"] = cached_methods
        if cache_policy is not None:
            self._values["cache_policy"] = cache_policy
        if compress is not None:
            self._values["compress"] = compress
        if edge_lambdas is not None:
            self._values["edge_lambdas"] = edge_lambdas
        if function_associations is not None:
            self._values["function_associations"] = function_associations
        if origin_request_policy is not None:
            self._values["origin_request_policy"] = origin_request_policy
        if response_headers_policy is not None:
            self._values["response_headers_policy"] = response_headers_policy
        if smooth_streaming is not None:
            self._values["smooth_streaming"] = smooth_streaming
        if trusted_key_groups is not None:
            self._values["trusted_key_groups"] = trusted_key_groups
        if viewer_protocol_policy is not None:
            self._values["viewer_protocol_policy"] = viewer_protocol_policy

    @builtins.property
    def allowed_methods(self) -> typing.Optional[AllowedMethods]:
        '''HTTP methods to allow for this behavior.

        :default: AllowedMethods.ALLOW_GET_HEAD
        '''
        result = self._values.get("allowed_methods")
        return typing.cast(typing.Optional[AllowedMethods], result)

    @builtins.property
    def cached_methods(self) -> typing.Optional["CachedMethods"]:
        '''HTTP methods to cache for this behavior.

        :default: CachedMethods.CACHE_GET_HEAD
        '''
        result = self._values.get("cached_methods")
        return typing.cast(typing.Optional["CachedMethods"], result)

    @builtins.property
    def cache_policy(self) -> typing.Optional["ICachePolicy"]:
        '''The cache policy for this behavior.

        The cache policy determines what values are included in the cache key,
        and the time-to-live (TTL) values for the cache.

        :default: CachePolicy.CACHING_OPTIMIZED

        :see: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html.
        '''
        result = self._values.get("cache_policy")
        return typing.cast(typing.Optional["ICachePolicy"], result)

    @builtins.property
    def compress(self) -> typing.Optional[builtins.bool]:
        '''Whether you want CloudFront to automatically compress certain files for this cache behavior.

        See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html#compressed-content-cloudfront-file-types
        for file types CloudFront will compress.

        :default: true
        '''
        result = self._values.get("compress")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def edge_lambdas(self) -> typing.Optional[typing.List["EdgeLambda"]]:
        '''The Lambda@Edge functions to invoke before serving the contents.

        :default: - no Lambda functions will be invoked

        :see: https://aws.amazon.com/lambda/edge
        '''
        result = self._values.get("edge_lambdas")
        return typing.cast(typing.Optional[typing.List["EdgeLambda"]], result)

    @builtins.property
    def function_associations(
        self,
    ) -> typing.Optional[typing.List["FunctionAssociation"]]:
        '''The CloudFront functions to invoke before serving the contents.

        :default: - no functions will be invoked
        '''
        result = self._values.get("function_associations")
        return typing.cast(typing.Optional[typing.List["FunctionAssociation"]], result)

    @builtins.property
    def origin_request_policy(self) -> typing.Optional["IOriginRequestPolicy"]:
        '''The origin request policy for this behavior.

        The origin request policy determines which values (e.g., headers, cookies)
        are included in requests that CloudFront sends to the origin.

        :default: - none
        '''
        result = self._values.get("origin_request_policy")
        return typing.cast(typing.Optional["IOriginRequestPolicy"], result)

    @builtins.property
    def response_headers_policy(self) -> typing.Optional["IResponseHeadersPolicy"]:
        '''The response headers policy for this behavior.

        The response headers policy determines which headers are included in responses

        :default: - none
        '''
        result = self._values.get("response_headers_policy")
        return typing.cast(typing.Optional["IResponseHeadersPolicy"], result)

    @builtins.property
    def smooth_streaming(self) -> typing.Optional[builtins.bool]:
        '''Set this to true to indicate you want to distribute media files in the Microsoft Smooth Streaming format using this behavior.

        :default: false
        '''
        result = self._values.get("smooth_streaming")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def trusted_key_groups(self) -> typing.Optional[typing.List["IKeyGroup"]]:
        '''A list of Key Groups that CloudFront can use to validate signed URLs or signed cookies.

        :default: - no KeyGroups are associated with cache behavior

        :see: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html
        '''
        result = self._values.get("trusted_key_groups")
        return typing.cast(typing.Optional[typing.List["IKeyGroup"]], result)

    @builtins.property
    def viewer_protocol_policy(self) -> typing.Optional["ViewerProtocolPolicy"]:
        '''The protocol that viewers can use to access the files controlled by this behavior.

        :default: ViewerProtocolPolicy.ALLOW_ALL
        '''
        result = self._values.get("viewer_protocol_policy")
        return typing.cast(typing.Optional["ViewerProtocolPolicy"], result)

    @builtins.property
    def origin(self) -> "IOrigin":
        '''The origin that you want CloudFront to route requests to when they match this behavior.'''
        result = self._values.get("origin")
        assert result is not None, "Required property 'origin' is missing"
        return typing.cast("IOrigin", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BehaviorOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CacheCookieBehavior(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CacheCookieBehavior",
):
    '''Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.

    :exampleMetadata: infused

    Example::

        # Creating a custom cache policy for a Distribution -- all parameters optional
        # bucket_origin: origins.S3Origin
        
        my_cache_policy = cloudfront.CachePolicy(self, "myCachePolicy",
            cache_policy_name="MyPolicy",
            comment="A default policy",
            default_ttl=Duration.days(2),
            min_ttl=Duration.minutes(1),
            max_ttl=Duration.days(10),
            cookie_behavior=cloudfront.CacheCookieBehavior.all(),
            header_behavior=cloudfront.CacheHeaderBehavior.allow_list("X-CustomHeader"),
            query_string_behavior=cloudfront.CacheQueryStringBehavior.deny_list("username"),
            enable_accept_encoding_gzip=True,
            enable_accept_encoding_brotli=True
        )
        cloudfront.Distribution(self, "myDistCustomPolicy",
            default_behavior=cloudfront.BehaviorOptions(
                origin=bucket_origin,
                cache_policy=my_cache_policy
            )
        )
    '''

    @jsii.member(jsii_name="all")
    @builtins.classmethod
    def all(cls) -> "CacheCookieBehavior":
        '''All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.'''
        return typing.cast("CacheCookieBehavior", jsii.sinvoke(cls, "all", []))

    @jsii.member(jsii_name="allowList")
    @builtins.classmethod
    def allow_list(cls, *cookies: builtins.str) -> "CacheCookieBehavior":
        '''Only the provided ``cookies`` are included in the cache key and automatically included in requests that CloudFront sends to the origin.

        :param cookies: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d1b200255e2997155a36ebae0418a4ff63ff30a3bec775cb7f0aa34a882b845)
            check_type(argname="argument cookies", value=cookies, expected_type=typing.Tuple[type_hints["cookies"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("CacheCookieBehavior", jsii.sinvoke(cls, "allowList", [*cookies]))

    @jsii.member(jsii_name="denyList")
    @builtins.classmethod
    def deny_list(cls, *cookies: builtins.str) -> "CacheCookieBehavior":
        '''All cookies except the provided ``cookies`` are included in the cache key and automatically included in requests that CloudFront sends to the origin.

        :param cookies: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4547ab98b70f7be01d53d6a9c1b7bd2bed21681c23c5b561a517569cc25de570)
            check_type(argname="argument cookies", value=cookies, expected_type=typing.Tuple[type_hints["cookies"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("CacheCookieBehavior", jsii.sinvoke(cls, "denyList", [*cookies]))

    @jsii.member(jsii_name="none")
    @builtins.classmethod
    def none(cls) -> "CacheCookieBehavior":
        '''Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.'''
        return typing.cast("CacheCookieBehavior", jsii.sinvoke(cls, "none", []))

    @builtins.property
    @jsii.member(jsii_name="behavior")
    def behavior(self) -> builtins.str:
        '''The behavior of cookies: allow all, none, an allow list, or a deny list.'''
        return typing.cast(builtins.str, jsii.get(self, "behavior"))

    @builtins.property
    @jsii.member(jsii_name="cookies")
    def cookies(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The cookies to allow or deny, if the behavior is an allow or deny list.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "cookies"))


class CacheHeaderBehavior(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CacheHeaderBehavior",
):
    '''Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.

    :exampleMetadata: infused

    Example::

        # Creating a custom cache policy for a Distribution -- all parameters optional
        # bucket_origin: origins.S3Origin
        
        my_cache_policy = cloudfront.CachePolicy(self, "myCachePolicy",
            cache_policy_name="MyPolicy",
            comment="A default policy",
            default_ttl=Duration.days(2),
            min_ttl=Duration.minutes(1),
            max_ttl=Duration.days(10),
            cookie_behavior=cloudfront.CacheCookieBehavior.all(),
            header_behavior=cloudfront.CacheHeaderBehavior.allow_list("X-CustomHeader"),
            query_string_behavior=cloudfront.CacheQueryStringBehavior.deny_list("username"),
            enable_accept_encoding_gzip=True,
            enable_accept_encoding_brotli=True
        )
        cloudfront.Distribution(self, "myDistCustomPolicy",
            default_behavior=cloudfront.BehaviorOptions(
                origin=bucket_origin,
                cache_policy=my_cache_policy
            )
        )
    '''

    @jsii.member(jsii_name="allowList")
    @builtins.classmethod
    def allow_list(cls, *headers: builtins.str) -> "CacheHeaderBehavior":
        '''Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin.

        :param headers: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5cc5b7ebcf922f18655084cd4eb48cdb91af00f957cd09f75a248cdc68f147fd)
            check_type(argname="argument headers", value=headers, expected_type=typing.Tuple[type_hints["headers"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("CacheHeaderBehavior", jsii.sinvoke(cls, "allowList", [*headers]))

    @jsii.member(jsii_name="none")
    @builtins.classmethod
    def none(cls) -> "CacheHeaderBehavior":
        '''HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.'''
        return typing.cast("CacheHeaderBehavior", jsii.sinvoke(cls, "none", []))

    @builtins.property
    @jsii.member(jsii_name="behavior")
    def behavior(self) -> builtins.str:
        '''If no headers will be passed, or an allow list of headers.'''
        return typing.cast(builtins.str, jsii.get(self, "behavior"))

    @builtins.property
    @jsii.member(jsii_name="headers")
    def headers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The headers for the allow/deny list, if applicable.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "headers"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CachePolicyProps",
    jsii_struct_bases=[],
    name_mapping={
        "cache_policy_name": "cachePolicyName",
        "comment": "comment",
        "cookie_behavior": "cookieBehavior",
        "default_ttl": "defaultTtl",
        "enable_accept_encoding_brotli": "enableAcceptEncodingBrotli",
        "enable_accept_encoding_gzip": "enableAcceptEncodingGzip",
        "header_behavior": "headerBehavior",
        "max_ttl": "maxTtl",
        "min_ttl": "minTtl",
        "query_string_behavior": "queryStringBehavior",
    },
)
class CachePolicyProps:
    def __init__(
        self,
        *,
        cache_policy_name: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        cookie_behavior: typing.Optional[CacheCookieBehavior] = None,
        default_ttl: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        enable_accept_encoding_brotli: typing.Optional[builtins.bool] = None,
        enable_accept_encoding_gzip: typing.Optional[builtins.bool] = None,
        header_behavior: typing.Optional[CacheHeaderBehavior] = None,
        max_ttl: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        min_ttl: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        query_string_behavior: typing.Optional["CacheQueryStringBehavior"] = None,
    ) -> None:
        '''Properties for creating a Cache Policy.

        :param cache_policy_name: A unique name to identify the cache policy. The name must only include '-', '_', or alphanumeric characters. Default: - generated from the ``id``
        :param comment: A comment to describe the cache policy. Default: - no comment
        :param cookie_behavior: Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin. Default: CacheCookieBehavior.none()
        :param default_ttl: The default amount of time for objects to stay in the CloudFront cache. Only used when the origin does not send Cache-Control or Expires headers with the object. Default: - The greater of 1 day and ``minTtl``
        :param enable_accept_encoding_brotli: Whether to normalize and include the ``Accept-Encoding`` header in the cache key when the ``Accept-Encoding`` header is 'br'. Default: false
        :param enable_accept_encoding_gzip: Whether to normalize and include the ``Accept-Encoding`` header in the cache key when the ``Accept-Encoding`` header is 'gzip'. Default: false
        :param header_behavior: Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin. Default: CacheHeaderBehavior.none()
        :param max_ttl: The maximum amount of time for objects to stay in the CloudFront cache. CloudFront uses this value only when the origin sends Cache-Control or Expires headers with the object. Default: - The greater of 1 year and ``defaultTtl``
        :param min_ttl: The minimum amount of time for objects to stay in the CloudFront cache. Default: Duration.seconds(0)
        :param query_string_behavior: Determines whether any query strings are included in the cache key and automatically included in requests that CloudFront sends to the origin. Default: CacheQueryStringBehavior.none()

        :exampleMetadata: infused

        Example::

            # Creating a custom cache policy for a Distribution -- all parameters optional
            # bucket_origin: origins.S3Origin
            
            my_cache_policy = cloudfront.CachePolicy(self, "myCachePolicy",
                cache_policy_name="MyPolicy",
                comment="A default policy",
                default_ttl=Duration.days(2),
                min_ttl=Duration.minutes(1),
                max_ttl=Duration.days(10),
                cookie_behavior=cloudfront.CacheCookieBehavior.all(),
                header_behavior=cloudfront.CacheHeaderBehavior.allow_list("X-CustomHeader"),
                query_string_behavior=cloudfront.CacheQueryStringBehavior.deny_list("username"),
                enable_accept_encoding_gzip=True,
                enable_accept_encoding_brotli=True
            )
            cloudfront.Distribution(self, "myDistCustomPolicy",
                default_behavior=cloudfront.BehaviorOptions(
                    origin=bucket_origin,
                    cache_policy=my_cache_policy
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__318679262f857779b61fd4a382e472950998585f7d42494d8b33b43cd2dd1a55)
            check_type(argname="argument cache_policy_name", value=cache_policy_name, expected_type=type_hints["cache_policy_name"])
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument cookie_behavior", value=cookie_behavior, expected_type=type_hints["cookie_behavior"])
            check_type(argname="argument default_ttl", value=default_ttl, expected_type=type_hints["default_ttl"])
            check_type(argname="argument enable_accept_encoding_brotli", value=enable_accept_encoding_brotli, expected_type=type_hints["enable_accept_encoding_brotli"])
            check_type(argname="argument enable_accept_encoding_gzip", value=enable_accept_encoding_gzip, expected_type=type_hints["enable_accept_encoding_gzip"])
            check_type(argname="argument header_behavior", value=header_behavior, expected_type=type_hints["header_behavior"])
            check_type(argname="argument max_ttl", value=max_ttl, expected_type=type_hints["max_ttl"])
            check_type(argname="argument min_ttl", value=min_ttl, expected_type=type_hints["min_ttl"])
            check_type(argname="argument query_string_behavior", value=query_string_behavior, expected_type=type_hints["query_string_behavior"])
        self._values: typing.Dict[builtins.str, typing.Any] = {}
        if cache_policy_name is not None:
            self._values["cache_policy_name"] = cache_policy_name
        if comment is not None:
            self._values["comment"] = comment
        if cookie_behavior is not None:
            self._values["cookie_behavior"] = cookie_behavior
        if default_ttl is not None:
            self._values["default_ttl"] = default_ttl
        if enable_accept_encoding_brotli is not None:
            self._values["enable_accept_encoding_brotli"] = enable_accept_encoding_brotli
        if enable_accept_encoding_gzip is not None:
            self._values["enable_accept_encoding_gzip"] = enable_accept_encoding_gzip
        if header_behavior is not None:
            self._values["header_behavior"] = header_behavior
        if max_ttl is not None:
            self._values["max_ttl"] = max_ttl
        if min_ttl is not None:
            self._values["min_ttl"] = min_ttl
        if query_string_behavior is not None:
            self._values["query_string_behavior"] = query_string_behavior

    @builtins.property
    def cache_policy_name(self) -> typing.Optional[builtins.str]:
        '''A unique name to identify the cache policy.

        The name must only include '-', '_', or alphanumeric characters.

        :default: - generated from the ``id``
        '''
        result = self._values.get("cache_policy_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment to describe the cache policy.

        :default: - no comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cookie_behavior(self) -> typing.Optional[CacheCookieBehavior]:
        '''Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.

        :default: CacheCookieBehavior.none()
        '''
        result = self._values.get("cookie_behavior")
        return typing.cast(typing.Optional[CacheCookieBehavior], result)

    @builtins.property
    def default_ttl(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The default amount of time for objects to stay in the CloudFront cache.

        Only used when the origin does not send Cache-Control or Expires headers with the object.

        :default: - The greater of 1 day and ``minTtl``
        '''
        result = self._values.get("default_ttl")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def enable_accept_encoding_brotli(self) -> typing.Optional[builtins.bool]:
        '''Whether to normalize and include the ``Accept-Encoding`` header in the cache key when the ``Accept-Encoding`` header is 'br'.

        :default: false
        '''
        result = self._values.get("enable_accept_encoding_brotli")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_accept_encoding_gzip(self) -> typing.Optional[builtins.bool]:
        '''Whether to normalize and include the ``Accept-Encoding`` header in the cache key when the ``Accept-Encoding`` header is 'gzip'.

        :default: false
        '''
        result = self._values.get("enable_accept_encoding_gzip")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def header_behavior(self) -> typing.Optional[CacheHeaderBehavior]:
        '''Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.

        :default: CacheHeaderBehavior.none()
        '''
        result = self._values.get("header_behavior")
        return typing.cast(typing.Optional[CacheHeaderBehavior], result)

    @builtins.property
    def max_ttl(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The maximum amount of time for objects to stay in the CloudFront cache.

        CloudFront uses this value only when the origin sends Cache-Control or Expires headers with the object.

        :default: - The greater of 1 year and ``defaultTtl``
        '''
        result = self._values.get("max_ttl")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def min_ttl(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The minimum amount of time for objects to stay in the CloudFront cache.

        :default: Duration.seconds(0)
        '''
        result = self._values.get("min_ttl")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def query_string_behavior(self) -> typing.Optional["CacheQueryStringBehavior"]:
        '''Determines whether any query strings are included in the cache key and automatically included in requests that CloudFront sends to the origin.

        :default: CacheQueryStringBehavior.none()
        '''
        result = self._values.get("query_string_behavior")
        return typing.cast(typing.Optional["CacheQueryStringBehavior"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CachePolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CacheQueryStringBehavior(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CacheQueryStringBehavior",
):
    '''Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.

    :exampleMetadata: infused

    Example::

        # Creating a custom cache policy for a Distribution -- all parameters optional
        # bucket_origin: origins.S3Origin
        
        my_cache_policy = cloudfront.CachePolicy(self, "myCachePolicy",
            cache_policy_name="MyPolicy",
            comment="A default policy",
            default_ttl=Duration.days(2),
            min_ttl=Duration.minutes(1),
            max_ttl=Duration.days(10),
            cookie_behavior=cloudfront.CacheCookieBehavior.all(),
            header_behavior=cloudfront.CacheHeaderBehavior.allow_list("X-CustomHeader"),
            query_string_behavior=cloudfront.CacheQueryStringBehavior.deny_list("username"),
            enable_accept_encoding_gzip=True,
            enable_accept_encoding_brotli=True
        )
        cloudfront.Distribution(self, "myDistCustomPolicy",
            default_behavior=cloudfront.BehaviorOptions(
                origin=bucket_origin,
                cache_policy=my_cache_policy
            )
        )
    '''

    @jsii.member(jsii_name="all")
    @builtins.classmethod
    def all(cls) -> "CacheQueryStringBehavior":
        '''All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.'''
        return typing.cast("CacheQueryStringBehavior", jsii.sinvoke(cls, "all", []))

    @jsii.member(jsii_name="allowList")
    @builtins.classmethod
    def allow_list(cls, *query_strings: builtins.str) -> "CacheQueryStringBehavior":
        '''Only the provided ``queryStrings`` are included in the cache key and automatically included in requests that CloudFront sends to the origin.

        :param query_strings: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5e9e684fc91bc5fc575080937cf4264822ae2979784b38f05941235fea3d848f)
            check_type(argname="argument query_strings", value=query_strings, expected_type=typing.Tuple[type_hints["query_strings"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("CacheQueryStringBehavior", jsii.sinvoke(cls, "allowList", [*query_strings]))

    @jsii.member(jsii_name="denyList")
    @builtins.classmethod
    def deny_list(cls, *query_strings: builtins.str) -> "CacheQueryStringBehavior":
        '''All query strings except the provided ``queryStrings`` are included in the cache key and automatically included in requests that CloudFront sends to the origin.

        :param query_strings: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__57a120c8c733bc30c05cc993bf9838f00ead49560068a6d56ec78e1fab2db3af)
            check_type(argname="argument query_strings", value=query_strings, expected_type=typing.Tuple[type_hints["query_strings"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("CacheQueryStringBehavior", jsii.sinvoke(cls, "denyList", [*query_strings]))

    @jsii.member(jsii_name="none")
    @builtins.classmethod
    def none(cls) -> "CacheQueryStringBehavior":
        '''Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.'''
        return typing.cast("CacheQueryStringBehavior", jsii.sinvoke(cls, "none", []))

    @builtins.property
    @jsii.member(jsii_name="behavior")
    def behavior(self) -> builtins.str:
        '''The behavior of query strings -- allow all, none, only an allow list, or a deny list.'''
        return typing.cast(builtins.str, jsii.get(self, "behavior"))

    @builtins.property
    @jsii.member(jsii_name="queryStrings")
    def query_strings(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The query strings to allow or deny, if the behavior is an allow or deny list.'''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "queryStrings"))


class CachedMethods(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CachedMethods",
):
    '''The HTTP methods that the Behavior will cache requests on.

    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cached_methods = cloudfront.CachedMethods.CACHE_GET_HEAD
    '''

    @jsii.python.classproperty
    @jsii.member(jsii_name="CACHE_GET_HEAD")
    def CACHE_GET_HEAD(cls) -> "CachedMethods":
        '''HEAD and GET.'''
        return typing.cast("CachedMethods", jsii.sget(cls, "CACHE_GET_HEAD"))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CACHE_GET_HEAD_OPTIONS")
    def CACHE_GET_HEAD_OPTIONS(cls) -> "CachedMethods":
        '''HEAD, GET, and OPTIONS.'''
        return typing.cast("CachedMethods", jsii.sget(cls, "CACHE_GET_HEAD_OPTIONS"))

    @builtins.property
    @jsii.member(jsii_name="methods")
    def methods(self) -> typing.List[builtins.str]:
        '''HTTP methods supported.'''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "methods"))


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnCachePolicy(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnCachePolicy",
):
    '''A CloudFormation ``AWS::CloudFront::CachePolicy``.

    A cache policy.

    When it’s attached to a cache behavior, the cache policy determines the following:

    - The values that CloudFront includes in the cache key. These values can include HTTP headers, cookies, and URL query strings. CloudFront uses the cache key to find an object in its cache that it can return to the viewer.
    - The default, minimum, and maximum time to live (TTL) values that you want objects to stay in the CloudFront cache.

    The headers, cookies, and query strings that are included in the cache key are automatically included in requests that CloudFront sends to the origin. CloudFront sends a request when it can’t find a valid object in its cache that matches the request’s cache key. If you want to send values to the origin but *not* include them in the cache key, use ``OriginRequestPolicy`` .

    :cloudformationResource: AWS::CloudFront::CachePolicy
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cachepolicy.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_cache_policy = cloudfront.CfnCachePolicy(self, "MyCfnCachePolicy",
            cache_policy_config=cloudfront.CfnCachePolicy.CachePolicyConfigProperty(
                default_ttl=123,
                max_ttl=123,
                min_ttl=123,
                name="name",
                parameters_in_cache_key_and_forwarded_to_origin=cloudfront.CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty(
                    cookies_config=cloudfront.CfnCachePolicy.CookiesConfigProperty(
                        cookie_behavior="cookieBehavior",
        
                        # the properties below are optional
                        cookies=["cookies"]
                    ),
                    enable_accept_encoding_gzip=False,
                    headers_config=cloudfront.CfnCachePolicy.HeadersConfigProperty(
                        header_behavior="headerBehavior",
        
                        # the properties below are optional
                        headers=["headers"]
                    ),
                    query_strings_config=cloudfront.CfnCachePolicy.QueryStringsConfigProperty(
                        query_string_behavior="queryStringBehavior",
        
                        # the properties below are optional
                        query_strings=["queryStrings"]
                    ),
        
                    # the properties below are optional
                    enable_accept_encoding_brotli=False
                ),
        
                # the properties below are optional
                comment="comment"
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        cache_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCachePolicy.CachePolicyConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::CachePolicy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cache_policy_config: The cache policy configuration.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0f137f8f408ba8d1a90a678f5ba4594f688b346cc3c4f148641e8427db1a69d3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCachePolicyProps(cache_policy_config=cache_policy_config)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6acaa7d26400b84857936b2739b952bea5519bca164ff700c5eeef388f004832)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__7213d756b676bbbc597bc431a8140b3550ce54b75cf3a8ae72bedebb2632fcd0)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The unique identifier for the cache policy.

        For example: ``2766f7b2-75c5-41c6-8f06-bf4303a2f2f5`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrLastModifiedTime")
    def attr_last_modified_time(self) -> builtins.str:
        '''The date and time when the cache policy was last modified.

        :cloudformationAttribute: LastModifiedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLastModifiedTime"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="cachePolicyConfig")
    def cache_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.CachePolicyConfigProperty"]:
        '''The cache policy configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cachepolicy.html#cfn-cloudfront-cachepolicy-cachepolicyconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.CachePolicyConfigProperty"], jsii.get(self, "cachePolicyConfig"))

    @cache_policy_config.setter
    def cache_policy_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.CachePolicyConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__264b0fd7cdce46d82fbc62dd541a4a3f41c0c644c5143585f9218dc3dffb490c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cachePolicyConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnCachePolicy.CachePolicyConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "default_ttl": "defaultTtl",
            "max_ttl": "maxTtl",
            "min_ttl": "minTtl",
            "name": "name",
            "parameters_in_cache_key_and_forwarded_to_origin": "parametersInCacheKeyAndForwardedToOrigin",
            "comment": "comment",
        },
    )
    class CachePolicyConfigProperty:
        def __init__(
            self,
            *,
            default_ttl: jsii.Number,
            max_ttl: jsii.Number,
            min_ttl: jsii.Number,
            name: builtins.str,
            parameters_in_cache_key_and_forwarded_to_origin: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty", typing.Dict[builtins.str, typing.Any]]],
            comment: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A cache policy configuration.

            This configuration determines the following:

            - The values that CloudFront includes in the cache key. These values can include HTTP headers, cookies, and URL query strings. CloudFront uses the cache key to find an object in its cache that it can return to the viewer.
            - The default, minimum, and maximum time to live (TTL) values that you want objects to stay in the CloudFront cache.

            The headers, cookies, and query strings that are included in the cache key are automatically included in requests that CloudFront sends to the origin. CloudFront sends a request when it can’t find a valid object in its cache that matches the request’s cache key. If you want to send values to the origin but *not* include them in the cache key, use ``OriginRequestPolicy`` .

            :param default_ttl: The default amount of time, in seconds, that you want objects to stay in the CloudFront cache before CloudFront sends another request to the origin to see if the object has been updated. CloudFront uses this value as the object’s time to live (TTL) only when the origin does *not* send ``Cache-Control`` or ``Expires`` headers with the object. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* . The default value for this field is 86400 seconds (one day). If the value of ``MinTTL`` is more than 86400 seconds, then the default value for this field is the same as the value of ``MinTTL`` .
            :param max_ttl: The maximum amount of time, in seconds, that objects stay in the CloudFront cache before CloudFront sends another request to the origin to see if the object has been updated. CloudFront uses this value only when the origin sends ``Cache-Control`` or ``Expires`` headers with the object. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* . The default value for this field is 31536000 seconds (one year). If the value of ``MinTTL`` or ``DefaultTTL`` is more than 31536000 seconds, then the default value for this field is the same as the value of ``DefaultTTL`` .
            :param min_ttl: The minimum amount of time, in seconds, that you want objects to stay in the CloudFront cache before CloudFront sends another request to the origin to see if the object has been updated. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param name: A unique name to identify the cache policy.
            :param parameters_in_cache_key_and_forwarded_to_origin: The HTTP headers, cookies, and URL query strings to include in the cache key. The values included in the cache key are automatically included in requests that CloudFront sends to the origin.
            :param comment: A comment to describe the cache policy. The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cachepolicyconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                cache_policy_config_property = cloudfront.CfnCachePolicy.CachePolicyConfigProperty(
                    default_ttl=123,
                    max_ttl=123,
                    min_ttl=123,
                    name="name",
                    parameters_in_cache_key_and_forwarded_to_origin=cloudfront.CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty(
                        cookies_config=cloudfront.CfnCachePolicy.CookiesConfigProperty(
                            cookie_behavior="cookieBehavior",
                
                            # the properties below are optional
                            cookies=["cookies"]
                        ),
                        enable_accept_encoding_gzip=False,
                        headers_config=cloudfront.CfnCachePolicy.HeadersConfigProperty(
                            header_behavior="headerBehavior",
                
                            # the properties below are optional
                            headers=["headers"]
                        ),
                        query_strings_config=cloudfront.CfnCachePolicy.QueryStringsConfigProperty(
                            query_string_behavior="queryStringBehavior",
                
                            # the properties below are optional
                            query_strings=["queryStrings"]
                        ),
                
                        # the properties below are optional
                        enable_accept_encoding_brotli=False
                    ),
                
                    # the properties below are optional
                    comment="comment"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6dc45e36bb50ad58f496056995646e15de61abdca3071d3dece2b94cbfda5ebf)
                check_type(argname="argument default_ttl", value=default_ttl, expected_type=type_hints["default_ttl"])
                check_type(argname="argument max_ttl", value=max_ttl, expected_type=type_hints["max_ttl"])
                check_type(argname="argument min_ttl", value=min_ttl, expected_type=type_hints["min_ttl"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument parameters_in_cache_key_and_forwarded_to_origin", value=parameters_in_cache_key_and_forwarded_to_origin, expected_type=type_hints["parameters_in_cache_key_and_forwarded_to_origin"])
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "default_ttl": default_ttl,
                "max_ttl": max_ttl,
                "min_ttl": min_ttl,
                "name": name,
                "parameters_in_cache_key_and_forwarded_to_origin": parameters_in_cache_key_and_forwarded_to_origin,
            }
            if comment is not None:
                self._values["comment"] = comment

        @builtins.property
        def default_ttl(self) -> jsii.Number:
            '''The default amount of time, in seconds, that you want objects to stay in the CloudFront cache before CloudFront sends another request to the origin to see if the object has been updated.

            CloudFront uses this value as the object’s time to live (TTL) only when the origin does *not* send ``Cache-Control`` or ``Expires`` headers with the object. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            The default value for this field is 86400 seconds (one day). If the value of ``MinTTL`` is more than 86400 seconds, then the default value for this field is the same as the value of ``MinTTL`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cachepolicyconfig.html#cfn-cloudfront-cachepolicy-cachepolicyconfig-defaultttl
            '''
            result = self._values.get("default_ttl")
            assert result is not None, "Required property 'default_ttl' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_ttl(self) -> jsii.Number:
            '''The maximum amount of time, in seconds, that objects stay in the CloudFront cache before CloudFront sends another request to the origin to see if the object has been updated.

            CloudFront uses this value only when the origin sends ``Cache-Control`` or ``Expires`` headers with the object. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            The default value for this field is 31536000 seconds (one year). If the value of ``MinTTL`` or ``DefaultTTL`` is more than 31536000 seconds, then the default value for this field is the same as the value of ``DefaultTTL`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cachepolicyconfig.html#cfn-cloudfront-cachepolicy-cachepolicyconfig-maxttl
            '''
            result = self._values.get("max_ttl")
            assert result is not None, "Required property 'max_ttl' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def min_ttl(self) -> jsii.Number:
            '''The minimum amount of time, in seconds, that you want objects to stay in the CloudFront cache before CloudFront sends another request to the origin to see if the object has been updated.

            For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cachepolicyconfig.html#cfn-cloudfront-cachepolicy-cachepolicyconfig-minttl
            '''
            result = self._values.get("min_ttl")
            assert result is not None, "Required property 'min_ttl' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''A unique name to identify the cache policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cachepolicyconfig.html#cfn-cloudfront-cachepolicy-cachepolicyconfig-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def parameters_in_cache_key_and_forwarded_to_origin(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty"]:
            '''The HTTP headers, cookies, and URL query strings to include in the cache key.

            The values included in the cache key are automatically included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cachepolicyconfig.html#cfn-cloudfront-cachepolicy-cachepolicyconfig-parametersincachekeyandforwardedtoorigin
            '''
            result = self._values.get("parameters_in_cache_key_and_forwarded_to_origin")
            assert result is not None, "Required property 'parameters_in_cache_key_and_forwarded_to_origin' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty"], result)

        @builtins.property
        def comment(self) -> typing.Optional[builtins.str]:
            '''A comment to describe the cache policy.

            The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cachepolicyconfig.html#cfn-cloudfront-cachepolicy-cachepolicyconfig-comment
            '''
            result = self._values.get("comment")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CachePolicyConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnCachePolicy.CookiesConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"cookie_behavior": "cookieBehavior", "cookies": "cookies"},
    )
    class CookiesConfigProperty:
        def __init__(
            self,
            *,
            cookie_behavior: builtins.str,
            cookies: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that determines whether any cookies in viewer requests (and if so, which cookies) are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            :param cookie_behavior: Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin. Valid values are: - ``none`` – Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any cookies that are listed in an ``OriginRequestPolicy`` *are* included in origin requests. - ``whitelist`` – The cookies in viewer requests that are listed in the ``CookieNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin. - ``allExcept`` – All cookies in viewer requests that are **not** listed in the ``CookieNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin. - ``all`` – All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
            :param cookies: Contains a list of cookie names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cookiesconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                cookies_config_property = cloudfront.CfnCachePolicy.CookiesConfigProperty(
                    cookie_behavior="cookieBehavior",
                
                    # the properties below are optional
                    cookies=["cookies"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__dcd762f41d7e9e15acf0ff0dc7dabe6ae3dc94e3e8c31974a81db93adface208)
                check_type(argname="argument cookie_behavior", value=cookie_behavior, expected_type=type_hints["cookie_behavior"])
                check_type(argname="argument cookies", value=cookies, expected_type=type_hints["cookies"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cookie_behavior": cookie_behavior,
            }
            if cookies is not None:
                self._values["cookies"] = cookies

        @builtins.property
        def cookie_behavior(self) -> builtins.str:
            '''Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            Valid values are:

            - ``none`` – Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any cookies that are listed in an ``OriginRequestPolicy`` *are* included in origin requests.
            - ``whitelist`` – The cookies in viewer requests that are listed in the ``CookieNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin.
            - ``allExcept`` – All cookies in viewer requests that are **not** listed in the ``CookieNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin.
            - ``all`` – All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cookiesconfig.html#cfn-cloudfront-cachepolicy-cookiesconfig-cookiebehavior
            '''
            result = self._values.get("cookie_behavior")
            assert result is not None, "Required property 'cookie_behavior' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def cookies(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Contains a list of cookie names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-cookiesconfig.html#cfn-cloudfront-cachepolicy-cookiesconfig-cookies
            '''
            result = self._values.get("cookies")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CookiesConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnCachePolicy.HeadersConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"header_behavior": "headerBehavior", "headers": "headers"},
    )
    class HeadersConfigProperty:
        def __init__(
            self,
            *,
            header_behavior: builtins.str,
            headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that determines whether any HTTP headers (and if so, which headers) are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            :param header_behavior: Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin. Valid values are: - ``none`` – HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any headers that are listed in an ``OriginRequestPolicy`` *are* included in origin requests. - ``whitelist`` – The HTTP headers that are listed in the ``Headers`` type are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
            :param headers: Contains a list of HTTP header names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-headersconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                headers_config_property = cloudfront.CfnCachePolicy.HeadersConfigProperty(
                    header_behavior="headerBehavior",
                
                    # the properties below are optional
                    headers=["headers"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c49c636af6fef97c011c7a0fedd6b00088b8c905f0206740669f58ede1ab9161)
                check_type(argname="argument header_behavior", value=header_behavior, expected_type=type_hints["header_behavior"])
                check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "header_behavior": header_behavior,
            }
            if headers is not None:
                self._values["headers"] = headers

        @builtins.property
        def header_behavior(self) -> builtins.str:
            '''Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            Valid values are:

            - ``none`` – HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any headers that are listed in an ``OriginRequestPolicy`` *are* included in origin requests.
            - ``whitelist`` – The HTTP headers that are listed in the ``Headers`` type are included in the cache key and are automatically included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-headersconfig.html#cfn-cloudfront-cachepolicy-headersconfig-headerbehavior
            '''
            result = self._values.get("header_behavior")
            assert result is not None, "Required property 'header_behavior' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Contains a list of HTTP header names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-headersconfig.html#cfn-cloudfront-cachepolicy-headersconfig-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HeadersConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cookies_config": "cookiesConfig",
            "enable_accept_encoding_gzip": "enableAcceptEncodingGzip",
            "headers_config": "headersConfig",
            "query_strings_config": "queryStringsConfig",
            "enable_accept_encoding_brotli": "enableAcceptEncodingBrotli",
        },
    )
    class ParametersInCacheKeyAndForwardedToOriginProperty:
        def __init__(
            self,
            *,
            cookies_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCachePolicy.CookiesConfigProperty", typing.Dict[builtins.str, typing.Any]]],
            enable_accept_encoding_gzip: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            headers_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCachePolicy.HeadersConfigProperty", typing.Dict[builtins.str, typing.Any]]],
            query_strings_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCachePolicy.QueryStringsConfigProperty", typing.Dict[builtins.str, typing.Any]]],
            enable_accept_encoding_brotli: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        ) -> None:
            '''This object determines the values that CloudFront includes in the cache key.

            These values can include HTTP headers, cookies, and URL query strings. CloudFront uses the cache key to find an object in its cache that it can return to the viewer.

            The headers, cookies, and query strings that are included in the cache key are automatically included in requests that CloudFront sends to the origin. CloudFront sends a request when it can’t find an object in its cache that matches the request’s cache key. If you want to send values to the origin but *not* include them in the cache key, use ``OriginRequestPolicy`` .

            :param cookies_config: An object that determines whether any cookies in viewer requests (and if so, which cookies) are included in the cache key and automatically included in requests that CloudFront sends to the origin.
            :param enable_accept_encoding_gzip: A flag that can affect whether the ``Accept-Encoding`` HTTP header is included in the cache key and included in requests that CloudFront sends to the origin. This field is related to the ``EnableAcceptEncodingBrotli`` field. If one or both of these fields is ``true`` *and* the viewer request includes the ``Accept-Encoding`` header, then CloudFront does the following: - Normalizes the value of the viewer’s ``Accept-Encoding`` header - Includes the normalized header in the cache key - Includes the normalized header in the request to the origin, if a request is necessary For more information, see `Compression support <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects>`_ in the *Amazon CloudFront Developer Guide* . If you set this value to ``true`` , and this cache behavior also has an origin request policy attached, do not include the ``Accept-Encoding`` header in the origin request policy. CloudFront always includes the ``Accept-Encoding`` header in origin requests when the value of this field is ``true`` , so including this header in an origin request policy has no effect. If both of these fields are ``false`` , then CloudFront treats the ``Accept-Encoding`` header the same as any other HTTP header in the viewer request. By default, it’s not included in the cache key and it’s not included in origin requests. In this case, you can manually add ``Accept-Encoding`` to the headers whitelist like any other HTTP header.
            :param headers_config: An object that determines whether any HTTP headers (and if so, which headers) are included in the cache key and automatically included in requests that CloudFront sends to the origin.
            :param query_strings_config: An object that determines whether any URL query strings in viewer requests (and if so, which query strings) are included in the cache key and automatically included in requests that CloudFront sends to the origin.
            :param enable_accept_encoding_brotli: A flag that can affect whether the ``Accept-Encoding`` HTTP header is included in the cache key and included in requests that CloudFront sends to the origin. This field is related to the ``EnableAcceptEncodingGzip`` field. If one or both of these fields is ``true`` *and* the viewer request includes the ``Accept-Encoding`` header, then CloudFront does the following: - Normalizes the value of the viewer’s ``Accept-Encoding`` header - Includes the normalized header in the cache key - Includes the normalized header in the request to the origin, if a request is necessary For more information, see `Compression support <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects>`_ in the *Amazon CloudFront Developer Guide* . If you set this value to ``true`` , and this cache behavior also has an origin request policy attached, do not include the ``Accept-Encoding`` header in the origin request policy. CloudFront always includes the ``Accept-Encoding`` header in origin requests when the value of this field is ``true`` , so including this header in an origin request policy has no effect. If both of these fields are ``false`` , then CloudFront treats the ``Accept-Encoding`` header the same as any other HTTP header in the viewer request. By default, it’s not included in the cache key and it’s not included in origin requests. In this case, you can manually add ``Accept-Encoding`` to the headers whitelist like any other HTTP header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                parameters_in_cache_key_and_forwarded_to_origin_property = cloudfront.CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty(
                    cookies_config=cloudfront.CfnCachePolicy.CookiesConfigProperty(
                        cookie_behavior="cookieBehavior",
                
                        # the properties below are optional
                        cookies=["cookies"]
                    ),
                    enable_accept_encoding_gzip=False,
                    headers_config=cloudfront.CfnCachePolicy.HeadersConfigProperty(
                        header_behavior="headerBehavior",
                
                        # the properties below are optional
                        headers=["headers"]
                    ),
                    query_strings_config=cloudfront.CfnCachePolicy.QueryStringsConfigProperty(
                        query_string_behavior="queryStringBehavior",
                
                        # the properties below are optional
                        query_strings=["queryStrings"]
                    ),
                
                    # the properties below are optional
                    enable_accept_encoding_brotli=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a343bdf14384faf36b75fd2ed6222bddacbe6c298d032bcf6f80ca7376544992)
                check_type(argname="argument cookies_config", value=cookies_config, expected_type=type_hints["cookies_config"])
                check_type(argname="argument enable_accept_encoding_gzip", value=enable_accept_encoding_gzip, expected_type=type_hints["enable_accept_encoding_gzip"])
                check_type(argname="argument headers_config", value=headers_config, expected_type=type_hints["headers_config"])
                check_type(argname="argument query_strings_config", value=query_strings_config, expected_type=type_hints["query_strings_config"])
                check_type(argname="argument enable_accept_encoding_brotli", value=enable_accept_encoding_brotli, expected_type=type_hints["enable_accept_encoding_brotli"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cookies_config": cookies_config,
                "enable_accept_encoding_gzip": enable_accept_encoding_gzip,
                "headers_config": headers_config,
                "query_strings_config": query_strings_config,
            }
            if enable_accept_encoding_brotli is not None:
                self._values["enable_accept_encoding_brotli"] = enable_accept_encoding_brotli

        @builtins.property
        def cookies_config(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.CookiesConfigProperty"]:
            '''An object that determines whether any cookies in viewer requests (and if so, which cookies) are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin.html#cfn-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin-cookiesconfig
            '''
            result = self._values.get("cookies_config")
            assert result is not None, "Required property 'cookies_config' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.CookiesConfigProperty"], result)

        @builtins.property
        def enable_accept_encoding_gzip(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A flag that can affect whether the ``Accept-Encoding`` HTTP header is included in the cache key and included in requests that CloudFront sends to the origin.

            This field is related to the ``EnableAcceptEncodingBrotli`` field. If one or both of these fields is ``true`` *and* the viewer request includes the ``Accept-Encoding`` header, then CloudFront does the following:

            - Normalizes the value of the viewer’s ``Accept-Encoding`` header
            - Includes the normalized header in the cache key
            - Includes the normalized header in the request to the origin, if a request is necessary

            For more information, see `Compression support <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects>`_ in the *Amazon CloudFront Developer Guide* .

            If you set this value to ``true`` , and this cache behavior also has an origin request policy attached, do not include the ``Accept-Encoding`` header in the origin request policy. CloudFront always includes the ``Accept-Encoding`` header in origin requests when the value of this field is ``true`` , so including this header in an origin request policy has no effect.

            If both of these fields are ``false`` , then CloudFront treats the ``Accept-Encoding`` header the same as any other HTTP header in the viewer request. By default, it’s not included in the cache key and it’s not included in origin requests. In this case, you can manually add ``Accept-Encoding`` to the headers whitelist like any other HTTP header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin.html#cfn-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin-enableacceptencodinggzip
            '''
            result = self._values.get("enable_accept_encoding_gzip")
            assert result is not None, "Required property 'enable_accept_encoding_gzip' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def headers_config(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.HeadersConfigProperty"]:
            '''An object that determines whether any HTTP headers (and if so, which headers) are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin.html#cfn-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin-headersconfig
            '''
            result = self._values.get("headers_config")
            assert result is not None, "Required property 'headers_config' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.HeadersConfigProperty"], result)

        @builtins.property
        def query_strings_config(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.QueryStringsConfigProperty"]:
            '''An object that determines whether any URL query strings in viewer requests (and if so, which query strings) are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin.html#cfn-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin-querystringsconfig
            '''
            result = self._values.get("query_strings_config")
            assert result is not None, "Required property 'query_strings_config' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCachePolicy.QueryStringsConfigProperty"], result)

        @builtins.property
        def enable_accept_encoding_brotli(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A flag that can affect whether the ``Accept-Encoding`` HTTP header is included in the cache key and included in requests that CloudFront sends to the origin.

            This field is related to the ``EnableAcceptEncodingGzip`` field. If one or both of these fields is ``true`` *and* the viewer request includes the ``Accept-Encoding`` header, then CloudFront does the following:

            - Normalizes the value of the viewer’s ``Accept-Encoding`` header
            - Includes the normalized header in the cache key
            - Includes the normalized header in the request to the origin, if a request is necessary

            For more information, see `Compression support <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-policy-compressed-objects>`_ in the *Amazon CloudFront Developer Guide* .

            If you set this value to ``true`` , and this cache behavior also has an origin request policy attached, do not include the ``Accept-Encoding`` header in the origin request policy. CloudFront always includes the ``Accept-Encoding`` header in origin requests when the value of this field is ``true`` , so including this header in an origin request policy has no effect.

            If both of these fields are ``false`` , then CloudFront treats the ``Accept-Encoding`` header the same as any other HTTP header in the viewer request. By default, it’s not included in the cache key and it’s not included in origin requests. In this case, you can manually add ``Accept-Encoding`` to the headers whitelist like any other HTTP header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin.html#cfn-cloudfront-cachepolicy-parametersincachekeyandforwardedtoorigin-enableacceptencodingbrotli
            '''
            result = self._values.get("enable_accept_encoding_brotli")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ParametersInCacheKeyAndForwardedToOriginProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnCachePolicy.QueryStringsConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "query_string_behavior": "queryStringBehavior",
            "query_strings": "queryStrings",
        },
    )
    class QueryStringsConfigProperty:
        def __init__(
            self,
            *,
            query_string_behavior: builtins.str,
            query_strings: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that determines whether any URL query strings in viewer requests (and if so, which query strings) are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            :param query_string_behavior: Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin. Valid values are: - ``none`` – Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any query strings that are listed in an ``OriginRequestPolicy`` *are* included in origin requests. - ``whitelist`` – The query strings in viewer requests that are listed in the ``QueryStringNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin. - ``allExcept`` – All query strings in viewer requests that are **not** listed in the ``QueryStringNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin. - ``all`` – All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
            :param query_strings: Contains a list of query string names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-querystringsconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                query_strings_config_property = cloudfront.CfnCachePolicy.QueryStringsConfigProperty(
                    query_string_behavior="queryStringBehavior",
                
                    # the properties below are optional
                    query_strings=["queryStrings"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ea860077fa7b4f8734153e6897fa0a07aef73bc3d49332f40f4c90d0b432cd56)
                check_type(argname="argument query_string_behavior", value=query_string_behavior, expected_type=type_hints["query_string_behavior"])
                check_type(argname="argument query_strings", value=query_strings, expected_type=type_hints["query_strings"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "query_string_behavior": query_string_behavior,
            }
            if query_strings is not None:
                self._values["query_strings"] = query_strings

        @builtins.property
        def query_string_behavior(self) -> builtins.str:
            '''Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.

            Valid values are:

            - ``none`` – Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any query strings that are listed in an ``OriginRequestPolicy`` *are* included in origin requests.
            - ``whitelist`` – The query strings in viewer requests that are listed in the ``QueryStringNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin.
            - ``allExcept`` – All query strings in viewer requests that are **not** listed in the ``QueryStringNames`` type are included in the cache key and automatically included in requests that CloudFront sends to the origin.
            - ``all`` – All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-querystringsconfig.html#cfn-cloudfront-cachepolicy-querystringsconfig-querystringbehavior
            '''
            result = self._values.get("query_string_behavior")
            assert result is not None, "Required property 'query_string_behavior' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def query_strings(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Contains a list of query string names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cachepolicy-querystringsconfig.html#cfn-cloudfront-cachepolicy-querystringsconfig-querystrings
            '''
            result = self._values.get("query_strings")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryStringsConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnCachePolicyProps",
    jsii_struct_bases=[],
    name_mapping={"cache_policy_config": "cachePolicyConfig"},
)
class CfnCachePolicyProps:
    def __init__(
        self,
        *,
        cache_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnCachePolicy.CachePolicyConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnCachePolicy``.

        :param cache_policy_config: The cache policy configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cachepolicy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_cache_policy_props = cloudfront.CfnCachePolicyProps(
                cache_policy_config=cloudfront.CfnCachePolicy.CachePolicyConfigProperty(
                    default_ttl=123,
                    max_ttl=123,
                    min_ttl=123,
                    name="name",
                    parameters_in_cache_key_and_forwarded_to_origin=cloudfront.CfnCachePolicy.ParametersInCacheKeyAndForwardedToOriginProperty(
                        cookies_config=cloudfront.CfnCachePolicy.CookiesConfigProperty(
                            cookie_behavior="cookieBehavior",
            
                            # the properties below are optional
                            cookies=["cookies"]
                        ),
                        enable_accept_encoding_gzip=False,
                        headers_config=cloudfront.CfnCachePolicy.HeadersConfigProperty(
                            header_behavior="headerBehavior",
            
                            # the properties below are optional
                            headers=["headers"]
                        ),
                        query_strings_config=cloudfront.CfnCachePolicy.QueryStringsConfigProperty(
                            query_string_behavior="queryStringBehavior",
            
                            # the properties below are optional
                            query_strings=["queryStrings"]
                        ),
            
                        # the properties below are optional
                        enable_accept_encoding_brotli=False
                    ),
            
                    # the properties below are optional
                    comment="comment"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__60a95c952526c03fda99da5f3c6cf40a813f24c6e8e1e6e9fd5fa0af4a8d0458)
            check_type(argname="argument cache_policy_config", value=cache_policy_config, expected_type=type_hints["cache_policy_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cache_policy_config": cache_policy_config,
        }

    @builtins.property
    def cache_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCachePolicy.CachePolicyConfigProperty]:
        '''The cache policy configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cachepolicy.html#cfn-cloudfront-cachepolicy-cachepolicyconfig
        '''
        result = self._values.get("cache_policy_config")
        assert result is not None, "Required property 'cache_policy_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCachePolicy.CachePolicyConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCachePolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnCloudFrontOriginAccessIdentity(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnCloudFrontOriginAccessIdentity",
):
    '''A CloudFormation ``AWS::CloudFront::CloudFrontOriginAccessIdentity``.

    The request to create a new origin access identity (OAI). An origin access identity is a special CloudFront user that you can associate with Amazon S3 origins, so that you can secure all or just some of your Amazon S3 content. For more information, see `Restricting Access to Amazon S3 Content by Using an Origin Access Identity <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html>`_ in the *Amazon CloudFront Developer Guide* .

    :cloudformationResource: AWS::CloudFront::CloudFrontOriginAccessIdentity
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cloudfrontoriginaccessidentity.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_cloud_front_origin_access_identity = cloudfront.CfnCloudFrontOriginAccessIdentity(self, "MyCfnCloudFrontOriginAccessIdentity",
            cloud_front_origin_access_identity_config=cloudfront.CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty(
                comment="comment"
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        cloud_front_origin_access_identity_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::CloudFrontOriginAccessIdentity``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cloud_front_origin_access_identity_config: The current configuration information for the identity.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4c378ee48f7e98c7a3a1e9c806236a80d81fc28579470c5d133ff9955484c4f4)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnCloudFrontOriginAccessIdentityProps(
            cloud_front_origin_access_identity_config=cloud_front_origin_access_identity_config,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b252c2371c4dae33ca47cb966edf4bd57f208308d2b647e06c5d04ce86cdcf72)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e85598ba7d685818a99b947c7f7ae2a476110d717e43b89d9a829c0dfd6b1c07)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID for the origin access identity, for example, ``E74FTE3AJFJ256A`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrS3CanonicalUserId")
    def attr_s3_canonical_user_id(self) -> builtins.str:
        '''The Amazon S3 canonical user ID for the origin access identity, used when giving the origin access identity read permission to an object in Amazon S3.

        For example: ``b970b42360b81c8ddbd79d2f5df0069ba9033c8a79655752abe380cd6d63ba8bcf23384d568fcf89fc49700b5e11a0fd`` .

        :cloudformationAttribute: S3CanonicalUserId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrS3CanonicalUserId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="cloudFrontOriginAccessIdentityConfig")
    def cloud_front_origin_access_identity_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty"]:
        '''The current configuration information for the identity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cloudfrontoriginaccessidentity.html#cfn-cloudfront-cloudfrontoriginaccessidentity-cloudfrontoriginaccessidentityconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty"], jsii.get(self, "cloudFrontOriginAccessIdentityConfig"))

    @cloud_front_origin_access_identity_config.setter
    def cloud_front_origin_access_identity_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1f203d33d8e20fd0b046ef8c5cadf8428853f8ad4352939e14ca03e58576f2d7)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "cloudFrontOriginAccessIdentityConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"comment": "comment"},
    )
    class CloudFrontOriginAccessIdentityConfigProperty:
        def __init__(self, *, comment: builtins.str) -> None:
            '''Origin access identity configuration.

            Send a ``GET`` request to the ``/ *CloudFront API version* /CloudFront/identity ID/config`` resource.

            :param comment: A comment to describe the origin access identity. The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cloudfrontoriginaccessidentity-cloudfrontoriginaccessidentityconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                cloud_front_origin_access_identity_config_property = cloudfront.CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty(
                    comment="comment"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__995fe14cf84326c7a8cee1779f38f3ddd16a3d01ea87bbaf7a2983be285613fc)
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "comment": comment,
            }

        @builtins.property
        def comment(self) -> builtins.str:
            '''A comment to describe the origin access identity.

            The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-cloudfrontoriginaccessidentity-cloudfrontoriginaccessidentityconfig.html#cfn-cloudfront-cloudfrontoriginaccessidentity-cloudfrontoriginaccessidentityconfig-comment
            '''
            result = self._values.get("comment")
            assert result is not None, "Required property 'comment' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CloudFrontOriginAccessIdentityConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnCloudFrontOriginAccessIdentityProps",
    jsii_struct_bases=[],
    name_mapping={
        "cloud_front_origin_access_identity_config": "cloudFrontOriginAccessIdentityConfig",
    },
)
class CfnCloudFrontOriginAccessIdentityProps:
    def __init__(
        self,
        *,
        cloud_front_origin_access_identity_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnCloudFrontOriginAccessIdentity``.

        :param cloud_front_origin_access_identity_config: The current configuration information for the identity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cloudfrontoriginaccessidentity.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_cloud_front_origin_access_identity_props = cloudfront.CfnCloudFrontOriginAccessIdentityProps(
                cloud_front_origin_access_identity_config=cloudfront.CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty(
                    comment="comment"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b09417a2ed6c05edd8f3b9282547f89a3be70f1a9133d99e7a5ade9533307afa)
            check_type(argname="argument cloud_front_origin_access_identity_config", value=cloud_front_origin_access_identity_config, expected_type=type_hints["cloud_front_origin_access_identity_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "cloud_front_origin_access_identity_config": cloud_front_origin_access_identity_config,
        }

    @builtins.property
    def cloud_front_origin_access_identity_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty]:
        '''The current configuration information for the identity.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-cloudfrontoriginaccessidentity.html#cfn-cloudfront-cloudfrontoriginaccessidentity-cloudfrontoriginaccessidentityconfig
        '''
        result = self._values.get("cloud_front_origin_access_identity_config")
        assert result is not None, "Required property 'cloud_front_origin_access_identity_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnCloudFrontOriginAccessIdentity.CloudFrontOriginAccessIdentityConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCloudFrontOriginAccessIdentityProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnContinuousDeploymentPolicy(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnContinuousDeploymentPolicy",
):
    '''A CloudFormation ``AWS::CloudFront::ContinuousDeploymentPolicy``.

    Creates a continuous deployment policy that routes a subset of production traffic from a primary distribution to a staging distribution.

    After you create and update a staging distribution, you can use a continuous deployment policy to incrementally move traffic to the staging distribution. This enables you to test changes to a distribution's configuration before moving all of your production traffic to the new configuration.

    For more information, see `Using CloudFront continuous deployment to safely test CDN configuration changes <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/continuous-deployment.html>`_ in the *Amazon CloudFront Developer Guide* .

    :cloudformationResource: AWS::CloudFront::ContinuousDeploymentPolicy
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-continuousdeploymentpolicy.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_continuous_deployment_policy = cloudfront.CfnContinuousDeploymentPolicy(self, "MyCfnContinuousDeploymentPolicy",
            continuous_deployment_policy_config=cloudfront.CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty(
                enabled=False,
                staging_distribution_dns_names=["stagingDistributionDnsNames"],
        
                # the properties below are optional
                traffic_config=cloudfront.CfnContinuousDeploymentPolicy.TrafficConfigProperty(
                    type="type",
        
                    # the properties below are optional
                    single_header_config=cloudfront.CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty(
                        header="header",
                        value="value"
                    ),
                    single_weight_config=cloudfront.CfnContinuousDeploymentPolicy.SingleWeightConfigProperty(
                        weight=123,
        
                        # the properties below are optional
                        session_stickiness_config=cloudfront.CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty(
                            idle_ttl=123,
                            maximum_ttl=123
                        )
                    )
                )
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        continuous_deployment_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::ContinuousDeploymentPolicy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param continuous_deployment_policy_config: Contains the configuration for a continuous deployment policy.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4d6353d4948ee0df074698cdabf005bcaeea484303f2d6093c756c81adecdf67)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnContinuousDeploymentPolicyProps(
            continuous_deployment_policy_config=continuous_deployment_policy_config
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__526ee24c9d92411ab0c1b5c5add50702d9806fef63565312210845797a6566b1)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b41f8afa9a0cb58f957b26e0398a3a23a8d7b910115de9af0d44c9cb8e924bb4)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The identifier of the cotinuous deployment policy.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrLastModifiedTime")
    def attr_last_modified_time(self) -> builtins.str:
        '''The date and time when the continuous deployment policy was last modified.

        :cloudformationAttribute: LastModifiedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLastModifiedTime"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="continuousDeploymentPolicyConfig")
    def continuous_deployment_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty"]:
        '''Contains the configuration for a continuous deployment policy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-continuousdeploymentpolicy.html#cfn-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty"], jsii.get(self, "continuousDeploymentPolicyConfig"))

    @continuous_deployment_policy_config.setter
    def continuous_deployment_policy_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0ba8fc6158b0fbad615b3e47d3373061574c9c01152f4034dc4f444770eec82d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "continuousDeploymentPolicyConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "staging_distribution_dns_names": "stagingDistributionDnsNames",
            "traffic_config": "trafficConfig",
        },
    )
    class ContinuousDeploymentPolicyConfigProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            staging_distribution_dns_names: typing.Sequence[builtins.str],
            traffic_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnContinuousDeploymentPolicy.TrafficConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''Contains the configuration for a continuous deployment policy.

            :param enabled: A Boolean that indicates whether this continuous deployment policy is enabled (in effect). When this value is ``true`` , this policy is enabled and in effect. When this value is ``false`` , this policy is not enabled and has no effect.
            :param staging_distribution_dns_names: The CloudFront domain name of the staging distribution. For example: ``d111111abcdef8.cloudfront.net`` .
            :param traffic_config: Contains the parameters for routing production traffic from your primary to staging distributions.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                continuous_deployment_policy_config_property = cloudfront.CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty(
                    enabled=False,
                    staging_distribution_dns_names=["stagingDistributionDnsNames"],
                
                    # the properties below are optional
                    traffic_config=cloudfront.CfnContinuousDeploymentPolicy.TrafficConfigProperty(
                        type="type",
                
                        # the properties below are optional
                        single_header_config=cloudfront.CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty(
                            header="header",
                            value="value"
                        ),
                        single_weight_config=cloudfront.CfnContinuousDeploymentPolicy.SingleWeightConfigProperty(
                            weight=123,
                
                            # the properties below are optional
                            session_stickiness_config=cloudfront.CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty(
                                idle_ttl=123,
                                maximum_ttl=123
                            )
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__63482b9dbdd659f77fe4c0101642618c3148ca1cf6eb21f726760bb0e754560e)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument staging_distribution_dns_names", value=staging_distribution_dns_names, expected_type=type_hints["staging_distribution_dns_names"])
                check_type(argname="argument traffic_config", value=traffic_config, expected_type=type_hints["traffic_config"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enabled": enabled,
                "staging_distribution_dns_names": staging_distribution_dns_names,
            }
            if traffic_config is not None:
                self._values["traffic_config"] = traffic_config

        @builtins.property
        def enabled(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that indicates whether this continuous deployment policy is enabled (in effect).

            When this value is ``true`` , this policy is enabled and in effect. When this value is ``false`` , this policy is not enabled and has no effect.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig.html#cfn-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def staging_distribution_dns_names(self) -> typing.List[builtins.str]:
            '''The CloudFront domain name of the staging distribution.

            For example: ``d111111abcdef8.cloudfront.net`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig.html#cfn-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig-stagingdistributiondnsnames
            '''
            result = self._values.get("staging_distribution_dns_names")
            assert result is not None, "Required property 'staging_distribution_dns_names' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def traffic_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.TrafficConfigProperty"]]:
            '''Contains the parameters for routing production traffic from your primary to staging distributions.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig.html#cfn-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig-trafficconfig
            '''
            result = self._values.get("traffic_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.TrafficConfigProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContinuousDeploymentPolicyConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"idle_ttl": "idleTtl", "maximum_ttl": "maximumTtl"},
    )
    class SessionStickinessConfigProperty:
        def __init__(self, *, idle_ttl: jsii.Number, maximum_ttl: jsii.Number) -> None:
            '''Session stickiness provides the ability to define multiple requests from a single viewer as a single session.

            This prevents the potentially inconsistent experience of sending some of a given user's requests to your staging distribution, while others are sent to your primary distribution. Define the session duration using TTL values.

            :param idle_ttl: The amount of time after which you want sessions to cease if no requests are received. Allowed values are 300–3600 seconds (5–60 minutes).
            :param maximum_ttl: The maximum amount of time to consider requests from the viewer as being part of the same session. Allowed values are 300–3600 seconds (5–60 minutes).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-sessionstickinessconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                session_stickiness_config_property = cloudfront.CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty(
                    idle_ttl=123,
                    maximum_ttl=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ab96c7e06e9a473a85a289aa9584b59553c3ef24a7b5be9090c52256c47cfed5)
                check_type(argname="argument idle_ttl", value=idle_ttl, expected_type=type_hints["idle_ttl"])
                check_type(argname="argument maximum_ttl", value=maximum_ttl, expected_type=type_hints["maximum_ttl"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "idle_ttl": idle_ttl,
                "maximum_ttl": maximum_ttl,
            }

        @builtins.property
        def idle_ttl(self) -> jsii.Number:
            '''The amount of time after which you want sessions to cease if no requests are received.

            Allowed values are 300–3600 seconds (5–60 minutes).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-sessionstickinessconfig.html#cfn-cloudfront-continuousdeploymentpolicy-sessionstickinessconfig-idlettl
            '''
            result = self._values.get("idle_ttl")
            assert result is not None, "Required property 'idle_ttl' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def maximum_ttl(self) -> jsii.Number:
            '''The maximum amount of time to consider requests from the viewer as being part of the same session.

            Allowed values are 300–3600 seconds (5–60 minutes).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-sessionstickinessconfig.html#cfn-cloudfront-continuousdeploymentpolicy-sessionstickinessconfig-maximumttl
            '''
            result = self._values.get("maximum_ttl")
            assert result is not None, "Required property 'maximum_ttl' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SessionStickinessConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"header": "header", "value": "value"},
    )
    class SingleHeaderConfigProperty:
        def __init__(self, *, header: builtins.str, value: builtins.str) -> None:
            '''Determines which HTTP requests are sent to the staging distribution.

            :param header: The request header name that you want CloudFront to send to your staging distribution.
            :param value: The request header value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-singleheaderconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                single_header_config_property = cloudfront.CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty(
                    header="header",
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b772943e8059700f0ecd30d665fd0a16b67afc3c1cd31b72996fdbcaebd04879)
                check_type(argname="argument header", value=header, expected_type=type_hints["header"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "header": header,
                "value": value,
            }

        @builtins.property
        def header(self) -> builtins.str:
            '''The request header name that you want CloudFront to send to your staging distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-singleheaderconfig.html#cfn-cloudfront-continuousdeploymentpolicy-singleheaderconfig-header
            '''
            result = self._values.get("header")
            assert result is not None, "Required property 'header' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The request header value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-singleheaderconfig.html#cfn-cloudfront-continuousdeploymentpolicy-singleheaderconfig-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SingleHeaderConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnContinuousDeploymentPolicy.SingleWeightConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "weight": "weight",
            "session_stickiness_config": "sessionStickinessConfig",
        },
    )
    class SingleWeightConfigProperty:
        def __init__(
            self,
            *,
            weight: jsii.Number,
            session_stickiness_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''This configuration determines the percentage of HTTP requests that are sent to the staging distribution.

            :param weight: The percentage of traffic to send to a staging distribution, expressed as a decimal number between 0 and .15.
            :param session_stickiness_config: Session stickiness provides the ability to define multiple requests from a single viewer as a single session. This prevents the potentially inconsistent experience of sending some of a given user's requests to your staging distribution, while others are sent to your primary distribution. Define the session duration using TTL values.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-singleweightconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                single_weight_config_property = cloudfront.CfnContinuousDeploymentPolicy.SingleWeightConfigProperty(
                    weight=123,
                
                    # the properties below are optional
                    session_stickiness_config=cloudfront.CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty(
                        idle_ttl=123,
                        maximum_ttl=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f3fa3e897d0c02b69f2331e02ffec00329964e51ae4d01dcd7f3f9e9e3b468e2)
                check_type(argname="argument weight", value=weight, expected_type=type_hints["weight"])
                check_type(argname="argument session_stickiness_config", value=session_stickiness_config, expected_type=type_hints["session_stickiness_config"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "weight": weight,
            }
            if session_stickiness_config is not None:
                self._values["session_stickiness_config"] = session_stickiness_config

        @builtins.property
        def weight(self) -> jsii.Number:
            '''The percentage of traffic to send to a staging distribution, expressed as a decimal number between 0 and .15.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-singleweightconfig.html#cfn-cloudfront-continuousdeploymentpolicy-singleweightconfig-weight
            '''
            result = self._values.get("weight")
            assert result is not None, "Required property 'weight' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def session_stickiness_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty"]]:
            '''Session stickiness provides the ability to define multiple requests from a single viewer as a single session.

            This prevents the potentially inconsistent experience of sending some of a given user's requests to your staging distribution, while others are sent to your primary distribution. Define the session duration using TTL values.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-singleweightconfig.html#cfn-cloudfront-continuousdeploymentpolicy-singleweightconfig-sessionstickinessconfig
            '''
            result = self._values.get("session_stickiness_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SingleWeightConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnContinuousDeploymentPolicy.TrafficConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "single_header_config": "singleHeaderConfig",
            "single_weight_config": "singleWeightConfig",
        },
    )
    class TrafficConfigProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            single_header_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            single_weight_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnContinuousDeploymentPolicy.SingleWeightConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''The traffic configuration of your continuous deployment.

            :param type: The type of traffic configuration.
            :param single_header_config: Determines which HTTP requests are sent to the staging distribution.
            :param single_weight_config: Contains the percentage of traffic to send to the staging distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-trafficconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                traffic_config_property = cloudfront.CfnContinuousDeploymentPolicy.TrafficConfigProperty(
                    type="type",
                
                    # the properties below are optional
                    single_header_config=cloudfront.CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty(
                        header="header",
                        value="value"
                    ),
                    single_weight_config=cloudfront.CfnContinuousDeploymentPolicy.SingleWeightConfigProperty(
                        weight=123,
                
                        # the properties below are optional
                        session_stickiness_config=cloudfront.CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty(
                            idle_ttl=123,
                            maximum_ttl=123
                        )
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5d5f82eacd0e24306a0d443ecb57032ede0cdfa639b7bc23b70ed25669317818)
                check_type(argname="argument type", value=type, expected_type=type_hints["type"])
                check_type(argname="argument single_header_config", value=single_header_config, expected_type=type_hints["single_header_config"])
                check_type(argname="argument single_weight_config", value=single_weight_config, expected_type=type_hints["single_weight_config"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "type": type,
            }
            if single_header_config is not None:
                self._values["single_header_config"] = single_header_config
            if single_weight_config is not None:
                self._values["single_weight_config"] = single_weight_config

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of traffic configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-trafficconfig.html#cfn-cloudfront-continuousdeploymentpolicy-trafficconfig-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def single_header_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty"]]:
            '''Determines which HTTP requests are sent to the staging distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-trafficconfig.html#cfn-cloudfront-continuousdeploymentpolicy-trafficconfig-singleheaderconfig
            '''
            result = self._values.get("single_header_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty"]], result)

        @builtins.property
        def single_weight_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.SingleWeightConfigProperty"]]:
            '''Contains the percentage of traffic to send to the staging distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-continuousdeploymentpolicy-trafficconfig.html#cfn-cloudfront-continuousdeploymentpolicy-trafficconfig-singleweightconfig
            '''
            result = self._values.get("single_weight_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnContinuousDeploymentPolicy.SingleWeightConfigProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TrafficConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnContinuousDeploymentPolicyProps",
    jsii_struct_bases=[],
    name_mapping={
        "continuous_deployment_policy_config": "continuousDeploymentPolicyConfig",
    },
)
class CfnContinuousDeploymentPolicyProps:
    def __init__(
        self,
        *,
        continuous_deployment_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnContinuousDeploymentPolicy``.

        :param continuous_deployment_policy_config: Contains the configuration for a continuous deployment policy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-continuousdeploymentpolicy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_continuous_deployment_policy_props = cloudfront.CfnContinuousDeploymentPolicyProps(
                continuous_deployment_policy_config=cloudfront.CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty(
                    enabled=False,
                    staging_distribution_dns_names=["stagingDistributionDnsNames"],
            
                    # the properties below are optional
                    traffic_config=cloudfront.CfnContinuousDeploymentPolicy.TrafficConfigProperty(
                        type="type",
            
                        # the properties below are optional
                        single_header_config=cloudfront.CfnContinuousDeploymentPolicy.SingleHeaderConfigProperty(
                            header="header",
                            value="value"
                        ),
                        single_weight_config=cloudfront.CfnContinuousDeploymentPolicy.SingleWeightConfigProperty(
                            weight=123,
            
                            # the properties below are optional
                            session_stickiness_config=cloudfront.CfnContinuousDeploymentPolicy.SessionStickinessConfigProperty(
                                idle_ttl=123,
                                maximum_ttl=123
                            )
                        )
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__89b9e167b8e278045b329f16e2fa5a65b0dd05265316faa0d480d3095a457569)
            check_type(argname="argument continuous_deployment_policy_config", value=continuous_deployment_policy_config, expected_type=type_hints["continuous_deployment_policy_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "continuous_deployment_policy_config": continuous_deployment_policy_config,
        }

    @builtins.property
    def continuous_deployment_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty]:
        '''Contains the configuration for a continuous deployment policy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-continuousdeploymentpolicy.html#cfn-cloudfront-continuousdeploymentpolicy-continuousdeploymentpolicyconfig
        '''
        result = self._values.get("continuous_deployment_policy_config")
        assert result is not None, "Required property 'continuous_deployment_policy_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnContinuousDeploymentPolicy.ContinuousDeploymentPolicyConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnContinuousDeploymentPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnDistribution(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution",
):
    '''A CloudFormation ``AWS::CloudFront::Distribution``.

    A distribution tells CloudFront where you want content to be delivered from, and the details about how to track and manage content delivery.

    :cloudformationResource: AWS::CloudFront::Distribution
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-distribution.html
    :exampleMetadata: infused

    Example::

        # source_bucket: s3.Bucket
        
        
        my_distribution = cloudfront.Distribution(self, "MyCfWebDistribution",
            default_behavior=cloudfront.BehaviorOptions(
                origin=origins.S3Origin(source_bucket)
            )
        )
        cfn_distribution = my_distribution.node.default_child
        cfn_distribution.override_logical_id("MyDistributionCFDistribution3H55TI9Q")
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        distribution_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.DistributionConfigProperty", typing.Dict[builtins.str, typing.Any]]],
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::CloudFront::Distribution``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param distribution_config: The distribution’s configuration.
        :param tags: A complex type that contains zero or more ``Tag`` elements.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3a9e64484ef2b0579c2fed1a982526b7c83535215c1e6a8b93b9ed4154269d3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnDistributionProps(
            distribution_config=distribution_config, tags=tags
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ed1cc17b060730fe7cae019c70265a7cdef5ab4eb0fe075049168b3c59182a0a)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a47617b0c0fb366e8f183a1070d67513e5d581b58719a496487ff0ef627d59aa)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDomainName")
    def attr_domain_name(self) -> builtins.str:
        '''The domain name of the resource, such as ``d111111abcdef8.cloudfront.net`` .

        :cloudformationAttribute: DomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDomainName"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The identifier for the distribution, for example ``EDFDVBD632BHDS5`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''A complex type that contains zero or more ``Tag`` elements.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-distribution.html#cfn-cloudfront-distribution-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="distributionConfig")
    def distribution_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.DistributionConfigProperty"]:
        '''The distribution’s configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-distribution.html#cfn-cloudfront-distribution-distributionconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.DistributionConfigProperty"], jsii.get(self, "distributionConfig"))

    @distribution_config.setter
    def distribution_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.DistributionConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bf4fb6492e79bc5dc59162ce80bb43408765bc340153531f30adc7a0b8993cfd)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "distributionConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.CacheBehaviorProperty",
        jsii_struct_bases=[],
        name_mapping={
            "path_pattern": "pathPattern",
            "target_origin_id": "targetOriginId",
            "viewer_protocol_policy": "viewerProtocolPolicy",
            "allowed_methods": "allowedMethods",
            "cached_methods": "cachedMethods",
            "cache_policy_id": "cachePolicyId",
            "compress": "compress",
            "default_ttl": "defaultTtl",
            "field_level_encryption_id": "fieldLevelEncryptionId",
            "forwarded_values": "forwardedValues",
            "function_associations": "functionAssociations",
            "lambda_function_associations": "lambdaFunctionAssociations",
            "max_ttl": "maxTtl",
            "min_ttl": "minTtl",
            "origin_request_policy_id": "originRequestPolicyId",
            "realtime_log_config_arn": "realtimeLogConfigArn",
            "response_headers_policy_id": "responseHeadersPolicyId",
            "smooth_streaming": "smoothStreaming",
            "trusted_key_groups": "trustedKeyGroups",
            "trusted_signers": "trustedSigners",
        },
    )
    class CacheBehaviorProperty:
        def __init__(
            self,
            *,
            path_pattern: builtins.str,
            target_origin_id: builtins.str,
            viewer_protocol_policy: builtins.str,
            allowed_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
            cached_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
            cache_policy_id: typing.Optional[builtins.str] = None,
            compress: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            default_ttl: typing.Optional[jsii.Number] = None,
            field_level_encryption_id: typing.Optional[builtins.str] = None,
            forwarded_values: typing.Optional[typing.Union[typing.Union["CfnDistribution.ForwardedValuesProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]] = None,
            function_associations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.FunctionAssociationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            lambda_function_associations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.LambdaFunctionAssociationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            max_ttl: typing.Optional[jsii.Number] = None,
            min_ttl: typing.Optional[jsii.Number] = None,
            origin_request_policy_id: typing.Optional[builtins.str] = None,
            realtime_log_config_arn: typing.Optional[builtins.str] = None,
            response_headers_policy_id: typing.Optional[builtins.str] = None,
            smooth_streaming: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            trusted_key_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            trusted_signers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''A complex type that describes how CloudFront processes requests.

            You must create at least as many cache behaviors (including the default cache behavior) as you have origins if you want CloudFront to serve objects from all of the origins. Each cache behavior specifies the one origin from which you want CloudFront to get objects. If you have two origins and only the default cache behavior, the default cache behavior will cause CloudFront to get objects from one of the origins, but the other origin is never used.

            For the current quota (formerly known as limit) on the number of cache behaviors that you can add to a distribution, see `Quotas <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cloudfront-limits.html>`_ in the *Amazon CloudFront Developer Guide* .

            If you don’t want to specify any cache behaviors, include only an empty ``CacheBehaviors`` element. Don’t include an empty ``CacheBehavior`` element because this is invalid.

            To delete all cache behaviors in an existing distribution, update the distribution configuration and include only an empty ``CacheBehaviors`` element.

            To add, change, or remove one or more cache behaviors, update the distribution configuration and specify all of the cache behaviors that you want to include in the updated distribution.

            For more information about cache behaviors, see `Cache Behavior Settings <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesCacheBehavior>`_ in the *Amazon CloudFront Developer Guide* .

            :param path_pattern: The pattern (for example, ``images/*.jpg`` ) that specifies which requests to apply the behavior to. When CloudFront receives a viewer request, the requested path is compared with path patterns in the order in which cache behaviors are listed in the distribution. .. epigraph:: You can optionally include a slash ( ``/`` ) at the beginning of the path pattern. For example, ``/images/*.jpg`` . CloudFront behavior is the same with or without the leading ``/`` . The path pattern for the default cache behavior is ``*`` and cannot be changed. If the request for an object does not match the path pattern for any cache behaviors, CloudFront applies the behavior in the default cache behavior. For more information, see `Path Pattern <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesPathPattern>`_ in the *Amazon CloudFront Developer Guide* .
            :param target_origin_id: The value of ``ID`` for the origin that you want CloudFront to route requests to when they match this cache behavior.
            :param viewer_protocol_policy: The protocol that viewers can use to access the files in the origin specified by ``TargetOriginId`` when a request matches the path pattern in ``PathPattern`` . You can specify the following options: - ``allow-all`` : Viewers can use HTTP or HTTPS. - ``redirect-to-https`` : If a viewer submits an HTTP request, CloudFront returns an HTTP status code of 301 (Moved Permanently) to the viewer along with the HTTPS URL. The viewer then resubmits the request using the new URL. - ``https-only`` : If a viewer sends an HTTP request, CloudFront returns an HTTP status code of 403 (Forbidden). For more information about requiring the HTTPS protocol, see `Requiring HTTPS Between Viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-viewers-to-cloudfront.html>`_ in the *Amazon CloudFront Developer Guide* . .. epigraph:: The only way to guarantee that viewers retrieve an object that was fetched from the origin using HTTPS is never to use any other protocol to fetch the object. If you have recently changed from HTTP to HTTPS, we recommend that you clear your objects’ cache because cached objects are protocol agnostic. That means that an edge location will return an object from the cache regardless of whether the current request protocol matches the protocol used previously. For more information, see `Managing Cache Expiration <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param allowed_methods: A complex type that controls which HTTP methods CloudFront processes and forwards to your Amazon S3 bucket or your custom origin. There are three choices: - CloudFront forwards only ``GET`` and ``HEAD`` requests. - CloudFront forwards only ``GET`` , ``HEAD`` , and ``OPTIONS`` requests. - CloudFront forwards ``GET, HEAD, OPTIONS, PUT, PATCH, POST`` , and ``DELETE`` requests. If you pick the third choice, you may need to restrict access to your Amazon S3 bucket or to your custom origin so users can't perform operations that you don't want them to. For example, you might not want users to have permissions to delete objects from your origin.
            :param cached_methods: A complex type that controls whether CloudFront caches the response to requests using the specified HTTP methods. There are two choices: - CloudFront caches responses to ``GET`` and ``HEAD`` requests. - CloudFront caches responses to ``GET`` , ``HEAD`` , and ``OPTIONS`` requests. If you pick the second choice for your Amazon S3 Origin, you may need to forward Access-Control-Request-Method, Access-Control-Request-Headers, and Origin headers for the responses to be cached correctly.
            :param cache_policy_id: The unique identifier of the cache policy that is attached to this cache behavior. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . A ``CacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` .
            :param compress: Whether you want CloudFront to automatically compress certain files for this cache behavior. If so, specify true; if not, specify false. For more information, see `Serving Compressed Files <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param default_ttl: This field is deprecated. We recommend that you use the ``DefaultTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . The default amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin does not add HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param field_level_encryption_id: The value of ``ID`` for the field-level encryption configuration that you want CloudFront to use for encrypting specific fields of data for this cache behavior.
            :param forwarded_values: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. For more information, see `Working with policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html>`_ in the *Amazon CloudFront Developer Guide* . If you want to include values in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . If you want to send values to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* . A ``CacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` . A complex type that specifies how CloudFront handles query strings, cookies, and HTTP headers.
            :param function_associations: A list of CloudFront functions that are associated with this cache behavior. CloudFront functions must be published to the ``LIVE`` stage to associate them with a cache behavior.
            :param lambda_function_associations: A complex type that contains zero or more Lambda@Edge function associations for a cache behavior.
            :param max_ttl: This field is deprecated. We recommend that you use the ``MaxTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . The maximum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin adds HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param min_ttl: This field is deprecated. We recommend that you use the ``MinTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . The minimum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* . You must specify ``0`` for ``MinTTL`` if you configure CloudFront to forward all headers to your origin (under ``Headers`` , if you specify ``1`` for ``Quantity`` and ``*`` for ``Name`` ).
            :param origin_request_policy_id: The unique identifier of the origin request policy that is attached to this cache behavior. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param realtime_log_config_arn: The Amazon Resource Name (ARN) of the real-time log configuration that is attached to this cache behavior. For more information, see `Real-time logs <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param response_headers_policy_id: The identifier for a response headers policy.
            :param smooth_streaming: Indicates whether you want to distribute media files in the Microsoft Smooth Streaming format using the origin that is associated with this cache behavior. If so, specify ``true`` ; if not, specify ``false`` . If you specify ``true`` for ``SmoothStreaming`` , you can still distribute other content using this cache behavior if the content matches the value of ``PathPattern`` .
            :param trusted_key_groups: A list of key groups that CloudFront can use to validate signed URLs or signed cookies. When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with a private key whose corresponding public key is in the key group. The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param trusted_signers: .. epigraph:: We recommend using ``TrustedKeyGroups`` instead of ``TrustedSigners`` . A list of AWS account IDs whose public keys CloudFront can use to validate signed URLs or signed cookies. When a cache behavior contains trusted signers, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with the private key of a CloudFront key pair in the trusted signer’s AWS account . The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                cache_behavior_property = cloudfront.CfnDistribution.CacheBehaviorProperty(
                    path_pattern="pathPattern",
                    target_origin_id="targetOriginId",
                    viewer_protocol_policy="viewerProtocolPolicy",
                
                    # the properties below are optional
                    allowed_methods=["allowedMethods"],
                    cached_methods=["cachedMethods"],
                    cache_policy_id="cachePolicyId",
                    compress=False,
                    default_ttl=123,
                    field_level_encryption_id="fieldLevelEncryptionId",
                    forwarded_values=cloudfront.CfnDistribution.ForwardedValuesProperty(
                        query_string=False,
                
                        # the properties below are optional
                        cookies=cloudfront.CfnDistribution.CookiesProperty(
                            forward="forward",
                
                            # the properties below are optional
                            whitelisted_names=["whitelistedNames"]
                        ),
                        headers=["headers"],
                        query_string_cache_keys=["queryStringCacheKeys"]
                    ),
                    function_associations=[cloudfront.CfnDistribution.FunctionAssociationProperty(
                        event_type="eventType",
                        function_arn="functionArn"
                    )],
                    lambda_function_associations=[cloudfront.CfnDistribution.LambdaFunctionAssociationProperty(
                        event_type="eventType",
                        include_body=False,
                        lambda_function_arn="lambdaFunctionArn"
                    )],
                    max_ttl=123,
                    min_ttl=123,
                    origin_request_policy_id="originRequestPolicyId",
                    realtime_log_config_arn="realtimeLogConfigArn",
                    response_headers_policy_id="responseHeadersPolicyId",
                    smooth_streaming=False,
                    trusted_key_groups=["trustedKeyGroups"],
                    trusted_signers=["trustedSigners"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__134093ce3fa9e509a5d8ff0ba411b76c2d9f1e5c47e8c2bed4e57d4b14c0cc54)
                check_type(argname="argument path_pattern", value=path_pattern, expected_type=type_hints["path_pattern"])
                check_type(argname="argument target_origin_id", value=target_origin_id, expected_type=type_hints["target_origin_id"])
                check_type(argname="argument viewer_protocol_policy", value=viewer_protocol_policy, expected_type=type_hints["viewer_protocol_policy"])
                check_type(argname="argument allowed_methods", value=allowed_methods, expected_type=type_hints["allowed_methods"])
                check_type(argname="argument cached_methods", value=cached_methods, expected_type=type_hints["cached_methods"])
                check_type(argname="argument cache_policy_id", value=cache_policy_id, expected_type=type_hints["cache_policy_id"])
                check_type(argname="argument compress", value=compress, expected_type=type_hints["compress"])
                check_type(argname="argument default_ttl", value=default_ttl, expected_type=type_hints["default_ttl"])
                check_type(argname="argument field_level_encryption_id", value=field_level_encryption_id, expected_type=type_hints["field_level_encryption_id"])
                check_type(argname="argument forwarded_values", value=forwarded_values, expected_type=type_hints["forwarded_values"])
                check_type(argname="argument function_associations", value=function_associations, expected_type=type_hints["function_associations"])
                check_type(argname="argument lambda_function_associations", value=lambda_function_associations, expected_type=type_hints["lambda_function_associations"])
                check_type(argname="argument max_ttl", value=max_ttl, expected_type=type_hints["max_ttl"])
                check_type(argname="argument min_ttl", value=min_ttl, expected_type=type_hints["min_ttl"])
                check_type(argname="argument origin_request_policy_id", value=origin_request_policy_id, expected_type=type_hints["origin_request_policy_id"])
                check_type(argname="argument realtime_log_config_arn", value=realtime_log_config_arn, expected_type=type_hints["realtime_log_config_arn"])
                check_type(argname="argument response_headers_policy_id", value=response_headers_policy_id, expected_type=type_hints["response_headers_policy_id"])
                check_type(argname="argument smooth_streaming", value=smooth_streaming, expected_type=type_hints["smooth_streaming"])
                check_type(argname="argument trusted_key_groups", value=trusted_key_groups, expected_type=type_hints["trusted_key_groups"])
                check_type(argname="argument trusted_signers", value=trusted_signers, expected_type=type_hints["trusted_signers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "path_pattern": path_pattern,
                "target_origin_id": target_origin_id,
                "viewer_protocol_policy": viewer_protocol_policy,
            }
            if allowed_methods is not None:
                self._values["allowed_methods"] = allowed_methods
            if cached_methods is not None:
                self._values["cached_methods"] = cached_methods
            if cache_policy_id is not None:
                self._values["cache_policy_id"] = cache_policy_id
            if compress is not None:
                self._values["compress"] = compress
            if default_ttl is not None:
                self._values["default_ttl"] = default_ttl
            if field_level_encryption_id is not None:
                self._values["field_level_encryption_id"] = field_level_encryption_id
            if forwarded_values is not None:
                self._values["forwarded_values"] = forwarded_values
            if function_associations is not None:
                self._values["function_associations"] = function_associations
            if lambda_function_associations is not None:
                self._values["lambda_function_associations"] = lambda_function_associations
            if max_ttl is not None:
                self._values["max_ttl"] = max_ttl
            if min_ttl is not None:
                self._values["min_ttl"] = min_ttl
            if origin_request_policy_id is not None:
                self._values["origin_request_policy_id"] = origin_request_policy_id
            if realtime_log_config_arn is not None:
                self._values["realtime_log_config_arn"] = realtime_log_config_arn
            if response_headers_policy_id is not None:
                self._values["response_headers_policy_id"] = response_headers_policy_id
            if smooth_streaming is not None:
                self._values["smooth_streaming"] = smooth_streaming
            if trusted_key_groups is not None:
                self._values["trusted_key_groups"] = trusted_key_groups
            if trusted_signers is not None:
                self._values["trusted_signers"] = trusted_signers

        @builtins.property
        def path_pattern(self) -> builtins.str:
            '''The pattern (for example, ``images/*.jpg`` ) that specifies which requests to apply the behavior to. When CloudFront receives a viewer request, the requested path is compared with path patterns in the order in which cache behaviors are listed in the distribution.

            .. epigraph::

               You can optionally include a slash ( ``/`` ) at the beginning of the path pattern. For example, ``/images/*.jpg`` . CloudFront behavior is the same with or without the leading ``/`` .

            The path pattern for the default cache behavior is ``*`` and cannot be changed. If the request for an object does not match the path pattern for any cache behaviors, CloudFront applies the behavior in the default cache behavior.

            For more information, see `Path Pattern <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesPathPattern>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-pathpattern
            '''
            result = self._values.get("path_pattern")
            assert result is not None, "Required property 'path_pattern' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def target_origin_id(self) -> builtins.str:
            '''The value of ``ID`` for the origin that you want CloudFront to route requests to when they match this cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-targetoriginid
            '''
            result = self._values.get("target_origin_id")
            assert result is not None, "Required property 'target_origin_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def viewer_protocol_policy(self) -> builtins.str:
            '''The protocol that viewers can use to access the files in the origin specified by ``TargetOriginId`` when a request matches the path pattern in ``PathPattern`` .

            You can specify the following options:

            - ``allow-all`` : Viewers can use HTTP or HTTPS.
            - ``redirect-to-https`` : If a viewer submits an HTTP request, CloudFront returns an HTTP status code of 301 (Moved Permanently) to the viewer along with the HTTPS URL. The viewer then resubmits the request using the new URL.
            - ``https-only`` : If a viewer sends an HTTP request, CloudFront returns an HTTP status code of 403 (Forbidden).

            For more information about requiring the HTTPS protocol, see `Requiring HTTPS Between Viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-viewers-to-cloudfront.html>`_ in the *Amazon CloudFront Developer Guide* .
            .. epigraph::

               The only way to guarantee that viewers retrieve an object that was fetched from the origin using HTTPS is never to use any other protocol to fetch the object. If you have recently changed from HTTP to HTTPS, we recommend that you clear your objects’ cache because cached objects are protocol agnostic. That means that an edge location will return an object from the cache regardless of whether the current request protocol matches the protocol used previously. For more information, see `Managing Cache Expiration <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-viewerprotocolpolicy
            '''
            result = self._values.get("viewer_protocol_policy")
            assert result is not None, "Required property 'viewer_protocol_policy' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def allowed_methods(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A complex type that controls which HTTP methods CloudFront processes and forwards to your Amazon S3 bucket or your custom origin.

            There are three choices:

            - CloudFront forwards only ``GET`` and ``HEAD`` requests.
            - CloudFront forwards only ``GET`` , ``HEAD`` , and ``OPTIONS`` requests.
            - CloudFront forwards ``GET, HEAD, OPTIONS, PUT, PATCH, POST`` , and ``DELETE`` requests.

            If you pick the third choice, you may need to restrict access to your Amazon S3 bucket or to your custom origin so users can't perform operations that you don't want them to. For example, you might not want users to have permissions to delete objects from your origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-allowedmethods
            '''
            result = self._values.get("allowed_methods")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cached_methods(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A complex type that controls whether CloudFront caches the response to requests using the specified HTTP methods.

            There are two choices:

            - CloudFront caches responses to ``GET`` and ``HEAD`` requests.
            - CloudFront caches responses to ``GET`` , ``HEAD`` , and ``OPTIONS`` requests.

            If you pick the second choice for your Amazon S3 Origin, you may need to forward Access-Control-Request-Method, Access-Control-Request-Headers, and Origin headers for the responses to be cached correctly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-cachedmethods
            '''
            result = self._values.get("cached_methods")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cache_policy_id(self) -> typing.Optional[builtins.str]:
            '''The unique identifier of the cache policy that is attached to this cache behavior.

            For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            A ``CacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-cachepolicyid
            '''
            result = self._values.get("cache_policy_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def compress(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''Whether you want CloudFront to automatically compress certain files for this cache behavior.

            If so, specify true; if not, specify false. For more information, see `Serving Compressed Files <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-compress
            '''
            result = self._values.get("compress")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def default_ttl(self) -> typing.Optional[jsii.Number]:
            '''This field is deprecated.

            We recommend that you use the ``DefaultTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            The default amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin does not add HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-defaultttl
            '''
            result = self._values.get("default_ttl")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def field_level_encryption_id(self) -> typing.Optional[builtins.str]:
            '''The value of ``ID`` for the field-level encryption configuration that you want CloudFront to use for encrypting specific fields of data for this cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-fieldlevelencryptionid
            '''
            result = self._values.get("field_level_encryption_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def forwarded_values(
            self,
        ) -> typing.Optional[typing.Union["CfnDistribution.ForwardedValuesProperty", _aws_cdk_core_f4b25747.IResolvable]]:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field. For more information, see `Working with policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to include values in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send values to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            A ``CacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` .

            A complex type that specifies how CloudFront handles query strings, cookies, and HTTP headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-forwardedvalues
            '''
            result = self._values.get("forwarded_values")
            return typing.cast(typing.Optional[typing.Union["CfnDistribution.ForwardedValuesProperty", _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def function_associations(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.FunctionAssociationProperty"]]]]:
            '''A list of CloudFront functions that are associated with this cache behavior.

            CloudFront functions must be published to the ``LIVE`` stage to associate them with a cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-functionassociations
            '''
            result = self._values.get("function_associations")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.FunctionAssociationProperty"]]]], result)

        @builtins.property
        def lambda_function_associations(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LambdaFunctionAssociationProperty"]]]]:
            '''A complex type that contains zero or more Lambda@Edge function associations for a cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-lambdafunctionassociations
            '''
            result = self._values.get("lambda_function_associations")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LambdaFunctionAssociationProperty"]]]], result)

        @builtins.property
        def max_ttl(self) -> typing.Optional[jsii.Number]:
            '''This field is deprecated.

            We recommend that you use the ``MaxTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            The maximum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin adds HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-maxttl
            '''
            result = self._values.get("max_ttl")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min_ttl(self) -> typing.Optional[jsii.Number]:
            '''This field is deprecated.

            We recommend that you use the ``MinTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            The minimum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            You must specify ``0`` for ``MinTTL`` if you configure CloudFront to forward all headers to your origin (under ``Headers`` , if you specify ``1`` for ``Quantity`` and ``*`` for ``Name`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-minttl
            '''
            result = self._values.get("min_ttl")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def origin_request_policy_id(self) -> typing.Optional[builtins.str]:
            '''The unique identifier of the origin request policy that is attached to this cache behavior.

            For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-originrequestpolicyid
            '''
            result = self._values.get("origin_request_policy_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def realtime_log_config_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the real-time log configuration that is attached to this cache behavior.

            For more information, see `Real-time logs <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-realtimelogconfigarn
            '''
            result = self._values.get("realtime_log_config_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def response_headers_policy_id(self) -> typing.Optional[builtins.str]:
            '''The identifier for a response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-responseheaderspolicyid
            '''
            result = self._values.get("response_headers_policy_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def smooth_streaming(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''Indicates whether you want to distribute media files in the Microsoft Smooth Streaming format using the origin that is associated with this cache behavior.

            If so, specify ``true`` ; if not, specify ``false`` . If you specify ``true`` for ``SmoothStreaming`` , you can still distribute other content using this cache behavior if the content matches the value of ``PathPattern`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-smoothstreaming
            '''
            result = self._values.get("smooth_streaming")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def trusted_key_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of key groups that CloudFront can use to validate signed URLs or signed cookies.

            When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with a private key whose corresponding public key is in the key group. The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-trustedkeygroups
            '''
            result = self._values.get("trusted_key_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def trusted_signers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''.. epigraph::

   We recommend using ``TrustedKeyGroups`` instead of ``TrustedSigners`` .

            A list of AWS account IDs whose public keys CloudFront can use to validate signed URLs or signed cookies.

            When a cache behavior contains trusted signers, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with the private key of a CloudFront key pair in the trusted signer’s AWS account . The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cachebehavior.html#cfn-cloudfront-distribution-cachebehavior-trustedsigners
            '''
            result = self._values.get("trusted_signers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CacheBehaviorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.CookiesProperty",
        jsii_struct_bases=[],
        name_mapping={"forward": "forward", "whitelisted_names": "whitelistedNames"},
    )
    class CookiesProperty:
        def __init__(
            self,
            *,
            forward: builtins.str,
            whitelisted_names: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include cookies in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send cookies to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            A complex type that specifies whether you want CloudFront to forward cookies to the origin and, if so, which ones. For more information about forwarding cookies to the origin, see `How CloudFront Forwards, Caches, and Logs Cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Cookies.html>`_ in the *Amazon CloudFront Developer Guide* .

            :param forward: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. If you want to include cookies in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* . If you want to send cookies to the origin but not include them in the cache key, use origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* . Specifies which cookies to forward to the origin for this cache behavior: all, none, or the list of cookies specified in the ``WhitelistedNames`` complex type. Amazon S3 doesn't process cookies. When the cache behavior is forwarding requests to an Amazon S3 origin, specify none for the ``Forward`` element.
            :param whitelisted_names: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. If you want to include cookies in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* . If you want to send cookies to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* . Required if you specify ``whitelist`` for the value of ``Forward`` . A complex type that specifies how many different cookies you want CloudFront to forward to the origin for this cache behavior and, if you want to forward selected cookies, the names of those cookies. If you specify ``all`` or ``none`` for the value of ``Forward`` , omit ``WhitelistedNames`` . If you change the value of ``Forward`` from ``whitelist`` to ``all`` or ``none`` and you don't delete the ``WhitelistedNames`` element and its child elements, CloudFront deletes them automatically. For the current limit on the number of cookie names that you can whitelist for each cache behavior, see `CloudFront Limits <https://docs.aws.amazon.com/general/latest/gr/xrefaws_service_limits.html#limits_cloudfront>`_ in the *AWS General Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cookies.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                cookies_property = cloudfront.CfnDistribution.CookiesProperty(
                    forward="forward",
                
                    # the properties below are optional
                    whitelisted_names=["whitelistedNames"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6efbf6253fcb0631609ab48d142d7e99e76ee055c33bb2f4376ed2b758237e1d)
                check_type(argname="argument forward", value=forward, expected_type=type_hints["forward"])
                check_type(argname="argument whitelisted_names", value=whitelisted_names, expected_type=type_hints["whitelisted_names"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "forward": forward,
            }
            if whitelisted_names is not None:
                self._values["whitelisted_names"] = whitelisted_names

        @builtins.property
        def forward(self) -> builtins.str:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include cookies in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send cookies to the origin but not include them in the cache key, use origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            Specifies which cookies to forward to the origin for this cache behavior: all, none, or the list of cookies specified in the ``WhitelistedNames`` complex type.

            Amazon S3 doesn't process cookies. When the cache behavior is forwarding requests to an Amazon S3 origin, specify none for the ``Forward`` element.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cookies.html#cfn-cloudfront-distribution-cookies-forward
            '''
            result = self._values.get("forward")
            assert result is not None, "Required property 'forward' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def whitelisted_names(self) -> typing.Optional[typing.List[builtins.str]]:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include cookies in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send cookies to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            Required if you specify ``whitelist`` for the value of ``Forward`` . A complex type that specifies how many different cookies you want CloudFront to forward to the origin for this cache behavior and, if you want to forward selected cookies, the names of those cookies.

            If you specify ``all`` or ``none`` for the value of ``Forward`` , omit ``WhitelistedNames`` . If you change the value of ``Forward`` from ``whitelist`` to ``all`` or ``none`` and you don't delete the ``WhitelistedNames`` element and its child elements, CloudFront deletes them automatically.

            For the current limit on the number of cookie names that you can whitelist for each cache behavior, see `CloudFront Limits <https://docs.aws.amazon.com/general/latest/gr/xrefaws_service_limits.html#limits_cloudfront>`_ in the *AWS General Reference* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-cookies.html#cfn-cloudfront-distribution-cookies-whitelistednames
            '''
            result = self._values.get("whitelisted_names")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CookiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.CustomErrorResponseProperty",
        jsii_struct_bases=[],
        name_mapping={
            "error_code": "errorCode",
            "error_caching_min_ttl": "errorCachingMinTtl",
            "response_code": "responseCode",
            "response_page_path": "responsePagePath",
        },
    )
    class CustomErrorResponseProperty:
        def __init__(
            self,
            *,
            error_code: jsii.Number,
            error_caching_min_ttl: typing.Optional[jsii.Number] = None,
            response_code: typing.Optional[jsii.Number] = None,
            response_page_path: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A complex type that controls:.

            - Whether CloudFront replaces HTTP status codes in the 4xx and 5xx range with custom error messages before returning the response to the viewer.
            - How long CloudFront caches HTTP status codes in the 4xx and 5xx range.

            For more information about custom error pages, see `Customizing Error Responses <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html>`_ in the *Amazon CloudFront Developer Guide* .

            :param error_code: The HTTP status code for which you want to specify a custom error page and/or a caching duration.
            :param error_caching_min_ttl: The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in ``ErrorCode`` . When this time period has elapsed, CloudFront queries your origin to see whether the problem that caused the error has been resolved and the requested object is now available. For more information, see `Customizing Error Responses <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param response_code: The HTTP status code that you want CloudFront to return to the viewer along with the custom error page. There are a variety of reasons that you might want CloudFront to return a status code different from the status code that your origin returned to CloudFront, for example: - Some Internet devices (some firewalls and corporate proxies, for example) intercept HTTP 4xx and 5xx and prevent the response from being returned to the viewer. If you substitute ``200`` , the response typically won't be intercepted. - If you don't care about distinguishing among different client errors or server errors, you can specify ``400`` or ``500`` as the ``ResponseCode`` for all 4xx or 5xx errors. - You might want to return a ``200`` status code (OK) and static website so your customers don't know that your website is down. If you specify a value for ``ResponseCode`` , you must also specify a value for ``ResponsePagePath`` .
            :param response_page_path: The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the HTTP status code specified by ``ErrorCode`` , for example, ``/4xx-errors/403-forbidden.html`` . If you want to store your objects and your custom error pages in different locations, your distribution must include a cache behavior for which the following is true:. - The value of ``PathPattern`` matches the path to your custom error messages. For example, suppose you saved custom error pages for 4xx errors in an Amazon S3 bucket in a directory named ``/4xx-errors`` . Your distribution must include a cache behavior for which the path pattern routes requests for your custom error pages to that location, for example, ``/4xx-errors/*`` . - The value of ``TargetOriginId`` specifies the value of the ``ID`` element for the origin that contains your custom error pages. If you specify a value for ``ResponsePagePath`` , you must also specify a value for ``ResponseCode`` . We recommend that you store custom error pages in an Amazon S3 bucket. If you store custom error pages on an HTTP server and the server starts to return 5xx errors, CloudFront can't get the files that you want to return to viewers because the origin server is unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customerrorresponse.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                custom_error_response_property = cloudfront.CfnDistribution.CustomErrorResponseProperty(
                    error_code=123,
                
                    # the properties below are optional
                    error_caching_min_ttl=123,
                    response_code=123,
                    response_page_path="responsePagePath"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__52d9a41d1ba8d8ebd0b3c7f29e54f30d1eab053e8ec0ba5bced0aa9dda4eb99a)
                check_type(argname="argument error_code", value=error_code, expected_type=type_hints["error_code"])
                check_type(argname="argument error_caching_min_ttl", value=error_caching_min_ttl, expected_type=type_hints["error_caching_min_ttl"])
                check_type(argname="argument response_code", value=response_code, expected_type=type_hints["response_code"])
                check_type(argname="argument response_page_path", value=response_page_path, expected_type=type_hints["response_page_path"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "error_code": error_code,
            }
            if error_caching_min_ttl is not None:
                self._values["error_caching_min_ttl"] = error_caching_min_ttl
            if response_code is not None:
                self._values["response_code"] = response_code
            if response_page_path is not None:
                self._values["response_page_path"] = response_page_path

        @builtins.property
        def error_code(self) -> jsii.Number:
            '''The HTTP status code for which you want to specify a custom error page and/or a caching duration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customerrorresponse.html#cfn-cloudfront-distribution-customerrorresponse-errorcode
            '''
            result = self._values.get("error_code")
            assert result is not None, "Required property 'error_code' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def error_caching_min_ttl(self) -> typing.Optional[jsii.Number]:
            '''The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in ``ErrorCode`` .

            When this time period has elapsed, CloudFront queries your origin to see whether the problem that caused the error has been resolved and the requested object is now available.

            For more information, see `Customizing Error Responses <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customerrorresponse.html#cfn-cloudfront-distribution-customerrorresponse-errorcachingminttl
            '''
            result = self._values.get("error_caching_min_ttl")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def response_code(self) -> typing.Optional[jsii.Number]:
            '''The HTTP status code that you want CloudFront to return to the viewer along with the custom error page.

            There are a variety of reasons that you might want CloudFront to return a status code different from the status code that your origin returned to CloudFront, for example:

            - Some Internet devices (some firewalls and corporate proxies, for example) intercept HTTP 4xx and 5xx and prevent the response from being returned to the viewer. If you substitute ``200`` , the response typically won't be intercepted.
            - If you don't care about distinguishing among different client errors or server errors, you can specify ``400`` or ``500`` as the ``ResponseCode`` for all 4xx or 5xx errors.
            - You might want to return a ``200`` status code (OK) and static website so your customers don't know that your website is down.

            If you specify a value for ``ResponseCode`` , you must also specify a value for ``ResponsePagePath`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customerrorresponse.html#cfn-cloudfront-distribution-customerrorresponse-responsecode
            '''
            result = self._values.get("response_code")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def response_page_path(self) -> typing.Optional[builtins.str]:
            '''The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the HTTP status code specified by ``ErrorCode`` , for example, ``/4xx-errors/403-forbidden.html`` . If you want to store your objects and your custom error pages in different locations, your distribution must include a cache behavior for which the following is true:.

            - The value of ``PathPattern`` matches the path to your custom error messages. For example, suppose you saved custom error pages for 4xx errors in an Amazon S3 bucket in a directory named ``/4xx-errors`` . Your distribution must include a cache behavior for which the path pattern routes requests for your custom error pages to that location, for example, ``/4xx-errors/*`` .
            - The value of ``TargetOriginId`` specifies the value of the ``ID`` element for the origin that contains your custom error pages.

            If you specify a value for ``ResponsePagePath`` , you must also specify a value for ``ResponseCode`` .

            We recommend that you store custom error pages in an Amazon S3 bucket. If you store custom error pages on an HTTP server and the server starts to return 5xx errors, CloudFront can't get the files that you want to return to viewers because the origin server is unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customerrorresponse.html#cfn-cloudfront-distribution-customerrorresponse-responsepagepath
            '''
            result = self._values.get("response_page_path")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomErrorResponseProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.CustomOriginConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "origin_protocol_policy": "originProtocolPolicy",
            "http_port": "httpPort",
            "https_port": "httpsPort",
            "origin_keepalive_timeout": "originKeepaliveTimeout",
            "origin_read_timeout": "originReadTimeout",
            "origin_ssl_protocols": "originSslProtocols",
        },
    )
    class CustomOriginConfigProperty:
        def __init__(
            self,
            *,
            origin_protocol_policy: builtins.str,
            http_port: typing.Optional[jsii.Number] = None,
            https_port: typing.Optional[jsii.Number] = None,
            origin_keepalive_timeout: typing.Optional[jsii.Number] = None,
            origin_read_timeout: typing.Optional[jsii.Number] = None,
            origin_ssl_protocols: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''A custom origin.

            A custom origin is any origin that is *not* an Amazon S3 bucket, with one exception. An Amazon S3 bucket that is `configured with static website hosting <https://docs.aws.amazon.com/AmazonS3/latest/dev/WebsiteHosting.html>`_ *is* a custom origin.

            :param origin_protocol_policy: Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values are:. - ``http-only`` – CloudFront always uses HTTP to connect to the origin. - ``match-viewer`` – CloudFront connects to the origin using the same protocol that the viewer used to connect to CloudFront. - ``https-only`` – CloudFront always uses HTTPS to connect to the origin.
            :param http_port: The HTTP port that CloudFront uses to connect to the origin. Specify the HTTP port that the origin listens on.
            :param https_port: The HTTPS port that CloudFront uses to connect to the origin. Specify the HTTPS port that the origin listens on.
            :param origin_keepalive_timeout: Specifies how long, in seconds, CloudFront persists its connection to the origin. The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you don’t specify otherwise) is 5 seconds. For more information, see `Origin Keep-alive Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginKeepaliveTimeout>`_ in the *Amazon CloudFront Developer Guide* .
            :param origin_read_timeout: Specifies how long, in seconds, CloudFront waits for a response from the origin. This is also known as the *origin response timeout* . The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you don’t specify otherwise) is 30 seconds. For more information, see `Origin Response Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout>`_ in the *Amazon CloudFront Developer Guide* .
            :param origin_ssl_protocols: Specifies the minimum SSL/TLS protocol that CloudFront uses when connecting to your origin over HTTPS. Valid values include ``SSLv3`` , ``TLSv1`` , ``TLSv1.1`` , and ``TLSv1.2`` . For more information, see `Minimum Origin SSL Protocol <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginSSLProtocols>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customoriginconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                custom_origin_config_property = cloudfront.CfnDistribution.CustomOriginConfigProperty(
                    origin_protocol_policy="originProtocolPolicy",
                
                    # the properties below are optional
                    http_port=123,
                    https_port=123,
                    origin_keepalive_timeout=123,
                    origin_read_timeout=123,
                    origin_ssl_protocols=["originSslProtocols"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fd7e4d97a19b14f875be3caf9aa641161114740bb4b3434fb45c73683f02b1dc)
                check_type(argname="argument origin_protocol_policy", value=origin_protocol_policy, expected_type=type_hints["origin_protocol_policy"])
                check_type(argname="argument http_port", value=http_port, expected_type=type_hints["http_port"])
                check_type(argname="argument https_port", value=https_port, expected_type=type_hints["https_port"])
                check_type(argname="argument origin_keepalive_timeout", value=origin_keepalive_timeout, expected_type=type_hints["origin_keepalive_timeout"])
                check_type(argname="argument origin_read_timeout", value=origin_read_timeout, expected_type=type_hints["origin_read_timeout"])
                check_type(argname="argument origin_ssl_protocols", value=origin_ssl_protocols, expected_type=type_hints["origin_ssl_protocols"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "origin_protocol_policy": origin_protocol_policy,
            }
            if http_port is not None:
                self._values["http_port"] = http_port
            if https_port is not None:
                self._values["https_port"] = https_port
            if origin_keepalive_timeout is not None:
                self._values["origin_keepalive_timeout"] = origin_keepalive_timeout
            if origin_read_timeout is not None:
                self._values["origin_read_timeout"] = origin_read_timeout
            if origin_ssl_protocols is not None:
                self._values["origin_ssl_protocols"] = origin_ssl_protocols

        @builtins.property
        def origin_protocol_policy(self) -> builtins.str:
            '''Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin. Valid values are:.

            - ``http-only`` – CloudFront always uses HTTP to connect to the origin.
            - ``match-viewer`` – CloudFront connects to the origin using the same protocol that the viewer used to connect to CloudFront.
            - ``https-only`` – CloudFront always uses HTTPS to connect to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customoriginconfig.html#cfn-cloudfront-distribution-customoriginconfig-originprotocolpolicy
            '''
            result = self._values.get("origin_protocol_policy")
            assert result is not None, "Required property 'origin_protocol_policy' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def http_port(self) -> typing.Optional[jsii.Number]:
            '''The HTTP port that CloudFront uses to connect to the origin.

            Specify the HTTP port that the origin listens on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customoriginconfig.html#cfn-cloudfront-distribution-customoriginconfig-httpport
            '''
            result = self._values.get("http_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def https_port(self) -> typing.Optional[jsii.Number]:
            '''The HTTPS port that CloudFront uses to connect to the origin.

            Specify the HTTPS port that the origin listens on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customoriginconfig.html#cfn-cloudfront-distribution-customoriginconfig-httpsport
            '''
            result = self._values.get("https_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def origin_keepalive_timeout(self) -> typing.Optional[jsii.Number]:
            '''Specifies how long, in seconds, CloudFront persists its connection to the origin.

            The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you don’t specify otherwise) is 5 seconds.

            For more information, see `Origin Keep-alive Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginKeepaliveTimeout>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customoriginconfig.html#cfn-cloudfront-distribution-customoriginconfig-originkeepalivetimeout
            '''
            result = self._values.get("origin_keepalive_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def origin_read_timeout(self) -> typing.Optional[jsii.Number]:
            '''Specifies how long, in seconds, CloudFront waits for a response from the origin.

            This is also known as the *origin response timeout* . The minimum timeout is 1 second, the maximum is 60 seconds, and the default (if you don’t specify otherwise) is 30 seconds.

            For more information, see `Origin Response Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customoriginconfig.html#cfn-cloudfront-distribution-customoriginconfig-originreadtimeout
            '''
            result = self._values.get("origin_read_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def origin_ssl_protocols(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specifies the minimum SSL/TLS protocol that CloudFront uses when connecting to your origin over HTTPS.

            Valid values include ``SSLv3`` , ``TLSv1`` , ``TLSv1.1`` , and ``TLSv1.2`` .

            For more information, see `Minimum Origin SSL Protocol <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginSSLProtocols>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-customoriginconfig.html#cfn-cloudfront-distribution-customoriginconfig-originsslprotocols
            '''
            result = self._values.get("origin_ssl_protocols")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomOriginConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.DefaultCacheBehaviorProperty",
        jsii_struct_bases=[],
        name_mapping={
            "target_origin_id": "targetOriginId",
            "viewer_protocol_policy": "viewerProtocolPolicy",
            "allowed_methods": "allowedMethods",
            "cached_methods": "cachedMethods",
            "cache_policy_id": "cachePolicyId",
            "compress": "compress",
            "default_ttl": "defaultTtl",
            "field_level_encryption_id": "fieldLevelEncryptionId",
            "forwarded_values": "forwardedValues",
            "function_associations": "functionAssociations",
            "lambda_function_associations": "lambdaFunctionAssociations",
            "max_ttl": "maxTtl",
            "min_ttl": "minTtl",
            "origin_request_policy_id": "originRequestPolicyId",
            "realtime_log_config_arn": "realtimeLogConfigArn",
            "response_headers_policy_id": "responseHeadersPolicyId",
            "smooth_streaming": "smoothStreaming",
            "trusted_key_groups": "trustedKeyGroups",
            "trusted_signers": "trustedSigners",
        },
    )
    class DefaultCacheBehaviorProperty:
        def __init__(
            self,
            *,
            target_origin_id: builtins.str,
            viewer_protocol_policy: builtins.str,
            allowed_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
            cached_methods: typing.Optional[typing.Sequence[builtins.str]] = None,
            cache_policy_id: typing.Optional[builtins.str] = None,
            compress: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            default_ttl: typing.Optional[jsii.Number] = None,
            field_level_encryption_id: typing.Optional[builtins.str] = None,
            forwarded_values: typing.Optional[typing.Union[typing.Union["CfnDistribution.ForwardedValuesProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]] = None,
            function_associations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.FunctionAssociationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            lambda_function_associations: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.LambdaFunctionAssociationProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            max_ttl: typing.Optional[jsii.Number] = None,
            min_ttl: typing.Optional[jsii.Number] = None,
            origin_request_policy_id: typing.Optional[builtins.str] = None,
            realtime_log_config_arn: typing.Optional[builtins.str] = None,
            response_headers_policy_id: typing.Optional[builtins.str] = None,
            smooth_streaming: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            trusted_key_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            trusted_signers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''A complex type that describes the default cache behavior if you don’t specify a ``CacheBehavior`` element or if request URLs don’t match any of the values of ``PathPattern`` in ``CacheBehavior`` elements.

            You must create exactly one default cache behavior.

            :param target_origin_id: The value of ``ID`` for the origin that you want CloudFront to route requests to when they use the default cache behavior.
            :param viewer_protocol_policy: The protocol that viewers can use to access the files in the origin specified by ``TargetOriginId`` when a request matches the path pattern in ``PathPattern`` . You can specify the following options: - ``allow-all`` : Viewers can use HTTP or HTTPS. - ``redirect-to-https`` : If a viewer submits an HTTP request, CloudFront returns an HTTP status code of 301 (Moved Permanently) to the viewer along with the HTTPS URL. The viewer then resubmits the request using the new URL. - ``https-only`` : If a viewer sends an HTTP request, CloudFront returns an HTTP status code of 403 (Forbidden). For more information about requiring the HTTPS protocol, see `Requiring HTTPS Between Viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-viewers-to-cloudfront.html>`_ in the *Amazon CloudFront Developer Guide* . .. epigraph:: The only way to guarantee that viewers retrieve an object that was fetched from the origin using HTTPS is never to use any other protocol to fetch the object. If you have recently changed from HTTP to HTTPS, we recommend that you clear your objects’ cache because cached objects are protocol agnostic. That means that an edge location will return an object from the cache regardless of whether the current request protocol matches the protocol used previously. For more information, see `Managing Cache Expiration <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param allowed_methods: A complex type that controls which HTTP methods CloudFront processes and forwards to your Amazon S3 bucket or your custom origin. There are three choices: - CloudFront forwards only ``GET`` and ``HEAD`` requests. - CloudFront forwards only ``GET`` , ``HEAD`` , and ``OPTIONS`` requests. - CloudFront forwards ``GET, HEAD, OPTIONS, PUT, PATCH, POST`` , and ``DELETE`` requests. If you pick the third choice, you may need to restrict access to your Amazon S3 bucket or to your custom origin so users can't perform operations that you don't want them to. For example, you might not want users to have permissions to delete objects from your origin.
            :param cached_methods: A complex type that controls whether CloudFront caches the response to requests using the specified HTTP methods. There are two choices: - CloudFront caches responses to ``GET`` and ``HEAD`` requests. - CloudFront caches responses to ``GET`` , ``HEAD`` , and ``OPTIONS`` requests. If you pick the second choice for your Amazon S3 Origin, you may need to forward Access-Control-Request-Method, Access-Control-Request-Headers, and Origin headers for the responses to be cached correctly.
            :param cache_policy_id: The unique identifier of the cache policy that is attached to the default cache behavior. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . A ``DefaultCacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` .
            :param compress: Whether you want CloudFront to automatically compress certain files for this cache behavior. If so, specify ``true`` ; if not, specify ``false`` . For more information, see `Serving Compressed Files <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param default_ttl: This field is deprecated. We recommend that you use the ``DefaultTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . The default amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin does not add HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param field_level_encryption_id: The value of ``ID`` for the field-level encryption configuration that you want CloudFront to use for encrypting specific fields of data for the default cache behavior.
            :param forwarded_values: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. For more information, see `Working with policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html>`_ in the *Amazon CloudFront Developer Guide* . If you want to include values in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . If you want to send values to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* . A ``DefaultCacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` . A complex type that specifies how CloudFront handles query strings, cookies, and HTTP headers.
            :param function_associations: A list of CloudFront functions that are associated with this cache behavior. CloudFront functions must be published to the ``LIVE`` stage to associate them with a cache behavior.
            :param lambda_function_associations: A complex type that contains zero or more Lambda@Edge function associations for a cache behavior.
            :param max_ttl: This field is deprecated. We recommend that you use the ``MaxTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . The maximum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin adds HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param min_ttl: This field is deprecated. We recommend that you use the ``MinTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* . The minimum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* . You must specify ``0`` for ``MinTTL`` if you configure CloudFront to forward all headers to your origin (under ``Headers`` , if you specify ``1`` for ``Quantity`` and ``*`` for ``Name`` ).
            :param origin_request_policy_id: The unique identifier of the origin request policy that is attached to the default cache behavior. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param realtime_log_config_arn: The Amazon Resource Name (ARN) of the real-time log configuration that is attached to this cache behavior. For more information, see `Real-time logs <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param response_headers_policy_id: The identifier for a response headers policy.
            :param smooth_streaming: Indicates whether you want to distribute media files in the Microsoft Smooth Streaming format using the origin that is associated with this cache behavior. If so, specify ``true`` ; if not, specify ``false`` . If you specify ``true`` for ``SmoothStreaming`` , you can still distribute other content using this cache behavior if the content matches the value of ``PathPattern`` .
            :param trusted_key_groups: A list of key groups that CloudFront can use to validate signed URLs or signed cookies. When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with a private key whose corresponding public key is in the key group. The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param trusted_signers: .. epigraph:: We recommend using ``TrustedKeyGroups`` instead of ``TrustedSigners`` . A list of AWS account IDs whose public keys CloudFront can use to validate signed URLs or signed cookies. When a cache behavior contains trusted signers, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with the private key of a CloudFront key pair in a trusted signer’s AWS account . The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                default_cache_behavior_property = cloudfront.CfnDistribution.DefaultCacheBehaviorProperty(
                    target_origin_id="targetOriginId",
                    viewer_protocol_policy="viewerProtocolPolicy",
                
                    # the properties below are optional
                    allowed_methods=["allowedMethods"],
                    cached_methods=["cachedMethods"],
                    cache_policy_id="cachePolicyId",
                    compress=False,
                    default_ttl=123,
                    field_level_encryption_id="fieldLevelEncryptionId",
                    forwarded_values=cloudfront.CfnDistribution.ForwardedValuesProperty(
                        query_string=False,
                
                        # the properties below are optional
                        cookies=cloudfront.CfnDistribution.CookiesProperty(
                            forward="forward",
                
                            # the properties below are optional
                            whitelisted_names=["whitelistedNames"]
                        ),
                        headers=["headers"],
                        query_string_cache_keys=["queryStringCacheKeys"]
                    ),
                    function_associations=[cloudfront.CfnDistribution.FunctionAssociationProperty(
                        event_type="eventType",
                        function_arn="functionArn"
                    )],
                    lambda_function_associations=[cloudfront.CfnDistribution.LambdaFunctionAssociationProperty(
                        event_type="eventType",
                        include_body=False,
                        lambda_function_arn="lambdaFunctionArn"
                    )],
                    max_ttl=123,
                    min_ttl=123,
                    origin_request_policy_id="originRequestPolicyId",
                    realtime_log_config_arn="realtimeLogConfigArn",
                    response_headers_policy_id="responseHeadersPolicyId",
                    smooth_streaming=False,
                    trusted_key_groups=["trustedKeyGroups"],
                    trusted_signers=["trustedSigners"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d0242760b4a75ccb8faa281b7cf2bf7470950820e1713dc6e716a08f35662fca)
                check_type(argname="argument target_origin_id", value=target_origin_id, expected_type=type_hints["target_origin_id"])
                check_type(argname="argument viewer_protocol_policy", value=viewer_protocol_policy, expected_type=type_hints["viewer_protocol_policy"])
                check_type(argname="argument allowed_methods", value=allowed_methods, expected_type=type_hints["allowed_methods"])
                check_type(argname="argument cached_methods", value=cached_methods, expected_type=type_hints["cached_methods"])
                check_type(argname="argument cache_policy_id", value=cache_policy_id, expected_type=type_hints["cache_policy_id"])
                check_type(argname="argument compress", value=compress, expected_type=type_hints["compress"])
                check_type(argname="argument default_ttl", value=default_ttl, expected_type=type_hints["default_ttl"])
                check_type(argname="argument field_level_encryption_id", value=field_level_encryption_id, expected_type=type_hints["field_level_encryption_id"])
                check_type(argname="argument forwarded_values", value=forwarded_values, expected_type=type_hints["forwarded_values"])
                check_type(argname="argument function_associations", value=function_associations, expected_type=type_hints["function_associations"])
                check_type(argname="argument lambda_function_associations", value=lambda_function_associations, expected_type=type_hints["lambda_function_associations"])
                check_type(argname="argument max_ttl", value=max_ttl, expected_type=type_hints["max_ttl"])
                check_type(argname="argument min_ttl", value=min_ttl, expected_type=type_hints["min_ttl"])
                check_type(argname="argument origin_request_policy_id", value=origin_request_policy_id, expected_type=type_hints["origin_request_policy_id"])
                check_type(argname="argument realtime_log_config_arn", value=realtime_log_config_arn, expected_type=type_hints["realtime_log_config_arn"])
                check_type(argname="argument response_headers_policy_id", value=response_headers_policy_id, expected_type=type_hints["response_headers_policy_id"])
                check_type(argname="argument smooth_streaming", value=smooth_streaming, expected_type=type_hints["smooth_streaming"])
                check_type(argname="argument trusted_key_groups", value=trusted_key_groups, expected_type=type_hints["trusted_key_groups"])
                check_type(argname="argument trusted_signers", value=trusted_signers, expected_type=type_hints["trusted_signers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "target_origin_id": target_origin_id,
                "viewer_protocol_policy": viewer_protocol_policy,
            }
            if allowed_methods is not None:
                self._values["allowed_methods"] = allowed_methods
            if cached_methods is not None:
                self._values["cached_methods"] = cached_methods
            if cache_policy_id is not None:
                self._values["cache_policy_id"] = cache_policy_id
            if compress is not None:
                self._values["compress"] = compress
            if default_ttl is not None:
                self._values["default_ttl"] = default_ttl
            if field_level_encryption_id is not None:
                self._values["field_level_encryption_id"] = field_level_encryption_id
            if forwarded_values is not None:
                self._values["forwarded_values"] = forwarded_values
            if function_associations is not None:
                self._values["function_associations"] = function_associations
            if lambda_function_associations is not None:
                self._values["lambda_function_associations"] = lambda_function_associations
            if max_ttl is not None:
                self._values["max_ttl"] = max_ttl
            if min_ttl is not None:
                self._values["min_ttl"] = min_ttl
            if origin_request_policy_id is not None:
                self._values["origin_request_policy_id"] = origin_request_policy_id
            if realtime_log_config_arn is not None:
                self._values["realtime_log_config_arn"] = realtime_log_config_arn
            if response_headers_policy_id is not None:
                self._values["response_headers_policy_id"] = response_headers_policy_id
            if smooth_streaming is not None:
                self._values["smooth_streaming"] = smooth_streaming
            if trusted_key_groups is not None:
                self._values["trusted_key_groups"] = trusted_key_groups
            if trusted_signers is not None:
                self._values["trusted_signers"] = trusted_signers

        @builtins.property
        def target_origin_id(self) -> builtins.str:
            '''The value of ``ID`` for the origin that you want CloudFront to route requests to when they use the default cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-targetoriginid
            '''
            result = self._values.get("target_origin_id")
            assert result is not None, "Required property 'target_origin_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def viewer_protocol_policy(self) -> builtins.str:
            '''The protocol that viewers can use to access the files in the origin specified by ``TargetOriginId`` when a request matches the path pattern in ``PathPattern`` .

            You can specify the following options:

            - ``allow-all`` : Viewers can use HTTP or HTTPS.
            - ``redirect-to-https`` : If a viewer submits an HTTP request, CloudFront returns an HTTP status code of 301 (Moved Permanently) to the viewer along with the HTTPS URL. The viewer then resubmits the request using the new URL.
            - ``https-only`` : If a viewer sends an HTTP request, CloudFront returns an HTTP status code of 403 (Forbidden).

            For more information about requiring the HTTPS protocol, see `Requiring HTTPS Between Viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-viewers-to-cloudfront.html>`_ in the *Amazon CloudFront Developer Guide* .
            .. epigraph::

               The only way to guarantee that viewers retrieve an object that was fetched from the origin using HTTPS is never to use any other protocol to fetch the object. If you have recently changed from HTTP to HTTPS, we recommend that you clear your objects’ cache because cached objects are protocol agnostic. That means that an edge location will return an object from the cache regardless of whether the current request protocol matches the protocol used previously. For more information, see `Managing Cache Expiration <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-viewerprotocolpolicy
            '''
            result = self._values.get("viewer_protocol_policy")
            assert result is not None, "Required property 'viewer_protocol_policy' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def allowed_methods(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A complex type that controls which HTTP methods CloudFront processes and forwards to your Amazon S3 bucket or your custom origin.

            There are three choices:

            - CloudFront forwards only ``GET`` and ``HEAD`` requests.
            - CloudFront forwards only ``GET`` , ``HEAD`` , and ``OPTIONS`` requests.
            - CloudFront forwards ``GET, HEAD, OPTIONS, PUT, PATCH, POST`` , and ``DELETE`` requests.

            If you pick the third choice, you may need to restrict access to your Amazon S3 bucket or to your custom origin so users can't perform operations that you don't want them to. For example, you might not want users to have permissions to delete objects from your origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-allowedmethods
            '''
            result = self._values.get("allowed_methods")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cached_methods(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A complex type that controls whether CloudFront caches the response to requests using the specified HTTP methods.

            There are two choices:

            - CloudFront caches responses to ``GET`` and ``HEAD`` requests.
            - CloudFront caches responses to ``GET`` , ``HEAD`` , and ``OPTIONS`` requests.

            If you pick the second choice for your Amazon S3 Origin, you may need to forward Access-Control-Request-Method, Access-Control-Request-Headers, and Origin headers for the responses to be cached correctly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-cachedmethods
            '''
            result = self._values.get("cached_methods")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cache_policy_id(self) -> typing.Optional[builtins.str]:
            '''The unique identifier of the cache policy that is attached to the default cache behavior.

            For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            A ``DefaultCacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-cachepolicyid
            '''
            result = self._values.get("cache_policy_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def compress(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''Whether you want CloudFront to automatically compress certain files for this cache behavior.

            If so, specify ``true`` ; if not, specify ``false`` . For more information, see `Serving Compressed Files <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/ServingCompressedFiles.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-compress
            '''
            result = self._values.get("compress")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def default_ttl(self) -> typing.Optional[jsii.Number]:
            '''This field is deprecated.

            We recommend that you use the ``DefaultTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            The default amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin does not add HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-defaultttl
            '''
            result = self._values.get("default_ttl")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def field_level_encryption_id(self) -> typing.Optional[builtins.str]:
            '''The value of ``ID`` for the field-level encryption configuration that you want CloudFront to use for encrypting specific fields of data for the default cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-fieldlevelencryptionid
            '''
            result = self._values.get("field_level_encryption_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def forwarded_values(
            self,
        ) -> typing.Optional[typing.Union["CfnDistribution.ForwardedValuesProperty", _aws_cdk_core_f4b25747.IResolvable]]:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field. For more information, see `Working with policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to include values in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send values to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            A ``DefaultCacheBehavior`` must include either a ``CachePolicyId`` or ``ForwardedValues`` . We recommend that you use a ``CachePolicyId`` .

            A complex type that specifies how CloudFront handles query strings, cookies, and HTTP headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-forwardedvalues
            '''
            result = self._values.get("forwarded_values")
            return typing.cast(typing.Optional[typing.Union["CfnDistribution.ForwardedValuesProperty", _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def function_associations(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.FunctionAssociationProperty"]]]]:
            '''A list of CloudFront functions that are associated with this cache behavior.

            CloudFront functions must be published to the ``LIVE`` stage to associate them with a cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-functionassociations
            '''
            result = self._values.get("function_associations")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.FunctionAssociationProperty"]]]], result)

        @builtins.property
        def lambda_function_associations(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LambdaFunctionAssociationProperty"]]]]:
            '''A complex type that contains zero or more Lambda@Edge function associations for a cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-lambdafunctionassociations
            '''
            result = self._values.get("lambda_function_associations")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LambdaFunctionAssociationProperty"]]]], result)

        @builtins.property
        def max_ttl(self) -> typing.Optional[jsii.Number]:
            '''This field is deprecated.

            We recommend that you use the ``MaxTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            The maximum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. The value that you specify applies only when your origin adds HTTP headers such as ``Cache-Control max-age`` , ``Cache-Control s-maxage`` , and ``Expires`` to objects. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-maxttl
            '''
            result = self._values.get("max_ttl")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def min_ttl(self) -> typing.Optional[jsii.Number]:
            '''This field is deprecated.

            We recommend that you use the ``MinTTL`` field in a cache policy instead of this field. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ or `Using the managed cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            The minimum amount of time that you want objects to stay in CloudFront caches before CloudFront forwards another request to your origin to determine whether the object has been updated. For more information, see `Managing How Long Content Stays in an Edge Cache (Expiration) <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Expiration.html>`_ in the *Amazon CloudFront Developer Guide* .

            You must specify ``0`` for ``MinTTL`` if you configure CloudFront to forward all headers to your origin (under ``Headers`` , if you specify ``1`` for ``Quantity`` and ``*`` for ``Name`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-minttl
            '''
            result = self._values.get("min_ttl")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def origin_request_policy_id(self) -> typing.Optional[builtins.str]:
            '''The unique identifier of the origin request policy that is attached to the default cache behavior.

            For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ or `Using the managed origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-origin-request-policies.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-originrequestpolicyid
            '''
            result = self._values.get("origin_request_policy_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def realtime_log_config_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the real-time log configuration that is attached to this cache behavior.

            For more information, see `Real-time logs <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-realtimelogconfigarn
            '''
            result = self._values.get("realtime_log_config_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def response_headers_policy_id(self) -> typing.Optional[builtins.str]:
            '''The identifier for a response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-responseheaderspolicyid
            '''
            result = self._values.get("response_headers_policy_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def smooth_streaming(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''Indicates whether you want to distribute media files in the Microsoft Smooth Streaming format using the origin that is associated with this cache behavior.

            If so, specify ``true`` ; if not, specify ``false`` . If you specify ``true`` for ``SmoothStreaming`` , you can still distribute other content using this cache behavior if the content matches the value of ``PathPattern`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-smoothstreaming
            '''
            result = self._values.get("smooth_streaming")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def trusted_key_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of key groups that CloudFront can use to validate signed URLs or signed cookies.

            When a cache behavior contains trusted key groups, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with a private key whose corresponding public key is in the key group. The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-trustedkeygroups
            '''
            result = self._values.get("trusted_key_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def trusted_signers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''.. epigraph::

   We recommend using ``TrustedKeyGroups`` instead of ``TrustedSigners`` .

            A list of AWS account IDs whose public keys CloudFront can use to validate signed URLs or signed cookies.

            When a cache behavior contains trusted signers, CloudFront requires signed URLs or signed cookies for all requests that match the cache behavior. The URLs or cookies must be signed with the private key of a CloudFront key pair in a trusted signer’s AWS account . The signed URL or cookie contains information about which public key CloudFront should use to verify the signature. For more information, see `Serving private content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-defaultcachebehavior.html#cfn-cloudfront-distribution-defaultcachebehavior-trustedsigners
            '''
            result = self._values.get("trusted_signers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DefaultCacheBehaviorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.DistributionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "default_cache_behavior": "defaultCacheBehavior",
            "enabled": "enabled",
            "aliases": "aliases",
            "cache_behaviors": "cacheBehaviors",
            "cnam_es": "cnamEs",
            "comment": "comment",
            "continuous_deployment_policy_id": "continuousDeploymentPolicyId",
            "custom_error_responses": "customErrorResponses",
            "custom_origin": "customOrigin",
            "default_root_object": "defaultRootObject",
            "http_version": "httpVersion",
            "ipv6_enabled": "ipv6Enabled",
            "logging": "logging",
            "origin_groups": "originGroups",
            "origins": "origins",
            "price_class": "priceClass",
            "restrictions": "restrictions",
            "s3_origin": "s3Origin",
            "staging": "staging",
            "viewer_certificate": "viewerCertificate",
            "web_acl_id": "webAclId",
        },
    )
    class DistributionConfigProperty:
        def __init__(
            self,
            *,
            default_cache_behavior: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.DefaultCacheBehaviorProperty", typing.Dict[builtins.str, typing.Any]]],
            enabled: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            aliases: typing.Optional[typing.Sequence[builtins.str]] = None,
            cache_behaviors: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.CacheBehaviorProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            cnam_es: typing.Optional[typing.Sequence[builtins.str]] = None,
            comment: typing.Optional[builtins.str] = None,
            continuous_deployment_policy_id: typing.Optional[builtins.str] = None,
            custom_error_responses: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.CustomErrorResponseProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            custom_origin: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.LegacyCustomOriginProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            default_root_object: typing.Optional[builtins.str] = None,
            http_version: typing.Optional[builtins.str] = None,
            ipv6_enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            logging: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.LoggingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            origin_groups: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.OriginGroupsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            origins: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[typing.Union["CfnDistribution.OriginProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]]]] = None,
            price_class: typing.Optional[builtins.str] = None,
            restrictions: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.RestrictionsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            s3_origin: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.LegacyS3OriginProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            staging: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            viewer_certificate: typing.Optional[typing.Union[typing.Union["CfnDistribution.ViewerCertificateProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]] = None,
            web_acl_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A distribution configuration.

            :param default_cache_behavior: A complex type that describes the default cache behavior if you don't specify a ``CacheBehavior`` element or if files don't match any of the values of ``PathPattern`` in ``CacheBehavior`` elements. You must create exactly one default cache behavior.
            :param enabled: From this field, you can enable or disable the selected distribution.
            :param aliases: A complex type that contains information about CNAMEs (alternate domain names), if any, for this distribution.
            :param cache_behaviors: A complex type that contains zero or more ``CacheBehavior`` elements.
            :param cnam_es: ``CfnDistribution.DistributionConfigProperty.CNAMEs``.
            :param comment: An optional comment to describe the distribution. The comment cannot be longer than 128 characters.
            :param continuous_deployment_policy_id: The identifier of a continuous deployment policy. For more information, see ``CreateContinuousDeploymentPolicy`` .
            :param custom_error_responses: A complex type that controls the following:. - Whether CloudFront replaces HTTP status codes in the 4xx and 5xx range with custom error messages before returning the response to the viewer. - How long CloudFront caches HTTP status codes in the 4xx and 5xx range. For more information about custom error pages, see `Customizing Error Responses <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param custom_origin: ``CfnDistribution.DistributionConfigProperty.CustomOrigin``.
            :param default_root_object: The object that you want CloudFront to request from your origin (for example, ``index.html`` ) when a viewer requests the root URL for your distribution ( ``https://www.example.com`` ) instead of an object in your distribution ( ``https://www.example.com/product-description.html`` ). Specifying a default root object avoids exposing the contents of your distribution. Specify only the object name, for example, ``index.html`` . Don't add a ``/`` before the object name. If you don't want to specify a default root object when you create a distribution, include an empty ``DefaultRootObject`` element. To delete the default root object from an existing distribution, update the distribution configuration and include an empty ``DefaultRootObject`` element. To replace the default root object, update the distribution configuration and specify the new object. For more information about the default root object, see `Creating a Default Root Object <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DefaultRootObject.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param http_version: (Optional) Specify the maximum HTTP version(s) that you want viewers to use to communicate with CloudFront . The default value for new distributions is ``http1.1`` . For viewers and CloudFront to use HTTP/2, viewers must support TLSv1.2 or later, and must support Server Name Indication (SNI). For viewers and CloudFront to use HTTP/3, viewers must support TLSv1.3 and Server Name Indication (SNI). CloudFront supports HTTP/3 connection migration to allow the viewer to switch networks without losing connection. For more information about connection migration, see `Connection Migration <https://docs.aws.amazon.com/https://www.rfc-editor.org/rfc/rfc9000.html#name-connection-migration>`_ at RFC 9000. For more information about supported TLSv1.3 ciphers, see `Supported protocols and ciphers between viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html>`_ .
            :param ipv6_enabled: If you want CloudFront to respond to IPv6 DNS requests with an IPv6 address for your distribution, specify ``true`` . If you specify ``false`` , CloudFront responds to IPv6 DNS requests with the DNS response code ``NOERROR`` and with no IP addresses. This allows viewers to submit a second request, for an IPv4 address for your distribution. In general, you should enable IPv6 if you have users on IPv6 networks who want to access your content. However, if you're using signed URLs or signed cookies to restrict access to your content, and if you're using a custom policy that includes the ``IpAddress`` parameter to restrict the IP addresses that can access your content, don't enable IPv6. If you want to restrict access to some content by IP address and not restrict access to other content (or restrict access but not by IP address), you can create two distributions. For more information, see `Creating a Signed URL Using a Custom Policy <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-creating-signed-url-custom-policy.html>`_ in the *Amazon CloudFront Developer Guide* . If you're using an Amazon Route 53 AWS Integration alias resource record set to route traffic to your CloudFront distribution, you need to create a second alias resource record set when both of the following are true: - You enable IPv6 for the distribution - You're using alternate domain names in the URLs for your objects For more information, see `Routing Traffic to an Amazon CloudFront Web Distribution by Using Your Domain Name <https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-to-cloudfront-distribution.html>`_ in the *Amazon Route 53 AWS Integration Developer Guide* . If you created a CNAME resource record set, either with Amazon Route 53 AWS Integration or with another DNS service, you don't need to make any changes. A CNAME record will route traffic to your distribution regardless of the IP address format of the viewer request.
            :param logging: A complex type that controls whether access logs are written for the distribution. For more information about logging, see `Access Logs <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AccessLogs.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param origin_groups: A complex type that contains information about origin groups for this distribution.
            :param origins: A complex type that contains information about origins for this distribution.
            :param price_class: The price class that corresponds with the maximum price that you want to pay for CloudFront service. If you specify ``PriceClass_All`` , CloudFront responds to requests for your objects from all CloudFront edge locations. If you specify a price class other than ``PriceClass_All`` , CloudFront serves your objects from the CloudFront edge location that has the lowest latency among the edge locations in your price class. Viewers who are in or near regions that are excluded from your specified price class may encounter slower performance. For more information about price classes, see `Choosing the Price Class for a CloudFront Distribution <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PriceClass.html>`_ in the *Amazon CloudFront Developer Guide* . For information about CloudFront pricing, including how price classes (such as Price Class 100) map to CloudFront regions, see `Amazon CloudFront Pricing <https://docs.aws.amazon.com/cloudfront/pricing/>`_ .
            :param restrictions: A complex type that identifies ways in which you want to restrict distribution of your content.
            :param s3_origin: ``CfnDistribution.DistributionConfigProperty.S3Origin``.
            :param staging: A Boolean that indicates whether this is a staging distribution. When this value is ``true`` , this is a staging distribution. When this value is ``false`` , this is not a staging distribution.
            :param viewer_certificate: A complex type that determines the distribution’s SSL/TLS configuration for communicating with viewers.
            :param web_acl_id: A unique identifier that specifies the AWS WAF web ACL, if any, to associate with this distribution. To specify a web ACL created using the latest version of AWS WAF , use the ACL ARN, for example ``arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a`` . To specify a web ACL created using AWS WAF Classic, use the ACL ID, for example ``473e64fd-f30b-4765-81a0-62ad96dd167a`` . AWS WAF is a web application firewall that lets you monitor the HTTP and HTTPS requests that are forwarded to CloudFront, and lets you control access to your content. Based on conditions that you specify, such as the IP addresses that requests originate from or the values of query strings, CloudFront responds to requests either with the requested content or with an HTTP 403 status code (Forbidden). You can also configure CloudFront to return a custom error page when a request is blocked. For more information about AWS WAF , see the `AWS WAF Developer Guide <https://docs.aws.amazon.com/waf/latest/developerguide/what-is-aws-waf.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                distribution_config_property = cloudfront.CfnDistribution.DistributionConfigProperty(
                    default_cache_behavior=cloudfront.CfnDistribution.DefaultCacheBehaviorProperty(
                        target_origin_id="targetOriginId",
                        viewer_protocol_policy="viewerProtocolPolicy",
                
                        # the properties below are optional
                        allowed_methods=["allowedMethods"],
                        cached_methods=["cachedMethods"],
                        cache_policy_id="cachePolicyId",
                        compress=False,
                        default_ttl=123,
                        field_level_encryption_id="fieldLevelEncryptionId",
                        forwarded_values=cloudfront.CfnDistribution.ForwardedValuesProperty(
                            query_string=False,
                
                            # the properties below are optional
                            cookies=cloudfront.CfnDistribution.CookiesProperty(
                                forward="forward",
                
                                # the properties below are optional
                                whitelisted_names=["whitelistedNames"]
                            ),
                            headers=["headers"],
                            query_string_cache_keys=["queryStringCacheKeys"]
                        ),
                        function_associations=[cloudfront.CfnDistribution.FunctionAssociationProperty(
                            event_type="eventType",
                            function_arn="functionArn"
                        )],
                        lambda_function_associations=[cloudfront.CfnDistribution.LambdaFunctionAssociationProperty(
                            event_type="eventType",
                            include_body=False,
                            lambda_function_arn="lambdaFunctionArn"
                        )],
                        max_ttl=123,
                        min_ttl=123,
                        origin_request_policy_id="originRequestPolicyId",
                        realtime_log_config_arn="realtimeLogConfigArn",
                        response_headers_policy_id="responseHeadersPolicyId",
                        smooth_streaming=False,
                        trusted_key_groups=["trustedKeyGroups"],
                        trusted_signers=["trustedSigners"]
                    ),
                    enabled=False,
                
                    # the properties below are optional
                    aliases=["aliases"],
                    cache_behaviors=[cloudfront.CfnDistribution.CacheBehaviorProperty(
                        path_pattern="pathPattern",
                        target_origin_id="targetOriginId",
                        viewer_protocol_policy="viewerProtocolPolicy",
                
                        # the properties below are optional
                        allowed_methods=["allowedMethods"],
                        cached_methods=["cachedMethods"],
                        cache_policy_id="cachePolicyId",
                        compress=False,
                        default_ttl=123,
                        field_level_encryption_id="fieldLevelEncryptionId",
                        forwarded_values=cloudfront.CfnDistribution.ForwardedValuesProperty(
                            query_string=False,
                
                            # the properties below are optional
                            cookies=cloudfront.CfnDistribution.CookiesProperty(
                                forward="forward",
                
                                # the properties below are optional
                                whitelisted_names=["whitelistedNames"]
                            ),
                            headers=["headers"],
                            query_string_cache_keys=["queryStringCacheKeys"]
                        ),
                        function_associations=[cloudfront.CfnDistribution.FunctionAssociationProperty(
                            event_type="eventType",
                            function_arn="functionArn"
                        )],
                        lambda_function_associations=[cloudfront.CfnDistribution.LambdaFunctionAssociationProperty(
                            event_type="eventType",
                            include_body=False,
                            lambda_function_arn="lambdaFunctionArn"
                        )],
                        max_ttl=123,
                        min_ttl=123,
                        origin_request_policy_id="originRequestPolicyId",
                        realtime_log_config_arn="realtimeLogConfigArn",
                        response_headers_policy_id="responseHeadersPolicyId",
                        smooth_streaming=False,
                        trusted_key_groups=["trustedKeyGroups"],
                        trusted_signers=["trustedSigners"]
                    )],
                    cnam_es=["cnamEs"],
                    comment="comment",
                    continuous_deployment_policy_id="continuousDeploymentPolicyId",
                    custom_error_responses=[cloudfront.CfnDistribution.CustomErrorResponseProperty(
                        error_code=123,
                
                        # the properties below are optional
                        error_caching_min_ttl=123,
                        response_code=123,
                        response_page_path="responsePagePath"
                    )],
                    custom_origin=cloudfront.CfnDistribution.LegacyCustomOriginProperty(
                        dns_name="dnsName",
                        origin_protocol_policy="originProtocolPolicy",
                        origin_ssl_protocols=["originSslProtocols"],
                
                        # the properties below are optional
                        http_port=123,
                        https_port=123
                    ),
                    default_root_object="defaultRootObject",
                    http_version="httpVersion",
                    ipv6_enabled=False,
                    logging=cloudfront.CfnDistribution.LoggingProperty(
                        bucket="bucket",
                
                        # the properties below are optional
                        include_cookies=False,
                        prefix="prefix"
                    ),
                    origin_groups=cloudfront.CfnDistribution.OriginGroupsProperty(
                        quantity=123,
                
                        # the properties below are optional
                        items=[cloudfront.CfnDistribution.OriginGroupProperty(
                            failover_criteria=cloudfront.CfnDistribution.OriginGroupFailoverCriteriaProperty(
                                status_codes=cloudfront.CfnDistribution.StatusCodesProperty(
                                    items=[123],
                                    quantity=123
                                )
                            ),
                            id="id",
                            members=cloudfront.CfnDistribution.OriginGroupMembersProperty(
                                items=[cloudfront.CfnDistribution.OriginGroupMemberProperty(
                                    origin_id="originId"
                                )],
                                quantity=123
                            )
                        )]
                    ),
                    origins=[cloudfront.CfnDistribution.OriginProperty(
                        domain_name="domainName",
                        id="id",
                
                        # the properties below are optional
                        connection_attempts=123,
                        connection_timeout=123,
                        custom_origin_config=cloudfront.CfnDistribution.CustomOriginConfigProperty(
                            origin_protocol_policy="originProtocolPolicy",
                
                            # the properties below are optional
                            http_port=123,
                            https_port=123,
                            origin_keepalive_timeout=123,
                            origin_read_timeout=123,
                            origin_ssl_protocols=["originSslProtocols"]
                        ),
                        origin_access_control_id="originAccessControlId",
                        origin_custom_headers=[cloudfront.CfnDistribution.OriginCustomHeaderProperty(
                            header_name="headerName",
                            header_value="headerValue"
                        )],
                        origin_path="originPath",
                        origin_shield=cloudfront.CfnDistribution.OriginShieldProperty(
                            enabled=False,
                            origin_shield_region="originShieldRegion"
                        ),
                        s3_origin_config=cloudfront.CfnDistribution.S3OriginConfigProperty(
                            origin_access_identity="originAccessIdentity"
                        )
                    )],
                    price_class="priceClass",
                    restrictions=cloudfront.CfnDistribution.RestrictionsProperty(
                        geo_restriction=cloudfront.CfnDistribution.GeoRestrictionProperty(
                            restriction_type="restrictionType",
                
                            # the properties below are optional
                            locations=["locations"]
                        )
                    ),
                    s3_origin=cloudfront.CfnDistribution.LegacyS3OriginProperty(
                        dns_name="dnsName",
                
                        # the properties below are optional
                        origin_access_identity="originAccessIdentity"
                    ),
                    staging=False,
                    viewer_certificate=cloudfront.CfnDistribution.ViewerCertificateProperty(
                        acm_certificate_arn="acmCertificateArn",
                        cloud_front_default_certificate=False,
                        iam_certificate_id="iamCertificateId",
                        minimum_protocol_version="minimumProtocolVersion",
                        ssl_support_method="sslSupportMethod"
                    ),
                    web_acl_id="webAclId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5a8d39c7babab9d86e33497d904562788a3b47873038142c77376f5d2d357833)
                check_type(argname="argument default_cache_behavior", value=default_cache_behavior, expected_type=type_hints["default_cache_behavior"])
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument aliases", value=aliases, expected_type=type_hints["aliases"])
                check_type(argname="argument cache_behaviors", value=cache_behaviors, expected_type=type_hints["cache_behaviors"])
                check_type(argname="argument cnam_es", value=cnam_es, expected_type=type_hints["cnam_es"])
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
                check_type(argname="argument continuous_deployment_policy_id", value=continuous_deployment_policy_id, expected_type=type_hints["continuous_deployment_policy_id"])
                check_type(argname="argument custom_error_responses", value=custom_error_responses, expected_type=type_hints["custom_error_responses"])
                check_type(argname="argument custom_origin", value=custom_origin, expected_type=type_hints["custom_origin"])
                check_type(argname="argument default_root_object", value=default_root_object, expected_type=type_hints["default_root_object"])
                check_type(argname="argument http_version", value=http_version, expected_type=type_hints["http_version"])
                check_type(argname="argument ipv6_enabled", value=ipv6_enabled, expected_type=type_hints["ipv6_enabled"])
                check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
                check_type(argname="argument origin_groups", value=origin_groups, expected_type=type_hints["origin_groups"])
                check_type(argname="argument origins", value=origins, expected_type=type_hints["origins"])
                check_type(argname="argument price_class", value=price_class, expected_type=type_hints["price_class"])
                check_type(argname="argument restrictions", value=restrictions, expected_type=type_hints["restrictions"])
                check_type(argname="argument s3_origin", value=s3_origin, expected_type=type_hints["s3_origin"])
                check_type(argname="argument staging", value=staging, expected_type=type_hints["staging"])
                check_type(argname="argument viewer_certificate", value=viewer_certificate, expected_type=type_hints["viewer_certificate"])
                check_type(argname="argument web_acl_id", value=web_acl_id, expected_type=type_hints["web_acl_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "default_cache_behavior": default_cache_behavior,
                "enabled": enabled,
            }
            if aliases is not None:
                self._values["aliases"] = aliases
            if cache_behaviors is not None:
                self._values["cache_behaviors"] = cache_behaviors
            if cnam_es is not None:
                self._values["cnam_es"] = cnam_es
            if comment is not None:
                self._values["comment"] = comment
            if continuous_deployment_policy_id is not None:
                self._values["continuous_deployment_policy_id"] = continuous_deployment_policy_id
            if custom_error_responses is not None:
                self._values["custom_error_responses"] = custom_error_responses
            if custom_origin is not None:
                self._values["custom_origin"] = custom_origin
            if default_root_object is not None:
                self._values["default_root_object"] = default_root_object
            if http_version is not None:
                self._values["http_version"] = http_version
            if ipv6_enabled is not None:
                self._values["ipv6_enabled"] = ipv6_enabled
            if logging is not None:
                self._values["logging"] = logging
            if origin_groups is not None:
                self._values["origin_groups"] = origin_groups
            if origins is not None:
                self._values["origins"] = origins
            if price_class is not None:
                self._values["price_class"] = price_class
            if restrictions is not None:
                self._values["restrictions"] = restrictions
            if s3_origin is not None:
                self._values["s3_origin"] = s3_origin
            if staging is not None:
                self._values["staging"] = staging
            if viewer_certificate is not None:
                self._values["viewer_certificate"] = viewer_certificate
            if web_acl_id is not None:
                self._values["web_acl_id"] = web_acl_id

        @builtins.property
        def default_cache_behavior(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.DefaultCacheBehaviorProperty"]:
            '''A complex type that describes the default cache behavior if you don't specify a ``CacheBehavior`` element or if files don't match any of the values of ``PathPattern`` in ``CacheBehavior`` elements.

            You must create exactly one default cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-defaultcachebehavior
            '''
            result = self._values.get("default_cache_behavior")
            assert result is not None, "Required property 'default_cache_behavior' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.DefaultCacheBehaviorProperty"], result)

        @builtins.property
        def enabled(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''From this field, you can enable or disable the selected distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def aliases(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A complex type that contains information about CNAMEs (alternate domain names), if any, for this distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-aliases
            '''
            result = self._values.get("aliases")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cache_behaviors(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.CacheBehaviorProperty"]]]]:
            '''A complex type that contains zero or more ``CacheBehavior`` elements.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-cachebehaviors
            '''
            result = self._values.get("cache_behaviors")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.CacheBehaviorProperty"]]]], result)

        @builtins.property
        def cnam_es(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnDistribution.DistributionConfigProperty.CNAMEs``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-cnames
            '''
            result = self._values.get("cnam_es")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def comment(self) -> typing.Optional[builtins.str]:
            '''An optional comment to describe the distribution.

            The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-comment
            '''
            result = self._values.get("comment")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def continuous_deployment_policy_id(self) -> typing.Optional[builtins.str]:
            '''The identifier of a continuous deployment policy.

            For more information, see ``CreateContinuousDeploymentPolicy`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-continuousdeploymentpolicyid
            '''
            result = self._values.get("continuous_deployment_policy_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def custom_error_responses(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.CustomErrorResponseProperty"]]]]:
            '''A complex type that controls the following:.

            - Whether CloudFront replaces HTTP status codes in the 4xx and 5xx range with custom error messages before returning the response to the viewer.
            - How long CloudFront caches HTTP status codes in the 4xx and 5xx range.

            For more information about custom error pages, see `Customizing Error Responses <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/custom-error-pages.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-customerrorresponses
            '''
            result = self._values.get("custom_error_responses")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.CustomErrorResponseProperty"]]]], result)

        @builtins.property
        def custom_origin(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LegacyCustomOriginProperty"]]:
            '''``CfnDistribution.DistributionConfigProperty.CustomOrigin``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-customorigin
            '''
            result = self._values.get("custom_origin")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LegacyCustomOriginProperty"]], result)

        @builtins.property
        def default_root_object(self) -> typing.Optional[builtins.str]:
            '''The object that you want CloudFront to request from your origin (for example, ``index.html`` ) when a viewer requests the root URL for your distribution ( ``https://www.example.com`` ) instead of an object in your distribution ( ``https://www.example.com/product-description.html`` ). Specifying a default root object avoids exposing the contents of your distribution.

            Specify only the object name, for example, ``index.html`` . Don't add a ``/`` before the object name.

            If you don't want to specify a default root object when you create a distribution, include an empty ``DefaultRootObject`` element.

            To delete the default root object from an existing distribution, update the distribution configuration and include an empty ``DefaultRootObject`` element.

            To replace the default root object, update the distribution configuration and specify the new object.

            For more information about the default root object, see `Creating a Default Root Object <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/DefaultRootObject.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-defaultrootobject
            '''
            result = self._values.get("default_root_object")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def http_version(self) -> typing.Optional[builtins.str]:
            '''(Optional) Specify the maximum HTTP version(s) that you want viewers to use to communicate with CloudFront .

            The default value for new distributions is ``http1.1`` .

            For viewers and CloudFront to use HTTP/2, viewers must support TLSv1.2 or later, and must support Server Name Indication (SNI).

            For viewers and CloudFront to use HTTP/3, viewers must support TLSv1.3 and Server Name Indication (SNI). CloudFront supports HTTP/3 connection migration to allow the viewer to switch networks without losing connection. For more information about connection migration, see `Connection Migration <https://docs.aws.amazon.com/https://www.rfc-editor.org/rfc/rfc9000.html#name-connection-migration>`_ at RFC 9000. For more information about supported TLSv1.3 ciphers, see `Supported protocols and ciphers between viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-httpversion
            '''
            result = self._values.get("http_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ipv6_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''If you want CloudFront to respond to IPv6 DNS requests with an IPv6 address for your distribution, specify ``true`` .

            If you specify ``false`` , CloudFront responds to IPv6 DNS requests with the DNS response code ``NOERROR`` and with no IP addresses. This allows viewers to submit a second request, for an IPv4 address for your distribution.

            In general, you should enable IPv6 if you have users on IPv6 networks who want to access your content. However, if you're using signed URLs or signed cookies to restrict access to your content, and if you're using a custom policy that includes the ``IpAddress`` parameter to restrict the IP addresses that can access your content, don't enable IPv6. If you want to restrict access to some content by IP address and not restrict access to other content (or restrict access but not by IP address), you can create two distributions. For more information, see `Creating a Signed URL Using a Custom Policy <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-creating-signed-url-custom-policy.html>`_ in the *Amazon CloudFront Developer Guide* .

            If you're using an Amazon Route 53 AWS Integration alias resource record set to route traffic to your CloudFront distribution, you need to create a second alias resource record set when both of the following are true:

            - You enable IPv6 for the distribution
            - You're using alternate domain names in the URLs for your objects

            For more information, see `Routing Traffic to an Amazon CloudFront Web Distribution by Using Your Domain Name <https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/routing-to-cloudfront-distribution.html>`_ in the *Amazon Route 53 AWS Integration Developer Guide* .

            If you created a CNAME resource record set, either with Amazon Route 53 AWS Integration or with another DNS service, you don't need to make any changes. A CNAME record will route traffic to your distribution regardless of the IP address format of the viewer request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-ipv6enabled
            '''
            result = self._values.get("ipv6_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LoggingProperty"]]:
            '''A complex type that controls whether access logs are written for the distribution.

            For more information about logging, see `Access Logs <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/AccessLogs.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LoggingProperty"]], result)

        @builtins.property
        def origin_groups(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupsProperty"]]:
            '''A complex type that contains information about origin groups for this distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-origingroups
            '''
            result = self._values.get("origin_groups")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupsProperty"]], result)

        @builtins.property
        def origins(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union["CfnDistribution.OriginProperty", _aws_cdk_core_f4b25747.IResolvable]]]]:
            '''A complex type that contains information about origins for this distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-origins
            '''
            result = self._values.get("origins")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union["CfnDistribution.OriginProperty", _aws_cdk_core_f4b25747.IResolvable]]]], result)

        @builtins.property
        def price_class(self) -> typing.Optional[builtins.str]:
            '''The price class that corresponds with the maximum price that you want to pay for CloudFront service.

            If you specify ``PriceClass_All`` , CloudFront responds to requests for your objects from all CloudFront edge locations.

            If you specify a price class other than ``PriceClass_All`` , CloudFront serves your objects from the CloudFront edge location that has the lowest latency among the edge locations in your price class. Viewers who are in or near regions that are excluded from your specified price class may encounter slower performance.

            For more information about price classes, see `Choosing the Price Class for a CloudFront Distribution <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PriceClass.html>`_ in the *Amazon CloudFront Developer Guide* . For information about CloudFront pricing, including how price classes (such as Price Class 100) map to CloudFront regions, see `Amazon CloudFront Pricing <https://docs.aws.amazon.com/cloudfront/pricing/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-priceclass
            '''
            result = self._values.get("price_class")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def restrictions(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.RestrictionsProperty"]]:
            '''A complex type that identifies ways in which you want to restrict distribution of your content.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-restrictions
            '''
            result = self._values.get("restrictions")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.RestrictionsProperty"]], result)

        @builtins.property
        def s3_origin(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LegacyS3OriginProperty"]]:
            '''``CfnDistribution.DistributionConfigProperty.S3Origin``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-s3origin
            '''
            result = self._values.get("s3_origin")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.LegacyS3OriginProperty"]], result)

        @builtins.property
        def staging(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A Boolean that indicates whether this is a staging distribution.

            When this value is ``true`` , this is a staging distribution. When this value is ``false`` , this is not a staging distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-staging
            '''
            result = self._values.get("staging")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def viewer_certificate(
            self,
        ) -> typing.Optional[typing.Union["CfnDistribution.ViewerCertificateProperty", _aws_cdk_core_f4b25747.IResolvable]]:
            '''A complex type that determines the distribution’s SSL/TLS configuration for communicating with viewers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-viewercertificate
            '''
            result = self._values.get("viewer_certificate")
            return typing.cast(typing.Optional[typing.Union["CfnDistribution.ViewerCertificateProperty", _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def web_acl_id(self) -> typing.Optional[builtins.str]:
            '''A unique identifier that specifies the AWS WAF web ACL, if any, to associate with this distribution.

            To specify a web ACL created using the latest version of AWS WAF , use the ACL ARN, for example ``arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a`` . To specify a web ACL created using AWS WAF Classic, use the ACL ID, for example ``473e64fd-f30b-4765-81a0-62ad96dd167a`` .

            AWS WAF is a web application firewall that lets you monitor the HTTP and HTTPS requests that are forwarded to CloudFront, and lets you control access to your content. Based on conditions that you specify, such as the IP addresses that requests originate from or the values of query strings, CloudFront responds to requests either with the requested content or with an HTTP 403 status code (Forbidden). You can also configure CloudFront to return a custom error page when a request is blocked. For more information about AWS WAF , see the `AWS WAF Developer Guide <https://docs.aws.amazon.com/waf/latest/developerguide/what-is-aws-waf.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-webaclid
            '''
            result = self._values.get("web_acl_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DistributionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.ForwardedValuesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "query_string": "queryString",
            "cookies": "cookies",
            "headers": "headers",
            "query_string_cache_keys": "queryStringCacheKeys",
        },
    )
    class ForwardedValuesProperty:
        def __init__(
            self,
            *,
            query_string: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            cookies: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.CookiesProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            headers: typing.Optional[typing.Sequence[builtins.str]] = None,
            query_string_cache_keys: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include values in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send values to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            A complex type that specifies how CloudFront handles query strings, cookies, and HTTP headers.

            :param query_string: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. If you want to include query strings in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* . If you want to send query strings to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* . Indicates whether you want CloudFront to forward query strings to the origin that is associated with this cache behavior and cache based on the query string parameters. CloudFront behavior depends on the value of ``QueryString`` and on the values that you specify for ``QueryStringCacheKeys`` , if any: If you specify true for ``QueryString`` and you don't specify any values for ``QueryStringCacheKeys`` , CloudFront forwards all query string parameters to the origin and caches based on all query string parameters. Depending on how many query string parameters and values you have, this can adversely affect performance because CloudFront must forward more requests to the origin. If you specify true for ``QueryString`` and you specify one or more values for ``QueryStringCacheKeys`` , CloudFront forwards all query string parameters to the origin, but it only caches based on the query string parameters that you specify. If you specify false for ``QueryString`` , CloudFront doesn't forward any query string parameters to the origin, and doesn't cache based on query string parameters. For more information, see `Configuring CloudFront to Cache Based on Query String Parameters <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/QueryStringParameters.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param cookies: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. If you want to include cookies in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* . If you want to send cookies to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* . A complex type that specifies whether you want CloudFront to forward cookies to the origin and, if so, which ones. For more information about forwarding cookies to the origin, see `How CloudFront Forwards, Caches, and Logs Cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Cookies.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param headers: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. If you want to include headers in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* . If you want to send headers to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* . A complex type that specifies the ``Headers`` , if any, that you want CloudFront to forward to the origin for this cache behavior (whitelisted headers). For the headers that you specify, CloudFront also caches separate versions of a specified object that is based on the header values in viewer requests. For more information, see `Caching Content Based on Request Headers <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/header-caching.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param query_string_cache_keys: This field is deprecated. We recommend that you use a cache policy or an origin request policy instead of this field. If you want to include query strings in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* . If you want to send query strings to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* . A complex type that contains information about the query string parameters that you want CloudFront to use for caching for this cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-forwardedvalues.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                forwarded_values_property = cloudfront.CfnDistribution.ForwardedValuesProperty(
                    query_string=False,
                
                    # the properties below are optional
                    cookies=cloudfront.CfnDistribution.CookiesProperty(
                        forward="forward",
                
                        # the properties below are optional
                        whitelisted_names=["whitelistedNames"]
                    ),
                    headers=["headers"],
                    query_string_cache_keys=["queryStringCacheKeys"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__02faf10dba0bbadaa0e858d0a0d44f2ffa8e4bffd30fcd1276e1662fe9f9f5bf)
                check_type(argname="argument query_string", value=query_string, expected_type=type_hints["query_string"])
                check_type(argname="argument cookies", value=cookies, expected_type=type_hints["cookies"])
                check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
                check_type(argname="argument query_string_cache_keys", value=query_string_cache_keys, expected_type=type_hints["query_string_cache_keys"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "query_string": query_string,
            }
            if cookies is not None:
                self._values["cookies"] = cookies
            if headers is not None:
                self._values["headers"] = headers
            if query_string_cache_keys is not None:
                self._values["query_string_cache_keys"] = query_string_cache_keys

        @builtins.property
        def query_string(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include query strings in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send query strings to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            Indicates whether you want CloudFront to forward query strings to the origin that is associated with this cache behavior and cache based on the query string parameters. CloudFront behavior depends on the value of ``QueryString`` and on the values that you specify for ``QueryStringCacheKeys`` , if any:

            If you specify true for ``QueryString`` and you don't specify any values for ``QueryStringCacheKeys`` , CloudFront forwards all query string parameters to the origin and caches based on all query string parameters. Depending on how many query string parameters and values you have, this can adversely affect performance because CloudFront must forward more requests to the origin.

            If you specify true for ``QueryString`` and you specify one or more values for ``QueryStringCacheKeys`` , CloudFront forwards all query string parameters to the origin, but it only caches based on the query string parameters that you specify.

            If you specify false for ``QueryString`` , CloudFront doesn't forward any query string parameters to the origin, and doesn't cache based on query string parameters.

            For more information, see `Configuring CloudFront to Cache Based on Query String Parameters <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/QueryStringParameters.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-forwardedvalues.html#cfn-cloudfront-distribution-forwardedvalues-querystring
            '''
            result = self._values.get("query_string")
            assert result is not None, "Required property 'query_string' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def cookies(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.CookiesProperty"]]:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include cookies in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send cookies to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            A complex type that specifies whether you want CloudFront to forward cookies to the origin and, if so, which ones. For more information about forwarding cookies to the origin, see `How CloudFront Forwards, Caches, and Logs Cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/Cookies.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-forwardedvalues.html#cfn-cloudfront-distribution-forwardedvalues-cookies
            '''
            result = self._values.get("cookies")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.CookiesProperty"]], result)

        @builtins.property
        def headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include headers in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send headers to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            A complex type that specifies the ``Headers`` , if any, that you want CloudFront to forward to the origin for this cache behavior (whitelisted headers). For the headers that you specify, CloudFront also caches separate versions of a specified object that is based on the header values in viewer requests.

            For more information, see `Caching Content Based on Request Headers <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/header-caching.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-forwardedvalues.html#cfn-cloudfront-distribution-forwardedvalues-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def query_string_cache_keys(self) -> typing.Optional[typing.List[builtins.str]]:
            '''This field is deprecated.

            We recommend that you use a cache policy or an origin request policy instead of this field.

            If you want to include query strings in the cache key, use a cache policy. For more information, see `Creating cache policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html#cache-key-create-cache-policy>`_ in the *Amazon CloudFront Developer Guide* .

            If you want to send query strings to the origin but not include them in the cache key, use an origin request policy. For more information, see `Creating origin request policies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-origin-requests.html#origin-request-create-origin-request-policy>`_ in the *Amazon CloudFront Developer Guide* .

            A complex type that contains information about the query string parameters that you want CloudFront to use for caching for this cache behavior.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-forwardedvalues.html#cfn-cloudfront-distribution-forwardedvalues-querystringcachekeys
            '''
            result = self._values.get("query_string_cache_keys")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ForwardedValuesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.FunctionAssociationProperty",
        jsii_struct_bases=[],
        name_mapping={"event_type": "eventType", "function_arn": "functionArn"},
    )
    class FunctionAssociationProperty:
        def __init__(
            self,
            *,
            event_type: typing.Optional[builtins.str] = None,
            function_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A CloudFront function that is associated with a cache behavior in a CloudFront distribution.

            :param event_type: The event type of the function, either ``viewer-request`` or ``viewer-response`` . You cannot use origin-facing event types ( ``origin-request`` and ``origin-response`` ) with a CloudFront function.
            :param function_arn: The Amazon Resource Name (ARN) of the function.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-functionassociation.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                function_association_property = cloudfront.CfnDistribution.FunctionAssociationProperty(
                    event_type="eventType",
                    function_arn="functionArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9b1c3d15af0235666c25baf0f8fb72dee7e10ccd7370627fa82ef1302bef82ca)
                check_type(argname="argument event_type", value=event_type, expected_type=type_hints["event_type"])
                check_type(argname="argument function_arn", value=function_arn, expected_type=type_hints["function_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if event_type is not None:
                self._values["event_type"] = event_type
            if function_arn is not None:
                self._values["function_arn"] = function_arn

        @builtins.property
        def event_type(self) -> typing.Optional[builtins.str]:
            '''The event type of the function, either ``viewer-request`` or ``viewer-response`` .

            You cannot use origin-facing event types ( ``origin-request`` and ``origin-response`` ) with a CloudFront function.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-functionassociation.html#cfn-cloudfront-distribution-functionassociation-eventtype
            '''
            result = self._values.get("event_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def function_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the function.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-functionassociation.html#cfn-cloudfront-distribution-functionassociation-functionarn
            '''
            result = self._values.get("function_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FunctionAssociationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.GeoRestrictionProperty",
        jsii_struct_bases=[],
        name_mapping={"restriction_type": "restrictionType", "locations": "locations"},
    )
    class GeoRestrictionProperty:
        def __init__(
            self,
            *,
            restriction_type: builtins.str,
            locations: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''A complex type that controls the countries in which your content is distributed.

            CloudFront determines the location of your users using ``MaxMind`` GeoIP databases. To disable geo restriction, remove the `Restrictions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-restrictions>`_ property from your stack template.

            :param restriction_type: The method that you want to use to restrict distribution of your content by country:. - ``none`` : No geo restriction is enabled, meaning access to content is not restricted by client geo location. - ``blacklist`` : The ``Location`` elements specify the countries in which you don't want CloudFront to distribute your content. - ``whitelist`` : The ``Location`` elements specify the countries in which you want CloudFront to distribute your content.
            :param locations: A complex type that contains a ``Location`` element for each country in which you want CloudFront either to distribute your content ( ``whitelist`` ) or not distribute your content ( ``blacklist`` ). The ``Location`` element is a two-letter, uppercase country code for a country that you want to include in your ``blacklist`` or ``whitelist`` . Include one ``Location`` element for each country. CloudFront and ``MaxMind`` both use ``ISO 3166`` country codes. For the current list of countries and the corresponding codes, see ``ISO 3166-1-alpha-2`` code on the *International Organization for Standardization* website. You can also refer to the country list on the CloudFront console, which includes both country names and codes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-georestriction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                geo_restriction_property = cloudfront.CfnDistribution.GeoRestrictionProperty(
                    restriction_type="restrictionType",
                
                    # the properties below are optional
                    locations=["locations"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b9a009985093b4b38d3c65f7ab86737218f349ad69be1dffdda8eaf3ba9c0c12)
                check_type(argname="argument restriction_type", value=restriction_type, expected_type=type_hints["restriction_type"])
                check_type(argname="argument locations", value=locations, expected_type=type_hints["locations"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "restriction_type": restriction_type,
            }
            if locations is not None:
                self._values["locations"] = locations

        @builtins.property
        def restriction_type(self) -> builtins.str:
            '''The method that you want to use to restrict distribution of your content by country:.

            - ``none`` : No geo restriction is enabled, meaning access to content is not restricted by client geo location.
            - ``blacklist`` : The ``Location`` elements specify the countries in which you don't want CloudFront to distribute your content.
            - ``whitelist`` : The ``Location`` elements specify the countries in which you want CloudFront to distribute your content.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-georestriction.html#cfn-cloudfront-distribution-georestriction-restrictiontype
            '''
            result = self._values.get("restriction_type")
            assert result is not None, "Required property 'restriction_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def locations(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A complex type that contains a ``Location`` element for each country in which you want CloudFront either to distribute your content ( ``whitelist`` ) or not distribute your content ( ``blacklist`` ).

            The ``Location`` element is a two-letter, uppercase country code for a country that you want to include in your ``blacklist`` or ``whitelist`` . Include one ``Location`` element for each country.

            CloudFront and ``MaxMind`` both use ``ISO 3166`` country codes. For the current list of countries and the corresponding codes, see ``ISO 3166-1-alpha-2`` code on the *International Organization for Standardization* website. You can also refer to the country list on the CloudFront console, which includes both country names and codes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-georestriction.html#cfn-cloudfront-distribution-georestriction-locations
            '''
            result = self._values.get("locations")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GeoRestrictionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.LambdaFunctionAssociationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "event_type": "eventType",
            "include_body": "includeBody",
            "lambda_function_arn": "lambdaFunctionArn",
        },
    )
    class LambdaFunctionAssociationProperty:
        def __init__(
            self,
            *,
            event_type: typing.Optional[builtins.str] = None,
            include_body: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            lambda_function_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A complex type that contains a Lambda@Edge function association.

            :param event_type: Specifies the event type that triggers a Lambda@Edge function invocation. You can specify the following values:. - ``viewer-request`` : The function executes when CloudFront receives a request from a viewer and before it checks to see whether the requested object is in the edge cache. - ``origin-request`` : The function executes only when CloudFront sends a request to your origin. When the requested object is in the edge cache, the function doesn't execute. - ``origin-response`` : The function executes after CloudFront receives a response from the origin and before it caches the object in the response. When the requested object is in the edge cache, the function doesn't execute. - ``viewer-response`` : The function executes before CloudFront returns the requested object to the viewer. The function executes regardless of whether the object was already in the edge cache. If the origin returns an HTTP status code other than HTTP 200 (OK), the function doesn't execute.
            :param include_body: A flag that allows a Lambda@Edge function to have read access to the body content. For more information, see `Accessing the Request Body by Choosing the Include Body Option <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-include-body-access.html>`_ in the Amazon CloudFront Developer Guide.
            :param lambda_function_arn: The ARN of the Lambda@Edge function. You must specify the ARN of a function version; you can't specify an alias or $LATEST.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-lambdafunctionassociation.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                lambda_function_association_property = cloudfront.CfnDistribution.LambdaFunctionAssociationProperty(
                    event_type="eventType",
                    include_body=False,
                    lambda_function_arn="lambdaFunctionArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__447882608abe2f9e3437548735295dd3139b9bf1f7c42cf665c9b8d8ff0ee359)
                check_type(argname="argument event_type", value=event_type, expected_type=type_hints["event_type"])
                check_type(argname="argument include_body", value=include_body, expected_type=type_hints["include_body"])
                check_type(argname="argument lambda_function_arn", value=lambda_function_arn, expected_type=type_hints["lambda_function_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if event_type is not None:
                self._values["event_type"] = event_type
            if include_body is not None:
                self._values["include_body"] = include_body
            if lambda_function_arn is not None:
                self._values["lambda_function_arn"] = lambda_function_arn

        @builtins.property
        def event_type(self) -> typing.Optional[builtins.str]:
            '''Specifies the event type that triggers a Lambda@Edge function invocation. You can specify the following values:.

            - ``viewer-request`` : The function executes when CloudFront receives a request from a viewer and before it checks to see whether the requested object is in the edge cache.
            - ``origin-request`` : The function executes only when CloudFront sends a request to your origin. When the requested object is in the edge cache, the function doesn't execute.
            - ``origin-response`` : The function executes after CloudFront receives a response from the origin and before it caches the object in the response. When the requested object is in the edge cache, the function doesn't execute.
            - ``viewer-response`` : The function executes before CloudFront returns the requested object to the viewer. The function executes regardless of whether the object was already in the edge cache.

            If the origin returns an HTTP status code other than HTTP 200 (OK), the function doesn't execute.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-lambdafunctionassociation.html#cfn-cloudfront-distribution-lambdafunctionassociation-eventtype
            '''
            result = self._values.get("event_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def include_body(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A flag that allows a Lambda@Edge function to have read access to the body content.

            For more information, see `Accessing the Request Body by Choosing the Include Body Option <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-include-body-access.html>`_ in the Amazon CloudFront Developer Guide.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-lambdafunctionassociation.html#cfn-cloudfront-distribution-lambdafunctionassociation-includebody
            '''
            result = self._values.get("include_body")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def lambda_function_arn(self) -> typing.Optional[builtins.str]:
            '''The ARN of the Lambda@Edge function.

            You must specify the ARN of a function version; you can't specify an alias or $LATEST.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-lambdafunctionassociation.html#cfn-cloudfront-distribution-lambdafunctionassociation-lambdafunctionarn
            '''
            result = self._values.get("lambda_function_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LambdaFunctionAssociationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.LegacyCustomOriginProperty",
        jsii_struct_bases=[],
        name_mapping={
            "dns_name": "dnsName",
            "origin_protocol_policy": "originProtocolPolicy",
            "origin_ssl_protocols": "originSslProtocols",
            "http_port": "httpPort",
            "https_port": "httpsPort",
        },
    )
    class LegacyCustomOriginProperty:
        def __init__(
            self,
            *,
            dns_name: builtins.str,
            origin_protocol_policy: builtins.str,
            origin_ssl_protocols: typing.Sequence[builtins.str],
            http_port: typing.Optional[jsii.Number] = None,
            https_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param dns_name: ``CfnDistribution.LegacyCustomOriginProperty.DNSName``.
            :param origin_protocol_policy: ``CfnDistribution.LegacyCustomOriginProperty.OriginProtocolPolicy``.
            :param origin_ssl_protocols: ``CfnDistribution.LegacyCustomOriginProperty.OriginSSLProtocols``.
            :param http_port: ``CfnDistribution.LegacyCustomOriginProperty.HTTPPort``.
            :param https_port: ``CfnDistribution.LegacyCustomOriginProperty.HTTPSPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacycustomorigin.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                legacy_custom_origin_property = cloudfront.CfnDistribution.LegacyCustomOriginProperty(
                    dns_name="dnsName",
                    origin_protocol_policy="originProtocolPolicy",
                    origin_ssl_protocols=["originSslProtocols"],
                
                    # the properties below are optional
                    http_port=123,
                    https_port=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3a47b5b382245b1f8ce900409ba6ef2c2aecc195911bdd3ddb429d1359e033da)
                check_type(argname="argument dns_name", value=dns_name, expected_type=type_hints["dns_name"])
                check_type(argname="argument origin_protocol_policy", value=origin_protocol_policy, expected_type=type_hints["origin_protocol_policy"])
                check_type(argname="argument origin_ssl_protocols", value=origin_ssl_protocols, expected_type=type_hints["origin_ssl_protocols"])
                check_type(argname="argument http_port", value=http_port, expected_type=type_hints["http_port"])
                check_type(argname="argument https_port", value=https_port, expected_type=type_hints["https_port"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "dns_name": dns_name,
                "origin_protocol_policy": origin_protocol_policy,
                "origin_ssl_protocols": origin_ssl_protocols,
            }
            if http_port is not None:
                self._values["http_port"] = http_port
            if https_port is not None:
                self._values["https_port"] = https_port

        @builtins.property
        def dns_name(self) -> builtins.str:
            '''``CfnDistribution.LegacyCustomOriginProperty.DNSName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacycustomorigin.html#cfn-cloudfront-distribution-legacycustomorigin-dnsname
            '''
            result = self._values.get("dns_name")
            assert result is not None, "Required property 'dns_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def origin_protocol_policy(self) -> builtins.str:
            '''``CfnDistribution.LegacyCustomOriginProperty.OriginProtocolPolicy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacycustomorigin.html#cfn-cloudfront-distribution-legacycustomorigin-originprotocolpolicy
            '''
            result = self._values.get("origin_protocol_policy")
            assert result is not None, "Required property 'origin_protocol_policy' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def origin_ssl_protocols(self) -> typing.List[builtins.str]:
            '''``CfnDistribution.LegacyCustomOriginProperty.OriginSSLProtocols``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacycustomorigin.html#cfn-cloudfront-distribution-legacycustomorigin-originsslprotocols
            '''
            result = self._values.get("origin_ssl_protocols")
            assert result is not None, "Required property 'origin_ssl_protocols' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def http_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnDistribution.LegacyCustomOriginProperty.HTTPPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacycustomorigin.html#cfn-cloudfront-distribution-legacycustomorigin-httpport
            '''
            result = self._values.get("http_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def https_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnDistribution.LegacyCustomOriginProperty.HTTPSPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacycustomorigin.html#cfn-cloudfront-distribution-legacycustomorigin-httpsport
            '''
            result = self._values.get("https_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LegacyCustomOriginProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.LegacyS3OriginProperty",
        jsii_struct_bases=[],
        name_mapping={
            "dns_name": "dnsName",
            "origin_access_identity": "originAccessIdentity",
        },
    )
    class LegacyS3OriginProperty:
        def __init__(
            self,
            *,
            dns_name: builtins.str,
            origin_access_identity: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param dns_name: ``CfnDistribution.LegacyS3OriginProperty.DNSName``.
            :param origin_access_identity: ``CfnDistribution.LegacyS3OriginProperty.OriginAccessIdentity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacys3origin.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                legacy_s3_origin_property = cloudfront.CfnDistribution.LegacyS3OriginProperty(
                    dns_name="dnsName",
                
                    # the properties below are optional
                    origin_access_identity="originAccessIdentity"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b552217703d785cebea5eb256534c6aea7c300241e724a42387aafeac00154a9)
                check_type(argname="argument dns_name", value=dns_name, expected_type=type_hints["dns_name"])
                check_type(argname="argument origin_access_identity", value=origin_access_identity, expected_type=type_hints["origin_access_identity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "dns_name": dns_name,
            }
            if origin_access_identity is not None:
                self._values["origin_access_identity"] = origin_access_identity

        @builtins.property
        def dns_name(self) -> builtins.str:
            '''``CfnDistribution.LegacyS3OriginProperty.DNSName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacys3origin.html#cfn-cloudfront-distribution-legacys3origin-dnsname
            '''
            result = self._values.get("dns_name")
            assert result is not None, "Required property 'dns_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def origin_access_identity(self) -> typing.Optional[builtins.str]:
            '''``CfnDistribution.LegacyS3OriginProperty.OriginAccessIdentity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-legacys3origin.html#cfn-cloudfront-distribution-legacys3origin-originaccessidentity
            '''
            result = self._values.get("origin_access_identity")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LegacyS3OriginProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.LoggingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "bucket": "bucket",
            "include_cookies": "includeCookies",
            "prefix": "prefix",
        },
    )
    class LoggingProperty:
        def __init__(
            self,
            *,
            bucket: builtins.str,
            include_cookies: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            prefix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A complex type that controls whether access logs are written for the distribution.

            :param bucket: The Amazon S3 bucket to store the access logs in, for example, ``myawslogbucket.s3.amazonaws.com`` .
            :param include_cookies: Specifies whether you want CloudFront to include cookies in access logs, specify ``true`` for ``IncludeCookies`` . If you choose to include cookies in logs, CloudFront logs all cookies regardless of how you configure the cache behaviors for this distribution. If you don't want to include cookies when you create a distribution or if you want to disable include cookies for an existing distribution, specify ``false`` for ``IncludeCookies`` .
            :param prefix: An optional string that you want CloudFront to prefix to the access log ``filenames`` for this distribution, for example, ``myprefix/`` . If you want to enable logging, but you don't want to specify a prefix, you still must include an empty ``Prefix`` element in the ``Logging`` element.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-logging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                logging_property = cloudfront.CfnDistribution.LoggingProperty(
                    bucket="bucket",
                
                    # the properties below are optional
                    include_cookies=False,
                    prefix="prefix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__eb13f72080bcffb4b86369e94c125a910e12703069abb7a6d488c7a08df9f4ed)
                check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
                check_type(argname="argument include_cookies", value=include_cookies, expected_type=type_hints["include_cookies"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bucket": bucket,
            }
            if include_cookies is not None:
                self._values["include_cookies"] = include_cookies
            if prefix is not None:
                self._values["prefix"] = prefix

        @builtins.property
        def bucket(self) -> builtins.str:
            '''The Amazon S3 bucket to store the access logs in, for example, ``myawslogbucket.s3.amazonaws.com`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-logging.html#cfn-cloudfront-distribution-logging-bucket
            '''
            result = self._values.get("bucket")
            assert result is not None, "Required property 'bucket' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def include_cookies(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''Specifies whether you want CloudFront to include cookies in access logs, specify ``true`` for ``IncludeCookies`` .

            If you choose to include cookies in logs, CloudFront logs all cookies regardless of how you configure the cache behaviors for this distribution. If you don't want to include cookies when you create a distribution or if you want to disable include cookies for an existing distribution, specify ``false`` for ``IncludeCookies`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-logging.html#cfn-cloudfront-distribution-logging-includecookies
            '''
            result = self._values.get("include_cookies")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''An optional string that you want CloudFront to prefix to the access log ``filenames`` for this distribution, for example, ``myprefix/`` .

            If you want to enable logging, but you don't want to specify a prefix, you still must include an empty ``Prefix`` element in the ``Logging`` element.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-logging.html#cfn-cloudfront-distribution-logging-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginCustomHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"header_name": "headerName", "header_value": "headerValue"},
    )
    class OriginCustomHeaderProperty:
        def __init__(
            self,
            *,
            header_name: builtins.str,
            header_value: builtins.str,
        ) -> None:
            '''A complex type that contains ``HeaderName`` and ``HeaderValue`` elements, if any, for this distribution.

            :param header_name: The name of a header that you want CloudFront to send to your origin. For more information, see `Adding Custom Headers to Origin Requests <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/forward-custom-headers.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param header_value: The value for the header that you specified in the ``HeaderName`` field.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origincustomheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_custom_header_property = cloudfront.CfnDistribution.OriginCustomHeaderProperty(
                    header_name="headerName",
                    header_value="headerValue"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0414e075e780ce27259828768ef36ec46daf1c96d3b1e882814f5fa29afcf14a)
                check_type(argname="argument header_name", value=header_name, expected_type=type_hints["header_name"])
                check_type(argname="argument header_value", value=header_value, expected_type=type_hints["header_value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "header_name": header_name,
                "header_value": header_value,
            }

        @builtins.property
        def header_name(self) -> builtins.str:
            '''The name of a header that you want CloudFront to send to your origin.

            For more information, see `Adding Custom Headers to Origin Requests <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/forward-custom-headers.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origincustomheader.html#cfn-cloudfront-distribution-origincustomheader-headername
            '''
            result = self._values.get("header_name")
            assert result is not None, "Required property 'header_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def header_value(self) -> builtins.str:
            '''The value for the header that you specified in the ``HeaderName`` field.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origincustomheader.html#cfn-cloudfront-distribution-origincustomheader-headervalue
            '''
            result = self._values.get("header_value")
            assert result is not None, "Required property 'header_value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginCustomHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginGroupFailoverCriteriaProperty",
        jsii_struct_bases=[],
        name_mapping={"status_codes": "statusCodes"},
    )
    class OriginGroupFailoverCriteriaProperty:
        def __init__(
            self,
            *,
            status_codes: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.StatusCodesProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''A complex data type that includes information about the failover criteria for an origin group, including the status codes for which CloudFront will failover from the primary origin to the second origin.

            :param status_codes: The status codes that, when returned from the primary origin, will trigger CloudFront to failover to the second origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroupfailovercriteria.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_group_failover_criteria_property = cloudfront.CfnDistribution.OriginGroupFailoverCriteriaProperty(
                    status_codes=cloudfront.CfnDistribution.StatusCodesProperty(
                        items=[123],
                        quantity=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2add44dff049bb11d5828ed46ac35c863fd01befac690538c23f2d5e5bae4b4d)
                check_type(argname="argument status_codes", value=status_codes, expected_type=type_hints["status_codes"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "status_codes": status_codes,
            }

        @builtins.property
        def status_codes(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.StatusCodesProperty"]:
            '''The status codes that, when returned from the primary origin, will trigger CloudFront to failover to the second origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroupfailovercriteria.html#cfn-cloudfront-distribution-origingroupfailovercriteria-statuscodes
            '''
            result = self._values.get("status_codes")
            assert result is not None, "Required property 'status_codes' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.StatusCodesProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginGroupFailoverCriteriaProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginGroupMemberProperty",
        jsii_struct_bases=[],
        name_mapping={"origin_id": "originId"},
    )
    class OriginGroupMemberProperty:
        def __init__(self, *, origin_id: builtins.str) -> None:
            '''An origin in an origin group.

            :param origin_id: The ID for an origin in an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroupmember.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_group_member_property = cloudfront.CfnDistribution.OriginGroupMemberProperty(
                    origin_id="originId"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c589842e40d25c3cc2cee884665ae135bbdd142ec0c09f29e700ea2febb46b45)
                check_type(argname="argument origin_id", value=origin_id, expected_type=type_hints["origin_id"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "origin_id": origin_id,
            }

        @builtins.property
        def origin_id(self) -> builtins.str:
            '''The ID for an origin in an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroupmember.html#cfn-cloudfront-distribution-origingroupmember-originid
            '''
            result = self._values.get("origin_id")
            assert result is not None, "Required property 'origin_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginGroupMemberProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginGroupMembersProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items", "quantity": "quantity"},
    )
    class OriginGroupMembersProperty:
        def __init__(
            self,
            *,
            items: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.OriginGroupMemberProperty", typing.Dict[builtins.str, typing.Any]]]]],
            quantity: jsii.Number,
        ) -> None:
            '''A complex data type for the origins included in an origin group.

            :param items: Items (origins) in an origin group.
            :param quantity: The number of origins in an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroupmembers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_group_members_property = cloudfront.CfnDistribution.OriginGroupMembersProperty(
                    items=[cloudfront.CfnDistribution.OriginGroupMemberProperty(
                        origin_id="originId"
                    )],
                    quantity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9684bf3209e60a7a48fe9632ffeb2ec3b24a6679ba52ad0c5ca33d53a78e4781)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
                check_type(argname="argument quantity", value=quantity, expected_type=type_hints["quantity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
                "quantity": quantity,
            }

        @builtins.property
        def items(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupMemberProperty"]]]:
            '''Items (origins) in an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroupmembers.html#cfn-cloudfront-distribution-origingroupmembers-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupMemberProperty"]]], result)

        @builtins.property
        def quantity(self) -> jsii.Number:
            '''The number of origins in an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroupmembers.html#cfn-cloudfront-distribution-origingroupmembers-quantity
            '''
            result = self._values.get("quantity")
            assert result is not None, "Required property 'quantity' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginGroupMembersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginGroupProperty",
        jsii_struct_bases=[],
        name_mapping={
            "failover_criteria": "failoverCriteria",
            "id": "id",
            "members": "members",
        },
    )
    class OriginGroupProperty:
        def __init__(
            self,
            *,
            failover_criteria: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.OriginGroupFailoverCriteriaProperty", typing.Dict[builtins.str, typing.Any]]],
            id: builtins.str,
            members: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.OriginGroupMembersProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''An origin group includes two origins (a primary origin and a second origin to failover to) and a failover criteria that you specify.

            You create an origin group to support origin failover in CloudFront. When you create or update a distribution, you can specifiy the origin group instead of a single origin, and CloudFront will failover from the primary origin to the second origin under the failover conditions that you've chosen.

            :param failover_criteria: A complex type that contains information about the failover criteria for an origin group.
            :param id: The origin group's ID.
            :param members: A complex type that contains information about the origins in an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroup.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_group_property = cloudfront.CfnDistribution.OriginGroupProperty(
                    failover_criteria=cloudfront.CfnDistribution.OriginGroupFailoverCriteriaProperty(
                        status_codes=cloudfront.CfnDistribution.StatusCodesProperty(
                            items=[123],
                            quantity=123
                        )
                    ),
                    id="id",
                    members=cloudfront.CfnDistribution.OriginGroupMembersProperty(
                        items=[cloudfront.CfnDistribution.OriginGroupMemberProperty(
                            origin_id="originId"
                        )],
                        quantity=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__79714b3c9a059285e158c7a58750db48e5f6f80a5ebca26369bee034b5df8530)
                check_type(argname="argument failover_criteria", value=failover_criteria, expected_type=type_hints["failover_criteria"])
                check_type(argname="argument id", value=id, expected_type=type_hints["id"])
                check_type(argname="argument members", value=members, expected_type=type_hints["members"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "failover_criteria": failover_criteria,
                "id": id,
                "members": members,
            }

        @builtins.property
        def failover_criteria(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupFailoverCriteriaProperty"]:
            '''A complex type that contains information about the failover criteria for an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroup.html#cfn-cloudfront-distribution-origingroup-failovercriteria
            '''
            result = self._values.get("failover_criteria")
            assert result is not None, "Required property 'failover_criteria' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupFailoverCriteriaProperty"], result)

        @builtins.property
        def id(self) -> builtins.str:
            '''The origin group's ID.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroup.html#cfn-cloudfront-distribution-origingroup-id
            '''
            result = self._values.get("id")
            assert result is not None, "Required property 'id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def members(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupMembersProperty"]:
            '''A complex type that contains information about the origins in an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroup.html#cfn-cloudfront-distribution-origingroup-members
            '''
            result = self._values.get("members")
            assert result is not None, "Required property 'members' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupMembersProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginGroupProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginGroupsProperty",
        jsii_struct_bases=[],
        name_mapping={"quantity": "quantity", "items": "items"},
    )
    class OriginGroupsProperty:
        def __init__(
            self,
            *,
            quantity: jsii.Number,
            items: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.OriginGroupProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
        ) -> None:
            '''A complex data type for the origin groups specified for a distribution.

            :param quantity: The number of origin groups.
            :param items: The items (origin groups) in a distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroups.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_groups_property = cloudfront.CfnDistribution.OriginGroupsProperty(
                    quantity=123,
                
                    # the properties below are optional
                    items=[cloudfront.CfnDistribution.OriginGroupProperty(
                        failover_criteria=cloudfront.CfnDistribution.OriginGroupFailoverCriteriaProperty(
                            status_codes=cloudfront.CfnDistribution.StatusCodesProperty(
                                items=[123],
                                quantity=123
                            )
                        ),
                        id="id",
                        members=cloudfront.CfnDistribution.OriginGroupMembersProperty(
                            items=[cloudfront.CfnDistribution.OriginGroupMemberProperty(
                                origin_id="originId"
                            )],
                            quantity=123
                        )
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5954baf2467903a574beee9e747443c9a94e762afbfb39a9fc2cc9e80c2b66fc)
                check_type(argname="argument quantity", value=quantity, expected_type=type_hints["quantity"])
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "quantity": quantity,
            }
            if items is not None:
                self._values["items"] = items

        @builtins.property
        def quantity(self) -> jsii.Number:
            '''The number of origin groups.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroups.html#cfn-cloudfront-distribution-origingroups-quantity
            '''
            result = self._values.get("quantity")
            assert result is not None, "Required property 'quantity' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def items(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupProperty"]]]]:
            '''The items (origin groups) in a distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origingroups.html#cfn-cloudfront-distribution-origingroups-items
            '''
            result = self._values.get("items")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginGroupProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginGroupsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginProperty",
        jsii_struct_bases=[],
        name_mapping={
            "domain_name": "domainName",
            "id": "id",
            "connection_attempts": "connectionAttempts",
            "connection_timeout": "connectionTimeout",
            "custom_origin_config": "customOriginConfig",
            "origin_access_control_id": "originAccessControlId",
            "origin_custom_headers": "originCustomHeaders",
            "origin_path": "originPath",
            "origin_shield": "originShield",
            "s3_origin_config": "s3OriginConfig",
        },
    )
    class OriginProperty:
        def __init__(
            self,
            *,
            domain_name: builtins.str,
            id: builtins.str,
            connection_attempts: typing.Optional[jsii.Number] = None,
            connection_timeout: typing.Optional[jsii.Number] = None,
            custom_origin_config: typing.Optional[typing.Union[typing.Union["CfnDistribution.CustomOriginConfigProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]] = None,
            origin_access_control_id: typing.Optional[builtins.str] = None,
            origin_custom_headers: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.OriginCustomHeaderProperty", typing.Dict[builtins.str, typing.Any]]]]]] = None,
            origin_path: typing.Optional[builtins.str] = None,
            origin_shield: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.OriginShieldProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            s3_origin_config: typing.Optional[typing.Union[typing.Union["CfnDistribution.S3OriginConfigProperty", typing.Dict[builtins.str, typing.Any]], _aws_cdk_core_f4b25747.IResolvable]] = None,
        ) -> None:
            '''An origin.

            An origin is the location where content is stored, and from which CloudFront gets content to serve to viewers. To specify an origin:

            - Use ``S3OriginConfig`` to specify an Amazon S3 bucket that is not configured with static website hosting.
            - Use ``CustomOriginConfig`` to specify all other kinds of origins, including:
            - An Amazon S3 bucket that is configured with static website hosting
            - An Elastic Load Balancing load balancer
            - An AWS Elemental MediaPackage endpoint
            - An AWS Elemental MediaStore container
            - Any other HTTP server, running on an Amazon EC2 instance or any other kind of host

            For the current maximum number of origins that you can specify per distribution, see `General Quotas on Web Distributions <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cloudfront-limits.html#limits-web-distributions>`_ in the *Amazon CloudFront Developer Guide* (quotas were formerly referred to as limits).

            :param domain_name: The domain name for the origin. For more information, see `Origin Domain Name <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName>`_ in the *Amazon CloudFront Developer Guide* .
            :param id: A unique identifier for the origin. This value must be unique within the distribution. Use this value to specify the ``TargetOriginId`` in a ``CacheBehavior`` or ``DefaultCacheBehavior`` .
            :param connection_attempts: The number of times that CloudFront attempts to connect to the origin. The minimum number is 1, the maximum is 3, and the default (if you don’t specify otherwise) is 3. For a custom origin (including an Amazon S3 bucket that’s configured with static website hosting), this value also specifies the number of times that CloudFront attempts to get a response from the origin, in the case of an `Origin Response Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout>`_ . For more information, see `Origin Connection Attempts <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts>`_ in the *Amazon CloudFront Developer Guide* .
            :param connection_timeout: The number of seconds that CloudFront waits when trying to establish a connection to the origin. The minimum timeout is 1 second, the maximum is 10 seconds, and the default (if you don’t specify otherwise) is 10 seconds. For more information, see `Origin Connection Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout>`_ in the *Amazon CloudFront Developer Guide* .
            :param custom_origin_config: Use this type to specify an origin that is not an Amazon S3 bucket, with one exception. If the Amazon S3 bucket is configured with static website hosting, use this type. If the Amazon S3 bucket is not configured with static website hosting, use the ``S3OriginConfig`` type instead.
            :param origin_access_control_id: The unique identifier of an origin access control for this origin. For more information, see `Restricting access to an Amazon S3 origin <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param origin_custom_headers: A list of HTTP header names and values that CloudFront adds to the requests that it sends to the origin. For more information, see `Adding Custom Headers to Origin Requests <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param origin_path: An optional path that CloudFront appends to the origin domain name when CloudFront requests content from the origin. For more information, see `Origin Path <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginPath>`_ in the *Amazon CloudFront Developer Guide* .
            :param origin_shield: CloudFront Origin Shield. Using Origin Shield can help reduce the load on your origin. For more information, see `Using Origin Shield <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/origin-shield.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param s3_origin_config: Use this type to specify an origin that is an Amazon S3 bucket that is not configured with static website hosting. To specify any other type of origin, including an Amazon S3 bucket that is configured with static website hosting, use the ``CustomOriginConfig`` type instead.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_property = cloudfront.CfnDistribution.OriginProperty(
                    domain_name="domainName",
                    id="id",
                
                    # the properties below are optional
                    connection_attempts=123,
                    connection_timeout=123,
                    custom_origin_config=cloudfront.CfnDistribution.CustomOriginConfigProperty(
                        origin_protocol_policy="originProtocolPolicy",
                
                        # the properties below are optional
                        http_port=123,
                        https_port=123,
                        origin_keepalive_timeout=123,
                        origin_read_timeout=123,
                        origin_ssl_protocols=["originSslProtocols"]
                    ),
                    origin_access_control_id="originAccessControlId",
                    origin_custom_headers=[cloudfront.CfnDistribution.OriginCustomHeaderProperty(
                        header_name="headerName",
                        header_value="headerValue"
                    )],
                    origin_path="originPath",
                    origin_shield=cloudfront.CfnDistribution.OriginShieldProperty(
                        enabled=False,
                        origin_shield_region="originShieldRegion"
                    ),
                    s3_origin_config=cloudfront.CfnDistribution.S3OriginConfigProperty(
                        origin_access_identity="originAccessIdentity"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__afd3ef5ed95f5370a02e53dae3bae7f3569d0e36ebcf6a63291aedeb9eed9412)
                check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
                check_type(argname="argument id", value=id, expected_type=type_hints["id"])
                check_type(argname="argument connection_attempts", value=connection_attempts, expected_type=type_hints["connection_attempts"])
                check_type(argname="argument connection_timeout", value=connection_timeout, expected_type=type_hints["connection_timeout"])
                check_type(argname="argument custom_origin_config", value=custom_origin_config, expected_type=type_hints["custom_origin_config"])
                check_type(argname="argument origin_access_control_id", value=origin_access_control_id, expected_type=type_hints["origin_access_control_id"])
                check_type(argname="argument origin_custom_headers", value=origin_custom_headers, expected_type=type_hints["origin_custom_headers"])
                check_type(argname="argument origin_path", value=origin_path, expected_type=type_hints["origin_path"])
                check_type(argname="argument origin_shield", value=origin_shield, expected_type=type_hints["origin_shield"])
                check_type(argname="argument s3_origin_config", value=s3_origin_config, expected_type=type_hints["s3_origin_config"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "domain_name": domain_name,
                "id": id,
            }
            if connection_attempts is not None:
                self._values["connection_attempts"] = connection_attempts
            if connection_timeout is not None:
                self._values["connection_timeout"] = connection_timeout
            if custom_origin_config is not None:
                self._values["custom_origin_config"] = custom_origin_config
            if origin_access_control_id is not None:
                self._values["origin_access_control_id"] = origin_access_control_id
            if origin_custom_headers is not None:
                self._values["origin_custom_headers"] = origin_custom_headers
            if origin_path is not None:
                self._values["origin_path"] = origin_path
            if origin_shield is not None:
                self._values["origin_shield"] = origin_shield
            if s3_origin_config is not None:
                self._values["s3_origin_config"] = s3_origin_config

        @builtins.property
        def domain_name(self) -> builtins.str:
            '''The domain name for the origin.

            For more information, see `Origin Domain Name <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesDomainName>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-domainname
            '''
            result = self._values.get("domain_name")
            assert result is not None, "Required property 'domain_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def id(self) -> builtins.str:
            '''A unique identifier for the origin. This value must be unique within the distribution.

            Use this value to specify the ``TargetOriginId`` in a ``CacheBehavior`` or ``DefaultCacheBehavior`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-id
            '''
            result = self._values.get("id")
            assert result is not None, "Required property 'id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def connection_attempts(self) -> typing.Optional[jsii.Number]:
            '''The number of times that CloudFront attempts to connect to the origin.

            The minimum number is 1, the maximum is 3, and the default (if you don’t specify otherwise) is 3.

            For a custom origin (including an Amazon S3 bucket that’s configured with static website hosting), this value also specifies the number of times that CloudFront attempts to get a response from the origin, in the case of an `Origin Response Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginResponseTimeout>`_ .

            For more information, see `Origin Connection Attempts <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-attempts>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-connectionattempts
            '''
            result = self._values.get("connection_attempts")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def connection_timeout(self) -> typing.Optional[jsii.Number]:
            '''The number of seconds that CloudFront waits when trying to establish a connection to the origin.

            The minimum timeout is 1 second, the maximum is 10 seconds, and the default (if you don’t specify otherwise) is 10 seconds.

            For more information, see `Origin Connection Timeout <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#origin-connection-timeout>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-connectiontimeout
            '''
            result = self._values.get("connection_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def custom_origin_config(
            self,
        ) -> typing.Optional[typing.Union["CfnDistribution.CustomOriginConfigProperty", _aws_cdk_core_f4b25747.IResolvable]]:
            '''Use this type to specify an origin that is not an Amazon S3 bucket, with one exception.

            If the Amazon S3 bucket is configured with static website hosting, use this type. If the Amazon S3 bucket is not configured with static website hosting, use the ``S3OriginConfig`` type instead.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-customoriginconfig
            '''
            result = self._values.get("custom_origin_config")
            return typing.cast(typing.Optional[typing.Union["CfnDistribution.CustomOriginConfigProperty", _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def origin_access_control_id(self) -> typing.Optional[builtins.str]:
            '''The unique identifier of an origin access control for this origin.

            For more information, see `Restricting access to an Amazon S3 origin <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-originaccesscontrolid
            '''
            result = self._values.get("origin_access_control_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def origin_custom_headers(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginCustomHeaderProperty"]]]]:
            '''A list of HTTP header names and values that CloudFront adds to the requests that it sends to the origin.

            For more information, see `Adding Custom Headers to Origin Requests <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-origincustomheaders
            '''
            result = self._values.get("origin_custom_headers")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginCustomHeaderProperty"]]]], result)

        @builtins.property
        def origin_path(self) -> typing.Optional[builtins.str]:
            '''An optional path that CloudFront appends to the origin domain name when CloudFront requests content from the origin.

            For more information, see `Origin Path <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValuesOriginPath>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-originpath
            '''
            result = self._values.get("origin_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def origin_shield(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginShieldProperty"]]:
            '''CloudFront Origin Shield. Using Origin Shield can help reduce the load on your origin.

            For more information, see `Using Origin Shield <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/origin-shield.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-originshield
            '''
            result = self._values.get("origin_shield")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.OriginShieldProperty"]], result)

        @builtins.property
        def s3_origin_config(
            self,
        ) -> typing.Optional[typing.Union["CfnDistribution.S3OriginConfigProperty", _aws_cdk_core_f4b25747.IResolvable]]:
            '''Use this type to specify an origin that is an Amazon S3 bucket that is not configured with static website hosting.

            To specify any other type of origin, including an Amazon S3 bucket that is configured with static website hosting, use the ``CustomOriginConfig`` type instead.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-origin.html#cfn-cloudfront-distribution-origin-s3originconfig
            '''
            result = self._values.get("s3_origin_config")
            return typing.cast(typing.Optional[typing.Union["CfnDistribution.S3OriginConfigProperty", _aws_cdk_core_f4b25747.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.OriginShieldProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "origin_shield_region": "originShieldRegion",
        },
    )
    class OriginShieldProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            origin_shield_region: typing.Optional[builtins.str] = None,
        ) -> None:
            '''CloudFront Origin Shield.

            Using Origin Shield can help reduce the load on your origin. For more information, see `Using Origin Shield <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/origin-shield.html>`_ in the *Amazon CloudFront Developer Guide* .

            :param enabled: A flag that specifies whether Origin Shield is enabled. When it’s enabled, CloudFront routes all requests through Origin Shield, which can help protect your origin. When it’s disabled, CloudFront might send requests directly to your origin from multiple edge locations or regional edge caches.
            :param origin_shield_region: The AWS Region for Origin Shield. Specify the AWS Region that has the lowest latency to your origin. To specify a region, use the region code, not the region name. For example, specify the US East (Ohio) region as ``us-east-2`` . When you enable CloudFront Origin Shield, you must specify the AWS Region for Origin Shield. For the list of AWS Regions that you can specify, and for help choosing the best Region for your origin, see `Choosing the AWS Region for Origin Shield <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/origin-shield.html#choose-origin-shield-region>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-originshield.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_shield_property = cloudfront.CfnDistribution.OriginShieldProperty(
                    enabled=False,
                    origin_shield_region="originShieldRegion"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__bcd9ea5d09032796b6c61759fc95e96e2b4cf8db60d81e7b9ec8ad757d5af926)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument origin_shield_region", value=origin_shield_region, expected_type=type_hints["origin_shield_region"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled
            if origin_shield_region is not None:
                self._values["origin_shield_region"] = origin_shield_region

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A flag that specifies whether Origin Shield is enabled.

            When it’s enabled, CloudFront routes all requests through Origin Shield, which can help protect your origin. When it’s disabled, CloudFront might send requests directly to your origin from multiple edge locations or regional edge caches.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-originshield.html#cfn-cloudfront-distribution-originshield-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def origin_shield_region(self) -> typing.Optional[builtins.str]:
            '''The AWS Region for Origin Shield.

            Specify the AWS Region that has the lowest latency to your origin. To specify a region, use the region code, not the region name. For example, specify the US East (Ohio) region as ``us-east-2`` .

            When you enable CloudFront Origin Shield, you must specify the AWS Region for Origin Shield. For the list of AWS Regions that you can specify, and for help choosing the best Region for your origin, see `Choosing the AWS Region for Origin Shield <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/origin-shield.html#choose-origin-shield-region>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-originshield.html#cfn-cloudfront-distribution-originshield-originshieldregion
            '''
            result = self._values.get("origin_shield_region")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginShieldProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.RestrictionsProperty",
        jsii_struct_bases=[],
        name_mapping={"geo_restriction": "geoRestriction"},
    )
    class RestrictionsProperty:
        def __init__(
            self,
            *,
            geo_restriction: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnDistribution.GeoRestrictionProperty", typing.Dict[builtins.str, typing.Any]]],
        ) -> None:
            '''A complex type that identifies ways in which you want to restrict distribution of your content.

            :param geo_restriction: A complex type that controls the countries in which your content is distributed. CloudFront determines the location of your users using ``MaxMind`` GeoIP databases. To disable geo restriction, remove the `Restrictions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-restrictions>`_ property from your stack template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-restrictions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                restrictions_property = cloudfront.CfnDistribution.RestrictionsProperty(
                    geo_restriction=cloudfront.CfnDistribution.GeoRestrictionProperty(
                        restriction_type="restrictionType",
                
                        # the properties below are optional
                        locations=["locations"]
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__3b7f7be738639fe1ffe374c22e3303d8ebe36f17f19a1058f4dda452179bb321)
                check_type(argname="argument geo_restriction", value=geo_restriction, expected_type=type_hints["geo_restriction"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "geo_restriction": geo_restriction,
            }

        @builtins.property
        def geo_restriction(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.GeoRestrictionProperty"]:
            '''A complex type that controls the countries in which your content is distributed.

            CloudFront determines the location of your users using ``MaxMind`` GeoIP databases. To disable geo restriction, remove the `Restrictions <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-distributionconfig.html#cfn-cloudfront-distribution-distributionconfig-restrictions>`_ property from your stack template.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-restrictions.html#cfn-cloudfront-distribution-restrictions-georestriction
            '''
            result = self._values.get("geo_restriction")
            assert result is not None, "Required property 'geo_restriction' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnDistribution.GeoRestrictionProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RestrictionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.S3OriginConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"origin_access_identity": "originAccessIdentity"},
    )
    class S3OriginConfigProperty:
        def __init__(
            self,
            *,
            origin_access_identity: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A complex type that contains information about the Amazon S3 origin.

            If the origin is a custom origin or an S3 bucket that is configured as a website endpoint, use the ``CustomOriginConfig`` element instead.

            :param origin_access_identity: The CloudFront origin access identity to associate with the origin. Use an origin access identity to configure the origin so that viewers can *only* access objects in an Amazon S3 bucket through CloudFront. The format of the value is: origin-access-identity/cloudfront/ *ID-of-origin-access-identity* where ``*ID-of-origin-access-identity*`` is the value that CloudFront returned in the ``ID`` element when you created the origin access identity. If you want viewers to be able to access objects using either the CloudFront URL or the Amazon S3 URL, specify an empty ``OriginAccessIdentity`` element. To delete the origin access identity from an existing distribution, update the distribution configuration and include an empty ``OriginAccessIdentity`` element. To replace the origin access identity, update the distribution configuration and specify the new origin access identity. For more information about the origin access identity, see `Serving Private Content through CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-s3originconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                s3_origin_config_property = cloudfront.CfnDistribution.S3OriginConfigProperty(
                    origin_access_identity="originAccessIdentity"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5045ffcff79ccc9b4345e02a845001fc5ee1a1ad1b336edb466f9d3f48b5d062)
                check_type(argname="argument origin_access_identity", value=origin_access_identity, expected_type=type_hints["origin_access_identity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if origin_access_identity is not None:
                self._values["origin_access_identity"] = origin_access_identity

        @builtins.property
        def origin_access_identity(self) -> typing.Optional[builtins.str]:
            '''The CloudFront origin access identity to associate with the origin.

            Use an origin access identity to configure the origin so that viewers can *only* access objects in an Amazon S3 bucket through CloudFront. The format of the value is:

            origin-access-identity/cloudfront/ *ID-of-origin-access-identity*

            where ``*ID-of-origin-access-identity*`` is the value that CloudFront returned in the ``ID`` element when you created the origin access identity.

            If you want viewers to be able to access objects using either the CloudFront URL or the Amazon S3 URL, specify an empty ``OriginAccessIdentity`` element.

            To delete the origin access identity from an existing distribution, update the distribution configuration and include an empty ``OriginAccessIdentity`` element.

            To replace the origin access identity, update the distribution configuration and specify the new origin access identity.

            For more information about the origin access identity, see `Serving Private Content through CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-s3originconfig.html#cfn-cloudfront-distribution-s3originconfig-originaccessidentity
            '''
            result = self._values.get("origin_access_identity")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3OriginConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.StatusCodesProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items", "quantity": "quantity"},
    )
    class StatusCodesProperty:
        def __init__(
            self,
            *,
            items: typing.Union[typing.Sequence[jsii.Number], _aws_cdk_core_f4b25747.IResolvable],
            quantity: jsii.Number,
        ) -> None:
            '''A complex data type for the status codes that you specify that, when returned by a primary origin, trigger CloudFront to failover to a second origin.

            :param items: The items (status codes) for an origin group.
            :param quantity: The number of status codes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-statuscodes.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                status_codes_property = cloudfront.CfnDistribution.StatusCodesProperty(
                    items=[123],
                    quantity=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8543c07746f70feaddebb1ddaa763505c5db9a4b5e7c55b0411b6fc488c26580)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
                check_type(argname="argument quantity", value=quantity, expected_type=type_hints["quantity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
                "quantity": quantity,
            }

        @builtins.property
        def items(
            self,
        ) -> typing.Union[typing.List[jsii.Number], _aws_cdk_core_f4b25747.IResolvable]:
            '''The items (status codes) for an origin group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-statuscodes.html#cfn-cloudfront-distribution-statuscodes-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.Union[typing.List[jsii.Number], _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def quantity(self) -> jsii.Number:
            '''The number of status codes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-statuscodes.html#cfn-cloudfront-distribution-statuscodes-quantity
            '''
            result = self._values.get("quantity")
            assert result is not None, "Required property 'quantity' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StatusCodesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnDistribution.ViewerCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "acm_certificate_arn": "acmCertificateArn",
            "cloud_front_default_certificate": "cloudFrontDefaultCertificate",
            "iam_certificate_id": "iamCertificateId",
            "minimum_protocol_version": "minimumProtocolVersion",
            "ssl_support_method": "sslSupportMethod",
        },
    )
    class ViewerCertificateProperty:
        def __init__(
            self,
            *,
            acm_certificate_arn: typing.Optional[builtins.str] = None,
            cloud_front_default_certificate: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            iam_certificate_id: typing.Optional[builtins.str] = None,
            minimum_protocol_version: typing.Optional[builtins.str] = None,
            ssl_support_method: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A complex type that determines the distribution’s SSL/TLS configuration for communicating with viewers.

            If the distribution doesn’t use ``Aliases`` (also known as alternate domain names or CNAMEs)—that is, if the distribution uses the CloudFront domain name such as ``d111111abcdef8.cloudfront.net`` —set ``CloudFrontDefaultCertificate`` to ``true`` and leave all other fields empty.

            If the distribution uses ``Aliases`` (alternate domain names or CNAMEs), use the fields in this type to specify the following settings:

            - Which viewers the distribution accepts HTTPS connections from: only viewers that support `server name indication (SNI) <https://docs.aws.amazon.com/https://en.wikipedia.org/wiki/Server_Name_Indication>`_ (recommended), or all viewers including those that don’t support SNI.
            - To accept HTTPS connections from only viewers that support SNI, set ``SSLSupportMethod`` to ``sni-only`` . This is recommended. Most browsers and clients support SNI. (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.)
            - To accept HTTPS connections from all viewers, including those that don’t support SNI, set ``SSLSupportMethod`` to ``vip`` . This is not recommended, and results in additional monthly charges from CloudFront. (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.)
            - The minimum SSL/TLS protocol version that the distribution can use to communicate with viewers. To specify a minimum version, choose a value for ``MinimumProtocolVersion`` . For more information, see `Security Policy <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy>`_ in the *Amazon CloudFront Developer Guide* .
            - The location of the SSL/TLS certificate, `AWS Certificate Manager (ACM) <https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html>`_ (recommended) or `AWS Identity and Access Management (IAM) <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html>`_ . You specify the location by setting a value in one of the following fields (not both):
            - ``ACMCertificateArn`` (In CloudFormation, this field name is ``AcmCertificateArn`` . Note the different capitalization.)
            - ``IAMCertificateId`` (In CloudFormation, this field name is ``IamCertificateId`` . Note the different capitalization.)

            All distributions support HTTPS connections from viewers. To require viewers to use HTTPS only, or to redirect them from HTTP to HTTPS, use ``ViewerProtocolPolicy`` in the ``CacheBehavior`` or ``DefaultCacheBehavior`` . To specify how CloudFront should use SSL/TLS to communicate with your custom origin, use ``CustomOriginConfig`` .

            For more information, see `Using HTTPS with CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https.html>`_ and `Using Alternate Domain Names and HTTPS <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-alternate-domain-names.html>`_ in the *Amazon CloudFront Developer Guide* .

            :param acm_certificate_arn: .. epigraph:: In CloudFormation, this field name is ``AcmCertificateArn`` . Note the different capitalization. If the distribution uses ``Aliases`` (alternate domain names or CNAMEs) and the SSL/TLS certificate is stored in `AWS Certificate Manager (ACM) <https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html>`_ , provide the Amazon Resource Name (ARN) of the ACM certificate. CloudFront only supports ACM certificates in the US East (N. Virginia) Region ( ``us-east-1`` ). If you specify an ACM certificate ARN, you must also specify values for ``MinimumProtocolVersion`` and ``SSLSupportMethod`` . (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.)
            :param cloud_front_default_certificate: If the distribution uses the CloudFront domain name such as ``d111111abcdef8.cloudfront.net`` , set this field to ``true`` . If the distribution uses ``Aliases`` (alternate domain names or CNAMEs), set this field to ``false`` and specify values for the following fields: - ``ACMCertificateArn`` or ``IAMCertificateId`` (specify a value for one, not both) In CloudFormation, these field names are ``AcmCertificateArn`` and ``IamCertificateId`` . Note the different capitalization. - ``MinimumProtocolVersion`` - ``SSLSupportMethod`` (In CloudFormation, this field name is ``SslSupportMethod`` . Note the different capitalization.)
            :param iam_certificate_id: .. epigraph:: In CloudFormation, this field name is ``IamCertificateId`` . Note the different capitalization. If the distribution uses ``Aliases`` (alternate domain names or CNAMEs) and the SSL/TLS certificate is stored in `AWS Identity and Access Management (IAM) <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html>`_ , provide the ID of the IAM certificate. If you specify an IAM certificate ID, you must also specify values for ``MinimumProtocolVersion`` and ``SSLSupportMethod`` . (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.)
            :param minimum_protocol_version: If the distribution uses ``Aliases`` (alternate domain names or CNAMEs), specify the security policy that you want CloudFront to use for HTTPS connections with viewers. The security policy determines two settings: - The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers. - The ciphers that CloudFront can use to encrypt the content that it returns to viewers. For more information, see `Security Policy <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy>`_ and `Supported Protocols and Ciphers Between Viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers>`_ in the *Amazon CloudFront Developer Guide* . .. epigraph:: On the CloudFront console, this setting is called *Security Policy* . When you’re using SNI only (you set ``SSLSupportMethod`` to ``sni-only`` ), you must specify ``TLSv1`` or higher. (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.) If the distribution uses the CloudFront domain name such as ``d111111abcdef8.cloudfront.net`` (you set ``CloudFrontDefaultCertificate`` to ``true`` ), CloudFront automatically sets the security policy to ``TLSv1`` regardless of the value that you set here.
            :param ssl_support_method: .. epigraph:: In CloudFormation, this field name is ``SslSupportMethod`` . Note the different capitalization. If the distribution uses ``Aliases`` (alternate domain names or CNAMEs), specify which viewers the distribution accepts HTTPS connections from. - ``sni-only`` – The distribution accepts HTTPS connections from only viewers that support `server name indication (SNI) <https://docs.aws.amazon.com/https://en.wikipedia.org/wiki/Server_Name_Indication>`_ . This is recommended. Most browsers and clients support SNI. - ``vip`` – The distribution accepts HTTPS connections from all viewers including those that don’t support SNI. This is not recommended, and results in additional monthly charges from CloudFront. - ``static-ip`` - Do not specify this value unless your distribution has been enabled for this feature by the CloudFront team. If you have a use case that requires static IP addresses for a distribution, contact CloudFront through the `AWS Support Center <https://docs.aws.amazon.com/support/home>`_ . If the distribution uses the CloudFront domain name such as ``d111111abcdef8.cloudfront.net`` , don’t set a value for this field.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-viewercertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                viewer_certificate_property = cloudfront.CfnDistribution.ViewerCertificateProperty(
                    acm_certificate_arn="acmCertificateArn",
                    cloud_front_default_certificate=False,
                    iam_certificate_id="iamCertificateId",
                    minimum_protocol_version="minimumProtocolVersion",
                    ssl_support_method="sslSupportMethod"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__fa6528e4af2cfba29f26ed26ff0c114ce048d67ca58ef787cab353a3a354d42c)
                check_type(argname="argument acm_certificate_arn", value=acm_certificate_arn, expected_type=type_hints["acm_certificate_arn"])
                check_type(argname="argument cloud_front_default_certificate", value=cloud_front_default_certificate, expected_type=type_hints["cloud_front_default_certificate"])
                check_type(argname="argument iam_certificate_id", value=iam_certificate_id, expected_type=type_hints["iam_certificate_id"])
                check_type(argname="argument minimum_protocol_version", value=minimum_protocol_version, expected_type=type_hints["minimum_protocol_version"])
                check_type(argname="argument ssl_support_method", value=ssl_support_method, expected_type=type_hints["ssl_support_method"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if acm_certificate_arn is not None:
                self._values["acm_certificate_arn"] = acm_certificate_arn
            if cloud_front_default_certificate is not None:
                self._values["cloud_front_default_certificate"] = cloud_front_default_certificate
            if iam_certificate_id is not None:
                self._values["iam_certificate_id"] = iam_certificate_id
            if minimum_protocol_version is not None:
                self._values["minimum_protocol_version"] = minimum_protocol_version
            if ssl_support_method is not None:
                self._values["ssl_support_method"] = ssl_support_method

        @builtins.property
        def acm_certificate_arn(self) -> typing.Optional[builtins.str]:
            '''.. epigraph::

   In CloudFormation, this field name is ``AcmCertificateArn`` . Note the different capitalization.

            If the distribution uses ``Aliases`` (alternate domain names or CNAMEs) and the SSL/TLS certificate is stored in `AWS Certificate Manager (ACM) <https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html>`_ , provide the Amazon Resource Name (ARN) of the ACM certificate. CloudFront only supports ACM certificates in the US East (N. Virginia) Region ( ``us-east-1`` ).

            If you specify an ACM certificate ARN, you must also specify values for ``MinimumProtocolVersion`` and ``SSLSupportMethod`` . (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.)

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-viewercertificate.html#cfn-cloudfront-distribution-viewercertificate-acmcertificatearn
            '''
            result = self._values.get("acm_certificate_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cloud_front_default_certificate(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''If the distribution uses the CloudFront domain name such as ``d111111abcdef8.cloudfront.net`` , set this field to ``true`` .

            If the distribution uses ``Aliases`` (alternate domain names or CNAMEs), set this field to ``false`` and specify values for the following fields:

            - ``ACMCertificateArn`` or ``IAMCertificateId`` (specify a value for one, not both)

            In CloudFormation, these field names are ``AcmCertificateArn`` and ``IamCertificateId`` . Note the different capitalization.

            - ``MinimumProtocolVersion``
            - ``SSLSupportMethod`` (In CloudFormation, this field name is ``SslSupportMethod`` . Note the different capitalization.)

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-viewercertificate.html#cfn-cloudfront-distribution-viewercertificate-cloudfrontdefaultcertificate
            '''
            result = self._values.get("cloud_front_default_certificate")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def iam_certificate_id(self) -> typing.Optional[builtins.str]:
            '''.. epigraph::

   In CloudFormation, this field name is ``IamCertificateId`` . Note the different capitalization.

            If the distribution uses ``Aliases`` (alternate domain names or CNAMEs) and the SSL/TLS certificate is stored in `AWS Identity and Access Management (IAM) <https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html>`_ , provide the ID of the IAM certificate.

            If you specify an IAM certificate ID, you must also specify values for ``MinimumProtocolVersion`` and ``SSLSupportMethod`` . (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.)

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-viewercertificate.html#cfn-cloudfront-distribution-viewercertificate-iamcertificateid
            '''
            result = self._values.get("iam_certificate_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def minimum_protocol_version(self) -> typing.Optional[builtins.str]:
            '''If the distribution uses ``Aliases`` (alternate domain names or CNAMEs), specify the security policy that you want CloudFront to use for HTTPS connections with viewers.

            The security policy determines two settings:

            - The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
            - The ciphers that CloudFront can use to encrypt the content that it returns to viewers.

            For more information, see `Security Policy <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy>`_ and `Supported Protocols and Ciphers Between Viewers and CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers>`_ in the *Amazon CloudFront Developer Guide* .
            .. epigraph::

               On the CloudFront console, this setting is called *Security Policy* .

            When you’re using SNI only (you set ``SSLSupportMethod`` to ``sni-only`` ), you must specify ``TLSv1`` or higher. (In CloudFormation, the field name is ``SslSupportMethod`` . Note the different capitalization.)

            If the distribution uses the CloudFront domain name such as ``d111111abcdef8.cloudfront.net`` (you set ``CloudFrontDefaultCertificate`` to ``true`` ), CloudFront automatically sets the security policy to ``TLSv1`` regardless of the value that you set here.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-viewercertificate.html#cfn-cloudfront-distribution-viewercertificate-minimumprotocolversion
            '''
            result = self._values.get("minimum_protocol_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ssl_support_method(self) -> typing.Optional[builtins.str]:
            '''.. epigraph::

   In CloudFormation, this field name is ``SslSupportMethod`` . Note the different capitalization.

            If the distribution uses ``Aliases`` (alternate domain names or CNAMEs), specify which viewers the distribution accepts HTTPS connections from.

            - ``sni-only`` – The distribution accepts HTTPS connections from only viewers that support `server name indication (SNI) <https://docs.aws.amazon.com/https://en.wikipedia.org/wiki/Server_Name_Indication>`_ . This is recommended. Most browsers and clients support SNI.
            - ``vip`` – The distribution accepts HTTPS connections from all viewers including those that don’t support SNI. This is not recommended, and results in additional monthly charges from CloudFront.
            - ``static-ip`` - Do not specify this value unless your distribution has been enabled for this feature by the CloudFront team. If you have a use case that requires static IP addresses for a distribution, contact CloudFront through the `AWS Support Center <https://docs.aws.amazon.com/support/home>`_ .

            If the distribution uses the CloudFront domain name such as ``d111111abcdef8.cloudfront.net`` , don’t set a value for this field.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-distribution-viewercertificate.html#cfn-cloudfront-distribution-viewercertificate-sslsupportmethod
            '''
            result = self._values.get("ssl_support_method")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ViewerCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnDistributionProps",
    jsii_struct_bases=[],
    name_mapping={"distribution_config": "distributionConfig", "tags": "tags"},
)
class CfnDistributionProps:
    def __init__(
        self,
        *,
        distribution_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnDistribution.DistributionConfigProperty, typing.Dict[builtins.str, typing.Any]]],
        tags: typing.Optional[typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnDistribution``.

        :param distribution_config: The distribution’s configuration.
        :param tags: A complex type that contains zero or more ``Tag`` elements.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-distribution.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_distribution_props = cloudfront.CfnDistributionProps(
                distribution_config=cloudfront.CfnDistribution.DistributionConfigProperty(
                    default_cache_behavior=cloudfront.CfnDistribution.DefaultCacheBehaviorProperty(
                        target_origin_id="targetOriginId",
                        viewer_protocol_policy="viewerProtocolPolicy",
            
                        # the properties below are optional
                        allowed_methods=["allowedMethods"],
                        cached_methods=["cachedMethods"],
                        cache_policy_id="cachePolicyId",
                        compress=False,
                        default_ttl=123,
                        field_level_encryption_id="fieldLevelEncryptionId",
                        forwarded_values=cloudfront.CfnDistribution.ForwardedValuesProperty(
                            query_string=False,
            
                            # the properties below are optional
                            cookies=cloudfront.CfnDistribution.CookiesProperty(
                                forward="forward",
            
                                # the properties below are optional
                                whitelisted_names=["whitelistedNames"]
                            ),
                            headers=["headers"],
                            query_string_cache_keys=["queryStringCacheKeys"]
                        ),
                        function_associations=[cloudfront.CfnDistribution.FunctionAssociationProperty(
                            event_type="eventType",
                            function_arn="functionArn"
                        )],
                        lambda_function_associations=[cloudfront.CfnDistribution.LambdaFunctionAssociationProperty(
                            event_type="eventType",
                            include_body=False,
                            lambda_function_arn="lambdaFunctionArn"
                        )],
                        max_ttl=123,
                        min_ttl=123,
                        origin_request_policy_id="originRequestPolicyId",
                        realtime_log_config_arn="realtimeLogConfigArn",
                        response_headers_policy_id="responseHeadersPolicyId",
                        smooth_streaming=False,
                        trusted_key_groups=["trustedKeyGroups"],
                        trusted_signers=["trustedSigners"]
                    ),
                    enabled=False,
            
                    # the properties below are optional
                    aliases=["aliases"],
                    cache_behaviors=[cloudfront.CfnDistribution.CacheBehaviorProperty(
                        path_pattern="pathPattern",
                        target_origin_id="targetOriginId",
                        viewer_protocol_policy="viewerProtocolPolicy",
            
                        # the properties below are optional
                        allowed_methods=["allowedMethods"],
                        cached_methods=["cachedMethods"],
                        cache_policy_id="cachePolicyId",
                        compress=False,
                        default_ttl=123,
                        field_level_encryption_id="fieldLevelEncryptionId",
                        forwarded_values=cloudfront.CfnDistribution.ForwardedValuesProperty(
                            query_string=False,
            
                            # the properties below are optional
                            cookies=cloudfront.CfnDistribution.CookiesProperty(
                                forward="forward",
            
                                # the properties below are optional
                                whitelisted_names=["whitelistedNames"]
                            ),
                            headers=["headers"],
                            query_string_cache_keys=["queryStringCacheKeys"]
                        ),
                        function_associations=[cloudfront.CfnDistribution.FunctionAssociationProperty(
                            event_type="eventType",
                            function_arn="functionArn"
                        )],
                        lambda_function_associations=[cloudfront.CfnDistribution.LambdaFunctionAssociationProperty(
                            event_type="eventType",
                            include_body=False,
                            lambda_function_arn="lambdaFunctionArn"
                        )],
                        max_ttl=123,
                        min_ttl=123,
                        origin_request_policy_id="originRequestPolicyId",
                        realtime_log_config_arn="realtimeLogConfigArn",
                        response_headers_policy_id="responseHeadersPolicyId",
                        smooth_streaming=False,
                        trusted_key_groups=["trustedKeyGroups"],
                        trusted_signers=["trustedSigners"]
                    )],
                    cnam_es=["cnamEs"],
                    comment="comment",
                    continuous_deployment_policy_id="continuousDeploymentPolicyId",
                    custom_error_responses=[cloudfront.CfnDistribution.CustomErrorResponseProperty(
                        error_code=123,
            
                        # the properties below are optional
                        error_caching_min_ttl=123,
                        response_code=123,
                        response_page_path="responsePagePath"
                    )],
                    custom_origin=cloudfront.CfnDistribution.LegacyCustomOriginProperty(
                        dns_name="dnsName",
                        origin_protocol_policy="originProtocolPolicy",
                        origin_ssl_protocols=["originSslProtocols"],
            
                        # the properties below are optional
                        http_port=123,
                        https_port=123
                    ),
                    default_root_object="defaultRootObject",
                    http_version="httpVersion",
                    ipv6_enabled=False,
                    logging=cloudfront.CfnDistribution.LoggingProperty(
                        bucket="bucket",
            
                        # the properties below are optional
                        include_cookies=False,
                        prefix="prefix"
                    ),
                    origin_groups=cloudfront.CfnDistribution.OriginGroupsProperty(
                        quantity=123,
            
                        # the properties below are optional
                        items=[cloudfront.CfnDistribution.OriginGroupProperty(
                            failover_criteria=cloudfront.CfnDistribution.OriginGroupFailoverCriteriaProperty(
                                status_codes=cloudfront.CfnDistribution.StatusCodesProperty(
                                    items=[123],
                                    quantity=123
                                )
                            ),
                            id="id",
                            members=cloudfront.CfnDistribution.OriginGroupMembersProperty(
                                items=[cloudfront.CfnDistribution.OriginGroupMemberProperty(
                                    origin_id="originId"
                                )],
                                quantity=123
                            )
                        )]
                    ),
                    origins=[cloudfront.CfnDistribution.OriginProperty(
                        domain_name="domainName",
                        id="id",
            
                        # the properties below are optional
                        connection_attempts=123,
                        connection_timeout=123,
                        custom_origin_config=cloudfront.CfnDistribution.CustomOriginConfigProperty(
                            origin_protocol_policy="originProtocolPolicy",
            
                            # the properties below are optional
                            http_port=123,
                            https_port=123,
                            origin_keepalive_timeout=123,
                            origin_read_timeout=123,
                            origin_ssl_protocols=["originSslProtocols"]
                        ),
                        origin_access_control_id="originAccessControlId",
                        origin_custom_headers=[cloudfront.CfnDistribution.OriginCustomHeaderProperty(
                            header_name="headerName",
                            header_value="headerValue"
                        )],
                        origin_path="originPath",
                        origin_shield=cloudfront.CfnDistribution.OriginShieldProperty(
                            enabled=False,
                            origin_shield_region="originShieldRegion"
                        ),
                        s3_origin_config=cloudfront.CfnDistribution.S3OriginConfigProperty(
                            origin_access_identity="originAccessIdentity"
                        )
                    )],
                    price_class="priceClass",
                    restrictions=cloudfront.CfnDistribution.RestrictionsProperty(
                        geo_restriction=cloudfront.CfnDistribution.GeoRestrictionProperty(
                            restriction_type="restrictionType",
            
                            # the properties below are optional
                            locations=["locations"]
                        )
                    ),
                    s3_origin=cloudfront.CfnDistribution.LegacyS3OriginProperty(
                        dns_name="dnsName",
            
                        # the properties below are optional
                        origin_access_identity="originAccessIdentity"
                    ),
                    staging=False,
                    viewer_certificate=cloudfront.CfnDistribution.ViewerCertificateProperty(
                        acm_certificate_arn="acmCertificateArn",
                        cloud_front_default_certificate=False,
                        iam_certificate_id="iamCertificateId",
                        minimum_protocol_version="minimumProtocolVersion",
                        ssl_support_method="sslSupportMethod"
                    ),
                    web_acl_id="webAclId"
                ),
            
                # the properties below are optional
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1189d2ef7617d30014a9bb338cb80c7094ceeefd215a1029129df6fa17cd49ce)
            check_type(argname="argument distribution_config", value=distribution_config, expected_type=type_hints["distribution_config"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "distribution_config": distribution_config,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def distribution_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDistribution.DistributionConfigProperty]:
        '''The distribution’s configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-distribution.html#cfn-cloudfront-distribution-distributionconfig
        '''
        result = self._values.get("distribution_config")
        assert result is not None, "Required property 'distribution_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnDistribution.DistributionConfigProperty], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]]:
        '''A complex type that contains zero or more ``Tag`` elements.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-distribution.html#cfn-cloudfront-distribution-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_aws_cdk_core_f4b25747.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDistributionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnFunction(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnFunction",
):
    '''A CloudFormation ``AWS::CloudFront::Function``.

    Creates a CloudFront function.

    To create a function, you provide the function code and some configuration information about the function. The response contains an Amazon Resource Name (ARN) that uniquely identifies the function, and the function’s stage.

    By default, when you create a function, it’s in the ``DEVELOPMENT`` stage. In this stage, you can `test the function <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/test-function.html>`_ in the CloudFront console (or with ``TestFunction`` in the CloudFront API).

    When you’re ready to use your function with a CloudFront distribution, publish the function to the ``LIVE`` stage. You can do this in the CloudFront console, with ``PublishFunction`` in the CloudFront API, or by updating the ``AWS::CloudFront::Function`` resource with the ``AutoPublish`` property set to ``true`` . When the function is published to the ``LIVE`` stage, you can attach it to a distribution’s cache behavior, using the function’s ARN.

    To automatically publish the function to the ``LIVE`` stage when it’s created, set the ``AutoPublish`` property to ``true`` .

    :cloudformationResource: AWS::CloudFront::Function
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_function = cloudfront.CfnFunction(self, "MyCfnFunction",
            function_code="functionCode",
            function_config=cloudfront.CfnFunction.FunctionConfigProperty(
                comment="comment",
                runtime="runtime"
            ),
            name="name",
        
            # the properties below are optional
            auto_publish=False,
            function_metadata=cloudfront.CfnFunction.FunctionMetadataProperty(
                function_arn="functionArn"
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        function_code: builtins.str,
        function_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.FunctionConfigProperty", typing.Dict[builtins.str, typing.Any]]],
        name: builtins.str,
        auto_publish: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        function_metadata: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnFunction.FunctionMetadataProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Create a new ``AWS::CloudFront::Function``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param function_code: The function code. For more information about writing a CloudFront function, see `Writing function code for CloudFront Functions <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html>`_ in the *Amazon CloudFront Developer Guide* .
        :param function_config: Contains configuration information about a CloudFront function.
        :param name: A name to identify the function.
        :param auto_publish: A flag that determines whether to automatically publish the function to the ``LIVE`` stage when it’s created. To automatically publish to the ``LIVE`` stage, set this property to ``true`` .
        :param function_metadata: Contains metadata about a CloudFront function.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ecb57a7c63fcb922165789bf36a0b4b830ada8e833348d9740378cd1181b14cd)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnFunctionProps(
            function_code=function_code,
            function_config=function_config,
            name=name,
            auto_publish=auto_publish,
            function_metadata=function_metadata,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__bdec0515dfad98e09aaa44b5b5f16250f9565f90bdd79412ea7e754133479d87)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9fd6c5d19905c472ca42750eb6095e0e3dcf296745215d41bd288eacce7f9d74)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrFunctionArn")
    def attr_function_arn(self) -> builtins.str:
        '''The ARN of the function. For example:.

        ``arn:aws:cloudfront::123456789012:function/ExampleFunction`` .

        To get the function ARN, use the following syntax:

        ``!GetAtt *Function_Logical_ID* .FunctionMetadata.FunctionARN``

        :cloudformationAttribute: FunctionARN
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFunctionArn"))

    @builtins.property
    @jsii.member(jsii_name="attrFunctionMetadataFunctionArn")
    def attr_function_metadata_function_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the function.

        The ARN uniquely identifies the function.

        :cloudformationAttribute: FunctionMetadata.FunctionARN
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFunctionMetadataFunctionArn"))

    @builtins.property
    @jsii.member(jsii_name="attrStage")
    def attr_stage(self) -> builtins.str:
        '''
        :cloudformationAttribute: Stage
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStage"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="functionCode")
    def function_code(self) -> builtins.str:
        '''The function code.

        For more information about writing a CloudFront function, see `Writing function code for CloudFront Functions <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html>`_ in the *Amazon CloudFront Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-functioncode
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionCode"))

    @function_code.setter
    def function_code(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6ededa96b23a18a06d818944a48b37258948c30ec7ba128d80dca107e053c45b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionCode", value)

    @builtins.property
    @jsii.member(jsii_name="functionConfig")
    def function_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FunctionConfigProperty"]:
        '''Contains configuration information about a CloudFront function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-functionconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FunctionConfigProperty"], jsii.get(self, "functionConfig"))

    @function_config.setter
    def function_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FunctionConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6bd36db1028d8cb5665d6a61bbf4aef7b4182141317b50bfd98270d051a4cc13)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionConfig", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''A name to identify the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-name
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d7030fdc47333f0e22f564f03e6e9d624e501ccdf4d16cee718c3bb394f27703)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="autoPublish")
    def auto_publish(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A flag that determines whether to automatically publish the function to the ``LIVE`` stage when it’s created.

        To automatically publish to the ``LIVE`` stage, set this property to ``true`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-autopublish
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], jsii.get(self, "autoPublish"))

    @auto_publish.setter
    def auto_publish(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d045ecdabb339f03d637f5b4ebcfc2f2bc19568e9bd0e4d06de18e3cab43c2ea)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "autoPublish", value)

    @builtins.property
    @jsii.member(jsii_name="functionMetadata")
    def function_metadata(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FunctionMetadataProperty"]]:
        '''Contains metadata about a CloudFront function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-functionmetadata
        '''
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FunctionMetadataProperty"]], jsii.get(self, "functionMetadata"))

    @function_metadata.setter
    def function_metadata(
        self,
        value: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnFunction.FunctionMetadataProperty"]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__089ee3212ab2f5f9d665d412ba84de43e5c53bd7d33f64b609b6a06c8e1ebfae)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "functionMetadata", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnFunction.FunctionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"comment": "comment", "runtime": "runtime"},
    )
    class FunctionConfigProperty:
        def __init__(self, *, comment: builtins.str, runtime: builtins.str) -> None:
            '''Contains configuration information about a CloudFront function.

            :param comment: A comment to describe the function.
            :param runtime: The function’s runtime environment. The only valid value is ``cloudfront-js-1.0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-function-functionconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                function_config_property = cloudfront.CfnFunction.FunctionConfigProperty(
                    comment="comment",
                    runtime="runtime"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0aded14f84ff1ebd71fadbe3a6f8464c971d903bc2494df69969a71cbc174700)
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
                check_type(argname="argument runtime", value=runtime, expected_type=type_hints["runtime"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "comment": comment,
                "runtime": runtime,
            }

        @builtins.property
        def comment(self) -> builtins.str:
            '''A comment to describe the function.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-function-functionconfig.html#cfn-cloudfront-function-functionconfig-comment
            '''
            result = self._values.get("comment")
            assert result is not None, "Required property 'comment' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def runtime(self) -> builtins.str:
            '''The function’s runtime environment.

            The only valid value is ``cloudfront-js-1.0`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-function-functionconfig.html#cfn-cloudfront-function-functionconfig-runtime
            '''
            result = self._values.get("runtime")
            assert result is not None, "Required property 'runtime' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FunctionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnFunction.FunctionMetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"function_arn": "functionArn"},
    )
    class FunctionMetadataProperty:
        def __init__(
            self,
            *,
            function_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Contains metadata about a CloudFront function.

            :param function_arn: The Amazon Resource Name (ARN) of the function. The ARN uniquely identifies the function.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-function-functionmetadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                function_metadata_property = cloudfront.CfnFunction.FunctionMetadataProperty(
                    function_arn="functionArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b081057bc1fabae33c1d5fddbb7cfed9d36c2d592275cdafa0ff7c1b62d4c4fd)
                check_type(argname="argument function_arn", value=function_arn, expected_type=type_hints["function_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if function_arn is not None:
                self._values["function_arn"] = function_arn

        @builtins.property
        def function_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the function.

            The ARN uniquely identifies the function.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-function-functionmetadata.html#cfn-cloudfront-function-functionmetadata-functionarn
            '''
            result = self._values.get("function_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FunctionMetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnFunctionProps",
    jsii_struct_bases=[],
    name_mapping={
        "function_code": "functionCode",
        "function_config": "functionConfig",
        "name": "name",
        "auto_publish": "autoPublish",
        "function_metadata": "functionMetadata",
    },
)
class CfnFunctionProps:
    def __init__(
        self,
        *,
        function_code: builtins.str,
        function_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.FunctionConfigProperty, typing.Dict[builtins.str, typing.Any]]],
        name: builtins.str,
        auto_publish: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        function_metadata: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnFunction.FunctionMetadataProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnFunction``.

        :param function_code: The function code. For more information about writing a CloudFront function, see `Writing function code for CloudFront Functions <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html>`_ in the *Amazon CloudFront Developer Guide* .
        :param function_config: Contains configuration information about a CloudFront function.
        :param name: A name to identify the function.
        :param auto_publish: A flag that determines whether to automatically publish the function to the ``LIVE`` stage when it’s created. To automatically publish to the ``LIVE`` stage, set this property to ``true`` .
        :param function_metadata: Contains metadata about a CloudFront function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_function_props = cloudfront.CfnFunctionProps(
                function_code="functionCode",
                function_config=cloudfront.CfnFunction.FunctionConfigProperty(
                    comment="comment",
                    runtime="runtime"
                ),
                name="name",
            
                # the properties below are optional
                auto_publish=False,
                function_metadata=cloudfront.CfnFunction.FunctionMetadataProperty(
                    function_arn="functionArn"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__46ea8b1ac82f50cfa944f6d0caa1d6d82f08bf160d945a19853912f99f8585e0)
            check_type(argname="argument function_code", value=function_code, expected_type=type_hints["function_code"])
            check_type(argname="argument function_config", value=function_config, expected_type=type_hints["function_config"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument auto_publish", value=auto_publish, expected_type=type_hints["auto_publish"])
            check_type(argname="argument function_metadata", value=function_metadata, expected_type=type_hints["function_metadata"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_code": function_code,
            "function_config": function_config,
            "name": name,
        }
        if auto_publish is not None:
            self._values["auto_publish"] = auto_publish
        if function_metadata is not None:
            self._values["function_metadata"] = function_metadata

    @builtins.property
    def function_code(self) -> builtins.str:
        '''The function code.

        For more information about writing a CloudFront function, see `Writing function code for CloudFront Functions <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/writing-function-code.html>`_ in the *Amazon CloudFront Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-functioncode
        '''
        result = self._values.get("function_code")
        assert result is not None, "Required property 'function_code' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def function_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.FunctionConfigProperty]:
        '''Contains configuration information about a CloudFront function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-functionconfig
        '''
        result = self._values.get("function_config")
        assert result is not None, "Required property 'function_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.FunctionConfigProperty], result)

    @builtins.property
    def name(self) -> builtins.str:
        '''A name to identify the function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def auto_publish(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
        '''A flag that determines whether to automatically publish the function to the ``LIVE`` stage when it’s created.

        To automatically publish to the ``LIVE`` stage, set this property to ``true`` .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-autopublish
        '''
        result = self._values.get("auto_publish")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

    @builtins.property
    def function_metadata(
        self,
    ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.FunctionMetadataProperty]]:
        '''Contains metadata about a CloudFront function.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-function.html#cfn-cloudfront-function-functionmetadata
        '''
        result = self._values.get("function_metadata")
        return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnFunction.FunctionMetadataProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnKeyGroup(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnKeyGroup",
):
    '''A CloudFormation ``AWS::CloudFront::KeyGroup``.

    A key group.

    A key group contains a list of public keys that you can use with `CloudFront signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ .

    :cloudformationResource: AWS::CloudFront::KeyGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-keygroup.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_key_group = cloudfront.CfnKeyGroup(self, "MyCfnKeyGroup",
            key_group_config=cloudfront.CfnKeyGroup.KeyGroupConfigProperty(
                items=["items"],
                name="name",
        
                # the properties below are optional
                comment="comment"
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        key_group_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnKeyGroup.KeyGroupConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::KeyGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param key_group_config: The key group configuration.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__fe752360f95de4b252b1d0fac282fb395884fda6aeecbaf29e81e71b565b650f)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnKeyGroupProps(key_group_config=key_group_config)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__271e32103f554c3e3164dfd9239889caf9fef0a712972226807d2a4e14ac95ac)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__30bb4fe9663f5baec5ca1514ecf9b2b102dd6e22e04885d6987f6e5662324358)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The identifier for the key group.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrLastModifiedTime")
    def attr_last_modified_time(self) -> builtins.str:
        '''The date and time when the key group was last modified.

        :cloudformationAttribute: LastModifiedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLastModifiedTime"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="keyGroupConfig")
    def key_group_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnKeyGroup.KeyGroupConfigProperty"]:
        '''The key group configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-keygroup.html#cfn-cloudfront-keygroup-keygroupconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnKeyGroup.KeyGroupConfigProperty"], jsii.get(self, "keyGroupConfig"))

    @key_group_config.setter
    def key_group_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnKeyGroup.KeyGroupConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06dcdc1241fa4ec27dc65bf330ad7e7e141d56c94fae57f1bbba10f9ac679085)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "keyGroupConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnKeyGroup.KeyGroupConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items", "name": "name", "comment": "comment"},
    )
    class KeyGroupConfigProperty:
        def __init__(
            self,
            *,
            items: typing.Sequence[builtins.str],
            name: builtins.str,
            comment: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A key group configuration.

            A key group contains a list of public keys that you can use with `CloudFront signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ .

            :param items: A list of the identifiers of the public keys in the key group.
            :param name: A name to identify the key group.
            :param comment: A comment to describe the key group. The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-keygroup-keygroupconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                key_group_config_property = cloudfront.CfnKeyGroup.KeyGroupConfigProperty(
                    items=["items"],
                    name="name",
                
                    # the properties below are optional
                    comment="comment"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__39c056c904cbb518d9e1555e509d46322405bd5131c3ff5f51aa05d596e54736)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
                "name": name,
            }
            if comment is not None:
                self._values["comment"] = comment

        @builtins.property
        def items(self) -> typing.List[builtins.str]:
            '''A list of the identifiers of the public keys in the key group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-keygroup-keygroupconfig.html#cfn-cloudfront-keygroup-keygroupconfig-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def name(self) -> builtins.str:
            '''A name to identify the key group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-keygroup-keygroupconfig.html#cfn-cloudfront-keygroup-keygroupconfig-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def comment(self) -> typing.Optional[builtins.str]:
            '''A comment to describe the key group.

            The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-keygroup-keygroupconfig.html#cfn-cloudfront-keygroup-keygroupconfig-comment
            '''
            result = self._values.get("comment")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KeyGroupConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnKeyGroupProps",
    jsii_struct_bases=[],
    name_mapping={"key_group_config": "keyGroupConfig"},
)
class CfnKeyGroupProps:
    def __init__(
        self,
        *,
        key_group_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnKeyGroup.KeyGroupConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnKeyGroup``.

        :param key_group_config: The key group configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-keygroup.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_key_group_props = cloudfront.CfnKeyGroupProps(
                key_group_config=cloudfront.CfnKeyGroup.KeyGroupConfigProperty(
                    items=["items"],
                    name="name",
            
                    # the properties below are optional
                    comment="comment"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__25eb319e291b4c4304c834d5b9d74cfc1c15e5f0b5ad9916ad109097d729a865)
            check_type(argname="argument key_group_config", value=key_group_config, expected_type=type_hints["key_group_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "key_group_config": key_group_config,
        }

    @builtins.property
    def key_group_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnKeyGroup.KeyGroupConfigProperty]:
        '''The key group configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-keygroup.html#cfn-cloudfront-keygroup-keygroupconfig
        '''
        result = self._values.get("key_group_config")
        assert result is not None, "Required property 'key_group_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnKeyGroup.KeyGroupConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnKeyGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnMonitoringSubscription(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnMonitoringSubscription",
):
    '''A CloudFormation ``AWS::CloudFront::MonitoringSubscription``.

    A monitoring subscription. This structure contains information about whether additional CloudWatch metrics are enabled for a given CloudFront distribution.

    :cloudformationResource: AWS::CloudFront::MonitoringSubscription
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-monitoringsubscription.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_monitoring_subscription = cloudfront.CfnMonitoringSubscription(self, "MyCfnMonitoringSubscription",
            distribution_id="distributionId",
            monitoring_subscription=cloudfront.CfnMonitoringSubscription.MonitoringSubscriptionProperty(
                realtime_metrics_subscription_config=cloudfront.CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty(
                    realtime_metrics_subscription_status="realtimeMetricsSubscriptionStatus"
                )
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        distribution_id: builtins.str,
        monitoring_subscription: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnMonitoringSubscription.MonitoringSubscriptionProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::MonitoringSubscription``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param distribution_id: The ID of the distribution that you are enabling metrics for.
        :param monitoring_subscription: A subscription configuration for additional CloudWatch metrics.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9621747bf5eb88b0b5feff9ea55c61ac81f8630280713579ffbf529cac709c75)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnMonitoringSubscriptionProps(
            distribution_id=distribution_id,
            monitoring_subscription=monitoring_subscription,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6d5a72c31d73a39d3f02d097db981ebc9e0c1a0fa60790643443bd5792ceb303)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b6abe2074caeb87877ff2676df9dc464eada8a8edf536cc168867a624e30b9d4)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="distributionId")
    def distribution_id(self) -> builtins.str:
        '''The ID of the distribution that you are enabling metrics for.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-monitoringsubscription.html#cfn-cloudfront-monitoringsubscription-distributionid
        '''
        return typing.cast(builtins.str, jsii.get(self, "distributionId"))

    @distribution_id.setter
    def distribution_id(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a92b6a04db6d033b8f0d21a6c39b0d4462a1e685331e665cca8c785272ad7488)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "distributionId", value)

    @builtins.property
    @jsii.member(jsii_name="monitoringSubscription")
    def monitoring_subscription(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnMonitoringSubscription.MonitoringSubscriptionProperty"]:
        '''A subscription configuration for additional CloudWatch metrics.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-monitoringsubscription.html#cfn-cloudfront-monitoringsubscription-monitoringsubscription
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnMonitoringSubscription.MonitoringSubscriptionProperty"], jsii.get(self, "monitoringSubscription"))

    @monitoring_subscription.setter
    def monitoring_subscription(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnMonitoringSubscription.MonitoringSubscriptionProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2b35a28860b28b033ce5e3c8bc10281006d5adb5403a2616b7f3d9196116ea9a)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "monitoringSubscription", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnMonitoringSubscription.MonitoringSubscriptionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "realtime_metrics_subscription_config": "realtimeMetricsSubscriptionConfig",
        },
    )
    class MonitoringSubscriptionProperty:
        def __init__(
            self,
            *,
            realtime_metrics_subscription_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A monitoring subscription.

            This structure contains information about whether additional CloudWatch metrics are enabled for a given CloudFront distribution.

            :param realtime_metrics_subscription_config: A subscription configuration for additional CloudWatch metrics.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-monitoringsubscription-monitoringsubscription.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                monitoring_subscription_property = cloudfront.CfnMonitoringSubscription.MonitoringSubscriptionProperty(
                    realtime_metrics_subscription_config=cloudfront.CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty(
                        realtime_metrics_subscription_status="realtimeMetricsSubscriptionStatus"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9de3eeeb5fae698145695aa0984fec935ba2bbc4d397d3720ee65b87ecd96bf5)
                check_type(argname="argument realtime_metrics_subscription_config", value=realtime_metrics_subscription_config, expected_type=type_hints["realtime_metrics_subscription_config"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if realtime_metrics_subscription_config is not None:
                self._values["realtime_metrics_subscription_config"] = realtime_metrics_subscription_config

        @builtins.property
        def realtime_metrics_subscription_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty"]]:
            '''A subscription configuration for additional CloudWatch metrics.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-monitoringsubscription-monitoringsubscription.html#cfn-cloudfront-monitoringsubscription-monitoringsubscription-realtimemetricssubscriptionconfig
            '''
            result = self._values.get("realtime_metrics_subscription_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MonitoringSubscriptionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "realtime_metrics_subscription_status": "realtimeMetricsSubscriptionStatus",
        },
    )
    class RealtimeMetricsSubscriptionConfigProperty:
        def __init__(
            self,
            *,
            realtime_metrics_subscription_status: builtins.str,
        ) -> None:
            '''A subscription configuration for additional CloudWatch metrics.

            :param realtime_metrics_subscription_status: A flag that indicates whether additional CloudWatch metrics are enabled for a given CloudFront distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-monitoringsubscription-realtimemetricssubscriptionconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                realtime_metrics_subscription_config_property = cloudfront.CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty(
                    realtime_metrics_subscription_status="realtimeMetricsSubscriptionStatus"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__62485e89d0b8c8c80f95bf647273c721dbc166ced3e5010c233cc23264f363f8)
                check_type(argname="argument realtime_metrics_subscription_status", value=realtime_metrics_subscription_status, expected_type=type_hints["realtime_metrics_subscription_status"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "realtime_metrics_subscription_status": realtime_metrics_subscription_status,
            }

        @builtins.property
        def realtime_metrics_subscription_status(self) -> builtins.str:
            '''A flag that indicates whether additional CloudWatch metrics are enabled for a given CloudFront distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-monitoringsubscription-realtimemetricssubscriptionconfig.html#cfn-cloudfront-monitoringsubscription-realtimemetricssubscriptionconfig-realtimemetricssubscriptionstatus
            '''
            result = self._values.get("realtime_metrics_subscription_status")
            assert result is not None, "Required property 'realtime_metrics_subscription_status' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RealtimeMetricsSubscriptionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnMonitoringSubscriptionProps",
    jsii_struct_bases=[],
    name_mapping={
        "distribution_id": "distributionId",
        "monitoring_subscription": "monitoringSubscription",
    },
)
class CfnMonitoringSubscriptionProps:
    def __init__(
        self,
        *,
        distribution_id: builtins.str,
        monitoring_subscription: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnMonitoringSubscription.MonitoringSubscriptionProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnMonitoringSubscription``.

        :param distribution_id: The ID of the distribution that you are enabling metrics for.
        :param monitoring_subscription: A subscription configuration for additional CloudWatch metrics.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-monitoringsubscription.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_monitoring_subscription_props = cloudfront.CfnMonitoringSubscriptionProps(
                distribution_id="distributionId",
                monitoring_subscription=cloudfront.CfnMonitoringSubscription.MonitoringSubscriptionProperty(
                    realtime_metrics_subscription_config=cloudfront.CfnMonitoringSubscription.RealtimeMetricsSubscriptionConfigProperty(
                        realtime_metrics_subscription_status="realtimeMetricsSubscriptionStatus"
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9aa23cfbb2837d1a2e29b1ea013fe6e2062aca26a1bab15d933f1f568fa9a502)
            check_type(argname="argument distribution_id", value=distribution_id, expected_type=type_hints["distribution_id"])
            check_type(argname="argument monitoring_subscription", value=monitoring_subscription, expected_type=type_hints["monitoring_subscription"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "distribution_id": distribution_id,
            "monitoring_subscription": monitoring_subscription,
        }

    @builtins.property
    def distribution_id(self) -> builtins.str:
        '''The ID of the distribution that you are enabling metrics for.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-monitoringsubscription.html#cfn-cloudfront-monitoringsubscription-distributionid
        '''
        result = self._values.get("distribution_id")
        assert result is not None, "Required property 'distribution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def monitoring_subscription(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnMonitoringSubscription.MonitoringSubscriptionProperty]:
        '''A subscription configuration for additional CloudWatch metrics.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-monitoringsubscription.html#cfn-cloudfront-monitoringsubscription-monitoringsubscription
        '''
        result = self._values.get("monitoring_subscription")
        assert result is not None, "Required property 'monitoring_subscription' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnMonitoringSubscription.MonitoringSubscriptionProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMonitoringSubscriptionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnOriginAccessControl(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnOriginAccessControl",
):
    '''A CloudFormation ``AWS::CloudFront::OriginAccessControl``.

    Creates a new origin access control in CloudFront. After you create an origin access control, you can add it to an origin in a CloudFront distribution so that CloudFront sends authenticated (signed) requests to the origin.

    For an Amazon S3 origin, this makes it possible to block public access to the Amazon S3 bucket so that viewers (users) can access the content in the bucket only through CloudFront.

    For more information about using a CloudFront origin access control, see `Restricting access to an Amazon S3 origin <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html>`_ in the *Amazon CloudFront Developer Guide* .

    :cloudformationResource: AWS::CloudFront::OriginAccessControl
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originaccesscontrol.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_origin_access_control = cloudfront.CfnOriginAccessControl(self, "MyCfnOriginAccessControl",
            origin_access_control_config=cloudfront.CfnOriginAccessControl.OriginAccessControlConfigProperty(
                name="name",
                origin_access_control_origin_type="originAccessControlOriginType",
                signing_behavior="signingBehavior",
                signing_protocol="signingProtocol",
        
                # the properties below are optional
                description="description"
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        origin_access_control_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnOriginAccessControl.OriginAccessControlConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::OriginAccessControl``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param origin_access_control_config: The origin access control.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__701c786d80db656b070b7c6001dc719cfb89231c5772eaa8e1d079a698b79121)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnOriginAccessControlProps(
            origin_access_control_config=origin_access_control_config
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__176fde4ae0708af0e3faf6f410823f57491d03e4484673bcafc15857b0f641b4)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2a48a75b8a822185f20079e775b8a1673f59f8038cb25b1b68aa5287ce742006)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The unique identifier of the origin access control.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="originAccessControlConfig")
    def origin_access_control_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginAccessControl.OriginAccessControlConfigProperty"]:
        '''The origin access control.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originaccesscontrol.html#cfn-cloudfront-originaccesscontrol-originaccesscontrolconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginAccessControl.OriginAccessControlConfigProperty"], jsii.get(self, "originAccessControlConfig"))

    @origin_access_control_config.setter
    def origin_access_control_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginAccessControl.OriginAccessControlConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__088a90b03a35ebbd11412c6522e055675754bc468dad456e2df474f6b605301b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "originAccessControlConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnOriginAccessControl.OriginAccessControlConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "origin_access_control_origin_type": "originAccessControlOriginType",
            "signing_behavior": "signingBehavior",
            "signing_protocol": "signingProtocol",
            "description": "description",
        },
    )
    class OriginAccessControlConfigProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            origin_access_control_origin_type: builtins.str,
            signing_behavior: builtins.str,
            signing_protocol: builtins.str,
            description: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Creates a new origin access control in CloudFront.

            After you create an origin access control, you can add it to an origin in a CloudFront distribution so that CloudFront sends authenticated (signed) requests to the origin.

            For an Amazon S3 origin, this makes it possible to block public access to the Amazon S3 bucket so that viewers (users) can access the content in the bucket only through CloudFront.

            For more information about using a CloudFront origin access control, see `Restricting access to an Amazon S3 origin <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html>`_ in the *Amazon CloudFront Developer Guide* .

            :param name: A name to identify the origin access control.
            :param origin_access_control_origin_type: The type of origin that this origin access control is for. The only valid value is ``s3`` .
            :param signing_behavior: Specifies which requests CloudFront signs (adds authentication information to). Specify ``always`` for the most common use case. For more information, see `origin access control advanced settings <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#oac-advanced-settings>`_ in the *Amazon CloudFront Developer Guide* . This field can have one of the following values: - ``always`` – CloudFront signs all origin requests, overwriting the ``Authorization`` header from the viewer request if one exists. - ``never`` – CloudFront doesn't sign any origin requests. This value turns off origin access control for all origins in all distributions that use this origin access control. - ``no-override`` – If the viewer request doesn't contain the ``Authorization`` header, then CloudFront signs the origin request. If the viewer request contains the ``Authorization`` header, then CloudFront doesn't sign the origin request and instead passes along the ``Authorization`` header from the viewer request. *WARNING: To pass along the ``Authorization`` header from the viewer request, you *must* add the ``Authorization`` header to a `cache policy <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html>`_ for all cache behaviors that use origins associated with this origin access control.*
            :param signing_protocol: The signing protocol of the origin access control, which determines how CloudFront signs (authenticates) requests. The only valid value is ``sigv4`` .
            :param description: A description of the origin access control.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originaccesscontrol-originaccesscontrolconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_access_control_config_property = cloudfront.CfnOriginAccessControl.OriginAccessControlConfigProperty(
                    name="name",
                    origin_access_control_origin_type="originAccessControlOriginType",
                    signing_behavior="signingBehavior",
                    signing_protocol="signingProtocol",
                
                    # the properties below are optional
                    description="description"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__d10699dd92dab4b50f441a521fe6076f3990b4614a33dbf81c7744d9a0f4928a)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument origin_access_control_origin_type", value=origin_access_control_origin_type, expected_type=type_hints["origin_access_control_origin_type"])
                check_type(argname="argument signing_behavior", value=signing_behavior, expected_type=type_hints["signing_behavior"])
                check_type(argname="argument signing_protocol", value=signing_protocol, expected_type=type_hints["signing_protocol"])
                check_type(argname="argument description", value=description, expected_type=type_hints["description"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
                "origin_access_control_origin_type": origin_access_control_origin_type,
                "signing_behavior": signing_behavior,
                "signing_protocol": signing_protocol,
            }
            if description is not None:
                self._values["description"] = description

        @builtins.property
        def name(self) -> builtins.str:
            '''A name to identify the origin access control.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originaccesscontrol-originaccesscontrolconfig.html#cfn-cloudfront-originaccesscontrol-originaccesscontrolconfig-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def origin_access_control_origin_type(self) -> builtins.str:
            '''The type of origin that this origin access control is for.

            The only valid value is ``s3`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originaccesscontrol-originaccesscontrolconfig.html#cfn-cloudfront-originaccesscontrol-originaccesscontrolconfig-originaccesscontrolorigintype
            '''
            result = self._values.get("origin_access_control_origin_type")
            assert result is not None, "Required property 'origin_access_control_origin_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def signing_behavior(self) -> builtins.str:
            '''Specifies which requests CloudFront signs (adds authentication information to).

            Specify ``always`` for the most common use case. For more information, see `origin access control advanced settings <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#oac-advanced-settings>`_ in the *Amazon CloudFront Developer Guide* .

            This field can have one of the following values:

            - ``always`` – CloudFront signs all origin requests, overwriting the ``Authorization`` header from the viewer request if one exists.
            - ``never`` – CloudFront doesn't sign any origin requests. This value turns off origin access control for all origins in all distributions that use this origin access control.
            - ``no-override`` – If the viewer request doesn't contain the ``Authorization`` header, then CloudFront signs the origin request. If the viewer request contains the ``Authorization`` header, then CloudFront doesn't sign the origin request and instead passes along the ``Authorization`` header from the viewer request. *WARNING: To pass along the ``Authorization`` header from the viewer request, you *must* add the ``Authorization`` header to a `cache policy <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/controlling-the-cache-key.html>`_ for all cache behaviors that use origins associated with this origin access control.*

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originaccesscontrol-originaccesscontrolconfig.html#cfn-cloudfront-originaccesscontrol-originaccesscontrolconfig-signingbehavior
            '''
            result = self._values.get("signing_behavior")
            assert result is not None, "Required property 'signing_behavior' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def signing_protocol(self) -> builtins.str:
            '''The signing protocol of the origin access control, which determines how CloudFront signs (authenticates) requests.

            The only valid value is ``sigv4`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originaccesscontrol-originaccesscontrolconfig.html#cfn-cloudfront-originaccesscontrol-originaccesscontrolconfig-signingprotocol
            '''
            result = self._values.get("signing_protocol")
            assert result is not None, "Required property 'signing_protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''A description of the origin access control.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originaccesscontrol-originaccesscontrolconfig.html#cfn-cloudfront-originaccesscontrol-originaccesscontrolconfig-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginAccessControlConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnOriginAccessControlProps",
    jsii_struct_bases=[],
    name_mapping={"origin_access_control_config": "originAccessControlConfig"},
)
class CfnOriginAccessControlProps:
    def __init__(
        self,
        *,
        origin_access_control_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnOriginAccessControl.OriginAccessControlConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnOriginAccessControl``.

        :param origin_access_control_config: The origin access control.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originaccesscontrol.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_origin_access_control_props = cloudfront.CfnOriginAccessControlProps(
                origin_access_control_config=cloudfront.CfnOriginAccessControl.OriginAccessControlConfigProperty(
                    name="name",
                    origin_access_control_origin_type="originAccessControlOriginType",
                    signing_behavior="signingBehavior",
                    signing_protocol="signingProtocol",
            
                    # the properties below are optional
                    description="description"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__de6e4bb7e3021f33175e34ed651d8c9ee267e7b433ea3e49cc32bdb5ee41ce09)
            check_type(argname="argument origin_access_control_config", value=origin_access_control_config, expected_type=type_hints["origin_access_control_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "origin_access_control_config": origin_access_control_config,
        }

    @builtins.property
    def origin_access_control_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnOriginAccessControl.OriginAccessControlConfigProperty]:
        '''The origin access control.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originaccesscontrol.html#cfn-cloudfront-originaccesscontrol-originaccesscontrolconfig
        '''
        result = self._values.get("origin_access_control_config")
        assert result is not None, "Required property 'origin_access_control_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnOriginAccessControl.OriginAccessControlConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnOriginAccessControlProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnOriginRequestPolicy(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnOriginRequestPolicy",
):
    '''A CloudFormation ``AWS::CloudFront::OriginRequestPolicy``.

    An origin request policy.

    When it’s attached to a cache behavior, the origin request policy determines the values that CloudFront includes in requests that it sends to the origin. Each request that CloudFront sends to the origin includes the following:

    - The request body and the URL path (without the domain name) from the viewer request.
    - The headers that CloudFront automatically includes in every origin request, including ``Host`` , ``User-Agent`` , and ``X-Amz-Cf-Id`` .
    - All HTTP headers, cookies, and URL query strings that are specified in the cache policy or the origin request policy. These can include items from the viewer request and, in the case of headers, additional ones that are added by CloudFront.

    CloudFront sends a request when it can’t find an object in its cache that matches the request. If you want to send values to the origin and also include them in the cache key, use ``CachePolicy`` .

    :cloudformationResource: AWS::CloudFront::OriginRequestPolicy
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originrequestpolicy.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_origin_request_policy = cloudfront.CfnOriginRequestPolicy(self, "MyCfnOriginRequestPolicy",
            origin_request_policy_config=cloudfront.CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty(
                cookies_config=cloudfront.CfnOriginRequestPolicy.CookiesConfigProperty(
                    cookie_behavior="cookieBehavior",
        
                    # the properties below are optional
                    cookies=["cookies"]
                ),
                headers_config=cloudfront.CfnOriginRequestPolicy.HeadersConfigProperty(
                    header_behavior="headerBehavior",
        
                    # the properties below are optional
                    headers=["headers"]
                ),
                name="name",
                query_strings_config=cloudfront.CfnOriginRequestPolicy.QueryStringsConfigProperty(
                    query_string_behavior="queryStringBehavior",
        
                    # the properties below are optional
                    query_strings=["queryStrings"]
                ),
        
                # the properties below are optional
                comment="comment"
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        origin_request_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::OriginRequestPolicy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param origin_request_policy_config: The origin request policy configuration.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__eb205e61fd04c0cd45946f26d2ab0693e9507e475c152dc7427c29bbb7da87f3)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnOriginRequestPolicyProps(
            origin_request_policy_config=origin_request_policy_config
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2e5f358d30a0145a276ce75e8bd40f1fa87b5698339eb857fdb17aced6f1fe2d)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__326f71b4fc4cc6e9e87e215b35f340ed47b54e5e6e881578aac54dbd93db9876)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The unique identifier for the origin request policy.

        For example: ``befd7079-9bbc-4ebf-8ade-498a3694176c`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrLastModifiedTime")
    def attr_last_modified_time(self) -> builtins.str:
        '''The date and time when the origin request policy was last modified.

        :cloudformationAttribute: LastModifiedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLastModifiedTime"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="originRequestPolicyConfig")
    def origin_request_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty"]:
        '''The origin request policy configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originrequestpolicy.html#cfn-cloudfront-originrequestpolicy-originrequestpolicyconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty"], jsii.get(self, "originRequestPolicyConfig"))

    @origin_request_policy_config.setter
    def origin_request_policy_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ae55632aee0e43f85b510f73cf8bdbda3e12b42526efd048ecd5bd9b46d1481d)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "originRequestPolicyConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnOriginRequestPolicy.CookiesConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"cookie_behavior": "cookieBehavior", "cookies": "cookies"},
    )
    class CookiesConfigProperty:
        def __init__(
            self,
            *,
            cookie_behavior: builtins.str,
            cookies: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that determines whether any cookies in viewer requests (and if so, which cookies) are included in requests that CloudFront sends to the origin.

            :param cookie_behavior: Determines whether cookies in viewer requests are included in requests that CloudFront sends to the origin. Valid values are:. - ``none`` – Cookies in viewer requests are not included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any cookies that are listed in a ``CachePolicy`` *are* included in origin requests. - ``whitelist`` – The cookies in viewer requests that are listed in the ``CookieNames`` type are included in requests that CloudFront sends to the origin. - ``all`` – All cookies in viewer requests are included in requests that CloudFront sends to the origin.
            :param cookies: Contains a list of cookie names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-cookiesconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                cookies_config_property = cloudfront.CfnOriginRequestPolicy.CookiesConfigProperty(
                    cookie_behavior="cookieBehavior",
                
                    # the properties below are optional
                    cookies=["cookies"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__6d994de74f66ca79a1419ccb1e3101b4be62813fc1b82e54246bbcebf89bebf4)
                check_type(argname="argument cookie_behavior", value=cookie_behavior, expected_type=type_hints["cookie_behavior"])
                check_type(argname="argument cookies", value=cookies, expected_type=type_hints["cookies"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cookie_behavior": cookie_behavior,
            }
            if cookies is not None:
                self._values["cookies"] = cookies

        @builtins.property
        def cookie_behavior(self) -> builtins.str:
            '''Determines whether cookies in viewer requests are included in requests that CloudFront sends to the origin. Valid values are:.

            - ``none`` – Cookies in viewer requests are not included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any cookies that are listed in a ``CachePolicy`` *are* included in origin requests.
            - ``whitelist`` – The cookies in viewer requests that are listed in the ``CookieNames`` type are included in requests that CloudFront sends to the origin.
            - ``all`` – All cookies in viewer requests are included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-cookiesconfig.html#cfn-cloudfront-originrequestpolicy-cookiesconfig-cookiebehavior
            '''
            result = self._values.get("cookie_behavior")
            assert result is not None, "Required property 'cookie_behavior' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def cookies(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Contains a list of cookie names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-cookiesconfig.html#cfn-cloudfront-originrequestpolicy-cookiesconfig-cookies
            '''
            result = self._values.get("cookies")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CookiesConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnOriginRequestPolicy.HeadersConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"header_behavior": "headerBehavior", "headers": "headers"},
    )
    class HeadersConfigProperty:
        def __init__(
            self,
            *,
            header_behavior: builtins.str,
            headers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.

            :param header_behavior: Determines whether any HTTP headers are included in requests that CloudFront sends to the origin. Valid values are:. - ``none`` – HTTP headers are not included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any headers that are listed in a ``CachePolicy`` *are* included in origin requests. - ``whitelist`` – The HTTP headers that are listed in the ``Headers`` type are included in requests that CloudFront sends to the origin. - ``allViewer`` – All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin. - ``allViewerAndWhitelistCloudFront`` – All HTTP headers in viewer requests and the additional CloudFront headers that are listed in the ``Headers`` type are included in requests that CloudFront sends to the origin. The additional headers are added by CloudFront.
            :param headers: Contains a list of HTTP header names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-headersconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                headers_config_property = cloudfront.CfnOriginRequestPolicy.HeadersConfigProperty(
                    header_behavior="headerBehavior",
                
                    # the properties below are optional
                    headers=["headers"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__79d358832f2cb5e87a5b89397bc2ca21ed387b9f87fa467ba1705b9b1606a9e2)
                check_type(argname="argument header_behavior", value=header_behavior, expected_type=type_hints["header_behavior"])
                check_type(argname="argument headers", value=headers, expected_type=type_hints["headers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "header_behavior": header_behavior,
            }
            if headers is not None:
                self._values["headers"] = headers

        @builtins.property
        def header_behavior(self) -> builtins.str:
            '''Determines whether any HTTP headers are included in requests that CloudFront sends to the origin. Valid values are:.

            - ``none`` – HTTP headers are not included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any headers that are listed in a ``CachePolicy`` *are* included in origin requests.
            - ``whitelist`` – The HTTP headers that are listed in the ``Headers`` type are included in requests that CloudFront sends to the origin.
            - ``allViewer`` – All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
            - ``allViewerAndWhitelistCloudFront`` – All HTTP headers in viewer requests and the additional CloudFront headers that are listed in the ``Headers`` type are included in requests that CloudFront sends to the origin. The additional headers are added by CloudFront.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-headersconfig.html#cfn-cloudfront-originrequestpolicy-headersconfig-headerbehavior
            '''
            result = self._values.get("header_behavior")
            assert result is not None, "Required property 'header_behavior' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def headers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Contains a list of HTTP header names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-headersconfig.html#cfn-cloudfront-originrequestpolicy-headersconfig-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HeadersConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cookies_config": "cookiesConfig",
            "headers_config": "headersConfig",
            "name": "name",
            "query_strings_config": "queryStringsConfig",
            "comment": "comment",
        },
    )
    class OriginRequestPolicyConfigProperty:
        def __init__(
            self,
            *,
            cookies_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnOriginRequestPolicy.CookiesConfigProperty", typing.Dict[builtins.str, typing.Any]]],
            headers_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnOriginRequestPolicy.HeadersConfigProperty", typing.Dict[builtins.str, typing.Any]]],
            name: builtins.str,
            query_strings_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnOriginRequestPolicy.QueryStringsConfigProperty", typing.Dict[builtins.str, typing.Any]]],
            comment: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An origin request policy configuration.

            This configuration determines the values that CloudFront includes in requests that it sends to the origin. Each request that CloudFront sends to the origin includes the following:

            - The request body and the URL path (without the domain name) from the viewer request.
            - The headers that CloudFront automatically includes in every origin request, including ``Host`` , ``User-Agent`` , and ``X-Amz-Cf-Id`` .
            - All HTTP headers, cookies, and URL query strings that are specified in the cache policy or the origin request policy. These can include items from the viewer request and, in the case of headers, additional ones that are added by CloudFront.

            CloudFront sends a request when it can’t find an object in its cache that matches the request. If you want to send values to the origin and also include them in the cache key, use ``CachePolicy`` .

            :param cookies_config: The cookies from viewer requests to include in origin requests.
            :param headers_config: The HTTP headers to include in origin requests. These can include headers from viewer requests and additional headers added by CloudFront.
            :param name: A unique name to identify the origin request policy.
            :param query_strings_config: The URL query strings from viewer requests to include in origin requests.
            :param comment: A comment to describe the origin request policy. The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-originrequestpolicyconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                origin_request_policy_config_property = cloudfront.CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty(
                    cookies_config=cloudfront.CfnOriginRequestPolicy.CookiesConfigProperty(
                        cookie_behavior="cookieBehavior",
                
                        # the properties below are optional
                        cookies=["cookies"]
                    ),
                    headers_config=cloudfront.CfnOriginRequestPolicy.HeadersConfigProperty(
                        header_behavior="headerBehavior",
                
                        # the properties below are optional
                        headers=["headers"]
                    ),
                    name="name",
                    query_strings_config=cloudfront.CfnOriginRequestPolicy.QueryStringsConfigProperty(
                        query_string_behavior="queryStringBehavior",
                
                        # the properties below are optional
                        query_strings=["queryStrings"]
                    ),
                
                    # the properties below are optional
                    comment="comment"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__69d8ca5b14e422f811ef260334fa8a115ed3f2a9591bba126369d2409c337ee2)
                check_type(argname="argument cookies_config", value=cookies_config, expected_type=type_hints["cookies_config"])
                check_type(argname="argument headers_config", value=headers_config, expected_type=type_hints["headers_config"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument query_strings_config", value=query_strings_config, expected_type=type_hints["query_strings_config"])
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "cookies_config": cookies_config,
                "headers_config": headers_config,
                "name": name,
                "query_strings_config": query_strings_config,
            }
            if comment is not None:
                self._values["comment"] = comment

        @builtins.property
        def cookies_config(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.CookiesConfigProperty"]:
            '''The cookies from viewer requests to include in origin requests.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-originrequestpolicyconfig.html#cfn-cloudfront-originrequestpolicy-originrequestpolicyconfig-cookiesconfig
            '''
            result = self._values.get("cookies_config")
            assert result is not None, "Required property 'cookies_config' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.CookiesConfigProperty"], result)

        @builtins.property
        def headers_config(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.HeadersConfigProperty"]:
            '''The HTTP headers to include in origin requests.

            These can include headers from viewer requests and additional headers added by CloudFront.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-originrequestpolicyconfig.html#cfn-cloudfront-originrequestpolicy-originrequestpolicyconfig-headersconfig
            '''
            result = self._values.get("headers_config")
            assert result is not None, "Required property 'headers_config' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.HeadersConfigProperty"], result)

        @builtins.property
        def name(self) -> builtins.str:
            '''A unique name to identify the origin request policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-originrequestpolicyconfig.html#cfn-cloudfront-originrequestpolicy-originrequestpolicyconfig-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def query_strings_config(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.QueryStringsConfigProperty"]:
            '''The URL query strings from viewer requests to include in origin requests.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-originrequestpolicyconfig.html#cfn-cloudfront-originrequestpolicy-originrequestpolicyconfig-querystringsconfig
            '''
            result = self._values.get("query_strings_config")
            assert result is not None, "Required property 'query_strings_config' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnOriginRequestPolicy.QueryStringsConfigProperty"], result)

        @builtins.property
        def comment(self) -> typing.Optional[builtins.str]:
            '''A comment to describe the origin request policy.

            The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-originrequestpolicyconfig.html#cfn-cloudfront-originrequestpolicy-originrequestpolicyconfig-comment
            '''
            result = self._values.get("comment")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OriginRequestPolicyConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnOriginRequestPolicy.QueryStringsConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "query_string_behavior": "queryStringBehavior",
            "query_strings": "queryStrings",
        },
    )
    class QueryStringsConfigProperty:
        def __init__(
            self,
            *,
            query_string_behavior: builtins.str,
            query_strings: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that determines whether any URL query strings in viewer requests (and if so, which query strings) are included in requests that CloudFront sends to the origin.

            :param query_string_behavior: Determines whether any URL query strings in viewer requests are included in requests that CloudFront sends to the origin. Valid values are: - ``none`` – Query strings in viewer requests are not included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any query strings that are listed in a ``CachePolicy`` *are* included in origin requests. - ``whitelist`` – The query strings in viewer requests that are listed in the ``QueryStringNames`` type are included in requests that CloudFront sends to the origin. - ``all`` – All query strings in viewer requests are included in requests that CloudFront sends to the origin.
            :param query_strings: Contains a list of query string names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-querystringsconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                query_strings_config_property = cloudfront.CfnOriginRequestPolicy.QueryStringsConfigProperty(
                    query_string_behavior="queryStringBehavior",
                
                    # the properties below are optional
                    query_strings=["queryStrings"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__7d120383b3c57305feaa3ca5a18a4458f3e4536845c498e40cd7594aee26f542)
                check_type(argname="argument query_string_behavior", value=query_string_behavior, expected_type=type_hints["query_string_behavior"])
                check_type(argname="argument query_strings", value=query_strings, expected_type=type_hints["query_strings"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "query_string_behavior": query_string_behavior,
            }
            if query_strings is not None:
                self._values["query_strings"] = query_strings

        @builtins.property
        def query_string_behavior(self) -> builtins.str:
            '''Determines whether any URL query strings in viewer requests are included in requests that CloudFront sends to the origin.

            Valid values are:

            - ``none`` – Query strings in viewer requests are not included in requests that CloudFront sends to the origin. Even when this field is set to ``none`` , any query strings that are listed in a ``CachePolicy`` *are* included in origin requests.
            - ``whitelist`` – The query strings in viewer requests that are listed in the ``QueryStringNames`` type are included in requests that CloudFront sends to the origin.
            - ``all`` – All query strings in viewer requests are included in requests that CloudFront sends to the origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-querystringsconfig.html#cfn-cloudfront-originrequestpolicy-querystringsconfig-querystringbehavior
            '''
            result = self._values.get("query_string_behavior")
            assert result is not None, "Required property 'query_string_behavior' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def query_strings(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Contains a list of query string names.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-originrequestpolicy-querystringsconfig.html#cfn-cloudfront-originrequestpolicy-querystringsconfig-querystrings
            '''
            result = self._values.get("query_strings")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryStringsConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnOriginRequestPolicyProps",
    jsii_struct_bases=[],
    name_mapping={"origin_request_policy_config": "originRequestPolicyConfig"},
)
class CfnOriginRequestPolicyProps:
    def __init__(
        self,
        *,
        origin_request_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnOriginRequestPolicy``.

        :param origin_request_policy_config: The origin request policy configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originrequestpolicy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_origin_request_policy_props = cloudfront.CfnOriginRequestPolicyProps(
                origin_request_policy_config=cloudfront.CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty(
                    cookies_config=cloudfront.CfnOriginRequestPolicy.CookiesConfigProperty(
                        cookie_behavior="cookieBehavior",
            
                        # the properties below are optional
                        cookies=["cookies"]
                    ),
                    headers_config=cloudfront.CfnOriginRequestPolicy.HeadersConfigProperty(
                        header_behavior="headerBehavior",
            
                        # the properties below are optional
                        headers=["headers"]
                    ),
                    name="name",
                    query_strings_config=cloudfront.CfnOriginRequestPolicy.QueryStringsConfigProperty(
                        query_string_behavior="queryStringBehavior",
            
                        # the properties below are optional
                        query_strings=["queryStrings"]
                    ),
            
                    # the properties below are optional
                    comment="comment"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__23062a3ff69491c66cf59be7b29908b59ee8f0bd626a3f2aecb9a9a5d5a961f7)
            check_type(argname="argument origin_request_policy_config", value=origin_request_policy_config, expected_type=type_hints["origin_request_policy_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "origin_request_policy_config": origin_request_policy_config,
        }

    @builtins.property
    def origin_request_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty]:
        '''The origin request policy configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-originrequestpolicy.html#cfn-cloudfront-originrequestpolicy-originrequestpolicyconfig
        '''
        result = self._values.get("origin_request_policy_config")
        assert result is not None, "Required property 'origin_request_policy_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnOriginRequestPolicy.OriginRequestPolicyConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnOriginRequestPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnPublicKey(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnPublicKey",
):
    '''A CloudFormation ``AWS::CloudFront::PublicKey``.

    A public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .

    :cloudformationResource: AWS::CloudFront::PublicKey
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-publickey.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_public_key = cloudfront.CfnPublicKey(self, "MyCfnPublicKey",
            public_key_config=cloudfront.CfnPublicKey.PublicKeyConfigProperty(
                caller_reference="callerReference",
                encoded_key="encodedKey",
                name="name",
        
                # the properties below are optional
                comment="comment"
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        public_key_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnPublicKey.PublicKeyConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::PublicKey``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param public_key_config: Configuration information about a public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c3f6e0b38974fd39081337582714ca933f8fe2e707be6f1964fe5940600c246e)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnPublicKeyProps(public_key_config=public_key_config)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__afc3f293746265e93b763b737eb3246840b0f2ac37421896a0413c32856b3a87)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__955fe1a25daf2b9392dde1e31de01102f004dfec00e9b2be4622989c9c7d8e81)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrCreatedTime")
    def attr_created_time(self) -> builtins.str:
        '''The date and time when the public key was uploaded.

        :cloudformationAttribute: CreatedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedTime"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The identifier of the public key.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="publicKeyConfig")
    def public_key_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnPublicKey.PublicKeyConfigProperty"]:
        '''Configuration information about a public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-publickey.html#cfn-cloudfront-publickey-publickeyconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnPublicKey.PublicKeyConfigProperty"], jsii.get(self, "publicKeyConfig"))

    @public_key_config.setter
    def public_key_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnPublicKey.PublicKeyConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__02bcc14f76f02be4f61437cf4547437ad61bb992ea5e349d659f0f2c107e504b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "publicKeyConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnPublicKey.PublicKeyConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "caller_reference": "callerReference",
            "encoded_key": "encodedKey",
            "name": "name",
            "comment": "comment",
        },
    )
    class PublicKeyConfigProperty:
        def __init__(
            self,
            *,
            caller_reference: builtins.str,
            encoded_key: builtins.str,
            name: builtins.str,
            comment: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Configuration information about a public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .

            :param caller_reference: A string included in the request to help make sure that the request can’t be replayed.
            :param encoded_key: The public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .
            :param name: A name to help identify the public key.
            :param comment: A comment to describe the public key. The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-publickey-publickeyconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                public_key_config_property = cloudfront.CfnPublicKey.PublicKeyConfigProperty(
                    caller_reference="callerReference",
                    encoded_key="encodedKey",
                    name="name",
                
                    # the properties below are optional
                    comment="comment"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__c5ab8276c62665475df0c52af9cac1e913a2fb7255e41584865fe59c98f07243)
                check_type(argname="argument caller_reference", value=caller_reference, expected_type=type_hints["caller_reference"])
                check_type(argname="argument encoded_key", value=encoded_key, expected_type=type_hints["encoded_key"])
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "caller_reference": caller_reference,
                "encoded_key": encoded_key,
                "name": name,
            }
            if comment is not None:
                self._values["comment"] = comment

        @builtins.property
        def caller_reference(self) -> builtins.str:
            '''A string included in the request to help make sure that the request can’t be replayed.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-publickey-publickeyconfig.html#cfn-cloudfront-publickey-publickeyconfig-callerreference
            '''
            result = self._values.get("caller_reference")
            assert result is not None, "Required property 'caller_reference' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def encoded_key(self) -> builtins.str:
            '''The public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-publickey-publickeyconfig.html#cfn-cloudfront-publickey-publickeyconfig-encodedkey
            '''
            result = self._values.get("encoded_key")
            assert result is not None, "Required property 'encoded_key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''A name to help identify the public key.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-publickey-publickeyconfig.html#cfn-cloudfront-publickey-publickeyconfig-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def comment(self) -> typing.Optional[builtins.str]:
            '''A comment to describe the public key.

            The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-publickey-publickeyconfig.html#cfn-cloudfront-publickey-publickeyconfig-comment
            '''
            result = self._values.get("comment")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PublicKeyConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnPublicKeyProps",
    jsii_struct_bases=[],
    name_mapping={"public_key_config": "publicKeyConfig"},
)
class CfnPublicKeyProps:
    def __init__(
        self,
        *,
        public_key_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnPublicKey.PublicKeyConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnPublicKey``.

        :param public_key_config: Configuration information about a public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-publickey.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_public_key_props = cloudfront.CfnPublicKeyProps(
                public_key_config=cloudfront.CfnPublicKey.PublicKeyConfigProperty(
                    caller_reference="callerReference",
                    encoded_key="encodedKey",
                    name="name",
            
                    # the properties below are optional
                    comment="comment"
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e318c20e32381b196cf5088aa73b90ca17b547f3eb70092f3989dbd452da4772)
            check_type(argname="argument public_key_config", value=public_key_config, expected_type=type_hints["public_key_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "public_key_config": public_key_config,
        }

    @builtins.property
    def public_key_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnPublicKey.PublicKeyConfigProperty]:
        '''Configuration information about a public key that you can use with `signed URLs and signed cookies <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ , or with `field-level encryption <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-publickey.html#cfn-cloudfront-publickey-publickeyconfig
        '''
        result = self._values.get("public_key_config")
        assert result is not None, "Required property 'public_key_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnPublicKey.PublicKeyConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPublicKeyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnRealtimeLogConfig(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnRealtimeLogConfig",
):
    '''A CloudFormation ``AWS::CloudFront::RealtimeLogConfig``.

    A real-time log configuration.

    :cloudformationResource: AWS::CloudFront::RealtimeLogConfig
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_realtime_log_config = cloudfront.CfnRealtimeLogConfig(self, "MyCfnRealtimeLogConfig",
            end_points=[cloudfront.CfnRealtimeLogConfig.EndPointProperty(
                kinesis_stream_config=cloudfront.CfnRealtimeLogConfig.KinesisStreamConfigProperty(
                    role_arn="roleArn",
                    stream_arn="streamArn"
                ),
                stream_type="streamType"
            )],
            fields=["fields"],
            name="name",
            sampling_rate=123
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        end_points: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnRealtimeLogConfig.EndPointProperty", typing.Dict[builtins.str, typing.Any]]]]],
        fields: typing.Sequence[builtins.str],
        name: builtins.str,
        sampling_rate: jsii.Number,
    ) -> None:
        '''Create a new ``AWS::CloudFront::RealtimeLogConfig``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param end_points: Contains information about the Amazon Kinesis data stream where you are sending real-time log data for this real-time log configuration.
        :param fields: A list of fields that are included in each real-time log record. In an API response, the fields are provided in the same order in which they are sent to the Amazon Kinesis data stream. For more information about fields, see `Real-time log configuration fields <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html#understand-real-time-log-config-fields>`_ in the *Amazon CloudFront Developer Guide* .
        :param name: The unique name of this real-time log configuration.
        :param sampling_rate: The sampling rate for this real-time log configuration. The sampling rate determines the percentage of viewer requests that are represented in the real-time log data. The sampling rate is an integer between 1 and 100, inclusive.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ecc03e482e41d939eb69e9410fd335ea38632fc7aa426a2aab183372793e07d5)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnRealtimeLogConfigProps(
            end_points=end_points,
            fields=fields,
            name=name,
            sampling_rate=sampling_rate,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__1a2469ded56bdd77deb5cd2be5324dd5d8b67a896750c4d67546ccf5b96762ec)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__16c002d04874e9dacccc97bce8c90fcd4c9ccd0028d8ca3212cc2cf5b377c43e)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the real-time log configuration.

        For example: ``arn:aws:cloudfront::111122223333:realtime-log-config/ExampleNameForRealtimeLogConfig`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="endPoints")
    def end_points(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnRealtimeLogConfig.EndPointProperty"]]]:
        '''Contains information about the Amazon Kinesis data stream where you are sending real-time log data for this real-time log configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-endpoints
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnRealtimeLogConfig.EndPointProperty"]]], jsii.get(self, "endPoints"))

    @end_points.setter
    def end_points(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnRealtimeLogConfig.EndPointProperty"]]],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__0a67a8bb65a48f5a88303482ef50258096719563b5456d041302c70e8092590f)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "endPoints", value)

    @builtins.property
    @jsii.member(jsii_name="fields")
    def fields(self) -> typing.List[builtins.str]:
        '''A list of fields that are included in each real-time log record.

        In an API response, the fields are provided in the same order in which they are sent to the Amazon Kinesis data stream.

        For more information about fields, see `Real-time log configuration fields <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html#understand-real-time-log-config-fields>`_ in the *Amazon CloudFront Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-fields
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "fields"))

    @fields.setter
    def fields(self, value: typing.List[builtins.str]) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__2789c5be557b19436df26d7f7c402567b884cd4de3973b8e0fb6bb3368a61002)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "fields", value)

    @builtins.property
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        '''The unique name of this real-time log configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-name
        '''
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @name.setter
    def name(self, value: builtins.str) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6ee531a2a7ec0372ee4473ba908a0f507c6a5a7461140351b757b6a37ccae77c)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "name", value)

    @builtins.property
    @jsii.member(jsii_name="samplingRate")
    def sampling_rate(self) -> jsii.Number:
        '''The sampling rate for this real-time log configuration.

        The sampling rate determines the percentage of viewer requests that are represented in the real-time log data. The sampling rate is an integer between 1 and 100, inclusive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-samplingrate
        '''
        return typing.cast(jsii.Number, jsii.get(self, "samplingRate"))

    @sampling_rate.setter
    def sampling_rate(self, value: jsii.Number) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f3d63cf3fd194f96135d7381839a06a147cdbc7b0e87e641e467878a1fd8fbb4)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "samplingRate", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnRealtimeLogConfig.EndPointProperty",
        jsii_struct_bases=[],
        name_mapping={
            "kinesis_stream_config": "kinesisStreamConfig",
            "stream_type": "streamType",
        },
    )
    class EndPointProperty:
        def __init__(
            self,
            *,
            kinesis_stream_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnRealtimeLogConfig.KinesisStreamConfigProperty", typing.Dict[builtins.str, typing.Any]]],
            stream_type: builtins.str,
        ) -> None:
            '''Contains information about the Amazon Kinesis data stream where you are sending real-time log data in a real-time log configuration.

            :param kinesis_stream_config: Contains information about the Amazon Kinesis data stream where you are sending real-time log data.
            :param stream_type: The type of data stream where you are sending real-time log data. The only valid value is ``Kinesis`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-realtimelogconfig-endpoint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                end_point_property = cloudfront.CfnRealtimeLogConfig.EndPointProperty(
                    kinesis_stream_config=cloudfront.CfnRealtimeLogConfig.KinesisStreamConfigProperty(
                        role_arn="roleArn",
                        stream_arn="streamArn"
                    ),
                    stream_type="streamType"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__afc4a1ac14633ee2fe0f3b86c21e4ca1a4c5e1318fd5161148b91a2b08d20ee7)
                check_type(argname="argument kinesis_stream_config", value=kinesis_stream_config, expected_type=type_hints["kinesis_stream_config"])
                check_type(argname="argument stream_type", value=stream_type, expected_type=type_hints["stream_type"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "kinesis_stream_config": kinesis_stream_config,
                "stream_type": stream_type,
            }

        @builtins.property
        def kinesis_stream_config(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnRealtimeLogConfig.KinesisStreamConfigProperty"]:
            '''Contains information about the Amazon Kinesis data stream where you are sending real-time log data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-realtimelogconfig-endpoint.html#cfn-cloudfront-realtimelogconfig-endpoint-kinesisstreamconfig
            '''
            result = self._values.get("kinesis_stream_config")
            assert result is not None, "Required property 'kinesis_stream_config' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnRealtimeLogConfig.KinesisStreamConfigProperty"], result)

        @builtins.property
        def stream_type(self) -> builtins.str:
            '''The type of data stream where you are sending real-time log data.

            The only valid value is ``Kinesis`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-realtimelogconfig-endpoint.html#cfn-cloudfront-realtimelogconfig-endpoint-streamtype
            '''
            result = self._values.get("stream_type")
            assert result is not None, "Required property 'stream_type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EndPointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnRealtimeLogConfig.KinesisStreamConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"role_arn": "roleArn", "stream_arn": "streamArn"},
    )
    class KinesisStreamConfigProperty:
        def __init__(self, *, role_arn: builtins.str, stream_arn: builtins.str) -> None:
            '''Contains information about the Amazon Kinesis data stream where you are sending real-time log data.

            :param role_arn: The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFront can use to send real-time log data to your Kinesis data stream. For more information the IAM role, see `Real-time log configuration IAM role <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html#understand-real-time-log-config-iam-role>`_ in the *Amazon CloudFront Developer Guide* .
            :param stream_arn: The Amazon Resource Name (ARN) of the Kinesis data stream where you are sending real-time log data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-realtimelogconfig-kinesisstreamconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                kinesis_stream_config_property = cloudfront.CfnRealtimeLogConfig.KinesisStreamConfigProperty(
                    role_arn="roleArn",
                    stream_arn="streamArn"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__65caf23c73c45a9bde9f0cf4f5fb694cdff1c46bcd80e4919e7a683da9ef8eda)
                check_type(argname="argument role_arn", value=role_arn, expected_type=type_hints["role_arn"])
                check_type(argname="argument stream_arn", value=stream_arn, expected_type=type_hints["stream_arn"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "role_arn": role_arn,
                "stream_arn": stream_arn,
            }

        @builtins.property
        def role_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFront can use to send real-time log data to your Kinesis data stream.

            For more information the IAM role, see `Real-time log configuration IAM role <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html#understand-real-time-log-config-iam-role>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-realtimelogconfig-kinesisstreamconfig.html#cfn-cloudfront-realtimelogconfig-kinesisstreamconfig-rolearn
            '''
            result = self._values.get("role_arn")
            assert result is not None, "Required property 'role_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def stream_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) of the Kinesis data stream where you are sending real-time log data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-realtimelogconfig-kinesisstreamconfig.html#cfn-cloudfront-realtimelogconfig-kinesisstreamconfig-streamarn
            '''
            result = self._values.get("stream_arn")
            assert result is not None, "Required property 'stream_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KinesisStreamConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnRealtimeLogConfigProps",
    jsii_struct_bases=[],
    name_mapping={
        "end_points": "endPoints",
        "fields": "fields",
        "name": "name",
        "sampling_rate": "samplingRate",
    },
)
class CfnRealtimeLogConfigProps:
    def __init__(
        self,
        *,
        end_points: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnRealtimeLogConfig.EndPointProperty, typing.Dict[builtins.str, typing.Any]]]]],
        fields: typing.Sequence[builtins.str],
        name: builtins.str,
        sampling_rate: jsii.Number,
    ) -> None:
        '''Properties for defining a ``CfnRealtimeLogConfig``.

        :param end_points: Contains information about the Amazon Kinesis data stream where you are sending real-time log data for this real-time log configuration.
        :param fields: A list of fields that are included in each real-time log record. In an API response, the fields are provided in the same order in which they are sent to the Amazon Kinesis data stream. For more information about fields, see `Real-time log configuration fields <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html#understand-real-time-log-config-fields>`_ in the *Amazon CloudFront Developer Guide* .
        :param name: The unique name of this real-time log configuration.
        :param sampling_rate: The sampling rate for this real-time log configuration. The sampling rate determines the percentage of viewer requests that are represented in the real-time log data. The sampling rate is an integer between 1 and 100, inclusive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_realtime_log_config_props = cloudfront.CfnRealtimeLogConfigProps(
                end_points=[cloudfront.CfnRealtimeLogConfig.EndPointProperty(
                    kinesis_stream_config=cloudfront.CfnRealtimeLogConfig.KinesisStreamConfigProperty(
                        role_arn="roleArn",
                        stream_arn="streamArn"
                    ),
                    stream_type="streamType"
                )],
                fields=["fields"],
                name="name",
                sampling_rate=123
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a836bff0640926362431fc04a8d451c409670e4107870209eaad4895bbe37ec3)
            check_type(argname="argument end_points", value=end_points, expected_type=type_hints["end_points"])
            check_type(argname="argument fields", value=fields, expected_type=type_hints["fields"])
            check_type(argname="argument name", value=name, expected_type=type_hints["name"])
            check_type(argname="argument sampling_rate", value=sampling_rate, expected_type=type_hints["sampling_rate"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "end_points": end_points,
            "fields": fields,
            "name": name,
            "sampling_rate": sampling_rate,
        }

    @builtins.property
    def end_points(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnRealtimeLogConfig.EndPointProperty]]]:
        '''Contains information about the Amazon Kinesis data stream where you are sending real-time log data for this real-time log configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-endpoints
        '''
        result = self._values.get("end_points")
        assert result is not None, "Required property 'end_points' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnRealtimeLogConfig.EndPointProperty]]], result)

    @builtins.property
    def fields(self) -> typing.List[builtins.str]:
        '''A list of fields that are included in each real-time log record.

        In an API response, the fields are provided in the same order in which they are sent to the Amazon Kinesis data stream.

        For more information about fields, see `Real-time log configuration fields <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html#understand-real-time-log-config-fields>`_ in the *Amazon CloudFront Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-fields
        '''
        result = self._values.get("fields")
        assert result is not None, "Required property 'fields' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def name(self) -> builtins.str:
        '''The unique name of this real-time log configuration.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-name
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def sampling_rate(self) -> jsii.Number:
        '''The sampling rate for this real-time log configuration.

        The sampling rate determines the percentage of viewer requests that are represented in the real-time log data. The sampling rate is an integer between 1 and 100, inclusive.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-realtimelogconfig.html#cfn-cloudfront-realtimelogconfig-samplingrate
        '''
        result = self._values.get("sampling_rate")
        assert result is not None, "Required property 'sampling_rate' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRealtimeLogConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnResponseHeadersPolicy(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy",
):
    '''A CloudFormation ``AWS::CloudFront::ResponseHeadersPolicy``.

    A response headers policy.

    A response headers policy contains information about a set of HTTP response headers and their values.

    After you create a response headers policy, you can use its ID to attach it to one or more cache behaviors in a CloudFront distribution. When it’s attached to a cache behavior, CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match the cache behavior.

    For more information, see `Adding HTTP headers to CloudFront responses <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/adding-response-headers.html>`_ in the *Amazon CloudFront Developer Guide* .

    :cloudformationResource: AWS::CloudFront::ResponseHeadersPolicy
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-responseheaderspolicy.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_response_headers_policy = cloudfront.CfnResponseHeadersPolicy(self, "MyCfnResponseHeadersPolicy",
            response_headers_policy_config=cloudfront.CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty(
                name="name",
        
                # the properties below are optional
                comment="comment",
                cors_config=cloudfront.CfnResponseHeadersPolicy.CorsConfigProperty(
                    access_control_allow_credentials=False,
                    access_control_allow_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty(
                        items=["items"]
                    ),
                    access_control_allow_methods=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty(
                        items=["items"]
                    ),
                    access_control_allow_origins=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty(
                        items=["items"]
                    ),
                    origin_override=False,
        
                    # the properties below are optional
                    access_control_expose_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty(
                        items=["items"]
                    ),
                    access_control_max_age_sec=123
                ),
                custom_headers_config=cloudfront.CfnResponseHeadersPolicy.CustomHeadersConfigProperty(
                    items=[cloudfront.CfnResponseHeadersPolicy.CustomHeaderProperty(
                        header="header",
                        override=False,
                        value="value"
                    )]
                ),
                remove_headers_config=cloudfront.CfnResponseHeadersPolicy.RemoveHeadersConfigProperty(
                    items=[cloudfront.CfnResponseHeadersPolicy.RemoveHeaderProperty(
                        header="header"
                    )]
                ),
                security_headers_config=cloudfront.CfnResponseHeadersPolicy.SecurityHeadersConfigProperty(
                    content_security_policy=cloudfront.CfnResponseHeadersPolicy.ContentSecurityPolicyProperty(
                        content_security_policy="contentSecurityPolicy",
                        override=False
                    ),
                    content_type_options=cloudfront.CfnResponseHeadersPolicy.ContentTypeOptionsProperty(
                        override=False
                    ),
                    frame_options=cloudfront.CfnResponseHeadersPolicy.FrameOptionsProperty(
                        frame_option="frameOption",
                        override=False
                    ),
                    referrer_policy=cloudfront.CfnResponseHeadersPolicy.ReferrerPolicyProperty(
                        override=False,
                        referrer_policy="referrerPolicy"
                    ),
                    strict_transport_security=cloudfront.CfnResponseHeadersPolicy.StrictTransportSecurityProperty(
                        access_control_max_age_sec=123,
                        override=False,
        
                        # the properties below are optional
                        include_subdomains=False,
                        preload=False
                    ),
                    xss_protection=cloudfront.CfnResponseHeadersPolicy.XSSProtectionProperty(
                        override=False,
                        protection=False,
        
                        # the properties below are optional
                        mode_block=False,
                        report_uri="reportUri"
                    )
                ),
                server_timing_headers_config=cloudfront.CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty(
                    enabled=False,
        
                    # the properties below are optional
                    sampling_rate=123
                )
            )
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        response_headers_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty", typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::ResponseHeadersPolicy``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param response_headers_policy_config: A response headers policy configuration. A response headers policy contains information about a set of HTTP response headers and their values. CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache behavior that’s associated with the policy.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__95407fadc47ed16a997343af92739fb0cd9270051cbf899d857ec982095e8bd1)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnResponseHeadersPolicyProps(
            response_headers_policy_config=response_headers_policy_config
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__42ba7ae037107a1b7d56bb8454343b8c4a04e98a9794a4db65f3b424ecd89daa)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__81869065fb6045644801f496b8411bdf7847fca2eb6ba4a7b771aafd697a0309)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The unique identifier for the cache policy.

        For example: ``57f99797-3b20-4e1b-a728-27972a74082a`` .

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property
    @jsii.member(jsii_name="attrLastModifiedTime")
    def attr_last_modified_time(self) -> builtins.str:
        '''The date and time when the response headers policy was last modified.

        :cloudformationAttribute: LastModifiedTime
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLastModifiedTime"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="responseHeadersPolicyConfig")
    def response_headers_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty"]:
        '''A response headers policy configuration.

        A response headers policy contains information about a set of HTTP response headers and their values. CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache behavior that’s associated with the policy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-responseheaderspolicy.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty"], jsii.get(self, "responseHeadersPolicyConfig"))

    @response_headers_policy_config.setter
    def response_headers_policy_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6c12b287f2ff7d5a18a9ee292c4fbe19cc3dc08ddcab9f4f23c756c3b5e2f40b)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "responseHeadersPolicyConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items"},
    )
    class AccessControlAllowHeadersProperty:
        def __init__(self, *, items: typing.Sequence[builtins.str]) -> None:
            '''A list of HTTP header names that CloudFront includes as values for the ``Access-Control-Allow-Headers`` HTTP response header.

            For more information about the ``Access-Control-Allow-Headers`` HTTP response header, see `Access-Control-Allow-Headers <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers>`_ in the MDN Web Docs.

            :param items: The list of HTTP header names. You can specify ``*`` to allow all headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolallowheaders.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                access_control_allow_headers_property = cloudfront.CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty(
                    items=["items"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__94ba0ab1fa09a9eefcb264892adfad272d91e6ffaf920db8fefa8662f4844d3b)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
            }

        @builtins.property
        def items(self) -> typing.List[builtins.str]:
            '''The list of HTTP header names.

            You can specify ``*`` to allow all headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolallowheaders.html#cfn-cloudfront-responseheaderspolicy-accesscontrolallowheaders-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessControlAllowHeadersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items"},
    )
    class AccessControlAllowMethodsProperty:
        def __init__(self, *, items: typing.Sequence[builtins.str]) -> None:
            '''A list of HTTP methods that CloudFront includes as values for the ``Access-Control-Allow-Methods`` HTTP response header.

            For more information about the ``Access-Control-Allow-Methods`` HTTP response header, see `Access-Control-Allow-Methods <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods>`_ in the MDN Web Docs.

            :param items: The list of HTTP methods. Valid values are:. - ``GET`` - ``DELETE`` - ``HEAD`` - ``OPTIONS`` - ``PATCH`` - ``POST`` - ``PUT`` - ``ALL`` ``ALL`` is a special value that includes all of the listed HTTP methods.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolallowmethods.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                access_control_allow_methods_property = cloudfront.CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty(
                    items=["items"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__0e1d4a9bdc04088bdfaf148ee988ac68f00e8d694563356ac3d544243db959d0)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
            }

        @builtins.property
        def items(self) -> typing.List[builtins.str]:
            '''The list of HTTP methods. Valid values are:.

            - ``GET``
            - ``DELETE``
            - ``HEAD``
            - ``OPTIONS``
            - ``PATCH``
            - ``POST``
            - ``PUT``
            - ``ALL``

            ``ALL`` is a special value that includes all of the listed HTTP methods.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolallowmethods.html#cfn-cloudfront-responseheaderspolicy-accesscontrolallowmethods-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessControlAllowMethodsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items"},
    )
    class AccessControlAllowOriginsProperty:
        def __init__(self, *, items: typing.Sequence[builtins.str]) -> None:
            '''A list of origins (domain names) that CloudFront can use as the value for the ``Access-Control-Allow-Origin`` HTTP response header.

            For more information about the ``Access-Control-Allow-Origin`` HTTP response header, see `Access-Control-Allow-Origin <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin>`_ in the MDN Web Docs.

            :param items: The list of origins (domain names). You can specify ``*`` to allow all origins.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolalloworigins.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                access_control_allow_origins_property = cloudfront.CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty(
                    items=["items"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a99d95bdbf9215dd52ddfe8c52e470b7ab148f83d6726aa5dc108f387ae6220c)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
            }

        @builtins.property
        def items(self) -> typing.List[builtins.str]:
            '''The list of origins (domain names).

            You can specify ``*`` to allow all origins.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolalloworigins.html#cfn-cloudfront-responseheaderspolicy-accesscontrolalloworigins-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessControlAllowOriginsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items"},
    )
    class AccessControlExposeHeadersProperty:
        def __init__(self, *, items: typing.Sequence[builtins.str]) -> None:
            '''A list of HTTP headers that CloudFront includes as values for the ``Access-Control-Expose-Headers`` HTTP response header.

            For more information about the ``Access-Control-Expose-Headers`` HTTP response header, see `Access-Control-Expose-Headers <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers>`_ in the MDN Web Docs.

            :param items: The list of HTTP headers. You can specify ``*`` to expose all headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolexposeheaders.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                access_control_expose_headers_property = cloudfront.CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty(
                    items=["items"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8cb99fca1661a1208c0074e0401e3ba73987f535463659250331b622da31038a)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
            }

        @builtins.property
        def items(self) -> typing.List[builtins.str]:
            '''The list of HTTP headers.

            You can specify ``*`` to expose all headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-accesscontrolexposeheaders.html#cfn-cloudfront-responseheaderspolicy-accesscontrolexposeheaders-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessControlExposeHeadersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.ContentSecurityPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "content_security_policy": "contentSecurityPolicy",
            "override": "override",
        },
    )
    class ContentSecurityPolicyProperty:
        def __init__(
            self,
            *,
            content_security_policy: builtins.str,
            override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
        ) -> None:
            '''The policy directives and their values that CloudFront includes as values for the ``Content-Security-Policy`` HTTP response header.

            For more information about the ``Content-Security-Policy`` HTTP response header, see `Content-Security-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy>`_ in the MDN Web Docs.

            :param content_security_policy: The policy directives and their values that CloudFront includes as values for the ``Content-Security-Policy`` HTTP response header.
            :param override: A Boolean that determines whether CloudFront overrides the ``Content-Security-Policy`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-contentsecuritypolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                content_security_policy_property = cloudfront.CfnResponseHeadersPolicy.ContentSecurityPolicyProperty(
                    content_security_policy="contentSecurityPolicy",
                    override=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f595bb24b7ed946a4130e1a17089ab34e5eaea1ccfd346a9a87a85c10760b3c4)
                check_type(argname="argument content_security_policy", value=content_security_policy, expected_type=type_hints["content_security_policy"])
                check_type(argname="argument override", value=override, expected_type=type_hints["override"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "content_security_policy": content_security_policy,
                "override": override,
            }

        @builtins.property
        def content_security_policy(self) -> builtins.str:
            '''The policy directives and their values that CloudFront includes as values for the ``Content-Security-Policy`` HTTP response header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-contentsecuritypolicy.html#cfn-cloudfront-responseheaderspolicy-contentsecuritypolicy-contentsecuritypolicy
            '''
            result = self._values.get("content_security_policy")
            assert result is not None, "Required property 'content_security_policy' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides the ``Content-Security-Policy`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-contentsecuritypolicy.html#cfn-cloudfront-responseheaderspolicy-contentsecuritypolicy-override
            '''
            result = self._values.get("override")
            assert result is not None, "Required property 'override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContentSecurityPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.ContentTypeOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"override": "override"},
    )
    class ContentTypeOptionsProperty:
        def __init__(
            self,
            *,
            override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
        ) -> None:
            '''Determines whether CloudFront includes the ``X-Content-Type-Options`` HTTP response header with its value set to ``nosniff`` .

            For more information about the ``X-Content-Type-Options`` HTTP response header, see `X-Content-Type-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options>`_ in the MDN Web Docs.

            :param override: A Boolean that determines whether CloudFront overrides the ``X-Content-Type-Options`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-contenttypeoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                content_type_options_property = cloudfront.CfnResponseHeadersPolicy.ContentTypeOptionsProperty(
                    override=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__391545a270cc2ebed4b991db3fc163d8be9bbb2bb05f9bc55c631c786a5f51a3)
                check_type(argname="argument override", value=override, expected_type=type_hints["override"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "override": override,
            }

        @builtins.property
        def override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides the ``X-Content-Type-Options`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-contenttypeoptions.html#cfn-cloudfront-responseheaderspolicy-contenttypeoptions-override
            '''
            result = self._values.get("override")
            assert result is not None, "Required property 'override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContentTypeOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.CorsConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "access_control_allow_credentials": "accessControlAllowCredentials",
            "access_control_allow_headers": "accessControlAllowHeaders",
            "access_control_allow_methods": "accessControlAllowMethods",
            "access_control_allow_origins": "accessControlAllowOrigins",
            "origin_override": "originOverride",
            "access_control_expose_headers": "accessControlExposeHeaders",
            "access_control_max_age_sec": "accessControlMaxAgeSec",
        },
    )
    class CorsConfigProperty:
        def __init__(
            self,
            *,
            access_control_allow_credentials: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            access_control_allow_headers: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty", typing.Dict[builtins.str, typing.Any]]],
            access_control_allow_methods: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty", typing.Dict[builtins.str, typing.Any]]],
            access_control_allow_origins: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty", typing.Dict[builtins.str, typing.Any]]],
            origin_override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            access_control_expose_headers: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            access_control_max_age_sec: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''A configuration for a set of HTTP response headers that are used for cross-origin resource sharing (CORS).

            CloudFront adds these headers to HTTP responses that it sends for CORS requests that match a cache behavior associated with this response headers policy.

            For more information about CORS, see `Cross-Origin Resource Sharing (CORS) <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS>`_ in the MDN Web Docs.

            :param access_control_allow_credentials: A Boolean that CloudFront uses as the value for the ``Access-Control-Allow-Credentials`` HTTP response header. For more information about the ``Access-Control-Allow-Credentials`` HTTP response header, see `Access-Control-Allow-Credentials <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Credentials>`_ in the MDN Web Docs.
            :param access_control_allow_headers: A list of HTTP header names that CloudFront includes as values for the ``Access-Control-Allow-Headers`` HTTP response header. For more information about the ``Access-Control-Allow-Headers`` HTTP response header, see `Access-Control-Allow-Headers <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers>`_ in the MDN Web Docs.
            :param access_control_allow_methods: A list of HTTP methods that CloudFront includes as values for the ``Access-Control-Allow-Methods`` HTTP response header. For more information about the ``Access-Control-Allow-Methods`` HTTP response header, see `Access-Control-Allow-Methods <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods>`_ in the MDN Web Docs.
            :param access_control_allow_origins: A list of origins (domain names) that CloudFront can use as the value for the ``Access-Control-Allow-Origin`` HTTP response header. For more information about the ``Access-Control-Allow-Origin`` HTTP response header, see `Access-Control-Allow-Origin <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin>`_ in the MDN Web Docs.
            :param origin_override: A Boolean that determines whether CloudFront overrides HTTP response headers received from the origin with the ones specified in this response headers policy.
            :param access_control_expose_headers: A list of HTTP headers that CloudFront includes as values for the ``Access-Control-Expose-Headers`` HTTP response header. For more information about the ``Access-Control-Expose-Headers`` HTTP response header, see `Access-Control-Expose-Headers <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers>`_ in the MDN Web Docs.
            :param access_control_max_age_sec: A number that CloudFront uses as the value for the ``Access-Control-Max-Age`` HTTP response header. For more information about the ``Access-Control-Max-Age`` HTTP response header, see `Access-Control-Max-Age <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Max-Age>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                cors_config_property = cloudfront.CfnResponseHeadersPolicy.CorsConfigProperty(
                    access_control_allow_credentials=False,
                    access_control_allow_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty(
                        items=["items"]
                    ),
                    access_control_allow_methods=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty(
                        items=["items"]
                    ),
                    access_control_allow_origins=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty(
                        items=["items"]
                    ),
                    origin_override=False,
                
                    # the properties below are optional
                    access_control_expose_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty(
                        items=["items"]
                    ),
                    access_control_max_age_sec=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__4c74936c70af2524231afd1986ba626c0dde004caffab0dde33d835d3838748e)
                check_type(argname="argument access_control_allow_credentials", value=access_control_allow_credentials, expected_type=type_hints["access_control_allow_credentials"])
                check_type(argname="argument access_control_allow_headers", value=access_control_allow_headers, expected_type=type_hints["access_control_allow_headers"])
                check_type(argname="argument access_control_allow_methods", value=access_control_allow_methods, expected_type=type_hints["access_control_allow_methods"])
                check_type(argname="argument access_control_allow_origins", value=access_control_allow_origins, expected_type=type_hints["access_control_allow_origins"])
                check_type(argname="argument origin_override", value=origin_override, expected_type=type_hints["origin_override"])
                check_type(argname="argument access_control_expose_headers", value=access_control_expose_headers, expected_type=type_hints["access_control_expose_headers"])
                check_type(argname="argument access_control_max_age_sec", value=access_control_max_age_sec, expected_type=type_hints["access_control_max_age_sec"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "access_control_allow_credentials": access_control_allow_credentials,
                "access_control_allow_headers": access_control_allow_headers,
                "access_control_allow_methods": access_control_allow_methods,
                "access_control_allow_origins": access_control_allow_origins,
                "origin_override": origin_override,
            }
            if access_control_expose_headers is not None:
                self._values["access_control_expose_headers"] = access_control_expose_headers
            if access_control_max_age_sec is not None:
                self._values["access_control_max_age_sec"] = access_control_max_age_sec

        @builtins.property
        def access_control_allow_credentials(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that CloudFront uses as the value for the ``Access-Control-Allow-Credentials`` HTTP response header.

            For more information about the ``Access-Control-Allow-Credentials`` HTTP response header, see `Access-Control-Allow-Credentials <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Credentials>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html#cfn-cloudfront-responseheaderspolicy-corsconfig-accesscontrolallowcredentials
            '''
            result = self._values.get("access_control_allow_credentials")
            assert result is not None, "Required property 'access_control_allow_credentials' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def access_control_allow_headers(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty"]:
            '''A list of HTTP header names that CloudFront includes as values for the ``Access-Control-Allow-Headers`` HTTP response header.

            For more information about the ``Access-Control-Allow-Headers`` HTTP response header, see `Access-Control-Allow-Headers <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html#cfn-cloudfront-responseheaderspolicy-corsconfig-accesscontrolallowheaders
            '''
            result = self._values.get("access_control_allow_headers")
            assert result is not None, "Required property 'access_control_allow_headers' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty"], result)

        @builtins.property
        def access_control_allow_methods(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty"]:
            '''A list of HTTP methods that CloudFront includes as values for the ``Access-Control-Allow-Methods`` HTTP response header.

            For more information about the ``Access-Control-Allow-Methods`` HTTP response header, see `Access-Control-Allow-Methods <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html#cfn-cloudfront-responseheaderspolicy-corsconfig-accesscontrolallowmethods
            '''
            result = self._values.get("access_control_allow_methods")
            assert result is not None, "Required property 'access_control_allow_methods' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty"], result)

        @builtins.property
        def access_control_allow_origins(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty"]:
            '''A list of origins (domain names) that CloudFront can use as the value for the ``Access-Control-Allow-Origin`` HTTP response header.

            For more information about the ``Access-Control-Allow-Origin`` HTTP response header, see `Access-Control-Allow-Origin <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html#cfn-cloudfront-responseheaderspolicy-corsconfig-accesscontrolalloworigins
            '''
            result = self._values.get("access_control_allow_origins")
            assert result is not None, "Required property 'access_control_allow_origins' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty"], result)

        @builtins.property
        def origin_override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides HTTP response headers received from the origin with the ones specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html#cfn-cloudfront-responseheaderspolicy-corsconfig-originoverride
            '''
            result = self._values.get("origin_override")
            assert result is not None, "Required property 'origin_override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def access_control_expose_headers(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty"]]:
            '''A list of HTTP headers that CloudFront includes as values for the ``Access-Control-Expose-Headers`` HTTP response header.

            For more information about the ``Access-Control-Expose-Headers`` HTTP response header, see `Access-Control-Expose-Headers <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html#cfn-cloudfront-responseheaderspolicy-corsconfig-accesscontrolexposeheaders
            '''
            result = self._values.get("access_control_expose_headers")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty"]], result)

        @builtins.property
        def access_control_max_age_sec(self) -> typing.Optional[jsii.Number]:
            '''A number that CloudFront uses as the value for the ``Access-Control-Max-Age`` HTTP response header.

            For more information about the ``Access-Control-Max-Age`` HTTP response header, see `Access-Control-Max-Age <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Max-Age>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-corsconfig.html#cfn-cloudfront-responseheaderspolicy-corsconfig-accesscontrolmaxagesec
            '''
            result = self._values.get("access_control_max_age_sec")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CorsConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.CustomHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"header": "header", "override": "override", "value": "value"},
    )
    class CustomHeaderProperty:
        def __init__(
            self,
            *,
            header: builtins.str,
            override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            value: builtins.str,
        ) -> None:
            '''An HTTP response header name and its value.

            CloudFront includes this header in HTTP responses that it sends for requests that match a cache behavior that’s associated with this response headers policy.

            :param header: The HTTP response header name.
            :param override: A Boolean that determines whether CloudFront overrides a response header with the same name received from the origin with the header specified here.
            :param value: The value for the HTTP response header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-customheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                custom_header_property = cloudfront.CfnResponseHeadersPolicy.CustomHeaderProperty(
                    header="header",
                    override=False,
                    value="value"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2f1ab737983027bddc3a2e207c43f2907e65637c7351a1c781dfd7db50589f39)
                check_type(argname="argument header", value=header, expected_type=type_hints["header"])
                check_type(argname="argument override", value=override, expected_type=type_hints["override"])
                check_type(argname="argument value", value=value, expected_type=type_hints["value"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "header": header,
                "override": override,
                "value": value,
            }

        @builtins.property
        def header(self) -> builtins.str:
            '''The HTTP response header name.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-customheader.html#cfn-cloudfront-responseheaderspolicy-customheader-header
            '''
            result = self._values.get("header")
            assert result is not None, "Required property 'header' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides a response header with the same name received from the origin with the header specified here.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-customheader.html#cfn-cloudfront-responseheaderspolicy-customheader-override
            '''
            result = self._values.get("override")
            assert result is not None, "Required property 'override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value for the HTTP response header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-customheader.html#cfn-cloudfront-responseheaderspolicy-customheader-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.CustomHeadersConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items"},
    )
    class CustomHeadersConfigProperty:
        def __init__(
            self,
            *,
            items: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.CustomHeaderProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''A list of HTTP response header names and their values.

            CloudFront includes these headers in HTTP responses that it sends for requests that match a cache behavior that’s associated with this response headers policy.

            :param items: The list of HTTP response headers and their values.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-customheadersconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                custom_headers_config_property = cloudfront.CfnResponseHeadersPolicy.CustomHeadersConfigProperty(
                    items=[cloudfront.CfnResponseHeadersPolicy.CustomHeaderProperty(
                        header="header",
                        override=False,
                        value="value"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__24a1757cd5aa54498a73c385343073ad632b23ccf65387aa10d86e26e6f03d44)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
            }

        @builtins.property
        def items(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.CustomHeaderProperty"]]]:
            '''The list of HTTP response headers and their values.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-customheadersconfig.html#cfn-cloudfront-responseheaderspolicy-customheadersconfig-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.CustomHeaderProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CustomHeadersConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.FrameOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"frame_option": "frameOption", "override": "override"},
    )
    class FrameOptionsProperty:
        def __init__(
            self,
            *,
            frame_option: builtins.str,
            override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
        ) -> None:
            '''Determines whether CloudFront includes the ``X-Frame-Options`` HTTP response header and the header’s value.

            For more information about the ``X-Frame-Options`` HTTP response header, see `X-Frame-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options>`_ in the MDN Web Docs.

            :param frame_option: The value of the ``X-Frame-Options`` HTTP response header. Valid values are ``DENY`` and ``SAMEORIGIN`` . For more information about these values, see `X-Frame-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options>`_ in the MDN Web Docs.
            :param override: A Boolean that determines whether CloudFront overrides the ``X-Frame-Options`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-frameoptions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                frame_options_property = cloudfront.CfnResponseHeadersPolicy.FrameOptionsProperty(
                    frame_option="frameOption",
                    override=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__9e4aa0b976fc82bbbb0580a3f90fc13bcf0d9d5842af8533b057190cc0835c73)
                check_type(argname="argument frame_option", value=frame_option, expected_type=type_hints["frame_option"])
                check_type(argname="argument override", value=override, expected_type=type_hints["override"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "frame_option": frame_option,
                "override": override,
            }

        @builtins.property
        def frame_option(self) -> builtins.str:
            '''The value of the ``X-Frame-Options`` HTTP response header. Valid values are ``DENY`` and ``SAMEORIGIN`` .

            For more information about these values, see `X-Frame-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-frameoptions.html#cfn-cloudfront-responseheaderspolicy-frameoptions-frameoption
            '''
            result = self._values.get("frame_option")
            assert result is not None, "Required property 'frame_option' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides the ``X-Frame-Options`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-frameoptions.html#cfn-cloudfront-responseheaderspolicy-frameoptions-override
            '''
            result = self._values.get("override")
            assert result is not None, "Required property 'override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FrameOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.ReferrerPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"override": "override", "referrer_policy": "referrerPolicy"},
    )
    class ReferrerPolicyProperty:
        def __init__(
            self,
            *,
            override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            referrer_policy: builtins.str,
        ) -> None:
            '''Determines whether CloudFront includes the ``Referrer-Policy`` HTTP response header and the header’s value.

            For more information about the ``Referrer-Policy`` HTTP response header, see `Referrer-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy>`_ in the MDN Web Docs.

            :param override: A Boolean that determines whether CloudFront overrides the ``Referrer-Policy`` HTTP response header received from the origin with the one specified in this response headers policy.
            :param referrer_policy: The value of the ``Referrer-Policy`` HTTP response header. Valid values are:. - ``no-referrer`` - ``no-referrer-when-downgrade`` - ``origin`` - ``origin-when-cross-origin`` - ``same-origin`` - ``strict-origin`` - ``strict-origin-when-cross-origin`` - ``unsafe-url`` For more information about these values, see `Referrer-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-referrerpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                referrer_policy_property = cloudfront.CfnResponseHeadersPolicy.ReferrerPolicyProperty(
                    override=False,
                    referrer_policy="referrerPolicy"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__511a97af670662e7c6e8fdfecfeaff1c42d55dccbd679d1eb7298a37e300a034)
                check_type(argname="argument override", value=override, expected_type=type_hints["override"])
                check_type(argname="argument referrer_policy", value=referrer_policy, expected_type=type_hints["referrer_policy"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "override": override,
                "referrer_policy": referrer_policy,
            }

        @builtins.property
        def override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides the ``Referrer-Policy`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-referrerpolicy.html#cfn-cloudfront-responseheaderspolicy-referrerpolicy-override
            '''
            result = self._values.get("override")
            assert result is not None, "Required property 'override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def referrer_policy(self) -> builtins.str:
            '''The value of the ``Referrer-Policy`` HTTP response header. Valid values are:.

            - ``no-referrer``
            - ``no-referrer-when-downgrade``
            - ``origin``
            - ``origin-when-cross-origin``
            - ``same-origin``
            - ``strict-origin``
            - ``strict-origin-when-cross-origin``
            - ``unsafe-url``

            For more information about these values, see `Referrer-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-referrerpolicy.html#cfn-cloudfront-responseheaderspolicy-referrerpolicy-referrerpolicy
            '''
            result = self._values.get("referrer_policy")
            assert result is not None, "Required property 'referrer_policy' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ReferrerPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.RemoveHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"header": "header"},
    )
    class RemoveHeaderProperty:
        def __init__(self, *, header: builtins.str) -> None:
            '''
            :param header: ``CfnResponseHeadersPolicy.RemoveHeaderProperty.Header``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-removeheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                remove_header_property = cloudfront.CfnResponseHeadersPolicy.RemoveHeaderProperty(
                    header="header"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__f4831de88739a3b95f51bd4536bf9e1cf7a197bb25fe9d9d5505d94d31ace316)
                check_type(argname="argument header", value=header, expected_type=type_hints["header"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "header": header,
            }

        @builtins.property
        def header(self) -> builtins.str:
            '''``CfnResponseHeadersPolicy.RemoveHeaderProperty.Header``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-removeheader.html#cfn-cloudfront-responseheaderspolicy-removeheader-header
            '''
            result = self._values.get("header")
            assert result is not None, "Required property 'header' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RemoveHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.RemoveHeadersConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"items": "items"},
    )
    class RemoveHeadersConfigProperty:
        def __init__(
            self,
            *,
            items: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.RemoveHeaderProperty", typing.Dict[builtins.str, typing.Any]]]]],
        ) -> None:
            '''
            :param items: ``CfnResponseHeadersPolicy.RemoveHeadersConfigProperty.Items``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-removeheadersconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                remove_headers_config_property = cloudfront.CfnResponseHeadersPolicy.RemoveHeadersConfigProperty(
                    items=[cloudfront.CfnResponseHeadersPolicy.RemoveHeaderProperty(
                        header="header"
                    )]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__b21c80da7ce8d2eea5e7fe648206cc6e3453c301a7b77b9eec4233370bd1ac58)
                check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "items": items,
            }

        @builtins.property
        def items(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.RemoveHeaderProperty"]]]:
            '''``CfnResponseHeadersPolicy.RemoveHeadersConfigProperty.Items``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-removeheadersconfig.html#cfn-cloudfront-responseheaderspolicy-removeheadersconfig-items
            '''
            result = self._values.get("items")
            assert result is not None, "Required property 'items' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.List[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.RemoveHeaderProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RemoveHeadersConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "comment": "comment",
            "cors_config": "corsConfig",
            "custom_headers_config": "customHeadersConfig",
            "remove_headers_config": "removeHeadersConfig",
            "security_headers_config": "securityHeadersConfig",
            "server_timing_headers_config": "serverTimingHeadersConfig",
        },
    )
    class ResponseHeadersPolicyConfigProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            comment: typing.Optional[builtins.str] = None,
            cors_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.CorsConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            custom_headers_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.CustomHeadersConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            remove_headers_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.RemoveHeadersConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            security_headers_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.SecurityHeadersConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            server_timing_headers_config: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A response headers policy configuration.

            A response headers policy configuration contains metadata about the response headers policy, and configurations for sets of HTTP response headers and their values. CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache behavior associated with the policy.

            :param name: A name to identify the response headers policy. The name must be unique for response headers policies in this AWS account .
            :param comment: A comment to describe the response headers policy. The comment cannot be longer than 128 characters.
            :param cors_config: A configuration for a set of HTTP response headers that are used for cross-origin resource sharing (CORS).
            :param custom_headers_config: A configuration for a set of custom HTTP response headers.
            :param remove_headers_config: ``CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty.RemoveHeadersConfig``.
            :param security_headers_config: A configuration for a set of security-related HTTP response headers.
            :param server_timing_headers_config: A configuration for enabling the ``Server-Timing`` header in HTTP responses sent from CloudFront.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                response_headers_policy_config_property = cloudfront.CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty(
                    name="name",
                
                    # the properties below are optional
                    comment="comment",
                    cors_config=cloudfront.CfnResponseHeadersPolicy.CorsConfigProperty(
                        access_control_allow_credentials=False,
                        access_control_allow_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty(
                            items=["items"]
                        ),
                        access_control_allow_methods=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty(
                            items=["items"]
                        ),
                        access_control_allow_origins=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty(
                            items=["items"]
                        ),
                        origin_override=False,
                
                        # the properties below are optional
                        access_control_expose_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty(
                            items=["items"]
                        ),
                        access_control_max_age_sec=123
                    ),
                    custom_headers_config=cloudfront.CfnResponseHeadersPolicy.CustomHeadersConfigProperty(
                        items=[cloudfront.CfnResponseHeadersPolicy.CustomHeaderProperty(
                            header="header",
                            override=False,
                            value="value"
                        )]
                    ),
                    remove_headers_config=cloudfront.CfnResponseHeadersPolicy.RemoveHeadersConfigProperty(
                        items=[cloudfront.CfnResponseHeadersPolicy.RemoveHeaderProperty(
                            header="header"
                        )]
                    ),
                    security_headers_config=cloudfront.CfnResponseHeadersPolicy.SecurityHeadersConfigProperty(
                        content_security_policy=cloudfront.CfnResponseHeadersPolicy.ContentSecurityPolicyProperty(
                            content_security_policy="contentSecurityPolicy",
                            override=False
                        ),
                        content_type_options=cloudfront.CfnResponseHeadersPolicy.ContentTypeOptionsProperty(
                            override=False
                        ),
                        frame_options=cloudfront.CfnResponseHeadersPolicy.FrameOptionsProperty(
                            frame_option="frameOption",
                            override=False
                        ),
                        referrer_policy=cloudfront.CfnResponseHeadersPolicy.ReferrerPolicyProperty(
                            override=False,
                            referrer_policy="referrerPolicy"
                        ),
                        strict_transport_security=cloudfront.CfnResponseHeadersPolicy.StrictTransportSecurityProperty(
                            access_control_max_age_sec=123,
                            override=False,
                
                            # the properties below are optional
                            include_subdomains=False,
                            preload=False
                        ),
                        xss_protection=cloudfront.CfnResponseHeadersPolicy.XSSProtectionProperty(
                            override=False,
                            protection=False,
                
                            # the properties below are optional
                            mode_block=False,
                            report_uri="reportUri"
                        )
                    ),
                    server_timing_headers_config=cloudfront.CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty(
                        enabled=False,
                
                        # the properties below are optional
                        sampling_rate=123
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2ceddc85fa93a8b5afda4d096a0e293324c9cff5c83a30695b8483743e59216f)
                check_type(argname="argument name", value=name, expected_type=type_hints["name"])
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
                check_type(argname="argument cors_config", value=cors_config, expected_type=type_hints["cors_config"])
                check_type(argname="argument custom_headers_config", value=custom_headers_config, expected_type=type_hints["custom_headers_config"])
                check_type(argname="argument remove_headers_config", value=remove_headers_config, expected_type=type_hints["remove_headers_config"])
                check_type(argname="argument security_headers_config", value=security_headers_config, expected_type=type_hints["security_headers_config"])
                check_type(argname="argument server_timing_headers_config", value=server_timing_headers_config, expected_type=type_hints["server_timing_headers_config"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "name": name,
            }
            if comment is not None:
                self._values["comment"] = comment
            if cors_config is not None:
                self._values["cors_config"] = cors_config
            if custom_headers_config is not None:
                self._values["custom_headers_config"] = custom_headers_config
            if remove_headers_config is not None:
                self._values["remove_headers_config"] = remove_headers_config
            if security_headers_config is not None:
                self._values["security_headers_config"] = security_headers_config
            if server_timing_headers_config is not None:
                self._values["server_timing_headers_config"] = server_timing_headers_config

        @builtins.property
        def name(self) -> builtins.str:
            '''A name to identify the response headers policy.

            The name must be unique for response headers policies in this AWS account .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def comment(self) -> typing.Optional[builtins.str]:
            '''A comment to describe the response headers policy.

            The comment cannot be longer than 128 characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig-comment
            '''
            result = self._values.get("comment")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cors_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.CorsConfigProperty"]]:
            '''A configuration for a set of HTTP response headers that are used for cross-origin resource sharing (CORS).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig-corsconfig
            '''
            result = self._values.get("cors_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.CorsConfigProperty"]], result)

        @builtins.property
        def custom_headers_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.CustomHeadersConfigProperty"]]:
            '''A configuration for a set of custom HTTP response headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig-customheadersconfig
            '''
            result = self._values.get("custom_headers_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.CustomHeadersConfigProperty"]], result)

        @builtins.property
        def remove_headers_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.RemoveHeadersConfigProperty"]]:
            '''``CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty.RemoveHeadersConfig``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig-removeheadersconfig
            '''
            result = self._values.get("remove_headers_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.RemoveHeadersConfigProperty"]], result)

        @builtins.property
        def security_headers_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.SecurityHeadersConfigProperty"]]:
            '''A configuration for a set of security-related HTTP response headers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig-securityheadersconfig
            '''
            result = self._values.get("security_headers_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.SecurityHeadersConfigProperty"]], result)

        @builtins.property
        def server_timing_headers_config(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty"]]:
            '''A configuration for enabling the ``Server-Timing`` header in HTTP responses sent from CloudFront.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-responseheaderspolicyconfig.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig-servertimingheadersconfig
            '''
            result = self._values.get("server_timing_headers_config")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResponseHeadersPolicyConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.SecurityHeadersConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "content_security_policy": "contentSecurityPolicy",
            "content_type_options": "contentTypeOptions",
            "frame_options": "frameOptions",
            "referrer_policy": "referrerPolicy",
            "strict_transport_security": "strictTransportSecurity",
            "xss_protection": "xssProtection",
        },
    )
    class SecurityHeadersConfigProperty:
        def __init__(
            self,
            *,
            content_security_policy: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.ContentSecurityPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            content_type_options: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.ContentTypeOptionsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            frame_options: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.FrameOptionsProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            referrer_policy: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.ReferrerPolicyProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            strict_transport_security: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.StrictTransportSecurityProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            xss_protection: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnResponseHeadersPolicy.XSSProtectionProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
        ) -> None:
            '''A configuration for a set of security-related HTTP response headers.

            CloudFront adds these headers to HTTP responses that it sends for requests that match a cache behavior associated with this response headers policy.

            :param content_security_policy: The policy directives and their values that CloudFront includes as values for the ``Content-Security-Policy`` HTTP response header. For more information about the ``Content-Security-Policy`` HTTP response header, see `Content-Security-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy>`_ in the MDN Web Docs.
            :param content_type_options: Determines whether CloudFront includes the ``X-Content-Type-Options`` HTTP response header with its value set to ``nosniff`` . For more information about the ``X-Content-Type-Options`` HTTP response header, see `X-Content-Type-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options>`_ in the MDN Web Docs.
            :param frame_options: Determines whether CloudFront includes the ``X-Frame-Options`` HTTP response header and the header’s value. For more information about the ``X-Frame-Options`` HTTP response header, see `X-Frame-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options>`_ in the MDN Web Docs.
            :param referrer_policy: Determines whether CloudFront includes the ``Referrer-Policy`` HTTP response header and the header’s value. For more information about the ``Referrer-Policy`` HTTP response header, see `Referrer-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy>`_ in the MDN Web Docs.
            :param strict_transport_security: Determines whether CloudFront includes the ``Strict-Transport-Security`` HTTP response header and the header’s value. For more information about the ``Strict-Transport-Security`` HTTP response header, see `Strict-Transport-Security <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security>`_ in the MDN Web Docs.
            :param xss_protection: Determines whether CloudFront includes the ``X-XSS-Protection`` HTTP response header and the header’s value. For more information about the ``X-XSS-Protection`` HTTP response header, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-securityheadersconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                security_headers_config_property = cloudfront.CfnResponseHeadersPolicy.SecurityHeadersConfigProperty(
                    content_security_policy=cloudfront.CfnResponseHeadersPolicy.ContentSecurityPolicyProperty(
                        content_security_policy="contentSecurityPolicy",
                        override=False
                    ),
                    content_type_options=cloudfront.CfnResponseHeadersPolicy.ContentTypeOptionsProperty(
                        override=False
                    ),
                    frame_options=cloudfront.CfnResponseHeadersPolicy.FrameOptionsProperty(
                        frame_option="frameOption",
                        override=False
                    ),
                    referrer_policy=cloudfront.CfnResponseHeadersPolicy.ReferrerPolicyProperty(
                        override=False,
                        referrer_policy="referrerPolicy"
                    ),
                    strict_transport_security=cloudfront.CfnResponseHeadersPolicy.StrictTransportSecurityProperty(
                        access_control_max_age_sec=123,
                        override=False,
                
                        # the properties below are optional
                        include_subdomains=False,
                        preload=False
                    ),
                    xss_protection=cloudfront.CfnResponseHeadersPolicy.XSSProtectionProperty(
                        override=False,
                        protection=False,
                
                        # the properties below are optional
                        mode_block=False,
                        report_uri="reportUri"
                    )
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__2a08c42f7048c0b94ae2b451dae0078c3504e4d02e1fec01987be362f7da3fcb)
                check_type(argname="argument content_security_policy", value=content_security_policy, expected_type=type_hints["content_security_policy"])
                check_type(argname="argument content_type_options", value=content_type_options, expected_type=type_hints["content_type_options"])
                check_type(argname="argument frame_options", value=frame_options, expected_type=type_hints["frame_options"])
                check_type(argname="argument referrer_policy", value=referrer_policy, expected_type=type_hints["referrer_policy"])
                check_type(argname="argument strict_transport_security", value=strict_transport_security, expected_type=type_hints["strict_transport_security"])
                check_type(argname="argument xss_protection", value=xss_protection, expected_type=type_hints["xss_protection"])
            self._values: typing.Dict[builtins.str, typing.Any] = {}
            if content_security_policy is not None:
                self._values["content_security_policy"] = content_security_policy
            if content_type_options is not None:
                self._values["content_type_options"] = content_type_options
            if frame_options is not None:
                self._values["frame_options"] = frame_options
            if referrer_policy is not None:
                self._values["referrer_policy"] = referrer_policy
            if strict_transport_security is not None:
                self._values["strict_transport_security"] = strict_transport_security
            if xss_protection is not None:
                self._values["xss_protection"] = xss_protection

        @builtins.property
        def content_security_policy(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ContentSecurityPolicyProperty"]]:
            '''The policy directives and their values that CloudFront includes as values for the ``Content-Security-Policy`` HTTP response header.

            For more information about the ``Content-Security-Policy`` HTTP response header, see `Content-Security-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-securityheadersconfig.html#cfn-cloudfront-responseheaderspolicy-securityheadersconfig-contentsecuritypolicy
            '''
            result = self._values.get("content_security_policy")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ContentSecurityPolicyProperty"]], result)

        @builtins.property
        def content_type_options(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ContentTypeOptionsProperty"]]:
            '''Determines whether CloudFront includes the ``X-Content-Type-Options`` HTTP response header with its value set to ``nosniff`` .

            For more information about the ``X-Content-Type-Options`` HTTP response header, see `X-Content-Type-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-securityheadersconfig.html#cfn-cloudfront-responseheaderspolicy-securityheadersconfig-contenttypeoptions
            '''
            result = self._values.get("content_type_options")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ContentTypeOptionsProperty"]], result)

        @builtins.property
        def frame_options(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.FrameOptionsProperty"]]:
            '''Determines whether CloudFront includes the ``X-Frame-Options`` HTTP response header and the header’s value.

            For more information about the ``X-Frame-Options`` HTTP response header, see `X-Frame-Options <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Frame-Options>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-securityheadersconfig.html#cfn-cloudfront-responseheaderspolicy-securityheadersconfig-frameoptions
            '''
            result = self._values.get("frame_options")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.FrameOptionsProperty"]], result)

        @builtins.property
        def referrer_policy(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ReferrerPolicyProperty"]]:
            '''Determines whether CloudFront includes the ``Referrer-Policy`` HTTP response header and the header’s value.

            For more information about the ``Referrer-Policy`` HTTP response header, see `Referrer-Policy <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Referrer-Policy>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-securityheadersconfig.html#cfn-cloudfront-responseheaderspolicy-securityheadersconfig-referrerpolicy
            '''
            result = self._values.get("referrer_policy")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.ReferrerPolicyProperty"]], result)

        @builtins.property
        def strict_transport_security(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.StrictTransportSecurityProperty"]]:
            '''Determines whether CloudFront includes the ``Strict-Transport-Security`` HTTP response header and the header’s value.

            For more information about the ``Strict-Transport-Security`` HTTP response header, see `Strict-Transport-Security <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-securityheadersconfig.html#cfn-cloudfront-responseheaderspolicy-securityheadersconfig-stricttransportsecurity
            '''
            result = self._values.get("strict_transport_security")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.StrictTransportSecurityProperty"]], result)

        @builtins.property
        def xss_protection(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.XSSProtectionProperty"]]:
            '''Determines whether CloudFront includes the ``X-XSS-Protection`` HTTP response header and the header’s value.

            For more information about the ``X-XSS-Protection`` HTTP response header, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-securityheadersconfig.html#cfn-cloudfront-responseheaderspolicy-securityheadersconfig-xssprotection
            '''
            result = self._values.get("xss_protection")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnResponseHeadersPolicy.XSSProtectionProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SecurityHeadersConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled", "sampling_rate": "samplingRate"},
    )
    class ServerTimingHeadersConfigProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            sampling_rate: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''A configuration for enabling the ``Server-Timing`` header in HTTP responses sent from CloudFront.

            :param enabled: A Boolean that determines whether CloudFront adds the ``Server-Timing`` header to HTTP responses that it sends in response to requests that match a cache behavior that's associated with this response headers policy.
            :param sampling_rate: A number 0–100 (inclusive) that specifies the percentage of responses that you want CloudFront to add the ``Server-Timing`` header to. When you set the sampling rate to 100, CloudFront adds the ``Server-Timing`` header to the HTTP response for every request that matches the cache behavior that this response headers policy is attached to. When you set it to 50, CloudFront adds the header to 50% of the responses for requests that match the cache behavior. You can set the sampling rate to any number 0–100 with up to four decimal places.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-servertimingheadersconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                server_timing_headers_config_property = cloudfront.CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty(
                    enabled=False,
                
                    # the properties below are optional
                    sampling_rate=123
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__536cdd76f23051eb8f7ff0b574986e1c63ce37bcad70140b59d34aac23979c54)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument sampling_rate", value=sampling_rate, expected_type=type_hints["sampling_rate"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enabled": enabled,
            }
            if sampling_rate is not None:
                self._values["sampling_rate"] = sampling_rate

        @builtins.property
        def enabled(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront adds the ``Server-Timing`` header to HTTP responses that it sends in response to requests that match a cache behavior that's associated with this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-servertimingheadersconfig.html#cfn-cloudfront-responseheaderspolicy-servertimingheadersconfig-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def sampling_rate(self) -> typing.Optional[jsii.Number]:
            '''A number 0–100 (inclusive) that specifies the percentage of responses that you want CloudFront to add the ``Server-Timing`` header to.

            When you set the sampling rate to 100, CloudFront adds the ``Server-Timing`` header to the HTTP response for every request that matches the cache behavior that this response headers policy is attached to. When you set it to 50, CloudFront adds the header to 50% of the responses for requests that match the cache behavior. You can set the sampling rate to any number 0–100 with up to four decimal places.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-servertimingheadersconfig.html#cfn-cloudfront-responseheaderspolicy-servertimingheadersconfig-samplingrate
            '''
            result = self._values.get("sampling_rate")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServerTimingHeadersConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.StrictTransportSecurityProperty",
        jsii_struct_bases=[],
        name_mapping={
            "access_control_max_age_sec": "accessControlMaxAgeSec",
            "override": "override",
            "include_subdomains": "includeSubdomains",
            "preload": "preload",
        },
    )
    class StrictTransportSecurityProperty:
        def __init__(
            self,
            *,
            access_control_max_age_sec: jsii.Number,
            override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            include_subdomains: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            preload: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
        ) -> None:
            '''Determines whether CloudFront includes the ``Strict-Transport-Security`` HTTP response header and the header’s value.

            For more information about the ``Strict-Transport-Security`` HTTP response header, see `Strict-Transport-Security <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security>`_ in the MDN Web Docs.

            :param access_control_max_age_sec: A number that CloudFront uses as the value for the ``max-age`` directive in the ``Strict-Transport-Security`` HTTP response header.
            :param override: A Boolean that determines whether CloudFront overrides the ``Strict-Transport-Security`` HTTP response header received from the origin with the one specified in this response headers policy.
            :param include_subdomains: A Boolean that determines whether CloudFront includes the ``includeSubDomains`` directive in the ``Strict-Transport-Security`` HTTP response header.
            :param preload: A Boolean that determines whether CloudFront includes the ``preload`` directive in the ``Strict-Transport-Security`` HTTP response header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-stricttransportsecurity.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                strict_transport_security_property = cloudfront.CfnResponseHeadersPolicy.StrictTransportSecurityProperty(
                    access_control_max_age_sec=123,
                    override=False,
                
                    # the properties below are optional
                    include_subdomains=False,
                    preload=False
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__617a6df75d42f2e188bb5f97fbbe0a3808e21a5d0c146a6ac03c02ea45fdfada)
                check_type(argname="argument access_control_max_age_sec", value=access_control_max_age_sec, expected_type=type_hints["access_control_max_age_sec"])
                check_type(argname="argument override", value=override, expected_type=type_hints["override"])
                check_type(argname="argument include_subdomains", value=include_subdomains, expected_type=type_hints["include_subdomains"])
                check_type(argname="argument preload", value=preload, expected_type=type_hints["preload"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "access_control_max_age_sec": access_control_max_age_sec,
                "override": override,
            }
            if include_subdomains is not None:
                self._values["include_subdomains"] = include_subdomains
            if preload is not None:
                self._values["preload"] = preload

        @builtins.property
        def access_control_max_age_sec(self) -> jsii.Number:
            '''A number that CloudFront uses as the value for the ``max-age`` directive in the ``Strict-Transport-Security`` HTTP response header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-stricttransportsecurity.html#cfn-cloudfront-responseheaderspolicy-stricttransportsecurity-accesscontrolmaxagesec
            '''
            result = self._values.get("access_control_max_age_sec")
            assert result is not None, "Required property 'access_control_max_age_sec' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides the ``Strict-Transport-Security`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-stricttransportsecurity.html#cfn-cloudfront-responseheaderspolicy-stricttransportsecurity-override
            '''
            result = self._values.get("override")
            assert result is not None, "Required property 'override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def include_subdomains(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A Boolean that determines whether CloudFront includes the ``includeSubDomains`` directive in the ``Strict-Transport-Security`` HTTP response header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-stricttransportsecurity.html#cfn-cloudfront-responseheaderspolicy-stricttransportsecurity-includesubdomains
            '''
            result = self._values.get("include_subdomains")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def preload(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A Boolean that determines whether CloudFront includes the ``preload`` directive in the ``Strict-Transport-Security`` HTTP response header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-stricttransportsecurity.html#cfn-cloudfront-responseheaderspolicy-stricttransportsecurity-preload
            '''
            result = self._values.get("preload")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StrictTransportSecurityProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicy.XSSProtectionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "override": "override",
            "protection": "protection",
            "mode_block": "modeBlock",
            "report_uri": "reportUri",
        },
    )
    class XSSProtectionProperty:
        def __init__(
            self,
            *,
            override: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            protection: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            mode_block: typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]] = None,
            report_uri: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Determines whether CloudFront includes the ``X-XSS-Protection`` HTTP response header and the header’s value.

            For more information about the ``X-XSS-Protection`` HTTP response header, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.

            :param override: A Boolean that determines whether CloudFront overrides the ``X-XSS-Protection`` HTTP response header received from the origin with the one specified in this response headers policy.
            :param protection: A Boolean that determines the value of the ``X-XSS-Protection`` HTTP response header. When this setting is ``true`` , the value of the ``X-XSS-Protection`` header is ``1`` . When this setting is ``false`` , the value of the ``X-XSS-Protection`` header is ``0`` . For more information about these settings, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.
            :param mode_block: A Boolean that determines whether CloudFront includes the ``mode=block`` directive in the ``X-XSS-Protection`` header. For more information about this directive, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.
            :param report_uri: A reporting URI, which CloudFront uses as the value of the ``report`` directive in the ``X-XSS-Protection`` header. You cannot specify a ``ReportUri`` when ``ModeBlock`` is ``true`` . For more information about using a reporting URL, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-xssprotection.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                x_sSProtection_property = cloudfront.CfnResponseHeadersPolicy.XSSProtectionProperty(
                    override=False,
                    protection=False,
                
                    # the properties below are optional
                    mode_block=False,
                    report_uri="reportUri"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__8c484006f957eeb92dc381ced4e6bfd5138e8c2733f06f038a06d06e6fb1d354)
                check_type(argname="argument override", value=override, expected_type=type_hints["override"])
                check_type(argname="argument protection", value=protection, expected_type=type_hints["protection"])
                check_type(argname="argument mode_block", value=mode_block, expected_type=type_hints["mode_block"])
                check_type(argname="argument report_uri", value=report_uri, expected_type=type_hints["report_uri"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "override": override,
                "protection": protection,
            }
            if mode_block is not None:
                self._values["mode_block"] = mode_block
            if report_uri is not None:
                self._values["report_uri"] = report_uri

        @builtins.property
        def override(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines whether CloudFront overrides the ``X-XSS-Protection`` HTTP response header received from the origin with the one specified in this response headers policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-xssprotection.html#cfn-cloudfront-responseheaderspolicy-xssprotection-override
            '''
            result = self._values.get("override")
            assert result is not None, "Required property 'override' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def protection(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''A Boolean that determines the value of the ``X-XSS-Protection`` HTTP response header.

            When this setting is ``true`` , the value of the ``X-XSS-Protection`` header is ``1`` . When this setting is ``false`` , the value of the ``X-XSS-Protection`` header is ``0`` .

            For more information about these settings, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-xssprotection.html#cfn-cloudfront-responseheaderspolicy-xssprotection-protection
            '''
            result = self._values.get("protection")
            assert result is not None, "Required property 'protection' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def mode_block(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]]:
            '''A Boolean that determines whether CloudFront includes the ``mode=block`` directive in the ``X-XSS-Protection`` header.

            For more information about this directive, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-xssprotection.html#cfn-cloudfront-responseheaderspolicy-xssprotection-modeblock
            '''
            result = self._values.get("mode_block")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]], result)

        @builtins.property
        def report_uri(self) -> typing.Optional[builtins.str]:
            '''A reporting URI, which CloudFront uses as the value of the ``report`` directive in the ``X-XSS-Protection`` header.

            You cannot specify a ``ReportUri`` when ``ModeBlock`` is ``true`` .

            For more information about using a reporting URL, see `X-XSS-Protection <https://docs.aws.amazon.com/https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection>`_ in the MDN Web Docs.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-responseheaderspolicy-xssprotection.html#cfn-cloudfront-responseheaderspolicy-xssprotection-reporturi
            '''
            result = self._values.get("report_uri")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "XSSProtectionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnResponseHeadersPolicyProps",
    jsii_struct_bases=[],
    name_mapping={"response_headers_policy_config": "responseHeadersPolicyConfig"},
)
class CfnResponseHeadersPolicyProps:
    def __init__(
        self,
        *,
        response_headers_policy_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnResponseHeadersPolicy``.

        :param response_headers_policy_config: A response headers policy configuration. A response headers policy contains information about a set of HTTP response headers and their values. CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache behavior that’s associated with the policy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-responseheaderspolicy.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_response_headers_policy_props = cloudfront.CfnResponseHeadersPolicyProps(
                response_headers_policy_config=cloudfront.CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty(
                    name="name",
            
                    # the properties below are optional
                    comment="comment",
                    cors_config=cloudfront.CfnResponseHeadersPolicy.CorsConfigProperty(
                        access_control_allow_credentials=False,
                        access_control_allow_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowHeadersProperty(
                            items=["items"]
                        ),
                        access_control_allow_methods=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowMethodsProperty(
                            items=["items"]
                        ),
                        access_control_allow_origins=cloudfront.CfnResponseHeadersPolicy.AccessControlAllowOriginsProperty(
                            items=["items"]
                        ),
                        origin_override=False,
            
                        # the properties below are optional
                        access_control_expose_headers=cloudfront.CfnResponseHeadersPolicy.AccessControlExposeHeadersProperty(
                            items=["items"]
                        ),
                        access_control_max_age_sec=123
                    ),
                    custom_headers_config=cloudfront.CfnResponseHeadersPolicy.CustomHeadersConfigProperty(
                        items=[cloudfront.CfnResponseHeadersPolicy.CustomHeaderProperty(
                            header="header",
                            override=False,
                            value="value"
                        )]
                    ),
                    remove_headers_config=cloudfront.CfnResponseHeadersPolicy.RemoveHeadersConfigProperty(
                        items=[cloudfront.CfnResponseHeadersPolicy.RemoveHeaderProperty(
                            header="header"
                        )]
                    ),
                    security_headers_config=cloudfront.CfnResponseHeadersPolicy.SecurityHeadersConfigProperty(
                        content_security_policy=cloudfront.CfnResponseHeadersPolicy.ContentSecurityPolicyProperty(
                            content_security_policy="contentSecurityPolicy",
                            override=False
                        ),
                        content_type_options=cloudfront.CfnResponseHeadersPolicy.ContentTypeOptionsProperty(
                            override=False
                        ),
                        frame_options=cloudfront.CfnResponseHeadersPolicy.FrameOptionsProperty(
                            frame_option="frameOption",
                            override=False
                        ),
                        referrer_policy=cloudfront.CfnResponseHeadersPolicy.ReferrerPolicyProperty(
                            override=False,
                            referrer_policy="referrerPolicy"
                        ),
                        strict_transport_security=cloudfront.CfnResponseHeadersPolicy.StrictTransportSecurityProperty(
                            access_control_max_age_sec=123,
                            override=False,
            
                            # the properties below are optional
                            include_subdomains=False,
                            preload=False
                        ),
                        xss_protection=cloudfront.CfnResponseHeadersPolicy.XSSProtectionProperty(
                            override=False,
                            protection=False,
            
                            # the properties below are optional
                            mode_block=False,
                            report_uri="reportUri"
                        )
                    ),
                    server_timing_headers_config=cloudfront.CfnResponseHeadersPolicy.ServerTimingHeadersConfigProperty(
                        enabled=False,
            
                        # the properties below are optional
                        sampling_rate=123
                    )
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__5f0302f76fc528b57913f3a2cc67326980cc912ac8644a81f0be61ab0cecc856)
            check_type(argname="argument response_headers_policy_config", value=response_headers_policy_config, expected_type=type_hints["response_headers_policy_config"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "response_headers_policy_config": response_headers_policy_config,
        }

    @builtins.property
    def response_headers_policy_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty]:
        '''A response headers policy configuration.

        A response headers policy contains information about a set of HTTP response headers and their values. CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache behavior that’s associated with the policy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-responseheaderspolicy.html#cfn-cloudfront-responseheaderspolicy-responseheaderspolicyconfig
        '''
        result = self._values.get("response_headers_policy_config")
        assert result is not None, "Required property 'response_headers_policy_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnResponseHeadersPolicy.ResponseHeadersPolicyConfigProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnResponseHeadersPolicyProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_aws_cdk_core_f4b25747.IInspectable)
class CfnStreamingDistribution(
    _aws_cdk_core_f4b25747.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.CfnStreamingDistribution",
):
    '''A CloudFormation ``AWS::CloudFront::StreamingDistribution``.

    This resource is deprecated. Amazon CloudFront is deprecating real-time messaging protocol (RTMP) distributions on December 31, 2020. For more information, `read the announcement <https://docs.aws.amazon.com/ann.jspa?annID=7356>`_ on the Amazon CloudFront discussion forum.

    :cloudformationResource: AWS::CloudFront::StreamingDistribution
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-streamingdistribution.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import aws_cdk.aws_cloudfront as cloudfront
        
        cfn_streaming_distribution = cloudfront.CfnStreamingDistribution(self, "MyCfnStreamingDistribution",
            streaming_distribution_config=cloudfront.CfnStreamingDistribution.StreamingDistributionConfigProperty(
                comment="comment",
                enabled=False,
                s3_origin=cloudfront.CfnStreamingDistribution.S3OriginProperty(
                    domain_name="domainName",
                    origin_access_identity="originAccessIdentity"
                ),
                trusted_signers=cloudfront.CfnStreamingDistribution.TrustedSignersProperty(
                    enabled=False,
        
                    # the properties below are optional
                    aws_account_numbers=["awsAccountNumbers"]
                ),
        
                # the properties below are optional
                aliases=["aliases"],
                logging=cloudfront.CfnStreamingDistribution.LoggingProperty(
                    bucket="bucket",
                    enabled=False,
                    prefix="prefix"
                ),
                price_class="priceClass"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        id: builtins.str,
        *,
        streaming_distribution_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnStreamingDistribution.StreamingDistributionConfigProperty", typing.Dict[builtins.str, typing.Any]]],
        tags: typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Create a new ``AWS::CloudFront::StreamingDistribution``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param streaming_distribution_config: The current configuration information for the RTMP distribution.
        :param tags: A complex type that contains zero or more ``Tag`` elements.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8177a8f6b4397fb3db26543040753ab09c314aa3f25a07c77ebf91e988cb6312)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = CfnStreamingDistributionProps(
            streaming_distribution_config=streaming_distribution_config, tags=tags
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _aws_cdk_core_f4b25747.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__49cb84a726e8cc070bc541c55ce74abca85f632df4aa071ba44c16c4bcef8487)
            check_type(argname="argument inspector", value=inspector, expected_type=type_hints["inspector"])
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__f92083a076067a270f085bf30357d8f616e28420517d2225b969d8aca391dfed)
            check_type(argname="argument props", value=props, expected_type=type_hints["props"])
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property
    @jsii.member(jsii_name="attrDomainName")
    def attr_domain_name(self) -> builtins.str:
        '''The domain name of the resource, such as ``d111111abcdef8.cloudfront.net`` .

        :cloudformationAttribute: DomainName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDomainName"))

    @builtins.property
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property
    @jsii.member(jsii_name="tags")
    def tags(self) -> _aws_cdk_core_f4b25747.TagManager:
        '''A complex type that contains zero or more ``Tag`` elements.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-streamingdistribution.html#cfn-cloudfront-streamingdistribution-tags
        '''
        return typing.cast(_aws_cdk_core_f4b25747.TagManager, jsii.get(self, "tags"))

    @builtins.property
    @jsii.member(jsii_name="streamingDistributionConfig")
    def streaming_distribution_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.StreamingDistributionConfigProperty"]:
        '''The current configuration information for the RTMP distribution.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-streamingdistribution.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig
        '''
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.StreamingDistributionConfigProperty"], jsii.get(self, "streamingDistributionConfig"))

    @streaming_distribution_config.setter
    def streaming_distribution_config(
        self,
        value: typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.StreamingDistributionConfigProperty"],
    ) -> None:
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9879308469561eef11c0af84467890538a58377b13ea8ee0a4cc10b07615ebe2)
            check_type(argname="argument value", value=value, expected_type=type_hints["value"])
        jsii.set(self, "streamingDistributionConfig", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnStreamingDistribution.LoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"bucket": "bucket", "enabled": "enabled", "prefix": "prefix"},
    )
    class LoggingProperty:
        def __init__(
            self,
            *,
            bucket: builtins.str,
            enabled: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            prefix: builtins.str,
        ) -> None:
            '''A complex type that controls whether access logs are written for the streaming distribution.

            :param bucket: The Amazon S3 bucket to store the access logs in, for example, ``myawslogbucket.s3.amazonaws.com`` .
            :param enabled: Specifies whether you want CloudFront to save access logs to an Amazon S3 bucket. If you don't want to enable logging when you create a streaming distribution or if you want to disable logging for an existing streaming distribution, specify ``false`` for ``Enabled`` , and specify ``empty Bucket`` and ``Prefix`` elements. If you specify ``false`` for ``Enabled`` but you specify values for ``Bucket`` and ``Prefix`` , the values are automatically deleted.
            :param prefix: An optional string that you want CloudFront to prefix to the access log filenames for this streaming distribution, for example, ``myprefix/`` . If you want to enable logging, but you don't want to specify a prefix, you still must include an empty ``Prefix`` element in the ``Logging`` element.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-logging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                logging_property = cloudfront.CfnStreamingDistribution.LoggingProperty(
                    bucket="bucket",
                    enabled=False,
                    prefix="prefix"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__a5f3a849d1668f11ddf59e06d57f938662108eaf1beeb771e92f101f7812b760)
                check_type(argname="argument bucket", value=bucket, expected_type=type_hints["bucket"])
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument prefix", value=prefix, expected_type=type_hints["prefix"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "bucket": bucket,
                "enabled": enabled,
                "prefix": prefix,
            }

        @builtins.property
        def bucket(self) -> builtins.str:
            '''The Amazon S3 bucket to store the access logs in, for example, ``myawslogbucket.s3.amazonaws.com`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-logging.html#cfn-cloudfront-streamingdistribution-logging-bucket
            '''
            result = self._values.get("bucket")
            assert result is not None, "Required property 'bucket' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def enabled(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''Specifies whether you want CloudFront to save access logs to an Amazon S3 bucket.

            If you don't want to enable logging when you create a streaming distribution or if you want to disable logging for an existing streaming distribution, specify ``false`` for ``Enabled`` , and specify ``empty Bucket`` and ``Prefix`` elements. If you specify ``false`` for ``Enabled`` but you specify values for ``Bucket`` and ``Prefix`` , the values are automatically deleted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-logging.html#cfn-cloudfront-streamingdistribution-logging-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def prefix(self) -> builtins.str:
            '''An optional string that you want CloudFront to prefix to the access log filenames for this streaming distribution, for example, ``myprefix/`` .

            If you want to enable logging, but you don't want to specify a prefix, you still must include an empty ``Prefix`` element in the ``Logging`` element.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-logging.html#cfn-cloudfront-streamingdistribution-logging-prefix
            '''
            result = self._values.get("prefix")
            assert result is not None, "Required property 'prefix' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnStreamingDistribution.S3OriginProperty",
        jsii_struct_bases=[],
        name_mapping={
            "domain_name": "domainName",
            "origin_access_identity": "originAccessIdentity",
        },
    )
    class S3OriginProperty:
        def __init__(
            self,
            *,
            domain_name: builtins.str,
            origin_access_identity: builtins.str,
        ) -> None:
            '''A complex type that contains information about the Amazon S3 bucket from which you want CloudFront to get your media files for distribution.

            :param domain_name: The DNS name of the Amazon S3 origin.
            :param origin_access_identity: The CloudFront origin access identity to associate with the distribution. Use an origin access identity to configure the distribution so that end users can only access objects in an Amazon S3 bucket through CloudFront. If you want end users to be able to access objects using either the CloudFront URL or the Amazon S3 URL, specify an empty ``OriginAccessIdentity`` element. To delete the origin access identity from an existing distribution, update the distribution configuration and include an empty ``OriginAccessIdentity`` element. To replace the origin access identity, update the distribution configuration and specify the new origin access identity. For more information, see `Using an Origin Access Identity to Restrict Access to Your Amazon S3 Content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-s3origin.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                s3_origin_property = cloudfront.CfnStreamingDistribution.S3OriginProperty(
                    domain_name="domainName",
                    origin_access_identity="originAccessIdentity"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__5d3332086e79ef33d790cddbbfc6fb7c8cb18fdab713859999dbd642a707c54b)
                check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
                check_type(argname="argument origin_access_identity", value=origin_access_identity, expected_type=type_hints["origin_access_identity"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "domain_name": domain_name,
                "origin_access_identity": origin_access_identity,
            }

        @builtins.property
        def domain_name(self) -> builtins.str:
            '''The DNS name of the Amazon S3 origin.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-s3origin.html#cfn-cloudfront-streamingdistribution-s3origin-domainname
            '''
            result = self._values.get("domain_name")
            assert result is not None, "Required property 'domain_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def origin_access_identity(self) -> builtins.str:
            '''The CloudFront origin access identity to associate with the distribution.

            Use an origin access identity to configure the distribution so that end users can only access objects in an Amazon S3 bucket through CloudFront.

            If you want end users to be able to access objects using either the CloudFront URL or the Amazon S3 URL, specify an empty ``OriginAccessIdentity`` element.

            To delete the origin access identity from an existing distribution, update the distribution configuration and include an empty ``OriginAccessIdentity`` element.

            To replace the origin access identity, update the distribution configuration and specify the new origin access identity.

            For more information, see `Using an Origin Access Identity to Restrict Access to Your Amazon S3 Content <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-s3origin.html#cfn-cloudfront-streamingdistribution-s3origin-originaccessidentity
            '''
            result = self._values.get("origin_access_identity")
            assert result is not None, "Required property 'origin_access_identity' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "S3OriginProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnStreamingDistribution.StreamingDistributionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "comment": "comment",
            "enabled": "enabled",
            "s3_origin": "s3Origin",
            "trusted_signers": "trustedSigners",
            "aliases": "aliases",
            "logging": "logging",
            "price_class": "priceClass",
        },
    )
    class StreamingDistributionConfigProperty:
        def __init__(
            self,
            *,
            comment: builtins.str,
            enabled: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            s3_origin: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnStreamingDistribution.S3OriginProperty", typing.Dict[builtins.str, typing.Any]]],
            trusted_signers: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnStreamingDistribution.TrustedSignersProperty", typing.Dict[builtins.str, typing.Any]]],
            aliases: typing.Optional[typing.Sequence[builtins.str]] = None,
            logging: typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union["CfnStreamingDistribution.LoggingProperty", typing.Dict[builtins.str, typing.Any]]]] = None,
            price_class: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The RTMP distribution's configuration information.

            :param comment: Any comments you want to include about the streaming distribution.
            :param enabled: Whether the streaming distribution is enabled to accept user requests for content.
            :param s3_origin: A complex type that contains information about the Amazon S3 bucket from which you want CloudFront to get your media files for distribution.
            :param trusted_signers: A complex type that specifies any AWS accounts that you want to permit to create signed URLs for private content. If you want the distribution to use signed URLs, include this element; if you want the distribution to use public URLs, remove this element. For more information, see `Serving Private Content through CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .
            :param aliases: A complex type that contains information about CNAMEs (alternate domain names), if any, for this streaming distribution.
            :param logging: A complex type that controls whether access logs are written for the streaming distribution.
            :param price_class: A complex type that contains information about price class for this streaming distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                streaming_distribution_config_property = cloudfront.CfnStreamingDistribution.StreamingDistributionConfigProperty(
                    comment="comment",
                    enabled=False,
                    s3_origin=cloudfront.CfnStreamingDistribution.S3OriginProperty(
                        domain_name="domainName",
                        origin_access_identity="originAccessIdentity"
                    ),
                    trusted_signers=cloudfront.CfnStreamingDistribution.TrustedSignersProperty(
                        enabled=False,
                
                        # the properties below are optional
                        aws_account_numbers=["awsAccountNumbers"]
                    ),
                
                    # the properties below are optional
                    aliases=["aliases"],
                    logging=cloudfront.CfnStreamingDistribution.LoggingProperty(
                        bucket="bucket",
                        enabled=False,
                        prefix="prefix"
                    ),
                    price_class="priceClass"
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__ec3d8ce3ecb8d9aaeecf2ccdaa8bd6778e0010c73112ddc0f223df3e47a3c749)
                check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument s3_origin", value=s3_origin, expected_type=type_hints["s3_origin"])
                check_type(argname="argument trusted_signers", value=trusted_signers, expected_type=type_hints["trusted_signers"])
                check_type(argname="argument aliases", value=aliases, expected_type=type_hints["aliases"])
                check_type(argname="argument logging", value=logging, expected_type=type_hints["logging"])
                check_type(argname="argument price_class", value=price_class, expected_type=type_hints["price_class"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "comment": comment,
                "enabled": enabled,
                "s3_origin": s3_origin,
                "trusted_signers": trusted_signers,
            }
            if aliases is not None:
                self._values["aliases"] = aliases
            if logging is not None:
                self._values["logging"] = logging
            if price_class is not None:
                self._values["price_class"] = price_class

        @builtins.property
        def comment(self) -> builtins.str:
            '''Any comments you want to include about the streaming distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig-comment
            '''
            result = self._values.get("comment")
            assert result is not None, "Required property 'comment' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def enabled(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''Whether the streaming distribution is enabled to accept user requests for content.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def s3_origin(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.S3OriginProperty"]:
            '''A complex type that contains information about the Amazon S3 bucket from which you want CloudFront to get your media files for distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig-s3origin
            '''
            result = self._values.get("s3_origin")
            assert result is not None, "Required property 's3_origin' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.S3OriginProperty"], result)

        @builtins.property
        def trusted_signers(
            self,
        ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.TrustedSignersProperty"]:
            '''A complex type that specifies any AWS accounts that you want to permit to create signed URLs for private content.

            If you want the distribution to use signed URLs, include this element; if you want the distribution to use public URLs, remove this element. For more information, see `Serving Private Content through CloudFront <https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html>`_ in the *Amazon CloudFront Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig-trustedsigners
            '''
            result = self._values.get("trusted_signers")
            assert result is not None, "Required property 'trusted_signers' is missing"
            return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.TrustedSignersProperty"], result)

        @builtins.property
        def aliases(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A complex type that contains information about CNAMEs (alternate domain names), if any, for this streaming distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig-aliases
            '''
            result = self._values.get("aliases")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.LoggingProperty"]]:
            '''A complex type that controls whether access logs are written for the streaming distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union[_aws_cdk_core_f4b25747.IResolvable, "CfnStreamingDistribution.LoggingProperty"]], result)

        @builtins.property
        def price_class(self) -> typing.Optional[builtins.str]:
            '''A complex type that contains information about price class for this streaming distribution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-streamingdistributionconfig.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig-priceclass
            '''
            result = self._values.get("price_class")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "StreamingDistributionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-cloudfront.CfnStreamingDistribution.TrustedSignersProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "aws_account_numbers": "awsAccountNumbers",
        },
    )
    class TrustedSignersProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable],
            aws_account_numbers: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''A list of AWS accounts whose public keys CloudFront can use to verify the signatures of signed URLs and signed cookies.

            :param enabled: This field is ``true`` if any of the AWS accounts have public keys that CloudFront can use to verify the signatures of signed URLs and signed cookies. If not, this field is ``false`` .
            :param aws_account_numbers: An AWS account number that contains active CloudFront key pairs that CloudFront can use to verify the signatures of signed URLs and signed cookies. If the AWS account that owns the key pairs is the same account that owns the CloudFront distribution, the value of this field is ``self`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-trustedsigners.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import aws_cdk.aws_cloudfront as cloudfront
                
                trusted_signers_property = cloudfront.CfnStreamingDistribution.TrustedSignersProperty(
                    enabled=False,
                
                    # the properties below are optional
                    aws_account_numbers=["awsAccountNumbers"]
                )
            '''
            if __debug__:
                type_hints = typing.get_type_hints(_typecheckingstub__550cffbc175f585e7d6d00f18c23b53e6c543af9a32aef421674b44669616eb3)
                check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
                check_type(argname="argument aws_account_numbers", value=aws_account_numbers, expected_type=type_hints["aws_account_numbers"])
            self._values: typing.Dict[builtins.str, typing.Any] = {
                "enabled": enabled,
            }
            if aws_account_numbers is not None:
                self._values["aws_account_numbers"] = aws_account_numbers

        @builtins.property
        def enabled(
            self,
        ) -> typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable]:
            '''This field is ``true`` if any of the AWS accounts have public keys that CloudFront can use to verify the signatures of signed URLs and signed cookies.

            If not, this field is ``false`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-trustedsigners.html#cfn-cloudfront-streamingdistribution-trustedsigners-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, _aws_cdk_core_f4b25747.IResolvable], result)

        @builtins.property
        def aws_account_numbers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''An AWS account number that contains active CloudFront key pairs that CloudFront can use to verify the signatures of signed URLs and signed cookies.

            If the AWS account that owns the key pairs is the same account that owns the CloudFront distribution, the value of this field is ``self`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-cloudfront-streamingdistribution-trustedsigners.html#cfn-cloudfront-streamingdistribution-trustedsigners-awsaccountnumbers
            '''
            result = self._values.get("aws_account_numbers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TrustedSignersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CfnStreamingDistributionProps",
    jsii_struct_bases=[],
    name_mapping={
        "streaming_distribution_config": "streamingDistributionConfig",
        "tags": "tags",
    },
)
class CfnStreamingDistributionProps:
    def __init__(
        self,
        *,
        streaming_distribution_config: typing.Union[_aws_cdk_core_f4b25747.IResolvable, typing.Union[CfnStreamingDistribution.StreamingDistributionConfigProperty, typing.Dict[builtins.str, typing.Any]]],
        tags: typing.Sequence[typing.Union[_aws_cdk_core_f4b25747.CfnTag, typing.Dict[builtins.str, typing.Any]]],
    ) -> None:
        '''Properties for defining a ``CfnStreamingDistribution``.

        :param streaming_distribution_config: The current configuration information for the RTMP distribution.
        :param tags: A complex type that contains zero or more ``Tag`` elements.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-streamingdistribution.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cfn_streaming_distribution_props = cloudfront.CfnStreamingDistributionProps(
                streaming_distribution_config=cloudfront.CfnStreamingDistribution.StreamingDistributionConfigProperty(
                    comment="comment",
                    enabled=False,
                    s3_origin=cloudfront.CfnStreamingDistribution.S3OriginProperty(
                        domain_name="domainName",
                        origin_access_identity="originAccessIdentity"
                    ),
                    trusted_signers=cloudfront.CfnStreamingDistribution.TrustedSignersProperty(
                        enabled=False,
            
                        # the properties below are optional
                        aws_account_numbers=["awsAccountNumbers"]
                    ),
            
                    # the properties below are optional
                    aliases=["aliases"],
                    logging=cloudfront.CfnStreamingDistribution.LoggingProperty(
                        bucket="bucket",
                        enabled=False,
                        prefix="prefix"
                    ),
                    price_class="priceClass"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__84b52aba6cc43c3837916d975a15222e4d4bbf857e61fee62b827911545bd352)
            check_type(argname="argument streaming_distribution_config", value=streaming_distribution_config, expected_type=type_hints["streaming_distribution_config"])
            check_type(argname="argument tags", value=tags, expected_type=type_hints["tags"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "streaming_distribution_config": streaming_distribution_config,
            "tags": tags,
        }

    @builtins.property
    def streaming_distribution_config(
        self,
    ) -> typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnStreamingDistribution.StreamingDistributionConfigProperty]:
        '''The current configuration information for the RTMP distribution.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-streamingdistribution.html#cfn-cloudfront-streamingdistribution-streamingdistributionconfig
        '''
        result = self._values.get("streaming_distribution_config")
        assert result is not None, "Required property 'streaming_distribution_config' is missing"
        return typing.cast(typing.Union[_aws_cdk_core_f4b25747.IResolvable, CfnStreamingDistribution.StreamingDistributionConfigProperty], result)

    @builtins.property
    def tags(self) -> typing.List[_aws_cdk_core_f4b25747.CfnTag]:
        '''A complex type that contains zero or more ``Tag`` elements.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudfront-streamingdistribution.html#cfn-cloudfront-streamingdistribution-tags
        '''
        result = self._values.get("tags")
        assert result is not None, "Required property 'tags' is missing"
        return typing.cast(typing.List[_aws_cdk_core_f4b25747.CfnTag], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnStreamingDistributionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.CloudFrontAllowedCachedMethods")
class CloudFrontAllowedCachedMethods(enum.Enum):
    '''Enums for the methods CloudFront can cache.'''

    GET_HEAD = "GET_HEAD"
    GET_HEAD_OPTIONS = "GET_HEAD_OPTIONS"


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.CloudFrontAllowedMethods")
class CloudFrontAllowedMethods(enum.Enum):
    '''An enum for the supported methods to a CloudFront distribution.'''

    GET_HEAD = "GET_HEAD"
    GET_HEAD_OPTIONS = "GET_HEAD_OPTIONS"
    ALL = "ALL"


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CloudFrontWebDistributionAttributes",
    jsii_struct_bases=[],
    name_mapping={"distribution_id": "distributionId", "domain_name": "domainName"},
)
class CloudFrontWebDistributionAttributes:
    def __init__(
        self,
        *,
        distribution_id: builtins.str,
        domain_name: builtins.str,
    ) -> None:
        '''Attributes used to import a Distribution.

        :param distribution_id: The distribution ID for this distribution.
        :param domain_name: The generated domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            cloud_front_web_distribution_attributes = cloudfront.CloudFrontWebDistributionAttributes(
                distribution_id="distributionId",
                domain_name="domainName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__b3ddc935b6219758024b22de1a28d47e2c1b7c6064518502ad36e2c0e614fe6c)
            check_type(argname="argument distribution_id", value=distribution_id, expected_type=type_hints["distribution_id"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "distribution_id": distribution_id,
            "domain_name": domain_name,
        }

    @builtins.property
    def distribution_id(self) -> builtins.str:
        '''The distribution ID for this distribution.

        :attribute: true
        '''
        result = self._values.get("distribution_id")
        assert result is not None, "Required property 'distribution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The generated domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :attribute: true
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CloudFrontWebDistributionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CloudFrontWebDistributionProps",
    jsii_struct_bases=[],
    name_mapping={
        "origin_configs": "originConfigs",
        "alias_configuration": "aliasConfiguration",
        "comment": "comment",
        "default_root_object": "defaultRootObject",
        "enabled": "enabled",
        "enable_ip_v6": "enableIpV6",
        "error_configurations": "errorConfigurations",
        "geo_restriction": "geoRestriction",
        "http_version": "httpVersion",
        "logging_config": "loggingConfig",
        "price_class": "priceClass",
        "viewer_certificate": "viewerCertificate",
        "viewer_protocol_policy": "viewerProtocolPolicy",
        "web_acl_id": "webACLId",
    },
)
class CloudFrontWebDistributionProps:
    def __init__(
        self,
        *,
        origin_configs: typing.Sequence[typing.Union["SourceConfiguration", typing.Dict[builtins.str, typing.Any]]],
        alias_configuration: typing.Optional[typing.Union[AliasConfiguration, typing.Dict[builtins.str, typing.Any]]] = None,
        comment: typing.Optional[builtins.str] = None,
        default_root_object: typing.Optional[builtins.str] = None,
        enabled: typing.Optional[builtins.bool] = None,
        enable_ip_v6: typing.Optional[builtins.bool] = None,
        error_configurations: typing.Optional[typing.Sequence[typing.Union[CfnDistribution.CustomErrorResponseProperty, typing.Dict[builtins.str, typing.Any]]]] = None,
        geo_restriction: typing.Optional["GeoRestriction"] = None,
        http_version: typing.Optional["HttpVersion"] = None,
        logging_config: typing.Optional[typing.Union["LoggingConfiguration", typing.Dict[builtins.str, typing.Any]]] = None,
        price_class: typing.Optional["PriceClass"] = None,
        viewer_certificate: typing.Optional["ViewerCertificate"] = None,
        viewer_protocol_policy: typing.Optional["ViewerProtocolPolicy"] = None,
        web_acl_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param origin_configs: The origin configurations for this distribution. Behaviors are a part of the origin.
        :param alias_configuration: (deprecated) AliasConfiguration is used to configured CloudFront to respond to requests on custom domain names. Default: - None.
        :param comment: A comment for this distribution in the CloudFront console. Default: - No comment is added to distribution.
        :param default_root_object: The default object to serve. Default: - "index.html" is served.
        :param enabled: Enable or disable the distribution. Default: true
        :param enable_ip_v6: If your distribution should have IPv6 enabled. Default: true
        :param error_configurations: How CloudFront should handle requests that are not successful (eg PageNotFound). By default, CloudFront does not replace HTTP status codes in the 4xx and 5xx range with custom error messages. CloudFront does not cache HTTP status codes. Default: - No custom error configuration.
        :param geo_restriction: Controls the countries in which your content is distributed. Default: No geo restriction
        :param http_version: The max supported HTTP Versions. Default: HttpVersion.HTTP2
        :param logging_config: Optional - if we should enable logging. You can pass an empty object ({}) to have us auto create a bucket for logging. Omission of this property indicates no logging is to be enabled. Default: - no logging is enabled by default.
        :param price_class: The price class for the distribution (this impacts how many locations CloudFront uses for your distribution, and billing). Default: PriceClass.PRICE_CLASS_100 the cheapest option for CloudFront is picked by default.
        :param viewer_certificate: Specifies whether you want viewers to use HTTP or HTTPS to request your objects, whether you're using an alternate domain name with HTTPS, and if so, if you're using AWS Certificate Manager (ACM) or a third-party certificate authority. Default: ViewerCertificate.fromCloudFrontDefaultCertificate()
        :param viewer_protocol_policy: The default viewer policy for incoming clients. Default: RedirectToHTTPs
        :param web_acl_id: Unique identifier that specifies the AWS WAF web ACL to associate with this CloudFront distribution. To specify a web ACL created using the latest version of AWS WAF, use the ACL ARN, for example ``arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a``. To specify a web ACL created using AWS WAF Classic, use the ACL ID, for example ``473e64fd-f30b-4765-81a0-62ad96dd167a``. Default: - No AWS Web Application Firewall web access control list (web ACL).

        :exampleMetadata: infused

        Example::

            # source_bucket: s3.Bucket
            
            viewer_certificate = cloudfront.ViewerCertificate.from_iam_certificate("MYIAMROLEIDENTIFIER",
                aliases=["MYALIAS"]
            )
            
            cloudfront.CloudFrontWebDistribution(self, "MyCfWebDistribution",
                origin_configs=[cloudfront.SourceConfiguration(
                    s3_origin_source=cloudfront.S3OriginConfig(
                        s3_bucket_source=source_bucket
                    ),
                    behaviors=[cloudfront.Behavior(is_default_behavior=True)]
                )
                ],
                viewer_certificate=viewer_certificate
            )
        '''
        if isinstance(alias_configuration, dict):
            alias_configuration = AliasConfiguration(**alias_configuration)
        if isinstance(logging_config, dict):
            logging_config = LoggingConfiguration(**logging_config)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__20ba4b3b8316ed2605e4c8b85c69f1afd58b2c1382683af4191b95b73d01a8c5)
            check_type(argname="argument origin_configs", value=origin_configs, expected_type=type_hints["origin_configs"])
            check_type(argname="argument alias_configuration", value=alias_configuration, expected_type=type_hints["alias_configuration"])
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument default_root_object", value=default_root_object, expected_type=type_hints["default_root_object"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument enable_ip_v6", value=enable_ip_v6, expected_type=type_hints["enable_ip_v6"])
            check_type(argname="argument error_configurations", value=error_configurations, expected_type=type_hints["error_configurations"])
            check_type(argname="argument geo_restriction", value=geo_restriction, expected_type=type_hints["geo_restriction"])
            check_type(argname="argument http_version", value=http_version, expected_type=type_hints["http_version"])
            check_type(argname="argument logging_config", value=logging_config, expected_type=type_hints["logging_config"])
            check_type(argname="argument price_class", value=price_class, expected_type=type_hints["price_class"])
            check_type(argname="argument viewer_certificate", value=viewer_certificate, expected_type=type_hints["viewer_certificate"])
            check_type(argname="argument viewer_protocol_policy", value=viewer_protocol_policy, expected_type=type_hints["viewer_protocol_policy"])
            check_type(argname="argument web_acl_id", value=web_acl_id, expected_type=type_hints["web_acl_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "origin_configs": origin_configs,
        }
        if alias_configuration is not None:
            self._values["alias_configuration"] = alias_configuration
        if comment is not None:
            self._values["comment"] = comment
        if default_root_object is not None:
            self._values["default_root_object"] = default_root_object
        if enabled is not None:
            self._values["enabled"] = enabled
        if enable_ip_v6 is not None:
            self._values["enable_ip_v6"] = enable_ip_v6
        if error_configurations is not None:
            self._values["error_configurations"] = error_configurations
        if geo_restriction is not None:
            self._values["geo_restriction"] = geo_restriction
        if http_version is not None:
            self._values["http_version"] = http_version
        if logging_config is not None:
            self._values["logging_config"] = logging_config
        if price_class is not None:
            self._values["price_class"] = price_class
        if viewer_certificate is not None:
            self._values["viewer_certificate"] = viewer_certificate
        if viewer_protocol_policy is not None:
            self._values["viewer_protocol_policy"] = viewer_protocol_policy
        if web_acl_id is not None:
            self._values["web_acl_id"] = web_acl_id

    @builtins.property
    def origin_configs(self) -> typing.List["SourceConfiguration"]:
        '''The origin configurations for this distribution.

        Behaviors are a part of the origin.
        '''
        result = self._values.get("origin_configs")
        assert result is not None, "Required property 'origin_configs' is missing"
        return typing.cast(typing.List["SourceConfiguration"], result)

    @builtins.property
    def alias_configuration(self) -> typing.Optional[AliasConfiguration]:
        '''(deprecated) AliasConfiguration is used to configured CloudFront to respond to requests on custom domain names.

        :default: - None.

        :deprecated: see {@link CloudFrontWebDistributionProps#viewerCertificate} with {@link ViewerCertificate#acmCertificate}

        :stability: deprecated
        '''
        result = self._values.get("alias_configuration")
        return typing.cast(typing.Optional[AliasConfiguration], result)

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment for this distribution in the CloudFront console.

        :default: - No comment is added to distribution.
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default_root_object(self) -> typing.Optional[builtins.str]:
        '''The default object to serve.

        :default: - "index.html" is served.
        '''
        result = self._values.get("default_root_object")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''Enable or disable the distribution.

        :default: true
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_ip_v6(self) -> typing.Optional[builtins.bool]:
        '''If your distribution should have IPv6 enabled.

        :default: true
        '''
        result = self._values.get("enable_ip_v6")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def error_configurations(
        self,
    ) -> typing.Optional[typing.List[CfnDistribution.CustomErrorResponseProperty]]:
        '''How CloudFront should handle requests that are not successful (eg PageNotFound).

        By default, CloudFront does not replace HTTP status codes in the 4xx and 5xx range
        with custom error messages. CloudFront does not cache HTTP status codes.

        :default: - No custom error configuration.
        '''
        result = self._values.get("error_configurations")
        return typing.cast(typing.Optional[typing.List[CfnDistribution.CustomErrorResponseProperty]], result)

    @builtins.property
    def geo_restriction(self) -> typing.Optional["GeoRestriction"]:
        '''Controls the countries in which your content is distributed.

        :default: No geo restriction
        '''
        result = self._values.get("geo_restriction")
        return typing.cast(typing.Optional["GeoRestriction"], result)

    @builtins.property
    def http_version(self) -> typing.Optional["HttpVersion"]:
        '''The max supported HTTP Versions.

        :default: HttpVersion.HTTP2
        '''
        result = self._values.get("http_version")
        return typing.cast(typing.Optional["HttpVersion"], result)

    @builtins.property
    def logging_config(self) -> typing.Optional["LoggingConfiguration"]:
        '''Optional - if we should enable logging.

        You can pass an empty object ({}) to have us auto create a bucket for logging.
        Omission of this property indicates no logging is to be enabled.

        :default: - no logging is enabled by default.
        '''
        result = self._values.get("logging_config")
        return typing.cast(typing.Optional["LoggingConfiguration"], result)

    @builtins.property
    def price_class(self) -> typing.Optional["PriceClass"]:
        '''The price class for the distribution (this impacts how many locations CloudFront uses for your distribution, and billing).

        :default: PriceClass.PRICE_CLASS_100 the cheapest option for CloudFront is picked by default.
        '''
        result = self._values.get("price_class")
        return typing.cast(typing.Optional["PriceClass"], result)

    @builtins.property
    def viewer_certificate(self) -> typing.Optional["ViewerCertificate"]:
        '''Specifies whether you want viewers to use HTTP or HTTPS to request your objects, whether you're using an alternate domain name with HTTPS, and if so, if you're using AWS Certificate Manager (ACM) or a third-party certificate authority.

        :default: ViewerCertificate.fromCloudFrontDefaultCertificate()

        :see: https://aws.amazon.com/premiumsupport/knowledge-center/custom-ssl-certificate-cloudfront/
        '''
        result = self._values.get("viewer_certificate")
        return typing.cast(typing.Optional["ViewerCertificate"], result)

    @builtins.property
    def viewer_protocol_policy(self) -> typing.Optional["ViewerProtocolPolicy"]:
        '''The default viewer policy for incoming clients.

        :default: RedirectToHTTPs
        '''
        result = self._values.get("viewer_protocol_policy")
        return typing.cast(typing.Optional["ViewerProtocolPolicy"], result)

    @builtins.property
    def web_acl_id(self) -> typing.Optional[builtins.str]:
        '''Unique identifier that specifies the AWS WAF web ACL to associate with this CloudFront distribution.

        To specify a web ACL created using the latest version of AWS WAF, use the ACL ARN, for example
        ``arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a``.

        To specify a web ACL created using AWS WAF Classic, use the ACL ID, for example ``473e64fd-f30b-4765-81a0-62ad96dd167a``.

        :default: - No AWS Web Application Firewall web access control list (web ACL).

        :see: https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html#API_CreateDistribution_RequestParameters.
        '''
        result = self._values.get("web_acl_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CloudFrontWebDistributionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.CustomOriginConfig",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "allowed_origin_ssl_versions": "allowedOriginSSLVersions",
        "http_port": "httpPort",
        "https_port": "httpsPort",
        "origin_headers": "originHeaders",
        "origin_keepalive_timeout": "originKeepaliveTimeout",
        "origin_path": "originPath",
        "origin_protocol_policy": "originProtocolPolicy",
        "origin_read_timeout": "originReadTimeout",
        "origin_shield_region": "originShieldRegion",
    },
)
class CustomOriginConfig:
    def __init__(
        self,
        *,
        domain_name: builtins.str,
        allowed_origin_ssl_versions: typing.Optional[typing.Sequence["OriginSslPolicy"]] = None,
        http_port: typing.Optional[jsii.Number] = None,
        https_port: typing.Optional[jsii.Number] = None,
        origin_headers: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        origin_keepalive_timeout: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        origin_path: typing.Optional[builtins.str] = None,
        origin_protocol_policy: typing.Optional["OriginProtocolPolicy"] = None,
        origin_read_timeout: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
        origin_shield_region: typing.Optional[builtins.str] = None,
    ) -> None:
        '''A custom origin configuration.

        :param domain_name: The domain name of the custom origin. Should not include the path - that should be in the parent SourceConfiguration
        :param allowed_origin_ssl_versions: The SSL versions to use when interacting with the origin. Default: OriginSslPolicy.TLS_V1_2
        :param http_port: The origin HTTP port. Default: 80
        :param https_port: The origin HTTPS port. Default: 443
        :param origin_headers: Any additional headers to pass to the origin. Default: - No additional headers are passed.
        :param origin_keepalive_timeout: The keep alive timeout when making calls in seconds. Default: Duration.seconds(5)
        :param origin_path: The relative path to the origin root to use for sources. Default: /
        :param origin_protocol_policy: The protocol (http or https) policy to use when interacting with the origin. Default: OriginProtocolPolicy.HttpsOnly
        :param origin_read_timeout: The read timeout when calling the origin in seconds. Default: Duration.seconds(30)
        :param origin_shield_region: When you enable Origin Shield in the AWS Region that has the lowest latency to your origin, you can get better network performance. Default: - origin shield not enabled

        :exampleMetadata: infused

        Example::

            # source_bucket: s3.Bucket
            # oai: cloudfront.OriginAccessIdentity
            
            
            cloudfront.CloudFrontWebDistribution(self, "MyCfWebDistribution",
                origin_configs=[cloudfront.SourceConfiguration(
                    s3_origin_source=cloudfront.S3OriginConfig(
                        s3_bucket_source=source_bucket,
                        origin_access_identity=oai
                    ),
                    behaviors=[cloudfront.Behavior(is_default_behavior=True)]
                ), cloudfront.SourceConfiguration(
                    custom_origin_source=cloudfront.CustomOriginConfig(
                        domain_name="MYALIAS"
                    ),
                    behaviors=[cloudfront.Behavior(path_pattern="/somewhere")]
                )
                ]
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__9dc5b1aaaefa5f9668bac796c095d6e35b0c332e8a5ad11e48ca32f6abe6b65a)
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
            check_type(argname="argument allowed_origin_ssl_versions", value=allowed_origin_ssl_versions, expected_type=type_hints["allowed_origin_ssl_versions"])
            check_type(argname="argument http_port", value=http_port, expected_type=type_hints["http_port"])
            check_type(argname="argument https_port", value=https_port, expected_type=type_hints["https_port"])
            check_type(argname="argument origin_headers", value=origin_headers, expected_type=type_hints["origin_headers"])
            check_type(argname="argument origin_keepalive_timeout", value=origin_keepalive_timeout, expected_type=type_hints["origin_keepalive_timeout"])
            check_type(argname="argument origin_path", value=origin_path, expected_type=type_hints["origin_path"])
            check_type(argname="argument origin_protocol_policy", value=origin_protocol_policy, expected_type=type_hints["origin_protocol_policy"])
            check_type(argname="argument origin_read_timeout", value=origin_read_timeout, expected_type=type_hints["origin_read_timeout"])
            check_type(argname="argument origin_shield_region", value=origin_shield_region, expected_type=type_hints["origin_shield_region"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "domain_name": domain_name,
        }
        if allowed_origin_ssl_versions is not None:
            self._values["allowed_origin_ssl_versions"] = allowed_origin_ssl_versions
        if http_port is not None:
            self._values["http_port"] = http_port
        if https_port is not None:
            self._values["https_port"] = https_port
        if origin_headers is not None:
            self._values["origin_headers"] = origin_headers
        if origin_keepalive_timeout is not None:
            self._values["origin_keepalive_timeout"] = origin_keepalive_timeout
        if origin_path is not None:
            self._values["origin_path"] = origin_path
        if origin_protocol_policy is not None:
            self._values["origin_protocol_policy"] = origin_protocol_policy
        if origin_read_timeout is not None:
            self._values["origin_read_timeout"] = origin_read_timeout
        if origin_shield_region is not None:
            self._values["origin_shield_region"] = origin_shield_region

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The domain name of the custom origin.

        Should not include the path - that should be in the parent SourceConfiguration
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def allowed_origin_ssl_versions(
        self,
    ) -> typing.Optional[typing.List["OriginSslPolicy"]]:
        '''The SSL versions to use when interacting with the origin.

        :default: OriginSslPolicy.TLS_V1_2
        '''
        result = self._values.get("allowed_origin_ssl_versions")
        return typing.cast(typing.Optional[typing.List["OriginSslPolicy"]], result)

    @builtins.property
    def http_port(self) -> typing.Optional[jsii.Number]:
        '''The origin HTTP port.

        :default: 80
        '''
        result = self._values.get("http_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def https_port(self) -> typing.Optional[jsii.Number]:
        '''The origin HTTPS port.

        :default: 443
        '''
        result = self._values.get("https_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def origin_headers(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''Any additional headers to pass to the origin.

        :default: - No additional headers are passed.
        '''
        result = self._values.get("origin_headers")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def origin_keepalive_timeout(
        self,
    ) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The keep alive timeout when making calls in seconds.

        :default: Duration.seconds(5)
        '''
        result = self._values.get("origin_keepalive_timeout")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def origin_path(self) -> typing.Optional[builtins.str]:
        '''The relative path to the origin root to use for sources.

        :default: /
        '''
        result = self._values.get("origin_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def origin_protocol_policy(self) -> typing.Optional["OriginProtocolPolicy"]:
        '''The protocol (http or https) policy to use when interacting with the origin.

        :default: OriginProtocolPolicy.HttpsOnly
        '''
        result = self._values.get("origin_protocol_policy")
        return typing.cast(typing.Optional["OriginProtocolPolicy"], result)

    @builtins.property
    def origin_read_timeout(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The read timeout when calling the origin in seconds.

        :default: Duration.seconds(30)
        '''
        result = self._values.get("origin_read_timeout")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    @builtins.property
    def origin_shield_region(self) -> typing.Optional[builtins.str]:
        '''When you enable Origin Shield in the AWS Region that has the lowest latency to your origin, you can get better network performance.

        :default: - origin shield not enabled
        '''
        result = self._values.get("origin_shield_region")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CustomOriginConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.DistributionAttributes",
    jsii_struct_bases=[],
    name_mapping={"distribution_id": "distributionId", "domain_name": "domainName"},
)
class DistributionAttributes:
    def __init__(
        self,
        *,
        distribution_id: builtins.str,
        domain_name: builtins.str,
    ) -> None:
        '''Attributes used to import a Distribution.

        :param distribution_id: The distribution ID for this distribution.
        :param domain_name: The generated domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :exampleMetadata: infused

        Example::

            # Using a reference to an imported Distribution
            distribution = cloudfront.Distribution.from_distribution_attributes(self, "ImportedDist",
                domain_name="d111111abcdef8.cloudfront.net",
                distribution_id="012345ABCDEF"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__e4d3022f5e906d489d1c6557395efcb34ee775909d2976a2539cebf74a79e11a)
            check_type(argname="argument distribution_id", value=distribution_id, expected_type=type_hints["distribution_id"])
            check_type(argname="argument domain_name", value=domain_name, expected_type=type_hints["domain_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "distribution_id": distribution_id,
            "domain_name": domain_name,
        }

    @builtins.property
    def distribution_id(self) -> builtins.str:
        '''The distribution ID for this distribution.

        :attribute: true
        '''
        result = self._values.get("distribution_id")
        assert result is not None, "Required property 'distribution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def domain_name(self) -> builtins.str:
        '''The generated domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :attribute: true
        '''
        result = self._values.get("domain_name")
        assert result is not None, "Required property 'domain_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DistributionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.DistributionProps",
    jsii_struct_bases=[],
    name_mapping={
        "default_behavior": "defaultBehavior",
        "additional_behaviors": "additionalBehaviors",
        "certificate": "certificate",
        "comment": "comment",
        "default_root_object": "defaultRootObject",
        "domain_names": "domainNames",
        "enabled": "enabled",
        "enable_ipv6": "enableIpv6",
        "enable_logging": "enableLogging",
        "error_responses": "errorResponses",
        "geo_restriction": "geoRestriction",
        "http_version": "httpVersion",
        "log_bucket": "logBucket",
        "log_file_prefix": "logFilePrefix",
        "log_includes_cookies": "logIncludesCookies",
        "minimum_protocol_version": "minimumProtocolVersion",
        "price_class": "priceClass",
        "ssl_support_method": "sslSupportMethod",
        "web_acl_id": "webAclId",
    },
)
class DistributionProps:
    def __init__(
        self,
        *,
        default_behavior: typing.Union[BehaviorOptions, typing.Dict[builtins.str, typing.Any]],
        additional_behaviors: typing.Optional[typing.Mapping[builtins.str, typing.Union[BehaviorOptions, typing.Dict[builtins.str, typing.Any]]]] = None,
        certificate: typing.Optional[_aws_cdk_aws_certificatemanager_1662be0d.ICertificate] = None,
        comment: typing.Optional[builtins.str] = None,
        default_root_object: typing.Optional[builtins.str] = None,
        domain_names: typing.Optional[typing.Sequence[builtins.str]] = None,
        enabled: typing.Optional[builtins.bool] = None,
        enable_ipv6: typing.Optional[builtins.bool] = None,
        enable_logging: typing.Optional[builtins.bool] = None,
        error_responses: typing.Optional[typing.Sequence[typing.Union["ErrorResponse", typing.Dict[builtins.str, typing.Any]]]] = None,
        geo_restriction: typing.Optional["GeoRestriction"] = None,
        http_version: typing.Optional["HttpVersion"] = None,
        log_bucket: typing.Optional[_aws_cdk_aws_s3_55f001a5.IBucket] = None,
        log_file_prefix: typing.Optional[builtins.str] = None,
        log_includes_cookies: typing.Optional[builtins.bool] = None,
        minimum_protocol_version: typing.Optional["SecurityPolicyProtocol"] = None,
        price_class: typing.Optional["PriceClass"] = None,
        ssl_support_method: typing.Optional["SSLMethod"] = None,
        web_acl_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for a Distribution.

        :param default_behavior: The default behavior for the distribution.
        :param additional_behaviors: Additional behaviors for the distribution, mapped by the pathPattern that specifies which requests to apply the behavior to. Default: - no additional behaviors are added.
        :param certificate: A certificate to associate with the distribution. The certificate must be located in N. Virginia (us-east-1). Default: - the CloudFront wildcard certificate (*.cloudfront.net) will be used.
        :param comment: Any comments you want to include about the distribution. Default: - no comment
        :param default_root_object: The object that you want CloudFront to request from your origin (for example, index.html) when a viewer requests the root URL for your distribution. If no default object is set, the request goes to the origin's root (e.g., example.com/). Default: - no default root object
        :param domain_names: Alternative domain names for this distribution. If you want to use your own domain name, such as www.example.com, instead of the cloudfront.net domain name, you can add an alternate domain name to your distribution. If you attach a certificate to the distribution, you must add (at least one of) the domain names of the certificate to this list. Default: - The distribution will only support the default generated name (e.g., d111111abcdef8.cloudfront.net)
        :param enabled: Enable or disable the distribution. Default: true
        :param enable_ipv6: Whether CloudFront will respond to IPv6 DNS requests with an IPv6 address. If you specify false, CloudFront responds to IPv6 DNS requests with the DNS response code NOERROR and with no IP addresses. This allows viewers to submit a second request, for an IPv4 address for your distribution. Default: true
        :param enable_logging: Enable access logging for the distribution. Default: - false, unless ``logBucket`` is specified.
        :param error_responses: How CloudFront should handle requests that are not successful (e.g., PageNotFound). Default: - No custom error responses.
        :param geo_restriction: Controls the countries in which your content is distributed. Default: - No geographic restrictions
        :param http_version: Specify the maximum HTTP version that you want viewers to use to communicate with CloudFront. For viewers and CloudFront to use HTTP/2, viewers must support TLS 1.2 or later, and must support server name identification (SNI). Default: HttpVersion.HTTP2
        :param log_bucket: The Amazon S3 bucket to store the access logs in. Default: - A bucket is created if ``enableLogging`` is true
        :param log_file_prefix: An optional string that you want CloudFront to prefix to the access log filenames for this distribution. Default: - no prefix
        :param log_includes_cookies: Specifies whether you want CloudFront to include cookies in access logs. Default: false
        :param minimum_protocol_version: The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections. CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify. Default: - SecurityPolicyProtocol.TLS_V1_2_2021 if the '
        :param price_class: The price class that corresponds with the maximum price that you want to pay for CloudFront service. If you specify PriceClass_All, CloudFront responds to requests for your objects from all CloudFront edge locations. If you specify a price class other than PriceClass_All, CloudFront serves your objects from the CloudFront edge location that has the lowest latency among the edge locations in your price class. Default: PriceClass.PRICE_CLASS_ALL
        :param ssl_support_method: The SSL method CloudFront will use for your distribution. Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate. CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support. If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP). See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/ Default: SSLMethod.SNI
        :param web_acl_id: Unique identifier that specifies the AWS WAF web ACL to associate with this CloudFront distribution. To specify a web ACL created using the latest version of AWS WAF, use the ACL ARN, for example ``arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a``. To specify a web ACL created using AWS WAF Classic, use the ACL ID, for example ``473e64fd-f30b-4765-81a0-62ad96dd167a``. Default: - No AWS Web Application Firewall web access control list (web ACL).

        :exampleMetadata: infused

        Example::

            # Adding an existing Lambda@Edge function created in a different stack
            # to a CloudFront distribution.
            # s3_bucket: s3.Bucket
            
            function_version = lambda_.Version.from_version_arn(self, "Version", "arn:aws:lambda:us-east-1:123456789012:function:functionName:1")
            
            cloudfront.Distribution(self, "distro",
                default_behavior=cloudfront.BehaviorOptions(
                    origin=origins.S3Origin(s3_bucket),
                    edge_lambdas=[cloudfront.EdgeLambda(
                        function_version=function_version,
                        event_type=cloudfront.LambdaEdgeEventType.VIEWER_REQUEST
                    )
                    ]
                )
            )
        '''
        if isinstance(default_behavior, dict):
            default_behavior = BehaviorOptions(**default_behavior)
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__c4fa896441126c2a75c39d52abe341d70818bd495d2a3f91f7685252c4b90047)
            check_type(argname="argument default_behavior", value=default_behavior, expected_type=type_hints["default_behavior"])
            check_type(argname="argument additional_behaviors", value=additional_behaviors, expected_type=type_hints["additional_behaviors"])
            check_type(argname="argument certificate", value=certificate, expected_type=type_hints["certificate"])
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument default_root_object", value=default_root_object, expected_type=type_hints["default_root_object"])
            check_type(argname="argument domain_names", value=domain_names, expected_type=type_hints["domain_names"])
            check_type(argname="argument enabled", value=enabled, expected_type=type_hints["enabled"])
            check_type(argname="argument enable_ipv6", value=enable_ipv6, expected_type=type_hints["enable_ipv6"])
            check_type(argname="argument enable_logging", value=enable_logging, expected_type=type_hints["enable_logging"])
            check_type(argname="argument error_responses", value=error_responses, expected_type=type_hints["error_responses"])
            check_type(argname="argument geo_restriction", value=geo_restriction, expected_type=type_hints["geo_restriction"])
            check_type(argname="argument http_version", value=http_version, expected_type=type_hints["http_version"])
            check_type(argname="argument log_bucket", value=log_bucket, expected_type=type_hints["log_bucket"])
            check_type(argname="argument log_file_prefix", value=log_file_prefix, expected_type=type_hints["log_file_prefix"])
            check_type(argname="argument log_includes_cookies", value=log_includes_cookies, expected_type=type_hints["log_includes_cookies"])
            check_type(argname="argument minimum_protocol_version", value=minimum_protocol_version, expected_type=type_hints["minimum_protocol_version"])
            check_type(argname="argument price_class", value=price_class, expected_type=type_hints["price_class"])
            check_type(argname="argument ssl_support_method", value=ssl_support_method, expected_type=type_hints["ssl_support_method"])
            check_type(argname="argument web_acl_id", value=web_acl_id, expected_type=type_hints["web_acl_id"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "default_behavior": default_behavior,
        }
        if additional_behaviors is not None:
            self._values["additional_behaviors"] = additional_behaviors
        if certificate is not None:
            self._values["certificate"] = certificate
        if comment is not None:
            self._values["comment"] = comment
        if default_root_object is not None:
            self._values["default_root_object"] = default_root_object
        if domain_names is not None:
            self._values["domain_names"] = domain_names
        if enabled is not None:
            self._values["enabled"] = enabled
        if enable_ipv6 is not None:
            self._values["enable_ipv6"] = enable_ipv6
        if enable_logging is not None:
            self._values["enable_logging"] = enable_logging
        if error_responses is not None:
            self._values["error_responses"] = error_responses
        if geo_restriction is not None:
            self._values["geo_restriction"] = geo_restriction
        if http_version is not None:
            self._values["http_version"] = http_version
        if log_bucket is not None:
            self._values["log_bucket"] = log_bucket
        if log_file_prefix is not None:
            self._values["log_file_prefix"] = log_file_prefix
        if log_includes_cookies is not None:
            self._values["log_includes_cookies"] = log_includes_cookies
        if minimum_protocol_version is not None:
            self._values["minimum_protocol_version"] = minimum_protocol_version
        if price_class is not None:
            self._values["price_class"] = price_class
        if ssl_support_method is not None:
            self._values["ssl_support_method"] = ssl_support_method
        if web_acl_id is not None:
            self._values["web_acl_id"] = web_acl_id

    @builtins.property
    def default_behavior(self) -> BehaviorOptions:
        '''The default behavior for the distribution.'''
        result = self._values.get("default_behavior")
        assert result is not None, "Required property 'default_behavior' is missing"
        return typing.cast(BehaviorOptions, result)

    @builtins.property
    def additional_behaviors(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, BehaviorOptions]]:
        '''Additional behaviors for the distribution, mapped by the pathPattern that specifies which requests to apply the behavior to.

        :default: - no additional behaviors are added.
        '''
        result = self._values.get("additional_behaviors")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, BehaviorOptions]], result)

    @builtins.property
    def certificate(
        self,
    ) -> typing.Optional[_aws_cdk_aws_certificatemanager_1662be0d.ICertificate]:
        '''A certificate to associate with the distribution.

        The certificate must be located in N. Virginia (us-east-1).

        :default: - the CloudFront wildcard certificate (*.cloudfront.net) will be used.
        '''
        result = self._values.get("certificate")
        return typing.cast(typing.Optional[_aws_cdk_aws_certificatemanager_1662be0d.ICertificate], result)

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''Any comments you want to include about the distribution.

        :default: - no comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default_root_object(self) -> typing.Optional[builtins.str]:
        '''The object that you want CloudFront to request from your origin (for example, index.html) when a viewer requests the root URL for your distribution. If no default object is set, the request goes to the origin's root (e.g., example.com/).

        :default: - no default root object
        '''
        result = self._values.get("default_root_object")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_names(self) -> typing.Optional[typing.List[builtins.str]]:
        '''Alternative domain names for this distribution.

        If you want to use your own domain name, such as www.example.com, instead of the cloudfront.net domain name,
        you can add an alternate domain name to your distribution. If you attach a certificate to the distribution,
        you must add (at least one of) the domain names of the certificate to this list.

        :default: - The distribution will only support the default generated name (e.g., d111111abcdef8.cloudfront.net)
        '''
        result = self._values.get("domain_names")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def enabled(self) -> typing.Optional[builtins.bool]:
        '''Enable or disable the distribution.

        :default: true
        '''
        result = self._values.get("enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_ipv6(self) -> typing.Optional[builtins.bool]:
        '''Whether CloudFront will respond to IPv6 DNS requests with an IPv6 address.

        If you specify false, CloudFront responds to IPv6 DNS requests with the DNS response code NOERROR and with no IP addresses.
        This allows viewers to submit a second request, for an IPv4 address for your distribution.

        :default: true
        '''
        result = self._values.get("enable_ipv6")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_logging(self) -> typing.Optional[builtins.bool]:
        '''Enable access logging for the distribution.

        :default: - false, unless ``logBucket`` is specified.
        '''
        result = self._values.get("enable_logging")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def error_responses(self) -> typing.Optional[typing.List["ErrorResponse"]]:
        '''How CloudFront should handle requests that are not successful (e.g., PageNotFound).

        :default: - No custom error responses.
        '''
        result = self._values.get("error_responses")
        return typing.cast(typing.Optional[typing.List["ErrorResponse"]], result)

    @builtins.property
    def geo_restriction(self) -> typing.Optional["GeoRestriction"]:
        '''Controls the countries in which your content is distributed.

        :default: - No geographic restrictions
        '''
        result = self._values.get("geo_restriction")
        return typing.cast(typing.Optional["GeoRestriction"], result)

    @builtins.property
    def http_version(self) -> typing.Optional["HttpVersion"]:
        '''Specify the maximum HTTP version that you want viewers to use to communicate with CloudFront.

        For viewers and CloudFront to use HTTP/2, viewers must support TLS 1.2 or later, and must support server name identification (SNI).

        :default: HttpVersion.HTTP2
        '''
        result = self._values.get("http_version")
        return typing.cast(typing.Optional["HttpVersion"], result)

    @builtins.property
    def log_bucket(self) -> typing.Optional[_aws_cdk_aws_s3_55f001a5.IBucket]:
        '''The Amazon S3 bucket to store the access logs in.

        :default: - A bucket is created if ``enableLogging`` is true
        '''
        result = self._values.get("log_bucket")
        return typing.cast(typing.Optional[_aws_cdk_aws_s3_55f001a5.IBucket], result)

    @builtins.property
    def log_file_prefix(self) -> typing.Optional[builtins.str]:
        '''An optional string that you want CloudFront to prefix to the access log filenames for this distribution.

        :default: - no prefix
        '''
        result = self._values.get("log_file_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_includes_cookies(self) -> typing.Optional[builtins.bool]:
        '''Specifies whether you want CloudFront to include cookies in access logs.

        :default: false
        '''
        result = self._values.get("log_includes_cookies")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def minimum_protocol_version(self) -> typing.Optional["SecurityPolicyProtocol"]:
        '''The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.

        CloudFront serves your objects only to browsers or devices that support at
        least the SSL version that you specify.

        :default: - SecurityPolicyProtocol.TLS_V1_2_2021 if the '

        :aws-cdk: /aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021' feature flag is set; otherwise, SecurityPolicyProtocol.TLS_V1_2_2019.
        '''
        result = self._values.get("minimum_protocol_version")
        return typing.cast(typing.Optional["SecurityPolicyProtocol"], result)

    @builtins.property
    def price_class(self) -> typing.Optional["PriceClass"]:
        '''The price class that corresponds with the maximum price that you want to pay for CloudFront service.

        If you specify PriceClass_All, CloudFront responds to requests for your objects from all CloudFront edge locations.
        If you specify a price class other than PriceClass_All, CloudFront serves your objects from the CloudFront edge location
        that has the lowest latency among the edge locations in your price class.

        :default: PriceClass.PRICE_CLASS_ALL
        '''
        result = self._values.get("price_class")
        return typing.cast(typing.Optional["PriceClass"], result)

    @builtins.property
    def ssl_support_method(self) -> typing.Optional["SSLMethod"]:
        '''The SSL method CloudFront will use for your distribution.

        Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
        which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
        multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
        (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.

        CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.

        If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
        using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).

        See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/

        :default: SSLMethod.SNI
        '''
        result = self._values.get("ssl_support_method")
        return typing.cast(typing.Optional["SSLMethod"], result)

    @builtins.property
    def web_acl_id(self) -> typing.Optional[builtins.str]:
        '''Unique identifier that specifies the AWS WAF web ACL to associate with this CloudFront distribution.

        To specify a web ACL created using the latest version of AWS WAF, use the ACL ARN, for example
        ``arn:aws:wafv2:us-east-1:123456789012:global/webacl/ExampleWebACL/473e64fd-f30b-4765-81a0-62ad96dd167a``.
        To specify a web ACL created using AWS WAF Classic, use the ACL ID, for example ``473e64fd-f30b-4765-81a0-62ad96dd167a``.

        :default: - No AWS Web Application Firewall web access control list (web ACL).

        :see: https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html#API_CreateDistribution_RequestParameters.
        '''
        result = self._values.get("web_acl_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DistributionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.EdgeLambda",
    jsii_struct_bases=[],
    name_mapping={
        "event_type": "eventType",
        "function_version": "functionVersion",
        "include_body": "includeBody",
    },
)
class EdgeLambda:
    def __init__(
        self,
        *,
        event_type: "LambdaEdgeEventType",
        function_version: _aws_cdk_aws_lambda_5443dbc3.IVersion,
        include_body: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Represents a Lambda function version and event type when using Lambda@Edge.

        The type of the {@link AddBehaviorOptions.edgeLambdas} property.

        :param event_type: The type of event in response to which should the function be invoked.
        :param function_version: The version of the Lambda function that will be invoked. **Note**: it's not possible to use the '$LATEST' function version for Lambda@Edge!
        :param include_body: Allows a Lambda function to have read access to the body content. Only valid for "request" event types (``ORIGIN_REQUEST`` or ``VIEWER_REQUEST``). See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-include-body-access.html Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            import aws_cdk.aws_lambda as lambda_
            
            # version: lambda.Version
            
            edge_lambda = cloudfront.EdgeLambda(
                event_type=cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                function_version=version,
            
                # the properties below are optional
                include_body=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__6efc4225db31fee873448bf4cae31d528e701dc5d45a80cfe3a437227509dea3)
            check_type(argname="argument event_type", value=event_type, expected_type=type_hints["event_type"])
            check_type(argname="argument function_version", value=function_version, expected_type=type_hints["function_version"])
            check_type(argname="argument include_body", value=include_body, expected_type=type_hints["include_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "event_type": event_type,
            "function_version": function_version,
        }
        if include_body is not None:
            self._values["include_body"] = include_body

    @builtins.property
    def event_type(self) -> "LambdaEdgeEventType":
        '''The type of event in response to which should the function be invoked.'''
        result = self._values.get("event_type")
        assert result is not None, "Required property 'event_type' is missing"
        return typing.cast("LambdaEdgeEventType", result)

    @builtins.property
    def function_version(self) -> _aws_cdk_aws_lambda_5443dbc3.IVersion:
        '''The version of the Lambda function that will be invoked.

        **Note**: it's not possible to use the '$LATEST' function version for Lambda@Edge!
        '''
        result = self._values.get("function_version")
        assert result is not None, "Required property 'function_version' is missing"
        return typing.cast(_aws_cdk_aws_lambda_5443dbc3.IVersion, result)

    @builtins.property
    def include_body(self) -> typing.Optional[builtins.bool]:
        '''Allows a Lambda function to have read access to the body content.

        Only valid for "request" event types (``ORIGIN_REQUEST`` or ``VIEWER_REQUEST``).
        See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-include-body-access.html

        :default: false
        '''
        result = self._values.get("include_body")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EdgeLambda(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.ErrorResponse",
    jsii_struct_bases=[],
    name_mapping={
        "http_status": "httpStatus",
        "response_http_status": "responseHttpStatus",
        "response_page_path": "responsePagePath",
        "ttl": "ttl",
    },
)
class ErrorResponse:
    def __init__(
        self,
        *,
        http_status: jsii.Number,
        response_http_status: typing.Optional[jsii.Number] = None,
        response_page_path: typing.Optional[builtins.str] = None,
        ttl: typing.Optional[_aws_cdk_core_f4b25747.Duration] = None,
    ) -> None:
        '''Options for configuring custom error responses.

        :param http_status: The HTTP status code for which you want to specify a custom error page and/or a caching duration.
        :param response_http_status: The HTTP status code that you want CloudFront to return to the viewer along with the custom error page. If you specify a value for ``responseHttpStatus``, you must also specify a value for ``responsePagePath``. Default: - the error code will be returned as the response code.
        :param response_page_path: The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the ``httpStatus``, for example, /4xx-errors/403-forbidden.html. Default: - the default CloudFront response is shown.
        :param ttl: The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in ErrorCode. Default: - the default caching TTL behavior applies

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            import aws_cdk.core as cdk
            
            error_response = cloudfront.ErrorResponse(
                http_status=123,
            
                # the properties below are optional
                response_http_status=123,
                response_page_path="responsePagePath",
                ttl=cdk.Duration.minutes(30)
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a3c34575cc8e7dc6cf96c31dd4daafffdde3ace578090b3de048b8572ab98273)
            check_type(argname="argument http_status", value=http_status, expected_type=type_hints["http_status"])
            check_type(argname="argument response_http_status", value=response_http_status, expected_type=type_hints["response_http_status"])
            check_type(argname="argument response_page_path", value=response_page_path, expected_type=type_hints["response_page_path"])
            check_type(argname="argument ttl", value=ttl, expected_type=type_hints["ttl"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "http_status": http_status,
        }
        if response_http_status is not None:
            self._values["response_http_status"] = response_http_status
        if response_page_path is not None:
            self._values["response_page_path"] = response_page_path
        if ttl is not None:
            self._values["ttl"] = ttl

    @builtins.property
    def http_status(self) -> jsii.Number:
        '''The HTTP status code for which you want to specify a custom error page and/or a caching duration.'''
        result = self._values.get("http_status")
        assert result is not None, "Required property 'http_status' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def response_http_status(self) -> typing.Optional[jsii.Number]:
        '''The HTTP status code that you want CloudFront to return to the viewer along with the custom error page.

        If you specify a value for ``responseHttpStatus``, you must also specify a value for ``responsePagePath``.

        :default: - the error code will be returned as the response code.
        '''
        result = self._values.get("response_http_status")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def response_page_path(self) -> typing.Optional[builtins.str]:
        '''The path to the custom error page that you want CloudFront to return to a viewer when your origin returns the ``httpStatus``, for example, /4xx-errors/403-forbidden.html.

        :default: - the default CloudFront response is shown.
        '''
        result = self._values.get("response_page_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ttl(self) -> typing.Optional[_aws_cdk_core_f4b25747.Duration]:
        '''The minimum amount of time, in seconds, that you want CloudFront to cache the HTTP status code specified in ErrorCode.

        :default: - the default caching TTL behavior applies
        '''
        result = self._values.get("ttl")
        return typing.cast(typing.Optional[_aws_cdk_core_f4b25747.Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ErrorResponse(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.FailoverStatusCode")
class FailoverStatusCode(enum.Enum):
    '''HTTP status code to failover to second origin.

    :exampleMetadata: infused

    Example::

        # Configuring origin fallback options for the CloudFrontWebDistribution
        cloudfront.CloudFrontWebDistribution(self, "ADistribution",
            origin_configs=[cloudfront.SourceConfiguration(
                s3_origin_source=cloudfront.S3OriginConfig(
                    s3_bucket_source=s3.Bucket.from_bucket_name(self, "aBucket", "myoriginbucket"),
                    origin_path="/",
                    origin_headers={
                        "myHeader": "42"
                    },
                    origin_shield_region="us-west-2"
                ),
                failover_s3_origin_source=cloudfront.S3OriginConfig(
                    s3_bucket_source=s3.Bucket.from_bucket_name(self, "aBucketFallback", "myoriginbucketfallback"),
                    origin_path="/somewhere",
                    origin_headers={
                        "myHeader2": "21"
                    },
                    origin_shield_region="us-east-1"
                ),
                failover_criteria_status_codes=[cloudfront.FailoverStatusCode.INTERNAL_SERVER_ERROR],
                behaviors=[cloudfront.Behavior(
                    is_default_behavior=True
                )
                ]
            )
            ]
        )
    '''

    FORBIDDEN = "FORBIDDEN"
    '''Forbidden (403).'''
    NOT_FOUND = "NOT_FOUND"
    '''Not found (404).'''
    INTERNAL_SERVER_ERROR = "INTERNAL_SERVER_ERROR"
    '''Internal Server Error (500).'''
    BAD_GATEWAY = "BAD_GATEWAY"
    '''Bad Gateway (502).'''
    SERVICE_UNAVAILABLE = "SERVICE_UNAVAILABLE"
    '''Service Unavailable (503).'''
    GATEWAY_TIMEOUT = "GATEWAY_TIMEOUT"
    '''Gateway Timeout (504).'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.FileCodeOptions",
    jsii_struct_bases=[],
    name_mapping={"file_path": "filePath"},
)
class FileCodeOptions:
    def __init__(self, *, file_path: builtins.str) -> None:
        '''Options when reading the function's code from an external file.

        :param file_path: The path of the file to read the code from.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            file_code_options = cloudfront.FileCodeOptions(
                file_path="filePath"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__091217409847d8f69c0d11a088ed2e7571b87f18ff52b19baedb7c9bb16f8f20)
            check_type(argname="argument file_path", value=file_path, expected_type=type_hints["file_path"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "file_path": file_path,
        }

    @builtins.property
    def file_path(self) -> builtins.str:
        '''The path of the file to read the code from.'''
        result = self._values.get("file_path")
        assert result is not None, "Required property 'file_path' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FileCodeOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.FunctionAssociation",
    jsii_struct_bases=[],
    name_mapping={"event_type": "eventType", "function": "function"},
)
class FunctionAssociation:
    def __init__(
        self,
        *,
        event_type: "FunctionEventType",
        function: "IFunction",
    ) -> None:
        '''Represents a CloudFront function and event type when using CF Functions.

        The type of the {@link AddBehaviorOptions.functionAssociations} property.

        :param event_type: The type of event which should invoke the function.
        :param function: The CloudFront function that will be invoked.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            # function_: cloudfront.Function
            
            function_association = cloudfront.FunctionAssociation(
                event_type=cloudfront.FunctionEventType.VIEWER_REQUEST,
                function=function_
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__a7e04e24123e028f451a64671ded9f6b4a291f11b4b1165f9fd8874459483fdc)
            check_type(argname="argument event_type", value=event_type, expected_type=type_hints["event_type"])
            check_type(argname="argument function", value=function, expected_type=type_hints["function"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "event_type": event_type,
            "function": function,
        }

    @builtins.property
    def event_type(self) -> "FunctionEventType":
        '''The type of event which should invoke the function.'''
        result = self._values.get("event_type")
        assert result is not None, "Required property 'event_type' is missing"
        return typing.cast("FunctionEventType", result)

    @builtins.property
    def function(self) -> "IFunction":
        '''The CloudFront function that will be invoked.'''
        result = self._values.get("function")
        assert result is not None, "Required property 'function' is missing"
        return typing.cast("IFunction", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionAssociation(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.FunctionAttributes",
    jsii_struct_bases=[],
    name_mapping={"function_arn": "functionArn", "function_name": "functionName"},
)
class FunctionAttributes:
    def __init__(
        self,
        *,
        function_arn: builtins.str,
        function_name: builtins.str,
    ) -> None:
        '''Attributes of an existing CloudFront Function to import it.

        :param function_arn: The ARN of the function.
        :param function_name: The name of the function.

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            
            function_attributes = cloudfront.FunctionAttributes(
                function_arn="functionArn",
                function_name="functionName"
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__06fbfb8312906d11239478b338f6844a6cd2249e88a1399bf73f22916996705a)
            check_type(argname="argument function_arn", value=function_arn, expected_type=type_hints["function_arn"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "function_arn": function_arn,
            "function_name": function_name,
        }

    @builtins.property
    def function_arn(self) -> builtins.str:
        '''The ARN of the function.'''
        result = self._values.get("function_arn")
        assert result is not None, "Required property 'function_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def function_name(self) -> builtins.str:
        '''The name of the function.'''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class FunctionCode(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-cloudfront.FunctionCode",
):
    '''Represents the function's source code.

    :exampleMetadata: infused

    Example::

        # s3_bucket: s3.Bucket
        # Add a cloudfront Function to a Distribution
        cf_function = cloudfront.Function(self, "Function",
            code=cloudfront.FunctionCode.from_inline("function handler(event) { return event.request }")
        )
        cloudfront.Distribution(self, "distro",
            default_behavior=cloudfront.BehaviorOptions(
                origin=origins.S3Origin(s3_bucket),
                function_associations=[cloudfront.FunctionAssociation(
                    function=cf_function,
                    event_type=cloudfront.FunctionEventType.VIEWER_REQUEST
                )]
            )
        )
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromFile")
    @builtins.classmethod
    def from_file(cls, *, file_path: builtins.str) -> "FunctionCode":
        '''Code from external file for function.

        :param file_path: The path of the file to read the code from.

        :return: code object with contents from file.
        '''
        options = FileCodeOptions(file_path=file_path)

        return typing.cast("FunctionCode", jsii.sinvoke(cls, "fromFile", [options]))

    @jsii.member(jsii_name="fromInline")
    @builtins.classmethod
    def from_inline(cls, code: builtins.str) -> "FunctionCode":
        '''Inline code for function.

        :param code: The actual function code.

        :return: code object with inline code.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__4e871c58941eb991e3a0e2d36dba1f35ab9e3fd216974a814fd7a975e0ed0457)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
        return typing.cast("FunctionCode", jsii.sinvoke(cls, "fromInline", [code]))

    @jsii.member(jsii_name="render")
    @abc.abstractmethod
    def render(self) -> builtins.str:
        '''renders the function code.'''
        ...


class _FunctionCodeProxy(FunctionCode):
    @jsii.member(jsii_name="render")
    def render(self) -> builtins.str:
        '''renders the function code.'''
        return typing.cast(builtins.str, jsii.invoke(self, "render", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, FunctionCode).__jsii_proxy_class__ = lambda : _FunctionCodeProxy


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.FunctionEventType")
class FunctionEventType(enum.Enum):
    '''The type of events that a CloudFront function can be invoked in response to.

    :exampleMetadata: infused

    Example::

        # s3_bucket: s3.Bucket
        # Add a cloudfront Function to a Distribution
        cf_function = cloudfront.Function(self, "Function",
            code=cloudfront.FunctionCode.from_inline("function handler(event) { return event.request }")
        )
        cloudfront.Distribution(self, "distro",
            default_behavior=cloudfront.BehaviorOptions(
                origin=origins.S3Origin(s3_bucket),
                function_associations=[cloudfront.FunctionAssociation(
                    function=cf_function,
                    event_type=cloudfront.FunctionEventType.VIEWER_REQUEST
                )]
            )
        )
    '''

    VIEWER_REQUEST = "VIEWER_REQUEST"
    '''The viewer-request specifies the incoming request.'''
    VIEWER_RESPONSE = "VIEWER_RESPONSE"
    '''The viewer-response specifies the outgoing response.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.FunctionProps",
    jsii_struct_bases=[],
    name_mapping={
        "code": "code",
        "comment": "comment",
        "function_name": "functionName",
    },
)
class FunctionProps:
    def __init__(
        self,
        *,
        code: FunctionCode,
        comment: typing.Optional[builtins.str] = None,
        function_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for creating a CloudFront Function.

        :param code: The source code of the function.
        :param comment: A comment to describe the function. Default: - same as ``functionName``
        :param function_name: A name to identify the function. Default: - generated from the ``id``

        :exampleMetadata: infused

        Example::

            # s3_bucket: s3.Bucket
            # Add a cloudfront Function to a Distribution
            cf_function = cloudfront.Function(self, "Function",
                code=cloudfront.FunctionCode.from_inline("function handler(event) { return event.request }")
            )
            cloudfront.Distribution(self, "distro",
                default_behavior=cloudfront.BehaviorOptions(
                    origin=origins.S3Origin(s3_bucket),
                    function_associations=[cloudfront.FunctionAssociation(
                        function=cf_function,
                        event_type=cloudfront.FunctionEventType.VIEWER_REQUEST
                    )]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__83786b817017c13cde5b254e11a9e472a4b0846728eaa720b663edc65d00de4c)
            check_type(argname="argument code", value=code, expected_type=type_hints["code"])
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument function_name", value=function_name, expected_type=type_hints["function_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "code": code,
        }
        if comment is not None:
            self._values["comment"] = comment
        if function_name is not None:
            self._values["function_name"] = function_name

    @builtins.property
    def code(self) -> FunctionCode:
        '''The source code of the function.'''
        result = self._values.get("code")
        assert result is not None, "Required property 'code' is missing"
        return typing.cast(FunctionCode, result)

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment to describe the function.

        :default: - same as ``functionName``
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def function_name(self) -> typing.Optional[builtins.str]:
        '''A name to identify the function.

        :default: - generated from the ``id``
        '''
        result = self._values.get("function_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GeoRestriction(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.GeoRestriction",
):
    '''Controls the countries in which content is distributed.

    :exampleMetadata: infused

    Example::

        # Adding restrictions to a Cloudfront Web Distribution.
        # source_bucket: s3.Bucket
        
        cloudfront.CloudFrontWebDistribution(self, "MyDistribution",
            origin_configs=[cloudfront.SourceConfiguration(
                s3_origin_source=cloudfront.S3OriginConfig(
                    s3_bucket_source=source_bucket
                ),
                behaviors=[cloudfront.Behavior(is_default_behavior=True)]
            )
            ],
            geo_restriction=cloudfront.GeoRestriction.allowlist("US", "GB")
        )
    '''

    @jsii.member(jsii_name="allowlist")
    @builtins.classmethod
    def allowlist(cls, *locations: builtins.str) -> "GeoRestriction":
        '''Allow specific countries which you want CloudFront to distribute your content.

        :param locations: Two-letter, uppercase country code for a country that you want to allow. Include one element for each country. See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__64ffce96c65be53cbe40e9f4cee9228d6078403ae532ed7a2cfb926146cb20b7)
            check_type(argname="argument locations", value=locations, expected_type=typing.Tuple[type_hints["locations"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("GeoRestriction", jsii.sinvoke(cls, "allowlist", [*locations]))

    @jsii.member(jsii_name="blacklist")
    @builtins.classmethod
    def blacklist(cls, *locations: builtins.str) -> "GeoRestriction":
        '''(deprecated) DEPRECATED.

        :param locations: -

        :deprecated: use ``denylist``

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__d54cbc73599ba2d1f932192986b849c57742ad6e0dd25125b63f276deb719826)
            check_type(argname="argument locations", value=locations, expected_type=typing.Tuple[type_hints["locations"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("GeoRestriction", jsii.sinvoke(cls, "blacklist", [*locations]))

    @jsii.member(jsii_name="denylist")
    @builtins.classmethod
    def denylist(cls, *locations: builtins.str) -> "GeoRestriction":
        '''Deny specific countries which you don't want CloudFront to distribute your content.

        :param locations: Two-letter, uppercase country code for a country that you want to deny. Include one element for each country. See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__50764c8a86ec044ebd6baee7c5cd33c108d4f2d04a9bf34c0d431deb039fb311)
            check_type(argname="argument locations", value=locations, expected_type=typing.Tuple[type_hints["locations"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("GeoRestriction", jsii.sinvoke(cls, "denylist", [*locations]))

    @jsii.member(jsii_name="whitelist")
    @builtins.classmethod
    def whitelist(cls, *locations: builtins.str) -> "GeoRestriction":
        '''(deprecated) DEPRECATED.

        :param locations: -

        :deprecated: use ``allowlist``

        :stability: deprecated
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__ee78ba4d521199f56f37e0da06fdf574068f5ec936d8e73014686db6801e9aec)
            check_type(argname="argument locations", value=locations, expected_type=typing.Tuple[type_hints["locations"], ...]) # pyright: ignore [reportGeneralTypeIssues]
        return typing.cast("GeoRestriction", jsii.sinvoke(cls, "whitelist", [*locations]))

    @builtins.property
    @jsii.member(jsii_name="locations")
    def locations(self) -> typing.List[builtins.str]:
        '''Two-letter, uppercase country code for a country that you want to allow/deny.

        Include one element for each country.
        See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "locations"))

    @builtins.property
    @jsii.member(jsii_name="restrictionType")
    def restriction_type(self) -> builtins.str:
        '''Specifies the restriction type to impose.'''
        return typing.cast(builtins.str, jsii.get(self, "restrictionType"))


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.HeadersFrameOption")
class HeadersFrameOption(enum.Enum):
    '''Enum representing possible values of the X-Frame-Options HTTP response header.

    :exampleMetadata: infused

    Example::

        # Using an existing managed response headers policy
        # bucket_origin: origins.S3Origin
        
        cloudfront.Distribution(self, "myDistManagedPolicy",
            default_behavior=cloudfront.BehaviorOptions(
                origin=bucket_origin,
                response_headers_policy=cloudfront.ResponseHeadersPolicy.CORS_ALLOW_ALL_ORIGINS
            )
        )
        
        # Creating a custom response headers policy -- all parameters optional
        my_response_headers_policy = cloudfront.ResponseHeadersPolicy(self, "ResponseHeadersPolicy",
            response_headers_policy_name="MyPolicy",
            comment="A default policy",
            cors_behavior=cloudfront.ResponseHeadersCorsBehavior(
                access_control_allow_credentials=False,
                access_control_allow_headers=["X-Custom-Header-1", "X-Custom-Header-2"],
                access_control_allow_methods=["GET", "POST"],
                access_control_allow_origins=["*"],
                access_control_expose_headers=["X-Custom-Header-1", "X-Custom-Header-2"],
                access_control_max_age=Duration.seconds(600),
                origin_override=True
            ),
            custom_headers_behavior=cloudfront.ResponseCustomHeadersBehavior(
                custom_headers=[cloudfront.ResponseCustomHeader(header="X-Amz-Date", value="some-value", override=True), cloudfront.ResponseCustomHeader(header="X-Amz-Security-Token", value="some-value", override=False)
                ]
            ),
            security_headers_behavior=cloudfront.ResponseSecurityHeadersBehavior(
                content_security_policy=cloudfront.ResponseHeadersContentSecurityPolicy(content_security_policy="default-src https:;", override=True),
                content_type_options=cloudfront.ResponseHeadersContentTypeOptions(override=True),
                frame_options=cloudfront.ResponseHeadersFrameOptions(frame_option=cloudfront.HeadersFrameOption.DENY, override=True),
                referrer_policy=cloudfront.ResponseHeadersReferrerPolicy(referrer_policy=cloudfront.HeadersReferrerPolicy.NO_REFERRER, override=True),
                strict_transport_security=cloudfront.ResponseHeadersStrictTransportSecurity(access_control_max_age=Duration.seconds(600), include_subdomains=True, override=True),
                xss_protection=cloudfront.ResponseHeadersXSSProtection(protection=True, mode_block=True, report_uri="https://example.com/csp-report", override=True)
            )
        )
        cloudfront.Distribution(self, "myDistCustomPolicy",
            default_behavior=cloudfront.BehaviorOptions(
                origin=bucket_origin,
                response_headers_policy=my_response_headers_policy
            )
        )
    '''

    DENY = "DENY"
    '''The page can only be displayed in a frame on the same origin as the page itself.'''
    SAMEORIGIN = "SAMEORIGIN"
    '''The page can only be displayed in a frame on the specified origin.'''


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.HeadersReferrerPolicy")
class HeadersReferrerPolicy(enum.Enum):
    '''Enum representing possible values of the Referrer-Policy HTTP response header.

    :exampleMetadata: infused

    Example::

        # Using an existing managed response headers policy
        # bucket_origin: origins.S3Origin
        
        cloudfront.Distribution(self, "myDistManagedPolicy",
            default_behavior=cloudfront.BehaviorOptions(
                origin=bucket_origin,
                response_headers_policy=cloudfront.ResponseHeadersPolicy.CORS_ALLOW_ALL_ORIGINS
            )
        )
        
        # Creating a custom response headers policy -- all parameters optional
        my_response_headers_policy = cloudfront.ResponseHeadersPolicy(self, "ResponseHeadersPolicy",
            response_headers_policy_name="MyPolicy",
            comment="A default policy",
            cors_behavior=cloudfront.ResponseHeadersCorsBehavior(
                access_control_allow_credentials=False,
                access_control_allow_headers=["X-Custom-Header-1", "X-Custom-Header-2"],
                access_control_allow_methods=["GET", "POST"],
                access_control_allow_origins=["*"],
                access_control_expose_headers=["X-Custom-Header-1", "X-Custom-Header-2"],
                access_control_max_age=Duration.seconds(600),
                origin_override=True
            ),
            custom_headers_behavior=cloudfront.ResponseCustomHeadersBehavior(
                custom_headers=[cloudfront.ResponseCustomHeader(header="X-Amz-Date", value="some-value", override=True), cloudfront.ResponseCustomHeader(header="X-Amz-Security-Token", value="some-value", override=False)
                ]
            ),
            security_headers_behavior=cloudfront.ResponseSecurityHeadersBehavior(
                content_security_policy=cloudfront.ResponseHeadersContentSecurityPolicy(content_security_policy="default-src https:;", override=True),
                content_type_options=cloudfront.ResponseHeadersContentTypeOptions(override=True),
                frame_options=cloudfront.ResponseHeadersFrameOptions(frame_option=cloudfront.HeadersFrameOption.DENY, override=True),
                referrer_policy=cloudfront.ResponseHeadersReferrerPolicy(referrer_policy=cloudfront.HeadersReferrerPolicy.NO_REFERRER, override=True),
                strict_transport_security=cloudfront.ResponseHeadersStrictTransportSecurity(access_control_max_age=Duration.seconds(600), include_subdomains=True, override=True),
                xss_protection=cloudfront.ResponseHeadersXSSProtection(protection=True, mode_block=True, report_uri="https://example.com/csp-report", override=True)
            )
        )
        cloudfront.Distribution(self, "myDistCustomPolicy",
            default_behavior=cloudfront.BehaviorOptions(
                origin=bucket_origin,
                response_headers_policy=my_response_headers_policy
            )
        )
    '''

    NO_REFERRER = "NO_REFERRER"
    '''The referrer policy is not set.'''
    NO_REFERRER_WHEN_DOWNGRADE = "NO_REFERRER_WHEN_DOWNGRADE"
    '''The referrer policy is no-referrer-when-downgrade.'''
    ORIGIN = "ORIGIN"
    '''The referrer policy is origin.'''
    ORIGIN_WHEN_CROSS_ORIGIN = "ORIGIN_WHEN_CROSS_ORIGIN"
    '''The referrer policy is origin-when-cross-origin.'''
    SAME_ORIGIN = "SAME_ORIGIN"
    '''The referrer policy is same-origin.'''
    STRICT_ORIGIN = "STRICT_ORIGIN"
    '''The referrer policy is strict-origin.'''
    STRICT_ORIGIN_WHEN_CROSS_ORIGIN = "STRICT_ORIGIN_WHEN_CROSS_ORIGIN"
    '''The referrer policy is strict-origin-when-cross-origin.'''
    UNSAFE_URL = "UNSAFE_URL"
    '''The referrer policy is unsafe-url.'''


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.HttpVersion")
class HttpVersion(enum.Enum):
    '''Maximum HTTP version to support.'''

    HTTP1_1 = "HTTP1_1"
    '''HTTP 1.1.'''
    HTTP2 = "HTTP2"
    '''HTTP 2.'''


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.ICachePolicy")
class ICachePolicy(typing_extensions.Protocol):
    '''Represents a Cache Policy.'''

    @builtins.property
    @jsii.member(jsii_name="cachePolicyId")
    def cache_policy_id(self) -> builtins.str:
        '''The ID of the cache policy.

        :attribute: true
        '''
        ...


class _ICachePolicyProxy:
    '''Represents a Cache Policy.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.ICachePolicy"

    @builtins.property
    @jsii.member(jsii_name="cachePolicyId")
    def cache_policy_id(self) -> builtins.str:
        '''The ID of the cache policy.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "cachePolicyId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICachePolicy).__jsii_proxy_class__ = lambda : _ICachePolicyProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IDistribution")
class IDistribution(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    '''Interface for CloudFront distributions.'''

    @builtins.property
    @jsii.member(jsii_name="distributionDomainName")
    def distribution_domain_name(self) -> builtins.str:
        '''The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="distributionId")
    def distribution_id(self) -> builtins.str:
        '''The distribution ID for this distribution.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''(deprecated) The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :deprecated: - Use ``distributionDomainName`` instead.

        :stability: deprecated
        :attribute: true
        '''
        ...


class _IDistributionProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    '''Interface for CloudFront distributions.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IDistribution"

    @builtins.property
    @jsii.member(jsii_name="distributionDomainName")
    def distribution_domain_name(self) -> builtins.str:
        '''The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "distributionDomainName"))

    @builtins.property
    @jsii.member(jsii_name="distributionId")
    def distribution_id(self) -> builtins.str:
        '''The distribution ID for this distribution.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "distributionId"))

    @builtins.property
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> builtins.str:
        '''(deprecated) The domain name of the Distribution, such as d111111abcdef8.cloudfront.net.

        :deprecated: - Use ``distributionDomainName`` instead.

        :stability: deprecated
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "domainName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDistribution).__jsii_proxy_class__ = lambda : _IDistributionProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IFunction")
class IFunction(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    '''Represents a CloudFront Function.'''

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function.

        :attribute: true
        '''
        ...

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the function.

        :attribute: true
        '''
        ...


class _IFunctionProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    '''Represents a CloudFront Function.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IFunction"

    @builtins.property
    @jsii.member(jsii_name="functionArn")
    def function_arn(self) -> builtins.str:
        '''The ARN of the function.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionArn"))

    @builtins.property
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> builtins.str:
        '''The name of the function.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "functionName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IFunction).__jsii_proxy_class__ = lambda : _IFunctionProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IKeyGroup")
class IKeyGroup(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    '''Represents a Key Group.'''

    @builtins.property
    @jsii.member(jsii_name="keyGroupId")
    def key_group_id(self) -> builtins.str:
        '''The ID of the key group.

        :attribute: true
        '''
        ...


class _IKeyGroupProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    '''Represents a Key Group.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IKeyGroup"

    @builtins.property
    @jsii.member(jsii_name="keyGroupId")
    def key_group_id(self) -> builtins.str:
        '''The ID of the key group.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "keyGroupId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IKeyGroup).__jsii_proxy_class__ = lambda : _IKeyGroupProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IOrigin")
class IOrigin(typing_extensions.Protocol):
    '''Represents the concept of a CloudFront Origin.

    You provide one or more origins when creating a Distribution.
    '''

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        *,
        origin_id: builtins.str,
    ) -> "OriginBindConfig":
        '''The method called when a given Origin is added (for the first time) to a Distribution.

        :param scope: -
        :param origin_id: The identifier of this Origin, as assigned by the Distribution this Origin has been used added to.
        '''
        ...


class _IOriginProxy:
    '''Represents the concept of a CloudFront Origin.

    You provide one or more origins when creating a Distribution.
    '''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IOrigin"

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _aws_cdk_core_f4b25747.Construct,
        *,
        origin_id: builtins.str,
    ) -> "OriginBindConfig":
        '''The method called when a given Origin is added (for the first time) to a Distribution.

        :param scope: -
        :param origin_id: The identifier of this Origin, as assigned by the Distribution this Origin has been used added to.
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__91422e667a84bc2a2164167816803b5eebbcee95478661c911a543ea27a6fa61)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
        options = OriginBindOptions(origin_id=origin_id)

        return typing.cast("OriginBindConfig", jsii.invoke(self, "bind", [scope, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IOrigin).__jsii_proxy_class__ = lambda : _IOriginProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IOriginAccessIdentity")
class IOriginAccessIdentity(
    _aws_cdk_core_f4b25747.IResource,
    _aws_cdk_aws_iam_940a1ce0.IGrantable,
    typing_extensions.Protocol,
):
    '''Interface for CloudFront OriginAccessIdentity.'''

    @builtins.property
    @jsii.member(jsii_name="originAccessIdentityName")
    def origin_access_identity_name(self) -> builtins.str:
        '''The Origin Access Identity Name.'''
        ...


class _IOriginAccessIdentityProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
    jsii.proxy_for(_aws_cdk_aws_iam_940a1ce0.IGrantable), # type: ignore[misc]
):
    '''Interface for CloudFront OriginAccessIdentity.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IOriginAccessIdentity"

    @builtins.property
    @jsii.member(jsii_name="originAccessIdentityName")
    def origin_access_identity_name(self) -> builtins.str:
        '''The Origin Access Identity Name.'''
        return typing.cast(builtins.str, jsii.get(self, "originAccessIdentityName"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IOriginAccessIdentity).__jsii_proxy_class__ = lambda : _IOriginAccessIdentityProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IOriginRequestPolicy")
class IOriginRequestPolicy(typing_extensions.Protocol):
    '''Represents a Origin Request Policy.'''

    @builtins.property
    @jsii.member(jsii_name="originRequestPolicyId")
    def origin_request_policy_id(self) -> builtins.str:
        '''The ID of the origin request policy.

        :attribute: true
        '''
        ...


class _IOriginRequestPolicyProxy:
    '''Represents a Origin Request Policy.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IOriginRequestPolicy"

    @builtins.property
    @jsii.member(jsii_name="originRequestPolicyId")
    def origin_request_policy_id(self) -> builtins.str:
        '''The ID of the origin request policy.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "originRequestPolicyId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IOriginRequestPolicy).__jsii_proxy_class__ = lambda : _IOriginRequestPolicyProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IPublicKey")
class IPublicKey(_aws_cdk_core_f4b25747.IResource, typing_extensions.Protocol):
    '''Represents a Public Key.'''

    @builtins.property
    @jsii.member(jsii_name="publicKeyId")
    def public_key_id(self) -> builtins.str:
        '''The ID of the key group.

        :attribute: true
        '''
        ...


class _IPublicKeyProxy(
    jsii.proxy_for(_aws_cdk_core_f4b25747.IResource), # type: ignore[misc]
):
    '''Represents a Public Key.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IPublicKey"

    @builtins.property
    @jsii.member(jsii_name="publicKeyId")
    def public_key_id(self) -> builtins.str:
        '''The ID of the key group.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "publicKeyId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPublicKey).__jsii_proxy_class__ = lambda : _IPublicKeyProxy


@jsii.interface(jsii_type="@aws-cdk/aws-cloudfront.IResponseHeadersPolicy")
class IResponseHeadersPolicy(typing_extensions.Protocol):
    '''Represents a response headers policy.'''

    @builtins.property
    @jsii.member(jsii_name="responseHeadersPolicyId")
    def response_headers_policy_id(self) -> builtins.str:
        '''The ID of the response headers policy.

        :attribute: true
        '''
        ...


class _IResponseHeadersPolicyProxy:
    '''Represents a response headers policy.'''

    __jsii_type__: typing.ClassVar[str] = "@aws-cdk/aws-cloudfront.IResponseHeadersPolicy"

    @builtins.property
    @jsii.member(jsii_name="responseHeadersPolicyId")
    def response_headers_policy_id(self) -> builtins.str:
        '''The ID of the response headers policy.

        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "responseHeadersPolicyId"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IResponseHeadersPolicy).__jsii_proxy_class__ = lambda : _IResponseHeadersPolicyProxy


@jsii.implements(IKeyGroup)
class KeyGroup(
    _aws_cdk_core_f4b25747.Resource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-cloudfront.KeyGroup",
):
    '''A Key Group configuration.

    :resource: AWS::CloudFront::KeyGroup
    :exampleMetadata: infused

    Example::

        # Validating signed URLs or signed cookies with Trusted Key Groups
        
        # public key in PEM format
        # public_key: str
        
        pub_key = cloudfront.PublicKey(self, "MyPubKey",
            encoded_key=public_key
        )
        
        key_group = cloudfront.KeyGroup(self, "MyKeyGroup",
            items=[pub_key
            ]
        )
        
        cloudfront.Distribution(self, "Dist",
            default_behavior=cloudfront.BehaviorOptions(
                origin=origins.HttpOrigin("www.example.com"),
                trusted_key_groups=[key_group
                ]
            )
        )
    '''

    def __init__(
        self,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        *,
        items: typing.Sequence[IPublicKey],
        comment: typing.Optional[builtins.str] = None,
        key_group_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param items: A list of public keys to add to the key group.
        :param comment: A comment to describe the key group. Default: - no comment
        :param key_group_name: A name to identify the key group. Default: - generated from the ``id``
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__aff1e6ec2ebb5cd105e92d2df9b50da0ce09da622b1a9fb1d32dfafbe2fbd943)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
        props = KeyGroupProps(
            items=items, comment=comment, key_group_name=key_group_name
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="fromKeyGroupId")
    @builtins.classmethod
    def from_key_group_id(
        cls,
        scope: _constructs_77d1e7e8.Construct,
        id: builtins.str,
        key_group_id: builtins.str,
    ) -> IKeyGroup:
        '''Imports a Key Group from its id.

        :param scope: -
        :param id: -
        :param key_group_id: -
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__8d264dbe59cce95e658b43d78627f99f1f7deee2bb97e9ccc865c8d668e9caa0)
            check_type(argname="argument scope", value=scope, expected_type=type_hints["scope"])
            check_type(argname="argument id", value=id, expected_type=type_hints["id"])
            check_type(argname="argument key_group_id", value=key_group_id, expected_type=type_hints["key_group_id"])
        return typing.cast(IKeyGroup, jsii.sinvoke(cls, "fromKeyGroupId", [scope, id, key_group_id]))

    @builtins.property
    @jsii.member(jsii_name="keyGroupId")
    def key_group_id(self) -> builtins.str:
        '''The ID of the key group.'''
        return typing.cast(builtins.str, jsii.get(self, "keyGroupId"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.KeyGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "items": "items",
        "comment": "comment",
        "key_group_name": "keyGroupName",
    },
)
class KeyGroupProps:
    def __init__(
        self,
        *,
        items: typing.Sequence[IPublicKey],
        comment: typing.Optional[builtins.str] = None,
        key_group_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for creating a Public Key.

        :param items: A list of public keys to add to the key group.
        :param comment: A comment to describe the key group. Default: - no comment
        :param key_group_name: A name to identify the key group. Default: - generated from the ``id``

        :exampleMetadata: infused

        Example::

            # Validating signed URLs or signed cookies with Trusted Key Groups
            
            # public key in PEM format
            # public_key: str
            
            pub_key = cloudfront.PublicKey(self, "MyPubKey",
                encoded_key=public_key
            )
            
            key_group = cloudfront.KeyGroup(self, "MyKeyGroup",
                items=[pub_key
                ]
            )
            
            cloudfront.Distribution(self, "Dist",
                default_behavior=cloudfront.BehaviorOptions(
                    origin=origins.HttpOrigin("www.example.com"),
                    trusted_key_groups=[key_group
                    ]
                )
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__24d1358aea0a314cb9d7de9138c5090056bf5bcfa79e376e485a64a9222e748b)
            check_type(argname="argument items", value=items, expected_type=type_hints["items"])
            check_type(argname="argument comment", value=comment, expected_type=type_hints["comment"])
            check_type(argname="argument key_group_name", value=key_group_name, expected_type=type_hints["key_group_name"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "items": items,
        }
        if comment is not None:
            self._values["comment"] = comment
        if key_group_name is not None:
            self._values["key_group_name"] = key_group_name

    @builtins.property
    def items(self) -> typing.List[IPublicKey]:
        '''A list of public keys to add to the key group.'''
        result = self._values.get("items")
        assert result is not None, "Required property 'items' is missing"
        return typing.cast(typing.List[IPublicKey], result)

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''A comment to describe the key group.

        :default: - no comment
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def key_group_name(self) -> typing.Optional[builtins.str]:
        '''A name to identify the key group.

        :default: - generated from the ``id``
        '''
        result = self._values.get("key_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "KeyGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-cloudfront.LambdaEdgeEventType")
class LambdaEdgeEventType(enum.Enum):
    '''The type of events that a Lambda@Edge function can be invoked in response to.

    :exampleMetadata: infused

    Example::

        # my_bucket: s3.Bucket
        # A Lambda@Edge function added to default behavior of a Distribution
        # and triggered on every request
        my_func = cloudfront.experimental.EdgeFunction(self, "MyFunction",
            runtime=lambda_.Runtime.NODEJS_14_X,
            handler="index.handler",
            code=lambda_.Code.from_asset(path.join(__dirname, "lambda-handler"))
        )
        cloudfront.Distribution(self, "myDist",
            default_behavior=cloudfront.BehaviorOptions(
                origin=origins.S3Origin(my_bucket),
                edge_lambdas=[cloudfront.EdgeLambda(
                    function_version=my_func.current_version,
                    event_type=cloudfront.LambdaEdgeEventType.VIEWER_REQUEST
                )
                ]
            )
        )
    '''

    ORIGIN_REQUEST = "ORIGIN_REQUEST"
    '''The origin-request specifies the request to the origin location (e.g. S3).'''
    ORIGIN_RESPONSE = "ORIGIN_RESPONSE"
    '''The origin-response specifies the response from the origin location (e.g. S3).'''
    VIEWER_REQUEST = "VIEWER_REQUEST"
    '''The viewer-request specifies the incoming request.'''
    VIEWER_RESPONSE = "VIEWER_RESPONSE"
    '''The viewer-response specifies the outgoing response.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.LambdaFunctionAssociation",
    jsii_struct_bases=[],
    name_mapping={
        "event_type": "eventType",
        "lambda_function": "lambdaFunction",
        "include_body": "includeBody",
    },
)
class LambdaFunctionAssociation:
    def __init__(
        self,
        *,
        event_type: LambdaEdgeEventType,
        lambda_function: _aws_cdk_aws_lambda_5443dbc3.IVersion,
        include_body: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''
        :param event_type: The lambda event type defines at which event the lambda is called during the request lifecycle.
        :param lambda_function: A version of the lambda to associate.
        :param include_body: Allows a Lambda function to have read access to the body content. Only valid for "request" event types (``ORIGIN_REQUEST`` or ``VIEWER_REQUEST``). See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-include-body-access.html Default: false

        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import aws_cdk.aws_cloudfront as cloudfront
            import aws_cdk.aws_lambda as lambda_
            
            # version: lambda.Version
            
            lambda_function_association = cloudfront.LambdaFunctionAssociation(
                event_type=cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                lambda_function=version,
            
                # the properties below are optional
                include_body=False
            )
        '''
        if __debug__:
            type_hints = typing.get_type_hints(_typecheckingstub__45061e1d35adf73cce1f4e0ae9f7526a51f6c01a5d7056ea0bd2b58a4aea8d89)
            check_type(argname="argument event_type", value=event_type, expected_type=type_hints["event_type"])
            check_type(argname="argument lambda_function", value=lambda_function, expected_type=type_hints["lambda_function"])
            check_type(argname="argument include_body", value=include_body, expected_type=type_hints["include_body"])
        self._values: typing.Dict[builtins.str, typing.Any] = {
            "event_type": event_type,
            "lambda_function": lambda_function,
        }
        if include_body is not None:
            self._values["include_body"] = include_body

    @builtins.property
    def event_type(self) -> LambdaEdgeEventType:
        '''The lambda event type defines at which event the lambda is called during the request lifecycle.'''
        result = self._values.get("event_type")
        assert result is not None, "Required property 'event_type' is missing"
        return typing.cast(LambdaEdgeEventType, result)

    @builtins.property
    def lambda_function(self) -> _aws_cdk_aws_lambda_5443dbc3.IVersion:
        '''A version of the lambda to associate.'''
        result = self._values.get("lambda_function")
        assert result is not None, "Required property 'lambda_function' is missing"
        return typing.cast(_aws_cdk_aws_lambda_5443dbc3.IVersion, result)

    @builtins.property
    def include_body(self) -> typing.Optional[builtins.bool]:
        '''Allows a Lambda function to have read access to the body content.

        Only valid for "request" event types (``ORIGIN_REQUEST`` or ``VIEWER_REQUEST``).
        See https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-include-body-access.html

        :default: false
        '''
        result = self._values.get("include_body")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdaFunctionAssociation(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-cloudfront.LoggingConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "bucke