"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoAttributeValue = exports.DynamoProjectionExpression = exports.DynamoReturnValues = exports.DynamoItemCollectionMetrics = exports.DynamoConsumedCapacity = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const utils_1 = require("./private/utils");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    /**
     * No ConsumedCapacity details are included in the response.
     */
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    /**
     * If set to NONE, no statistics are returned.
     */
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    /**
     * Nothing is returned
     */
    DynamoReturnValues["NONE"] = "NONE";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate projection expression
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
_a = JSII_RTTI_SYMBOL_1;
DynamoProjectionExpression[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.DynamoProjectionExpression", version: "1.193.0" };
/**
 * Represents the data for an attribute.
 * Each attribute value is described as a name-value pair.
 * The name is the data type, and the value is the data itself.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
 */
class DynamoAttributeValue {
    constructor(value) {
        this.attributeValue = value;
    }
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     * Strings may be literal values or as JsonPath. Example values:
     *
     * - `DynamoAttributeValue.fromString('someValue')`
     * - `DynamoAttributeValue.fromString(JsonPath.stringAt('$.bar'))`
     */
    static fromString(value) {
        return new DynamoAttributeValue({ S: value });
    }
    /**
     * Sets a literal number. For example: 1234
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumber(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberFromString(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     *
     * @param value base-64 encoded string
     */
    static fromBinary(value) {
        return new DynamoAttributeValue({ B: value });
    }
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    static fromStringSet(value) {
        return new DynamoAttributeValue({ SS: value });
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumberSet(value) {
        return new DynamoAttributeValue({ NS: value.map(String) });
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberSetFromStrings(value) {
        return new DynamoAttributeValue({ NS: value });
    }
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    static fromBinarySet(value) {
        return new DynamoAttributeValue({ BS: value });
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    static fromMap(value) {
        return new DynamoAttributeValue({ M: utils_1.transformAttributeValueMap(value) });
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @param value Json path that specifies state input to be used
     */
    static mapFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ 'M.$': value });
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    static fromList(value) {
        return new DynamoAttributeValue({ L: value.map((val) => val.toObject()) });
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"S", "Veggies"}]
     *
     * @param value Json path that specifies state input to be used
     */
    static listFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ L: value });
    }
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    static fromNull(value) {
        return new DynamoAttributeValue({ NULL: value });
    }
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    static fromBoolean(value) {
        return new DynamoAttributeValue({ BOOL: value });
    }
    /**
     * Sets an attribute of type Boolean from state input through Json path.
     * For example:  "BOOL": true
     *
     * @param value Json path that specifies state input to be used
     */
    static booleanFromJsonPath(value) {
        utils_1.validateJsonPath(value);
        return new DynamoAttributeValue({ BOOL: value.toString() });
    }
    /**
     * Returns the DynamoDB attribute value
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
_b = JSII_RTTI_SYMBOL_1;
DynamoAttributeValue[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.DynamoAttributeValue", version: "1.193.0" };
//# sourceMappingURL=data:application/json;base64,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