"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_helper_1 = require("../github-helper");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log('Create new ActionEnvironmentSecret with props ' + JSON.stringify(props));
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log(`Update ActionEnvironmentSecret ${secretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onDelete = async (event, octokit) => {
    const secretName = event.ResourceProperties.repositorySecretName;
    console.log('Delete ActionEnvironmentSecret ' + secretName);
    await deleteEnvironmentSecret(event, octokit);
    return { PhysicalResourceId: secretName };
};
const createOrUpdateEnvironmentSecret = async (event, octokit, smClient) => {
    const { repositoryOwner, repositoryName: repo, repositorySecretName: secret_name, environment: environment_name, sourceSecretArn: secretId, } = event.ResourceProperties;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const repository_id = await getRepositoryId(event, octokit, owner);
    const secretResponse = await octokit.request('PUT /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteEnvironmentSecret = async (event, octokit) => {
    const { environment: environment_name, repositorySecretName: secret_name, repositoryOwner } = event.ResourceProperties;
    const repository_id = await getRepositoryId(event, octokit, repositoryOwner);
    const deleteSecretResponse = await octokit.request('DELETE /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const getRepositoryId = async (event, octokit, repositoryOwner) => {
    const { repositoryName: repo } = event.ResourceProperties;
    const owner = await github_helper_1.getOwner(octokit, repositoryOwner);
    const { data } = await octokit.request('GET /repos/{owner}/{repo}', {
        owner,
        repo,
    });
    return data.id;
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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