/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.prec;

import java.util.Arrays;

import mt.DenseVector;
import mt.Matrix;
import mt.Vector;
import smt.SparseVector;

/**
 * An ICC preconditioner. Uses an incomplete Cholesky decomposition as
 * preconditioner. Applicable to symmetrical, positive definite problems.
 */
public class ICC extends DecompositionPreconditioner {

    /**
     * Constructor for ICC
     * 
     * @param A
     *            Matrix to decompose. Not modified
     * @param initShift
     *            Initial diagonal shift on the factor matrix
     * @param shift
     *            Progressive diagonal shifts used every time a non-positive
     *            pivot is found
     */
    public ICC(Matrix A, double initShift, double shift) {
        super(A, initShift, shift);
    }

    /**
     * Constructor for ICC. Uses no initial shift, and a progressive shift of
     * <code>0.1</code>
     * 
     * @param A
     *            Matrix to decompose. Not modified
     */
    public ICC(Matrix A) {
        super(A);
    }

    public Vector apply(Vector b, Vector x) {
        if (!(x instanceof DenseVector))
            throw new IllegalArgumentException("x must be a DenseVector");

        // Copy b into x, and get the array
        x.set(b);
        double[] xd = ((DenseVector) x).getData();

        // First solve F^T y=b, then Fx=y
        solveUT(xd, true);
        solveU(xd, true);

        return x;
    }

    public Vector transApply(Vector b, Vector x) {
        return apply(b, x);
    }

    void factor(Matrix A) {
        double diagMod = initShift;
        int n = A.numRows();
        diagInd = new int[n];
        DenseVector actRow = new DenseVector(n);

        outer: while (true) {

            // Copy A into F, and ensure non-zero diagonal
            F.set(A).addDiagonal(diagMod);

            // Extract the upper triangular part
            for (int i = 0; i < n; ++i) {

                // Get the whole row
                SparseVector curRow = F.getRow(i);
                int[] curInd = curRow.getIndex();
                double[] curDat = curRow.getData();
                int length = curRow.used();

                // Gets its diagonal index, and also check for missing or
                // negative diagonal entries
                int diagInd = smt.util.Arrays
                        .binarySearch(curInd, i, 0, length);
                if (diagInd < 0 || curDat[diagInd] <= 0) {
                    diagMod += shift;
                    continue outer;
                }

                // Copy into the upper triangular part
                int[] index = new int[length - diagInd];
                double[] data = new double[length - diagInd];
                System.arraycopy(curInd, diagInd, index, 0, index.length);
                System.arraycopy(curDat, diagInd, data, 0, data.length);

                F.setRow(i, new SparseVector(n, index, data, false));
            }

            // Factorize
            for (int i = 0; i < n; ++i) {

                // Get current row
                SparseVector curRow = F.getRow(i);
                int length = curRow.used();
                double[] curDat = curRow.getData();
                int[] curInd = curRow.getIndex();

                // We must have positive entries on the diagonal
                if (length == 0 || curDat[0] <= 0) {
                    diagMod += shift;
                    continue outer;
                }

                // Divide by rooted diagonal
                double diag = curDat[0] = Math.sqrt(curDat[0]);
                for (int j = 1; j < length; ++j)
                    curDat[j] /= diag;

                // Reduce submatrix using current row
                for (int j = 1; j < length; ++j) {

                    double factor = curDat[j];

                    // Scatter active row into dense format
                    SparseVector actRowI = F.getRow(curInd[j]);
                    scatter(actRow, actRowI);

                    // Reduce active row
                    for (int k = 0; k < length; ++k)
                        actRow.add(curInd[k], -factor * curDat[k]);

                    // Gather the active row back to sparse
                    gather(actRowI, actRow);
                }
            }

            // Compress the matrix, and ensure that all diagonals exists with
            // correct indices
            F.compact();
            Arrays.fill(diagInd, -1);
            for (int i = 0; i < n; ++i) {
                getDiagInd(i);
                if (diagInd[i] < 0) {
                    diagMod += shift;
                    continue outer;
                }
            }

            // Done
            break;
        }
    }

}
