/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt;

import mt.DenseVector;
import mt.Matrix;
import mt.Vector;

/**
 * Compressed column storage (CCS) matrix built from sparse vectors
 */
public class FlexCompColMatrix extends AbstractColMatrix {

    private static final long serialVersionUID = 3256446897597003572L;

    /**
     * Constructor for FlexCompColMatrix
     * 
     * @param numRows
     *            Number of rows
     * @param numColumns
     *            Number of column
     * @param numThreads
     *            Number of threads to use in parallelization
     */
    public FlexCompColMatrix(int numRows, int numColumns, int numThreads) {
        super(numRows, numColumns, new SparseVector(numRows), numThreads);
    }

    /**
     * Constructor for FlexCompColMatrix
     * 
     * @param A
     *            Matrix to copy contents from
     * @param numThreads
     *            Number of threads to use in parallelization
     * @param deep
     *            True for a deep copy, false for a reference copy. A reference
     *            copy can only be made of a <code>FlexCompColMatrix</code>
     */
    public FlexCompColMatrix(Matrix A, int numThreads, boolean deep) {
        super(A, new SparseVector(A.numRows()), numThreads, deep);
    }

    /**
     * Constructor for FlexCompColMatrix
     * 
     * @param A
     *            Matrix to copy contents from. The copy will be deep
     * @param numThreads
     *            Number of threads to use in parallelization
     */
    public FlexCompColMatrix(Matrix A, int numThreads) {
        super(A, new SparseVector(A.numRows()), numThreads);
    }

    /**
     * Constructor for FlexCompColMatrix. Tries to use as many threads as the
     * current machine has processors
     * 
     * @param numRows
     *            Number of rows
     * @param numColumns
     *            Number of column
     */
    public FlexCompColMatrix(int numRows, int numColumns) {
        super(numRows, numColumns, new SparseVector(numRows));
    }

    /**
     * Constructor for FlexCompColMatrix. Tries to use as many threads as the
     * current machine has processors
     * 
     * @param A
     *            Matrix to copy contents from. The copy will be deep
     */
    public FlexCompColMatrix(Matrix A) {
        super(A, new SparseVector(A.numRows()));
    }

    /**
     * Constructor for FlexCompColMatrix. Tries to use as many threads as the
     * current machine has processors
     * 
     * @param A
     *            Matrix to copy contents from
     * @param deep
     *            True for a deep copy, false for a reference copy. A reference
     *            copy can only be made of a <code>FlexCompColMatrix</code>
     */
    public FlexCompColMatrix(Matrix A, boolean deep) {
        super(A, new SparseVector(A.numRows()), deep);
    }

    public Vector multAdd(final double alpha, final Vector x,
            final double beta, final Vector y, final Vector z) {
        if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
            return super.multAdd(alpha, x, beta, y, z);

        checkMultAdd(x, y, z);

        double[] xd = ((DenseVector) x).getData(), zd = ((DenseVector) z)
                .getData();

        // z = beta/alpha * y
        z.set(beta / alpha, y);

        // z = A*x + z
        for (int i = 0; i < numColumns; ++i) {
            SparseVector v = (SparseVector) colD[i];
            int[] index = v.getIndex();
            double[] data = v.getData();
            int length = v.used();
            for (int j = 0; j < length; ++j)
                zd[index[j]] += data[j] * xd[i];
        }

        // z = alpha*z = alpha * A'x + beta*y
        return z.scale(alpha);
    }

    public FlexCompColMatrix copy() {
        return new FlexCompColMatrix(this);
    }

    public SparseVector getColumn(int i) {
        return (SparseVector) colD[i];
    }

    /**
     * Sets the given column equal the passed vector
     */
    public void setColumn(int i, SparseVector x) {
        if (x.size() != numRows)
            throw new IllegalArgumentException(
                    "New column must be of the same size as existing column");
        colD[i] = x;
    }

    /**
     * Tries to store the matrix as compactly as possible
     */
    public void compact() {
        for (Vector v : colD)
            ((SparseVector) v).compact();
    }

}
