/*
 * Copyright (C) 2003-2005 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.DenseMatrix;
import mt.Matrix;

abstract class OrthogonalComputer {

    /**
     * Factorisation sizes
     */
    final int m, n, k;

    /**
     * Work arrays
     */
    double[] work, workGen;

    /**
     * Scales for the reflectors
     */
    final double[] tau;

    /**
     * The Householder reflectors. Also used to extract the triangular factor
     */
    final DenseMatrix H;

    /**
     * True for storing an upper triangular factor, false for a lower triangular
     * factor
     */
    final boolean upper;

    /**
     * Constructor for OrthogonalComputer
     * 
     * @param m
     *            Number of rows
     * @param n
     *            Number of columns
     * @param upper
     *            True for storing an upper triangular factor, false for a lower
     *            triangular factor
     */
    OrthogonalComputer(int m, int n, boolean upper) {
        this.m = m;
        this.n = n;
        this.k = Math.min(m, n);

        tau = new double[k];
        H = new DenseMatrix(m, n);

        this.upper = upper;
    }

    /**
     * Computes an orthogonal decomposition
     * 
     * @param A
     *            Matrix to decompose. Not modified
     * @return Newly allocated decomposition
     */
    public OrthogonalDecomposition factor(Matrix A) {
        return factor(A, new OrthogonalDecomposition(m, n, upper));
    }

    /**
     * Computes an orthogonal decomposition
     * 
     * @param A
     *            Matrix to decompose. Not modified
     * @param od
     *            Decomposition is written here
     * @return The passed decomposition
     */
    public abstract OrthogonalDecomposition factor(Matrix A,
            OrthogonalDecomposition od);
}
