/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.DenseMatrix;
import mt.Matrix;
import mt.ll.Interface;
import mt.ll.LAPACKkernel.JobEig;

/**
 * Computes eigenvalues of general matrices
 */
public class EigenvalueComputer {

    /**
     * Double work array
     */
    private double[] work;

    /**
     * Size of the matrix
     */
    private int n;

    /**
     * Job to do on the left and right eigenvectors
     */
    private JobEig jobLeft, jobRight;

    /**
     * Creates a new eigenvalue computer for general matrices. Computes all the
     * eigenvectors (left and right)
     * 
     * @param n
     *            Size of the matrix
     */
    public EigenvalueComputer(int n) {
        this(n, JobEig.All, JobEig.All);
    }

    /**
     * Creates a new eigenvalue computer for general matrices
     * 
     * @param n
     *            Size of the matrix
     * @param jobLeft
     *            Whether to compute the left eigenvectors or not
     * @param jobRight
     *            Whether to compute the right eigenvectors or not
     */
    public EigenvalueComputer(int n, JobEig jobLeft, JobEig jobRight) {
        this.n = n;
        this.jobLeft = jobLeft;
        this.jobRight = jobRight;

        // Find the needed workspace
        work = new double[1];
        int info = Interface.lapack().geev(jobLeft, jobRight, n, new double[0],
                new double[0], new double[0], new double[0], new double[0],
                work, -1);

        // Allocate workspace
        int lwork = 0;
        if (info != 0) {
            if (jobLeft == JobEig.All || jobRight == JobEig.All)
                lwork = 4 * n;
            else
                lwork = 3 * n;
        } else
            lwork = (int) work[0];
        lwork = Math.max(1, lwork);
        work = new double[lwork];
    }

    /**
     * Creates a new eigenvalue computer for general matrices
     * 
     * @param n
     *            Size of the matrix
     * @param job
     *            Whether or not to compute the left and right eigenvectors
     */
    public EigenvalueComputer(int n, JobEig job) {
        this(n, job, job);
    }

    /**
     * Creates a new eigenvalue computer for general matrices
     * 
     * @param n
     *            Size of the matrix
     * @param job
     *            Whether or not to compute the left and right eigenvectors
     */
    public EigenvalueComputer(int n, boolean job) {
        this(n, job ? JobEig.All : JobEig.Eigenvalues);
    }

    /**
     * Convience method for computing the complete eigenvalue decomposition of
     * the given matrix
     * 
     * @param A
     *            Matrix to factorize. Not modified
     * @return Newly allocated decomposition
     * @throws NotConvergedException
     */
    public static EigenvalueDecomposition factorize(Matrix A)
            throws NotConvergedException {
        return new EigenvalueComputer(A.numRows()).factor(new DenseMatrix(A));
    }

    /**
     * Computes the eigenvalue decomposition of the given matrix
     * 
     * @param A
     *            Matrix to factorize. Overwritten on return
     * @return Newly allocated decomposition
     * @throws NotConvergedException
     */
    public EigenvalueDecomposition factor(DenseMatrix A)
            throws NotConvergedException {
        return factor(A, new EigenvalueDecomposition(n, jobLeft, jobRight));
    }

    /**
     * Computes the eigenvalue decomposition of the given matrix
     * 
     * @param A
     *            Matrix to factorize. Overwritten on return
     * @param evd
     *            Factorization is written here
     * @return evd
     * @throws NotConvergedException
     */
    public EigenvalueDecomposition factor(DenseMatrix A,
            EigenvalueDecomposition evd) throws NotConvergedException {
        if (!A.isSquare())
            throw new IllegalArgumentException("!A.isSquare()");
        else if (A.numRows() != n)
            throw new IllegalArgumentException("A.numRows() != n");
        else if (evd.getRealEigenvalues().length != n)
            throw new IllegalArgumentException(
                    "evd.getRealEigenvalues().length != n");
        else if (evd.getImaginaryEigenvalues().length != n)
            throw new IllegalArgumentException(
                    "evd.getImaginaryEigenvalues().length != n");
        else if (jobLeft == JobEig.All) {
            if (!evd.hasLeftEigenvectors())
                throw new IllegalArgumentException(
                        "Eigenvalue decomposition does not store left eigenvectors");
            else if (evd.getLeftEigenvectors().numRows() != n)
                throw new IllegalArgumentException(
                        "evd.getLeftEigenvectors().numRows() != n");
            else if (evd.getLeftEigenvectors().numColumns() != n)
                throw new IllegalArgumentException(
                        "evd.getLeftEigenvectors().numColumns() != n");
        } else if (jobRight == JobEig.All) {
            if (!evd.hasRightEigenvectors())
                throw new IllegalArgumentException(
                        "Eigenvalue decomposition does not store right eigenvectors");
            else if (evd.getRightEigenvectors().numRows() != n)
                throw new IllegalArgumentException(
                        "evd.getRightEigenvectors().numRows() != n");
            else if (evd.getRightEigenvectors().numColumns() != n)
                throw new IllegalArgumentException(
                        "evd.getRightEigenvectors().numColumns() != n");
        }

        int info = Interface.lapack().geev(
                jobLeft,
                jobRight,
                n,
                A.getData(),
                evd.getRealEigenvalues(),
                evd.getImaginaryEigenvalues(),
                jobLeft == JobEig.All ? evd.getLeftEigenvectors().getData()
                        : new double[0],
                jobRight == JobEig.All ? evd.getRightEigenvectors().getData()
                        : new double[0], work, work.length);

        if (info > 0)
            throw new NotConvergedException(
                    NotConvergedException.Reason.Iterations);
        else if (info < 0)
            throw new IllegalArgumentException();

        return evd;
    }

}
