/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.DenseMatrix;
import mt.LowerSPDDenseMatrix;
import mt.LowerTriangDenseMatrix;
import mt.Matrix;
import mt.Matrix.Norm;
import mt.MatrixNotSPDException;
import mt.UpperSPDDenseMatrix;
import mt.UpperTriangDenseMatrix;
import mt.ll.BLASkernel.UpLo;
import mt.ll.Interface;

/**
 * Dense Cholesky decomposition
 */
public class DenseCholesky {

    /**
     * Cholesky decomposition of a lower matrix
     */
    private LowerTriangDenseMatrix Cl;

    /**
     * Cholesky decomposition of an upper matrix
     */
    private UpperTriangDenseMatrix Cu;

    /**
     * If the matrix is SPD or not
     */
    private boolean notspd;

    /**
     * True for upper part, else false
     */
    private boolean upper;

    /**
     * Constructor for DenseCholesky
     * 
     * @param A
     *            Matrix to decompose. Overwritten on exit
     */
    public DenseCholesky(UpperSPDDenseMatrix A) {
        Cu = new UpperTriangDenseMatrix(A, false);
        decompose(upper = true);
    }

    /**
     * Constructor for DenseCholesky
     * 
     * @param A
     *            Matrix to decompose. Overwritten on exit
     */
    public DenseCholesky(LowerSPDDenseMatrix A) {
        Cl = new LowerTriangDenseMatrix(A, false);
        decompose(upper = false);
    }

    private void decompose(boolean upper) {
        int info = 0;
        if (upper)
            info = Interface.lapack().potrf(UpLo.Upper, Cu.numRows(),
                    Cu.getData());
        else
            info = Interface.lapack().potrf(UpLo.Lower, Cl.numRows(),
                    Cl.getData());

        if (info > 0)
            notspd = true;
        else if (info < 0)
            throw new IllegalArgumentException();
    }

    /**
     * Returns true if the matrix decomposed is symmetrical, positive definite
     */
    public boolean isSPD() {
        return !notspd;
    }

    /**
     * Returns the decomposition matrix. Only valid for decomposition of a lower
     * SPD matrix
     */
    public LowerTriangDenseMatrix getL() {
        if (!upper)
            return Cl;
        else
            throw new UnsupportedOperationException();
    }

    /**
     * Returns the decomposition matrix. Only valid for decomposition of a upper
     * SPD matrix
     */
    public UpperTriangDenseMatrix getU() {
        if (upper)
            return Cu;
        else
            throw new UnsupportedOperationException();
    }

    /**
     * Solves for <code>B</code>, overwriting it on return
     */
    public DenseMatrix solve(DenseMatrix B) {
        if (notspd)
            throw new MatrixNotSPDException();

        int info = 0;
        if (upper)
            info = Interface.lapack().potrs(UpLo.Upper, Cu.numRows(),
                    B.numColumns(), Cu.getData(), B.getData());
        else
            info = Interface.lapack().potrs(UpLo.Lower, Cl.numRows(),
                    B.numColumns(), Cl.getData(), B.getData());

        if (info < 0)
            throw new IllegalArgumentException();

        return B;
    }

    /**
     * Computes the reciprocal condition number
     * 
     * @param A
     *            The matrix this is a decomposition of
     * @return The reciprocal condition number. Values close to unity indicate a
     *         well-conditioned system, while numbers close to zero do not.
     */
    public double rcond(Matrix A) {
        double anorm = A.norm(Norm.One);

        int n = A.numRows();
        double[] work = new double[3 * n];
        int[] iwork = new int[n];

        double[] rcond = new double[1];
        int info = 0;
        if (upper)
            info = Interface.lapack().pocon(UpLo.Upper, n, Cu.getData(), anorm,
                    rcond, work, iwork);
        else
            info = Interface.lapack().pocon(UpLo.Lower, n, Cu.getData(), anorm,
                    rcond, work, iwork);

        if (info < 0)
            throw new IllegalArgumentException();

        return rcond[0];
    }

}
