/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MPP.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mpp;

import java.io.IOException;
import java.io.Serializable;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * Logs the sends and recieves of a communicator
 */
public class LoggingCommunicator extends Communicator {

    /**
     * Delegate communicator
     */
    private Communicator comm;

    /**
     * Keeps the logs
     */
    private List[] sends, recvs;

    /**
     * Constructor for LoggingCommunicator
     */
    public LoggingCommunicator(Communicator comm) {
        super(comm.rank(), comm.size());
        this.comm = comm;
        sends = new List[size];
        recvs = new List[size];
        for (int i = 0; i < size; ++i) {
            sends[i] = Collections.synchronizedList(new LinkedList<Event>());
            recvs[i] = Collections.synchronizedList(new LinkedList<Event>());
        }
    }

    /**
     * Returns a list of send events. The members of the lists are Event objects
     */
    public List[] sends() {
        return sends;
    }

    /**
     * Returns a list of recieve events. The members of the lists are Event
     * objects
     */
    public List[] recvs() {
        return recvs;
    }

    public void send(byte[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.send(data, offset, length, peer);
        sends[peer].add(new Event(start, length));
    }

    public void recv(byte[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.recv(data, offset, length, peer);
        recvs[peer].add(new Event(start, length));
    }

    public void send(char[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.send(data, offset, length, peer);
        sends[peer].add(new Event(start, length << 1));
    }

    public void recv(char[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.recv(data, offset, length, peer);
        recvs[peer].add(new Event(start, length << 1));
    }

    public void send(short[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.send(data, offset, length, peer);
        sends[peer].add(new Event(start, length << 1));
    }

    public void recv(short[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.recv(data, offset, length, peer);
        recvs[peer].add(new Event(start, length << 1));
    }

    public void send(int[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.send(data, offset, length, peer);
        sends[peer].add(new Event(start, length << 2));
    }

    public void recv(int[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.recv(data, offset, length, peer);
        recvs[peer].add(new Event(start, length << 2));
    }

    public void send(float[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.send(data, offset, length, peer);
        sends[peer].add(new Event(start, length << 2));
    }

    public void recv(float[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.recv(data, offset, length, peer);
        recvs[peer].add(new Event(start, length << 2));
    }

    public void send(long[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.send(data, offset, length, peer);
        sends[peer].add(new Event(start, length << 3));
    }

    public void recv(long[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.recv(data, offset, length, peer);
        recvs[peer].add(new Event(start, length << 3));
    }

    public void send(double[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.send(data, offset, length, peer);
        sends[peer].add(new Event(start, length << 3));
    }

    public void recv(double[] data, int offset, int length, int peer)
            throws IOException {
        long start = System.currentTimeMillis();
        comm.recv(data, offset, length, peer);
        recvs[peer].add(new Event(start, length << 3));
    }

    /**
     * A send or recieve event
     */
    public class Event implements Serializable {

        static final long serialVersionUID = 378126958044795132L;

        /**
         * Time of the event
         */
        private long time;

        /**
         * Number of bytes in the event
         */
        private int bytes;

        /**
         * Constructor for Event
         */
        Event(long offset, int bytes) {
            time = System.currentTimeMillis() - offset;
            this.bytes = bytes;
        }

        /**
         * Duration of the event in milliseconds
         */
        public long time() {
            return time;
        }

        /**
         * Number of bytes transmitted
         */
        public int bytes() {
            return bytes;
        }
    }

}
