from typing import Dict, Optional, List, Callable

import dspy
from dspy.teleprompt import MIPROv2

from fmcore.prompt_tuner.dspy.datasets.base_dataset import DspyDataset
from fmcore.prompt_tuner.dspy.lm_adapters.dspy_adapter import DSPyLLMAdapter
from fmcore.prompt_tuner.dspy.optimizer_wrapper.base_dspy_optimizer_wrapper import (
    BaseDspyOptimizerWrapper,
)
from fmcore.prompt_tuner.dspy.utils.dspy_utils import DSPyUtils
from fmcore.prompt_tuner.types.enums.optimizer_enums import DSPyOptimizerType
from fmcore.prompt_tuner.types.optimizer_types import BaseOptimizerConfig
from fmcore.prompt_tuner.types.prompt_tuner_types import PromptTunerConfig
from fmcore.utils.introspection_utils import IntrospectionUtils


class MIPROV2OptimizerWrapper(BaseDspyOptimizerWrapper):
    """
    Optimizer class for prompt tuning using DSPy's MIPROv2.

    This optimizer leverages DSPy's MIPROv2 for optimizing prompt generation
    by utilizing both a student model and an optional teacher model. The
    configuration supports default parameters with runtime overrides for
    flexibility and customization.

    Attributes:
        aliases (list): List of supported optimizer types.
        student (dspy.LM): The student model used for prompt generation.
        teacher (Optional[dspy.LM]): The optional teacher model used for guidance.
    """

    aliases = [DSPyOptimizerType.MIPRO_V2]

    student: dspy.LM
    teacher: Optional[dspy.LM]

    @classmethod
    def _get_instance(cls, prompt_tuner_config: PromptTunerConfig) -> "MIPROV2OptimizerWrapper":
        """
        Constructs and configures the necessary components for DSPy prompt tuning.

        This method initializes the signature, module, student model, teacher model (if specified),
        and evaluation function used for optimizing the prompt.

        Args:
            prompt_tuner_config (PromptTunerConfig): Configuration containing prompt tuning settings.

        Returns:
            Dict: A dictionary of parameters used to initialize the DSPy prompt tuning process.
        """
        # Initialize student model and configure DSPy
        optimizer_config: BaseOptimizerConfig = prompt_tuner_config.optimizer_config
        student_model = DSPyLLMAdapter(llm_config=optimizer_config.student_config)
        dspy.configure(lm=student_model)

        # Initialize teacher model (if provided) or use student as fallback
        teacher_model = (
            DSPyLLMAdapter(llm_config=optimizer_config.teacher_config)
            if optimizer_config.teacher_config
            else student_model
        )

        module: dspy.Module = DSPyUtils.create_dspy_signature_from_prompt_config(
            prompt_config=prompt_tuner_config.prompt_config
        )

        # Initialize evaluator for optimization
        # We assume that the evaluator used here would by default return boolean
        # TODO Add validations to ensure only evaluators will bool return types can be used
        evaluate: Callable = DSPyUtils.create_dspy_evaluate_from_evaluator_config(
            evaluator_config=prompt_tuner_config.optimizer_config.evaluator_config
        )

        return MIPROV2OptimizerWrapper(
            student=student_model,
            teacher=teacher_model,
            module=module,
            evaluate=evaluate,
            optimizer_config=optimizer_config,
        )

    def optimize(self, dataset: DspyDataset) -> List[dspy.Module]:
        """
        Optimizes prompts using MIPROv2 with both default and runtime parameters.

        This method runs the MIPROv2 optimization process on the given dataset. It
        compiles the optimizer, runs the optimization on the training and validation
        datasets, and returns a list of optimized prompts.

        Args:
            dataset (DspyDataset): The dataset containing training and validation data.

        Returns:
            List[dspy.Module]: A list of optimized prompts generated by the optimizer.
        """
        # Initialize the MIPROv2 optimizer with filtered parameters
        optimizer_params = self.optimizer_config.optimizer_params.model_dump() or {}
        constructor_params = IntrospectionUtils.filter_params(func=MIPROv2, params=optimizer_params)
        optimizer = MIPROv2(
            metric=self.evaluate,
            prompt_model=self.teacher,
            task_model=self.student,
            **constructor_params,
        )

        # Compile the optimized program using filtered parameters
        compile_params = IntrospectionUtils.filter_params(func=MIPROv2.compile, params=optimizer_params)
        optimized_program = optimizer.compile(
            student=self.module,
            trainset=dataset.train,
            valset=dataset.dev,
            requires_permission_to_run=False,
            **compile_params,
        )

        # Extract optimized candidates from the compiled program
        optimized_candidates = [
            candidate["program"].predictor.predict for candidate in optimized_program.candidate_programs
        ]
        return optimized_candidates
