# Copyright (c) 2013- The Spyder Development Team and Docrepr Contributors
#
# Distributed under the terms of the BSD BSD 3-Clause License

"""Setup for Pytest."""

# Standard library imports
import shutil
import sys
import warnings
import webbrowser
from pathlib import Path

# Third party imports
import pytest
from PIL import Image, ImageChops

# ---- Constants

TEST_DIR = Path(__file__).parent / 'docrepr' / 'tests'

OPEN_BROWSER_OPTION = '--open-browser'
COMPARE_SCREENSHOTS_OPTION = '--compare-screenshots'
UPDATE_REFERENCE_SCREENSHOTS_OPTION = '--update-reference-screenshots'


# ---- Pytest hooks

def pytest_addoption(parser):
    """Add command line options."""
    parser.addoption(
        OPEN_BROWSER_OPTION,
        action='store_true',
        default=False,
        help='For tests that generate HTML output, open it in a web browser',
    )
    parser.addoption(
        COMPARE_SCREENSHOTS_OPTION,
        action='store_true',
        default=False,
        help=('For tests that generate HTML output, '
              'run visual regression tests on it'),
    )
    parser.addoption(
        UPDATE_REFERENCE_SCREENSHOTS_OPTION,
        action='store_true',
        default=False,
        help=('For tests that generate HTML output, '
              'update reference screenshots for the visual regression tests'),
    )


# ---- Fixtures

@pytest.fixture
def open_browser(request):
    """Show the passed URL in the user's web browser if passed."""
    def _open_browser(url):
        if request.config.getoption(OPEN_BROWSER_OPTION):
            warnings.filterwarnings(
                'ignore', category=ResourceWarning, module='subprocess.*')

            webbrowser.open_new_tab(url)
    return _open_browser


@pytest.fixture
async def compare_screenshots(request):
    """Run visual regression test on the output."""
    async def _compare_screenshots(test_id, url):
        if (request.config.getoption(COMPARE_SCREENSHOTS_OPTION) or
           request.config.getoption(UPDATE_REFERENCE_SCREENSHOTS_OPTION)):
            from playwright.async_api import async_playwright

            # Filter warnings generated by playwright
            warnings.filterwarnings(
                'ignore', category=DeprecationWarning, module='pyee.*')

            image = f'test-{test_id}-{sys.platform}.png'
            reference = (TEST_DIR / 'reference_screenshots' / image).resolve()
            screenshot = (TEST_DIR / 'screenshots' / image).resolve()
            diff = (TEST_DIR / 'diffs' / image).resolve()

            # Create directories
            (TEST_DIR / 'screenshots').mkdir(parents=True, exist_ok=True)
            (TEST_DIR / 'diffs').mkdir(parents=True, exist_ok=True)

            # Take a screenshot of the generated HTML
            async with async_playwright() as p:
                browser = await p.firefox.launch()
                page = await browser.new_page()
                await page.goto(f'file://{url}', wait_until='networkidle')

                # Wait for mathjax to finish rendering
                await page.wait_for_selector(
                    '#MathJax_Message', state='hidden')

                await page.screenshot(path=screenshot)
                await browser.close()

            # Compress the screenshot
            screenshot_im = Image.open(screenshot).convert('RGB')
            screenshot_im.save(screenshot, optimize=True, quality=70)

            # If a reference update has been requested, we copy the screenshot
            # and consider the test as "passing"
            if request.config.getoption(UPDATE_REFERENCE_SCREENSHOTS_OPTION):
                shutil.copyfile(screenshot, reference)
                return

            # Compare the screenshot with the reference
            reference_im = Image.open(reference).convert('RGB')
            diff_im = ImageChops.difference(screenshot_im, reference_im)

            bbox = diff_im.getbbox()
            if bbox is not None:
                diff_im.save(diff)

            assert bbox is None, \
                f'{test_id} screenshot and reference do not match'
    return _compare_screenshots
