from pathlib import Path
from dataclasses import dataclass

from simple_slurm import Slurm
from paramiko.client import SSHClient as SSHClient
from ataraxis_data_structures import YamlConfig

def generate_server_credentials(
    output_directory: Path, username: str, password: str, host: str = "cbsuwsun.biohpc.cornell.edu"
) -> None:
    """Generates a new server_credentials.yaml file under the specified directory, using input information.

    This function provides a convenience interface for generating new BioHPC server credential files. Generally, this is
    only used when setting up new host-computers in the lab.
    """
@dataclass()
class ServerCredentials(YamlConfig):
    """This class stores the hostname and credentials used to log into the BioHPC cluster to run Sun lab processing
    pipelines.

    Primarily, this is used as part of the sl-experiment library runtime to start data processing once it is
    transferred to the BioHPC server during preprocessing.
    """

    username: str = ...
    password: str = ...
    host: str = ...

class Server:
    """Encapsulates access to the Sun lab BioHPC processing server.

    This class provides the API that allows accessing the BioHPC server and creating and submitting various
    SLURM-managed jobs to the server. It functions as the central interface used by all processing pipelines in the
    lab to execute costly data processing on the server.

    Notes:
        All lab processing pipelines expect the data to be stored on the server and all processing logic to be packaged
        and installed into dedicated conda environments on the server.

    Args:
        credentials_path: The path to the.yaml file containing the server hostname and access credentials.

        Attributes:
            _open: Tracks whether the connection to the server is open or not.
            _client: Stores the initialized SSHClient instance used to interface with the server.
    """

    _open: bool
    _credentials: ServerCredentials
    _client: SSHClient
    def __init__(self, credentials_path: Path) -> None: ...
    def __del__(self) -> None:
        """If the instance is connected to the server, terminates the connection before the instance is destroyed."""
    @staticmethod
    def generate_slurm_header(
        job_name: str, output_log: Path, error_log: Path, cpus_to_use: int = 20, ram_gb: int = 4, time_limit: int = 60
    ) -> Slurm:
        """Creates a SLURM command object and fills it with initial job configuration data.

        This method is used to generate the initial SLURM command object and fill it with job (SLURM) configuration and
        (general!) conda initialization data. It is used by all processing pipelines in the lab as the initial
        configuration point when writing job shell scripts.

        Notes:
            The command header generated by this method does not contain the command to initialize the specific conda
            environment to be used during processing. This has to be provided as part of the additional command
            configuration, typically by adding the "source activate {ENV_NAME}" subcommand to the end of the header
            returned by this method.

        Args:
            job_name: The descriptive name of the SLURM job to be created.
            output_log: The path to the .txt file on the processing server, where to store the standard output of the
                job.
            error_log: The path to the .txt file on the processing server, where to store the standard error of the
                job.
            cpus_to_use: The number of CPUs to use for the job.
            ram_gb: The amount of RAM to allocate for the job in Gigabytes.
            time_limit: The maximum time limit for the job, in minutes. It is highly advised to set an adequate maximum
                runtime limit to prevent jobs from hogging the server for a long period of time.
        """
    def submit_job(self, slurm_command: Slurm, working_directory: Path) -> str:
        """Submits the input SLURM command to the managed BioHPC server via the shell script.

        This method submits various commands for execution via SLURM-managed BioHPC cluster. As part of its runtime, the
        method translates the Slurm object into the shell script, moves the script to the target working directory on
        the server, and instructs the server to execute the shell script (via SLURM).

        Args:
            slurm_command: The Slurm (command) object containing the job configuration and individual commands to run
                as part of the processing pipeline.
            working_directory: The path to the working directory on the server where the shell script is moved
                and executed.

        Returns:
            The job ID assigned to the job by SLURM manager if the command submission is successful.

        Raises:
            RuntimeError: If the command submission to the server fails.
        """
    def job_complete(self, job_id: str) -> bool:
        """Returns True if the job with the given ID has been completed or terminated its runtime due to an error.

        If the job is still running or is waiting inside the execution queue, returns False.

        Args:
            job_id: The numeric ID of the job to check, assigned by SLURM.
        """
    def close(self) -> None:
        """Closes the SSH connection to the server.

        This method has to be called before destroying the class instance to ensure proper resource cleanup.
        """
