# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['prisma_cloud_pipeline']

package_data = \
{'': ['*']}

install_requires = \
['jq>=1.1,<1.2',
 'jsonschema>=3.2,<3.3',
 'pyyaml>=5.3,<5.5',
 'requests>=2.25,<2.26',
 'tabulate>=0.8,<0.9']

entry_points = \
{'console_scripts': ['prisma-cloud-pipeline = '
                     'prisma_cloud_pipeline.__main__:main']}

setup_kwargs = {
    'name': 'prisma-cloud-pipeline',
    'version': '0.1.3',
    'description': 'Export Prisma Cloud container findings to a CI pipeline, and identify un-triaged findings',
    'long_description': '# Prisma Cloud Pipeline Triage\n\nExport Prisma Cloud container findings to a CI pipeline, and identify un-triaged findings.\n\nPrisma Cloud\'s container scanning feature (formerly called Twistlock) has a web UI to review findings in. You can also\ndefine\n[triage rules](https://docs.twistlock.com/docs/compute_edition/vulnerability_management/vuln_management_rules.html) to\nignore findings. There are a number of [example integrations](https://github.com/twistlock/sample-code/blob/master/CI/)\ninto CI pipelines, which all follow the same pattern: scan a specific docker image that is present in the pipeline,\nreport any issues found, and optionally fail if a certain \'badness\' threshold is met. These are useful, but limited.\n\nThe motivation for this project is to get _all_ findings closer to developers, not just findings for a specific\ncontainer, and integrate the entire process with existing project CI pipelines. All findings (for specified collections)\nare retrieved from the Prisma Cloud API, and a set of locally defined triage rules are applied to suppress specific\nfindings or containers from the main output.\n\nThis:\n\n1.  Provides clear visibility of any new un-triaged findings, wherever they are in a deployment, and optionally allows\n    the pipeline to fail if there are any.\n\n2.  Enables triage of issues to follow the usual merge/pull request approach. If there are any new findings or false\n    positives, they need to be either fixed or added to the triage rules file before the pipeline passes again.\n\n3.  Means development teams don\'t need to use or learn the web UI unless they want the extra functionality it offers.\n\n## Example\n\nTriage rules look like this:\n\n```yaml\n- matches: GKE system components\n  containerFilter: .namespace == "kube-system"\n  rationale:\n    We use an auto-updating GKE instance that gets patches - these will either be false positives or will get patched\n    shortly by Google. There\'s nothing we can do about them.\n\n- matches: Limit memory\n  complianceFilter: .id == 510\n  rationale: We don\'t run untrusted containers; whilst this is a nice to have, it is very low priority.\n\n- matches: twistlock defender privileges\n  containerFilter: .namespace == "twistlockdefender"\n  complianceFilter: .id | IN([599,515,59,520,525,531,55,528,521,51][])\n  rationale: >-\n    The defender needs full access to the host to monitor everything.\n\n     Issues ignored: 599-root; 515-host PID ns; 59-host network ns; 520-host UTS ns; 525-extra privs; \n     531-docker socket; 55-sensitive dir mounts; 528-pid cgroup limit; 521-default seccomp profile; 51-apparmor profile\n\n     The PID cgroup limit could be implemented, but it\'s of negligible importance given the rest.\n\n- matches: heartbleed for the proj/foo-* containers\n  containerFilter: .imageName | test("gcr.io/proj/foo-")\n  vulnFilter: .cve == "CVE-2014-0160"\n  rationale:\n    We aren\'t exposed to heartbleed in foo-x, foo-y, or foo-zzz because we manually disable heartbeats. Raised an issue\n    to fix it anyway.\n  issue: JIRA-1234\n```\n\nAfter filtering out any findings that match any of these rules, a summary of the remaining findings are presented in a\ncondensed textual output:\n\n```\nPrior to triage filter, got 146 distinct running containers with findings, 45 vulnerabilities, 862 compliance issues\nAfter triage filter, got 2 distinct running containers with findings, 1 vulnerabilities, 15 compliance issues\n\n+-------------------------------------------------+-------------+-----------------+--------------------+\n| Triage Rule                                     |   Container |   Vulnerability |   Compliance Issue |\n|                                                 |     Matches |         Matches |            Matches |\n|-------------------------------------------------+-------------+-----------------+--------------------|\n| GKE system components                           |          24 |               4 |                158 |\n| Limit memory                                    |           0 |               0 |                 17 |\n| heartbleed for the foo containers               |           0 |               3 |                  0 |\n.                                                 .             .                 .                    .\n.                                                 .             .                 .                    .\n+-------------------------------------------------+-------------+-----------------+--------------------+\nFor container rules, the entries in the Vulnerabilities and Compliance Issues columns refer to the number of findings the matched containers had.\nFor vuln/compliance rules, the entries in the Containers column refer to the number of containers that had no findings left after this rule was processed.\nFor details on what each rule matched, review the file specified with the --triaged-findings flag.\n\n╒═════════════════════════════════════════╤════════════════════════════════════════════╤════════════════════════════════════════════════════════════════════════════════════╕\n│ Container                               │ Vulnerabilities                            │ Compliance Issues                                                                  │\n╞═════════════════════════════════════════╪════════════════════════════════════════════╪════════════════════════════════════════════════════════════════════════════════════╡\n│ tmp-shell                               │ CVE-2020-11984 in apache2 (critical) fixed │ 599: Container is running as root                                                  │\n│ ns:   debug                             │  in 2.4.46-r0                              │ 512: (CIS_Docker_CE_v1.1.0 - 5.12) Mount container\'s root filesystem as read only  │\n│ img:  netshoot:latest                   │                                            │ 521: (CIS_Docker_CE_v1.1.0 - 5.21) Do not disable default seccomp profile          │\n│ acct: proj-dev-eu-1                     │                                            │ 525: (CIS_Docker_CE_v1.1.0 - 5.25) Restrict container from acquiring additional    │\n│                                         │                                            │   privileges                                                                       │\n│                                         │                                            │ 528: (CIS_Docker_CE_v1.1.0 - 5.28) Use PIDs cgroup limit                           │\n├─────────────────────────────────────────┼────────────────────────────────────────────┼────────────────────────────────────────────────────────────────────────────────────┤\n│ twistlock-defender                      │                                            │ 599: Container is running as root                                                  │\n│ ns:   twistlockdefender                 │                                            │ 51: (CIS_Docker_CE_v1.1.0 - 5.1) Verify AppArmor profile, if applicable            │\n│ img:  defender:defender_20_04_163       │                                            │ 515: (CIS_Docker_CE_v1.1.0 - 5.15) Do not share the host\'s process namespace       │\n│ acct: proj-prod-eu-1                    │                                            │ 59: (CIS_Docker_CE_v1.1.0 - 5.9) Do not share the host\'s network namespace         │\n│                                         │                                            │ 520: (CIS_Docker_CE_v1.1.0 - 5.20) Do not share the host\'s UTS namespace           │\n│                                         │                                            │ 521: (CIS_Docker_CE_v1.1.0 - 5.21) Do not disable default seccomp profile          │\n│                                         │                                            │ 525: (CIS_Docker_CE_v1.1.0 - 5.25) Restrict container from acquiring additional    │\n│                                         │                                            │   privileges                                                                       │\n│                                         │                                            │ 528: (CIS_Docker_CE_v1.1.0 - 5.28) Use PIDs cgroup limit                           │\n│                                         │                                            │ 531: (CIS_Docker_CE_v1.1.0 - 5.31) Do not mount the Docker socket inside any       │\n│                                         │                                            │   containers                                                                       │\n│                                         │                                            │ 55: (CIS_Docker_CE_v1.1.0 - 5.5) Do not mount sensitive host system directories on │\n│                                         │                                            │   containers                                                                       │\n╘═════════════════════════════════════════╧════════════════════════════════════════════╧════════════════════════════════════════════════════════════════════════════════════╛\n\nOutstanding issues in triage rules:\n    PROJ-321\nOnce an issue is closed, the corresponding triage rule should be removed so regressions will be detected.\n```\n\n## Usage\n\nThe recommended way to run the tool is via the Docker container in your pipeline. Here\'s an example Gitlab job\ndefinition, where USER and PASS are predefined CI variables for an account that can read from the API:\n\n```yaml\nscan:\n  image:\n    name: safenetlabs/prisma-cloud-pipeline\n    entrypoint: [\'\'] # allow gitlab to run its own commands\n  variables:\n    API: https//twistlock.example.com:8083/api\n  script:\n    - export TOKEN=$(http --ignore-stdin $API/v1/authenticate username="$USER" password="$PASS" | jq -r .token)\n    # this command and hence job will fail (due to --check) if there are any findings in the col1 or col2 collections\n    # that are not matched by a rule in prisma-triage.yaml in the repo\n    - prisma-cloud-pipeline --api=$API --collections=col1,col2 --rules=prisma-triage.yaml --check\n      --results=untriaged-findings.json --triaged-findings=triaged-findings.json\n  artifacts:\n    when: always\n    paths:\n      - untriaged-findings.json # refer to this artifact or the Prisma Cloud Compute UI if the summary output in the job is insufficient\n      - triaged-findings.json # refer to this artifact to validate that the triage rules aren\'t ignoring more than they should\n  allow_failure: true # we want to be alerted if there is a new finding, but we don\'t want it to stop the pipeline from working\n```\n\nFull usage can be found with `docker run --rm safenetlabs/prisma-cloud-pipeline --help`.\n\nThe text output from the tool provides a summary of all of the untriaged findings; the full details (as returned by the\nAPI) are saved to the file specified by the --results flag (if present).\n\nYou can also use it in an offline manner, where it doesn\'t have direct access to the twistlock API. One invocation with\n`--save` is used to retrieve the results, and a later invocation with `--data` can process that file, instead of\naccessing the API.\n\nIf your API has a certificate from an untrusted root, set the REQUESTS_CA_BUNDLE environment variable, e.g.:\n`REQUESTS_CA_BUNDLE=mycert.pem prisma-cloud-pipeline $API ...`\n\nSpecify `--finding-stats` to get a count of how many times each untriaged finding occurred.\n\n### Local use\n\nTo run the tool locally, Python 3.8+ is required. Try this from a directory that contains a `triage.yaml` file with your\nrules:\n\n```sh\npip install prisma-cloud-pipeline\nexport TOKEN=$(http $API/v1/authenticate username=$USER password=$PASS | jq -r .token)\nprisma-cloud-pipeline --api=$API --rules=triage.yaml --collections=mycol,anothercol --results=results.json\n```\n\nYou can of course also use the Docker container locally, instead of installing via pip.\n\n## Triage Rules\n\nYou\'ll want to write triage rules for various reasons:\n\n1.  A finding doesn\'t apply - it\'s a false positive, or it doesn\'t matter in the circumstances of a particular\n    container.\n2.  It\'s a valid finding, but you don\'t care: it\'s for a container you can\'t control; it\'s not serious.\n3.  You\'re going to fix a finding, but haven\'t yet, and in the mean time don\'t want your pipeline failing.\n\nA triage rules file has up to three keys: `containers`, `vulnerabilities`, and `complianceIssues`. Each key can have any\nnumber of rules associated with it.\n\nThe basic format of a triage rule is:\n\n```yaml\n- matches: <a title for the rule>\n  rationale: <why this rule exists, for example why a reported vulnerability isn\'t a problem in this case>\n  issue: <an optional reference to an issue to fix this finding, e.g. in Jira or GitHub>\n  containerFilter: <see below>\n  vulnFilter:\n  complianceFilter:\n  expires: <date in YYYY-MM-DD format>\n```\n\nThe rules under the `containers` key can only contain a `containerFilter`. Any container that matches this filter will\nbe excluded, so be very sure you don\'t care about any possible finding before using this.\n\nThe rules under `vulnerabilities` and `complianceIssues` must include either a `vulnFilter` or a `complianceFilter`\nrespectively, and both can optionally include a `containerFilter`. If a `containerFilter` is not specified, then such a\nrule matches _every_ occurrence of a matching vulnerability or compliance issue, wherever it is found. If a\n`containerFilter` is specified, then the rule will only exclude findings from those containers that match.\n\nAny rule can have an `expires` key; this tells the tool to ignore the rule after it has expired. This is useful for when\nyou have fixed an issue, but the fix hasn\'t propagated across the whole system yet (or across any of it) - you want to\ntemporarily ignore that finding, on the assumption that it is going to go away shortly. Once the rule expires, if the\nfinding is still present then you will be notified - apparently the fix didn\'t work or took longer than you expected to\npropagate.\n\n### Writing Filters\n\nThe format of the filters is a [jq filter](https://stedolan.github.io/jq/manual/) that outputs `true` if the filter\nmatches (i.e. the container/finding has been triaged) and `false` otherwise.\n\nThe input to the filters - the value of `.` - is data taken directly from the Prisma Cloud API - this means you can\nfilter on any attributes that Prisma reports. The container filter input is an entry from the `radar/containers` API\nendpoint; the vulnerability filter input is the matching `vulnerabilities` field from the `images` API endpoint; the\ncompliance issues filter input is the matching `info/complianceIssues` field from the `containers` API endpoint. To see\nthe full set of data on which the filters operate, run the tool with the `--results=file.json` option, and inspect the\nresults file.\n\nHere are some more example filter combinations to show what you can do (note that a valid rule also needs a `rationale`\nentry, and must be under one of the three top-level keys in the rules file):\n\n```yaml\n- matches: All containers in staging\n  containerFilter: .accountID | test("my-staging-AWS-account")\n\n- matches: Running a particular set of containers as root\n  containerFilter: .imageName | test("datadog/cluster-agent")\n  complianceFilter: .id == 599\n\n- matches: Any compliance issue coming from a particular pod\n  containerFilter: .labels | map(test("^io.kubernetes.pod.name:verycompliantpod-")) | any\n  complianceFilter: \'true\'\n\n- matches: All low sev compliance issues\n  complianceFilter: .severity == "low"\n\n- matches: all python vulnerabilities\n  vulnFilter: \'.packageName | test("^python[.0-9]+$")\'\n```\n\nBe careful when writing filters - if your filter is overly broad you can easily lose findings you care about. The output\nreports how many containers/findings each rule suppressed - you can review this to check it matches your expectations.\nFor a more thorough review, use the `--triaged-findings` flag to specify a file to save details on what each rule\nmatched.\n\n## Limitations\n\nThis tool does not handle "runtime events" findings. Whilst they could be incorporated in the same manner that\nvulnerabilities and compliance issues are currently handled, runtime events are inherently more ephemeral and thus less\nwell suited to being managed in the same pipeline that builds and deployment use. If you want to follow a similar\napproach to triage and handling of runtime events, perhaps running it in a dedicated secops pipeline, PRs are welcome!\n',
    'author': 'Michael Macnair',
    'author_email': 'michael.macnair@thalesgroup.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/thalescpl-io/prisma-cloud-pipeline',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
