"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenerateCSV = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda_ = require("aws-cdk-lib/aws-lambda");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
class TextractGenerateCSV extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenerateCSV.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var lambdaMemoryMB = 1048;
        if (typeof (props.lambdaMemoryMB != 0) && props.lambdaMemoryMB) {
            lambdaMemoryMB = props.lambdaMemoryMB;
        }
        var outputType = props.outputType === undefined ? 'CSV' : props.outputType;
        var lambdaLogLevel = 'INFO';
        if (typeof (props.lambdaLogLevel) != 'undefined' && !props.lambdaMemoryMB) {
            lambdaLogLevel = props.lambdaLogLevel;
        }
        var lambdaTimeout = 900;
        if (typeof (props.lambdaTimeout != 0) && props.lambdaTimeout) {
            lambdaTimeout = props.lambdaTimeout;
        }
        const csvGeneratorFunction = new lambda_.DockerImageFunction(this, 'TextractCSVGenerator', {
            code: lambda_.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/generatecsv/')),
            memorySize: lambdaMemoryMB,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            environment: {
                CSV_S3_OUTPUT_BUCKET: props.csvS3OutputBucket,
                CSV_S3_OUTPUT_PREFIX: props.csvS3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                OUTPUT_TYPE: outputType,
            },
        });
        csvGeneratorFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['s3:PutObject', 's3:Get*', 's3:List*'], resources: ['*'] }));
        csvGeneratorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.generateCSVLambda = csvGeneratorFunction;
        this.generateCSVLogGroup = csvGeneratorFunction.logGroup;
        const csvGeneratorLambdaInvoke = new tasks.LambdaInvoke(this, 'csvGeneratorInvoke', {
            lambdaFunction: csvGeneratorFunction,
        });
        csvGeneratorLambdaInvoke.addCatch(new sfn.Fail(this, 'csvGenerationFailure'), {
            errors: [sfn.Errors.ALL],
        });
        const workflow_chain = sfn.Chain.start(csvGeneratorLambdaInvoke);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            tracingEnabled: true,
        });
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
           * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
           *
           * the scoped access policy should be generated accordingly.
           *
           * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
           */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenerateCSV = TextractGenerateCSV;
_a = JSII_RTTI_SYMBOL_1;
TextractGenerateCSV[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenerateCSV", version: "0.0.1" };
TextractGenerateCSV.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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