"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericAsyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sns = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
var textractSync_1 = require("./textractSync");
Object.defineProperty(exports, "TextractGenericSyncSfnTask", { enumerable: true, get: function () { return textractSync_1.TextractGenericSyncSfnTask; } });
var textractDecider_1 = require("./textractDecider");
Object.defineProperty(exports, "TextractPOCDecider", { enumerable: true, get: function () { return textractDecider_1.TextractPOCDecider; } });
var textractClassificationConfigurator_1 = require("./textractClassificationConfigurator");
Object.defineProperty(exports, "TextractClassificationConfigurator", { enumerable: true, get: function () { return textractClassificationConfigurator_1.TextractClassificationConfigurator; } });
var textractOutputConfigToJSON_1 = require("./textractOutputConfigToJSON");
Object.defineProperty(exports, "TextractAsyncToJSON", { enumerable: true, get: function () { return textractOutputConfigToJSON_1.TextractAsyncToJSON; } });
var textractGenerateCSV_1 = require("./textractGenerateCSV");
Object.defineProperty(exports, "TextractGenerateCSV", { enumerable: true, get: function () { return textractGenerateCSV_1.TextractGenerateCSV; } });
var textractA2I_1 = require("./textractA2I");
Object.defineProperty(exports, "TextractA2ISfnTask", { enumerable: true, get: function () { return textractA2I_1.TextractA2ISfnTask; } });
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
class TextractGenericAsyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractAsyncCallMaxRetries = props.textractAsyncCallMaxRetries === undefined ? 100 : props.textractAsyncCallMaxRetries;
        var textractAsyncCallBackoffRate = props.textractAsyncCallBackoffRate === undefined ? 1.1 : props.textractAsyncCallBackoffRate;
        var textractAsyncCallInterval = props.textractAsyncCallInterval === undefined ? 1 : props.textractAsyncCallInterval;
        const taskTokenTable = new dynamodb.Table(this, 'TextractTaskTokenTable', {
            partitionKey: {
                name: 'ID',
                type: dynamodb.AttributeType.STRING,
            },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            timeToLiveAttribute: 'ttltimestamp',
        });
        this.taskTokenTableName = taskTokenTable.tableName;
        // const duration: number = 30;
        // this.textractAsyncSQS = new sqs.Queue(this, 'StartTPSRequests', {
        //   visibilityTimeout: Duration.seconds(6*duration),
        // });
        // this.textractPutOnSQSFunction = new lambda.DockerImageFunction(this, 'PutOnSQS', {
        //   code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/put_on_sqs/')),
        //   memorySize: 128,
        //   timeout: Duration.seconds(duration),
        //   environment: {
        //     LOG_LEVEL: lambdaLogLevel,
        //     SQS_QUEUE_URL: this.textractAsyncSQS.queueUrl,
        //   },
        // });
        // this.textractPutOnSQSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sqs:SendMessage'], resources: [this.textractAsyncSQS.queueArn] }));
        // this.putOnSQSLambdaLogGroup = (<lambda.Function> this.textractPutOnSQSFunction).logGroup;
        // const putOnSQSFunctionInvoke = new tasks.LambdaInvoke(this, 'PutOnSQSFunctionInvoke', {
        //   lambdaFunction: this.textractPutOnSQSFunction,
        // });
        this.textractAsyncSNSRole = new iam.Role(this, 'TextractAsyncSNSRole', {
            assumedBy: new iam.ServicePrincipal('textract.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSQSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSNSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonTextractFullAccess')],
        });
        this.textractAsyncSNS = new sns.Topic(this, 'TextractAsyncSNS');
        this.textractAsyncCallFunction = new lambda.DockerImageFunction(this, 'TextractAsyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async/')),
            memorySize: 128,
            environment: {
                NOTIFICATION_SNS: this.textractAsyncSNS.topicArn,
                NOTIFICATION_ROLE_ARN: this.textractAsyncSNSRole.roleArn,
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                // SQS_QUEUE_URL: this.textractAsyncSQS.queueName,
                TEXTRACT_API: textractAPI,
            },
        });
        const textractAsyncCallTask = new tasks.LambdaInvoke(this, 'TextractAsyncCallTask', { lambdaFunction: this.textractAsyncCallFunction });
        textractAsyncCallTask.addRetry({
            maxAttempts: textractAsyncCallMaxRetries,
            backoffRate: textractAsyncCallBackoffRate,
            interval: aws_cdk_lib_1.Duration.seconds(textractAsyncCallInterval),
            errors: ['ThrottlingException', 'LimitExceededException', 'InternalServerError', 'ProvisionedThroughputExceededException'],
        });
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'textract:Start*',
                'textract:Get*',
            ],
            resources: ['*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: [this.textractAsyncSNS.topicArn] }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:PutItem', 'dynamodb:GetItem'], resources: [taskTokenTable.tableArn] }));
        this.startTextractLambdaLogGroup = this.textractAsyncCallFunction.logGroup;
        this.textractAsyncReceiveSNSFunction = new lambda.DockerImageFunction(this, 'TextractAsyncSNSFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async_sns_listener/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            environment: {
                TOKEN_STORE_DDB: this.taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
            },
        });
        this.textractAsyncSNS.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.textractAsyncReceiveSNSFunction));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:GetItem'], resources: ['*'] }));
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:Put*', 's3:List*'],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/${props.s3TempOutputPrefix}/*`],
        }));
        this.receiveStartSNSLambdaLogGroup = this.textractAsyncReceiveSNSFunction.logGroup;
        const workflow_chain = sfn.Chain.start(textractAsyncCallTask);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.textractAsyncReceiveSNSFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        const appName = this.node.tryGetContext('appName');
        const customMetricNamespace = 'TextractConstructGenericAsync';
        // OPERATIONAL
        const asyncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Duration',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                .whereString('message', '=', `textract_async_${textractAPI}_job_duration_in_ms:`),
            metricValue: '$durationMs',
        });
        const asyncDurationMetric = asyncDurationMetricFilter.metric({ statistic: 'avg' });
        const asyncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
            metricValue: '$pages',
        });
        const asyncNumberPagesMetric = asyncNumberPagesMetricFilter.metric({ statistic: 'sum' });
        // TODO: expose all filters as properties of Construct, so users can build their own metrics
        const asyncJobFinshedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsFinishedFilter`, {
            logGroup: this.textractAsyncReceiveSNSFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsFinished',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_processed:`),
            metricValue: '1',
        });
        const asyncJobFinshedMetric = asyncJobFinshedMetricFilter.metric({ statistic: 'sum' });
        const asyncJobStartedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsStartedFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsStarted',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message')
                .whereString('message', '=', `textract_async_${textractAPI}_job_started`),
            metricValue: '1',
        });
        const asyncJobStartedMetric = asyncJobStartedMetricFilter.metric({ statistic: 'sum' });
        const asyncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', `textract_async_${textractAPI}_number_of_pages_send_to_process:`),
            metricValue: '$pages',
        });
        const asyncNumberPagesSendMetric = asyncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
        // OPERATIONAL STOP
        // CALCUATED OPERATIONAL METRICS
        const pagesPerSecond = new cloudwatch.MathExpression({
            expression: 'pages / (duration / 1000)',
            usingMetrics: {
                pages: asyncNumberPagesMetric,
                duration: asyncDurationMetric,
            },
        });
        const openJobs = new cloudwatch.MathExpression({
            expression: 'startedJobs - finishedJobs',
            usingMetrics: {
                startedJobs: asyncJobStartedMetric,
                finishedJobs: asyncJobFinshedMetric,
            },
        });
        // CALCUATED OPERATIONAL METRICS STOP
        const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Errors',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
            metricValue: '1',
        });
        const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'LimitExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
            metricValue: '1',
        });
        const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ThrottlingException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
            metricValue: '1',
        });
        const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ProvisionedThroughputExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
            metricValue: '1',
        });
        const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
            logGroup: this.textractAsyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'InternalServerError',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
            metricValue: '1',
        });
        const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
        const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
        const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
        const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
        const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
        const textractStartDocumentTextThrottle = new cloudwatch.Metric({
            namespace: 'AWS/Textract',
            metricName: 'ThrottledCount',
            dimensionsMap: {
                Operation: 'StartDocumentTextDetection',
            },
            statistic: 'sum',
        });
        // DASHBOARD LAMBDA
        const dashboardWidth = 24;
        // const widgetStandardHeight=9;
        // DASHBOARD
        new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
            end: 'end',
            periodOverride: cloudwatch.PeriodOverride.AUTO,
            start: 'start',
            widgets: [
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'OpenJobs', left: [openJobs], width: Math.floor(dashboardWidth / 2) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [asyncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [asyncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPages', left: [asyncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'JobsStartedAndFinished', left: [asyncJobFinshedMetric], right: [asyncJobStartedMetric], width: Math.floor(dashboardWidth / 4) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractAsyncCallFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Errors', left: [this.textractAsyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Invocations', left: [this.textractAsyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'Async-Function-Throttles', left: [this.textractAsyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractAsyncReceiveSNSFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Errors', left: [this.textractAsyncReceiveSNSFunction.metricErrors()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Invocations', left: [this.textractAsyncReceiveSNSFunction.metricInvocations()], width: Math.floor(dashboardWidth / 3) }),
                    new cloudwatch.GraphWidget({ title: 'SNS-Function-Throttles', left: [this.textractAsyncReceiveSNSFunction.metricThrottles()], width: Math.floor(dashboardWidth / 3) }),
                ],
            ],
        });
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericAsyncSfnTask = TextractGenericAsyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericAsyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenericAsyncSfnTask", version: "0.0.1" };
TextractGenericAsyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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