
# -*- coding: utf-8 -*_
#
# Copyright (c) 2020, Pureport, Inc.
# All Rights Reserved

"""
The Pureport exceptions module provies all error classes used in this
project.  It provides a class hierarchy where all errors derive from the
```PureportClientError``` exception.

::

    PureportClientError
      +-- PureportConnectionError
      |     +-- ConnectionOperationTimeoutError
      |     +-- ConnectionOperationFailedError
      +-- MissingAccessTokenError
      +-- ClientHttpError


The exceptions module is pureposefully kept small and additional exception
classes should ony be added if there is a specific reason to do so.
Good reasons include adding mandatory arguments such as the case with
```ClientHttpError```.
"""

from __future__ import absolute_import


class PureportClientError(Exception):
    """ Base error class for all pureport-client exceptions

    This exception class represents the base exception from which all
    exceptions generated by pureport-client should derive from.  It exposes
    two properties: message and exc.

    :param message: human readable error message
    :type message: str

    :param exc: optional inner exception to expose
    :type exc: Error
    """

    def __init__(self, message, exc=None):
        self._message = message
        self._exc = exc

    @property
    def message(self):
        return self._message

    @property
    def exc(self):
        return self._exc


class PureportConnectionError(PureportClientError):

    def __init__(self, *args, **kwargs):
        """ Base error class for all Pureport connection errors

        :param connection: the connection
        :type connection: Connection
        """
        self._connection = kwargs.pop('connection', None)
        super(PureportConnectionError, self).__init__(*args, **kwargs)

    @property
    def connection(self):
        return self._connection


class MissingAccessTokenError(PureportClientError):
    pass


class ConnectionOperationTimeoutError(PureportConnectionError):
    """A connection operation that too long to complete
    """
    pass


class ConnectionOperationFailedError(PureportConnectionError):
    """A connection opertion that failed to complete
    """
    pass


class ClientHttpError(PureportClientError):

    def __init__(self, status_code, reason, *args, **kwargs):
        """ An exception representing a bad http call from the client

        :param status_code: The numeric HTTP status code return from
            the remote server
        :type status_code: int

        :param reason: Short string description of the status code
        :type reason: str
        """
        self._status_code = status_code
        self._reason = reason
        super(ClientHttpError, self).__init__(*args, **kwargs)

    @property
    def status_code(self):
        return self._status_code

    @property
    def reason(self):
        return self._reason
