# Migrate to Pyoframe v1.0

Pyoframe version 1.0 brings major improvements :tada:

- **New features** including support for the [Ipopt](https://coin-or.github.io/Ipopt/) solver and the ability to improve performance with [`pf.Config.maintain_order=False`][pyoframe._Config.maintain_order].
- **Improved performance** after rewriting internal functions based on extensive benchmarking.
- **Smoother developer experience**, notably error messages explain the mathematical operations that led to an error and Pyoframe objects now print as legible tables.
- **Improved documentation** including a revamped [API page](../../reference/index.md), an [examples page](../../examples/index.md), new explanations of functions like [`.over(…)`](../concepts/special-functions.md#adding-expressions-with-differing-dimensions-using-over), and the ability to view previous versions of the documentation.
- **More readable syntax** like our decision to rename `add_dim` to `over` and the changes to `.sum()` (see below).

Unfortunately, these improvements involve some breaking changes. This page will guide you in migrating your code to v1.0.

## Breaking changes

<!-- skip: start "Deprecated code would throw errors" -->

Summary:

1. [`pf.Model(…)` has a new signature](#1-update-all-calls-to-pfmodel)
2. [`add_dim(…)` was renamed to `over(…)`](#2-replace-pfsum-obj-with-objsum)
3. [`pf.sum(…, obj)` was replaced by `obj.sum(…)`](#3-rename-add_dim-to-over)
4. [Other less common breaking changes](#4-review-code-for-other-breaking-changes)

Follow the steps below to upgrade to v1.0.

### 1. Update all calls to `pf.Model(…)`

The signature of [`pf.Model(…)`][pyoframe.Model] has changed from

```
Model(name=None, solver=None, solver_env=None, use_var_names=False, sense=None)
```

to

```
Model(solver=None, solver_env=None, *, name=None, solver_uses_variable_names=False, print_uses_variable_names=True, sense=None)
```

Importantly, notice how,

- `name` is now a _named-only_ parameter (you must write `name=`), and

- `use_var_names` was renamed to `solver_uses_variable_names` (to avoid confusion with the new option `print_uses_variable_names`).

Please update all calls to `pf.Model(…)` accordingly.

```python
import pyoframe as pf

pf.Model("my-model")  # before
pf.Model(name="my-model")  # after

pf.Model(use_var_names=True)  # before
pf.Model(solver_uses_variable_names=True)  # after
```

### 2. Replace `pf.sum(…, obj)` with `obj.sum(…)`

To improve readability and align with Python best practices, both `pf.sum(…, obj)` and `pf.sum_by(…, obj)` have been replaced by `obj.sum(…)` and `obj.sum_by(…)`. Here `obj` represents any Pyoframe [Variable][pyoframe.Variable] or [Expression][pyoframe.Expression].[^1] For example:

```python
pf.sum(m.X)  # before
m.X.sum()  # after

pf.sum(["day", "hour"], m.X)  # before
m.X.sum("day", "hour")  # after

pf.sum_by("city", m.X)  # before
m.X.sum_by("city")  # after
```

Please update all uses of `pf.sum` and `pf.sum_by` accordingly.

[^1]: If you were using `pf.sum` or `pf.sum_by` on a Polars or Pandas DataFrame instead of on a Pyoframe object, you should use the `.sum()` function native to Pandas/Polars.

### 3. Rename `.add_dim(…)` to `.over(…)`

For clarity, `.add_dim(…)` was renamed to [`.over(…)`](../concepts/special-functions.md#adding-expressions-with-differing-dimensions-using-over). Please update your code accordingly.

```python
m.X.add_dim("time")  # before
m.X.over("time")  # after
```

### 4. Review code for other breaking changes

Steps 1 to 3 should cover most breaking changes. For completeness, below are
some additional breaking changes that you are less likely to encounter.

1. All submodules (e.g. `pyoframe.core`) and some attributes (`obj.unmatched_strategy`, `obj.allowed_new_dims`, `obj.dimensions_unsafe`, `Config.enable_is_duplicated_expression_safety_check`) have been renamed to begin with an underscore (e.g. now `pyoframe._core`) to indicate that these elements are no longer part of the public API.

2. If you try to read [`model.objective`][pyoframe.Model.objective] before having defined an objective, an error will be raised. Prior to v1.0 `None` was returned.

3. `.over(…)`, `.keep_unmatched()`, and `.drop_unmatched()` now have a well-defined "order of operations." They must be applied *after* all other transforms. For example, `my_obj.drop_unmatched().sum("time")` no longer works because `my_obj.sum("time").drop_unmatched()` should be used instead. [Learn more](../concepts/special-functions.md#addition-modifiers).

4. `pf.Config.print_max_set_elements`, `pf.Config.print_max_line_length`, and `pf.Config.print_max_lines` no longer exist. Use [`pf.Config.print_max_terms`][pyoframe._Config.print_max_terms] [`pf.Config.print_polars_config`][pyoframe._Config.print_polars_config] instead.

5. `pf.Config.print_uses_variable_names` no longer exists. Use the equivalent option in [`pf.Model(…)`][pyoframe.Model]

6. `pf.Config.default_solver` now defaults to `"auto"` instead of `None` to clarify that by default, the solver will be automatically detected. Don't set `pf.Config.default_solver = None`.

7. [`Model.write(…)`][pyoframe.Model.write] is no longer available when using Gurobi and [`solver_uses_variable_names=False`][pyoframe.Model].

<!-- skip: end -->