#!/usr/bin/env python3
""""""
from configparser import ConfigParser
from argparse import (
    ArgumentParser,
    ArgumentDefaultsHelpFormatter,
    RawDescriptionHelpFormatter,
)
from os import environ
import json
from pathlib import Path
from importlib import import_module
import sys
from typing import Optional
from subprocess import check_output
from dataclasses import asdict


sys.path.insert(0, Path(__file__).parent.parent.name)
aws_myfa = import_module('aws_myfa')


class ArgparseFormatter(ArgumentDefaultsHelpFormatter, RawDescriptionHelpFormatter):
    """A formatter class for showing default values and not inserting new line
       when showing the help text header
    """


def get_parser():
    """
    """

    parser = ArgumentParser(
        prog=Path(__file__).parent.name.replace('_', '-'),
        description='\n'.join([aws_myfa.__doc__, __doc__]),
        epilog = 'Your current AWS CLI profiles will be retained.',
        formatter_class = ArgparseFormatter,
    )

    parser.add_argument('session-profile-name',
        help = "name of AWS CLI profile to create a new source profile (credentials) for",
        metavar = "TARGET_PROFILE",
    )

    parser.add_argument('principal-profile-name',
        help = "name of AWS CLI profile to use when requesting an STS session token with MFA (leave it empty, to delegate switching to your environment variables or defaults)",
        metavar = "PRINCIPAL_PROFILE",
        nargs = "?",
    )

    parser.add_argument('--config',
        help="filename",
        metavar="FILE",
        default=str(Path.home() / '.aws' / 'config')
    )

    parser.add_argument('--yes', '-y',
        help="acknowledge all binary (yes/no) prompts",
        action="store_true"
    )

    parser.add_argument('--mfa-device-arn',
        help="ARN of AWS IAM virtual MFA device",
        metavar="FILE",
    )

    parser.add_argument('--mfa-token-code',
        help="A recent code generated by your MFA token device, will be prompted for if not supplied",
        metavar="TOKEN_CODE",
    )

    parser.add_argument('--credentials',
        help="filename",
        metavar="FILE",
        default=str(Path.home() / '.aws' / 'credentials')
    )

    return parser


def winnfield(country: str, index: 0):
    return[
        'What country you from?',
        f'{country} ain\'t no country i ever heard of. They can answer y(es), or n(o) in {country}?',
        f'y(es), or n(o) mothertrucker! Have you decided?',
        f'Then you know what i\'m saying.'
    ][index]


def sub(context: aws_myfa.Context) -> int:
    """
    """

    assert Path(context.config_path).exists(), f"{context.config_path}: AWS profile ""config path does not exist"

    assert Path(context.credentials_path).exists(), f"{context.credentials_path}: AWS profile credentials path does not exist"

    config = ConfigParser()

    credentials = ConfigParser()

    config.read(context.config_path)

    credentials.read(context.credentials_path)

    profiles: Dict[str, Profile] = {}

    source_profiles: Dict[str, SourceProfile] = {}

    for object_id in context.profile_names.keys():

        profile_name: Optional[str] = ''

        if context.profile_names[object_id]:

            profile_name = context.profile_names[object_id]

        elif object_id == 'principal' and \
             environ.get('AWS_PROFILE', None) != None:

            profile_name = environ['AWS_PROFILE']
        else:

            profile_name = "default"

        try:

            profiles[object_id] = aws_myfa.get_profile(profile_name, config)

        except aws_myfa.NoSuchProfileException as err:

            if object_id == 'session':

                continue

            raise Exception(f"get_profile(\"{profile_name}\", ...): {err}"
            ) from err

        if profiles[object_id].source_profile not in source_profiles.keys():

            source_profile_name = profiles[object_id].source_profile

            try:

                source_profiles[source_profile_name] = aws_myfa.get_source_profile(
                    profiles[object_id].source_profile,
                    credentials
                )
            except aws_myfa.NoSuchProfileException as err:

                raise Exception(f"get_source_profile(\"{source_profile_name}\""
                                f", ...): {err}") from err

    mfa_device_arn = context.mfa_device_arn

    if context.profile_names['session'] in source_profiles.keys():

        if source_profiles[context.profile_names['session']].aws_mfa_device_arn:

            mfa_device_arn = source_profiles[context.profile_names['session']].aws_mfa_device_arn

    if not mfa_device_arn:

        raise Exception('no MFA device ARN provided, neither with'
                        '\'--mfa-device-arn\' command-line argument, nor with '
                        '\'aws_mfa_device_arn\' configuration section '
                        'property')

    env = environ.copy()

    env['AWS_PROFILE'] = profiles['principal'].source_profile

    cmd = [
        'aws', 'sts', 'get-caller-identity',
        '--output', 'json',
        '--profile', profiles['principal'].name
    ]

    print("\033[94m%s\033[0m" % ' '.join(cmd))

    data = json.loads(check_output(cmd, env=env))

    print(f"Principal Access Key ID ({profiles['principal'].name}):\n\t"
          f"{data['UserId']}\n")

    print(f"Principal ARN ({profiles['principal'].name}):\n\t{data['Arn']}\n")

    choice = None

    dialog_index = 0

    prompt = 'Get STS session token for this principal (y|n)?: '

    while True:

        sys.stdout.write(prompt)

        if context.acknowledge_prompts:

            print('y')

            break

        choice = input()

        prompt='%s ' % winnfield(choice, dialog_index)

        if choice == 'y':

            if (dialog_index == 3):

                print(prompt)

            break

        elif choice == 'n':

            return

        if dialog_index == 3:

            return(1)

        dialog_index += 1

    mfa_token_code: str = context.mfa_token_code

    session_source_profile = None

    while True:

        if not mfa_token_code:

            sys.stdout.write('Enter MFA Token Code: ')

            mfa_token_code = ''.join(input().split(' '))

            if not mfa_token_code:

                continue

        try:

            session_source_profile = aws_myfa.get_session_source_profile(
                context.profile_names['session'],
                mfa_device_arn,
                mfa_token_code,
                env
            )
        except KeyboardInterrupt:

            break

        except Exception as err:

            if context.mfa_token_code:

                raise err from err

            mfa_token_code = ''

            continue
        else:
            break

    credentials[context.profile_names['session']] = asdict(
        session_source_profile
    )

    del credentials[context.profile_names['session']]['name']

    profile = None

    if profiles.get('session'):

        profile = aws_myfa.Profile(**{
            **asdict(profiles['session']),
            **{
                'source_profile': session_source_profile.name
            }
        })
    else:

        profile = aws_myfa.Profile(**{
            **asdict(profiles['principal']),
            **{
                'name': context.profile_names['session'],
                'source_profile': session_source_profile.name
            }
        })

    config[f"profile {context.profile_names['session']}"] = asdict(profile)

    with open(context.credentials_path, 'w') as credentials_file:

        credentials.write(credentials_file)

        print(f"Written to [{context.profile_names['session']}] in: "
              f"{context.credentials_path}")

    with open(context.config_path, 'w') as config_file:

        config.write(config_file)

        print(f"Written to [profile {context.profile_names['session']}] in: "
              f"{context.config_path}")

    env = environ.copy()

    env['AWS_PROFILE'] = context.profile_names['session']

    cmd = [
        'aws',
        'sts',
        'get-caller-identity',
        '--output', 'json',
        '--profile', context.profile_names['session']
    ]

    print("\033[94m%s\033[0m" % ' '.join(cmd))

    data = json.loads(check_output(cmd, env=env))

    print(f"Session Access Key ID ({session_source_profile.name}):\n\t"
          f"{session_source_profile.aws_access_key_id}\n")

    print(f"Session Principal ARN ({session_source_profile.name}):\n\t"
          f"{data['Arn']}\n")


def main():
    """
    """
    argv = vars(get_parser().parse_args())

    sys.exit(sub(aws_myfa.Context(
        config_path = Path(argv['config']),
        credentials_path = Path(argv['credentials']),
        mfa_device_arn=argv['mfa_device_arn'],
        profile_names={
            'session': str(argv['session-profile-name'] or ''),
            'principal': str(argv['principal-profile-name'] or ''),
        },
        acknowledge_prompts = argv['yes'],
        mfa_token_code = argv['mfa_token_code']
    )))


if __name__ == '__main__':

    main()
