# Copyright 2019 Coop IT Easy SCRL fs
#   Robin Keunen <robin@coopiteasy.be>
# License AGPL-3.0 or later (https://www.gnu.org/licenses/agpl.html).
# pylint: disable=consider-merging-classes-inherited

import logging
from werkzeug.exceptions import BadRequest, NotFound
from odoo import _
from odoo.addons.base_rest.http import wrapJsonException
from odoo.addons.component.core import Component
from odoo.addons.base_rest import restapi
from . import schemas


_logger = logging.getLogger(__name__)

class PermsService(Component):
    _inherit = "base.rest.private_abstract_service"
    _name = "ce.services"
    _usage = "member"
    _description = """
        CE Member roles requests
    """

    @restapi.method(
        [(["/<string:keycloak_id>"], "GET")],
        output_param=restapi.CerberusValidator("_validator_return_get"),
        auth="api_key",
    )
    def get(self, _keycloak_id):
        user, partner, roles, email = self._get_member_profile_objs(_keycloak_id)
        return self._to_dict(user, partner, role, email)
    
    def _validator_return_get(self):
        return schemas.S_MEMBER_PROFILE_RETURN_GET

    @restapi.method(
        [(["/<string:keycloak_id>"], "PUT")],
        input_param=restapi.CerberusValidator("_validator_update"),
        output_param=restapi.CerberusValidator("_validator_return_update"),
        auth="api_key",
    )
    def update(self, _keycloak_id, **params):
        _logger.info("Requested role update, user: {}".format(_keycloak_id))

        user, partner, role, email = self._get_member_profile_objs(_keycloak_id)
        roles_map = user.ce_user_roles_mapping()
        ce_admin_key = self.env['ir.config_parameter'].sudo().get_param('ce.ck_user_group_mapped_to_odoo_group_ce_admin')
        ce_admin_value = roles_map[ce_admin_key]

        new_role = params['role']
        if not new_role in roles_map.keys():
            raise wrapJsonException(
                BadRequest(
                    _("Role {} not found!").format(
                        new_role)
                ),
                include_description=True,
            )

        RoleLineSudo = self.env['res.users.role.line'].sudo()
        for role_line in user.role_line_ids:
            role_id = role_line.role_id.id
            if role_id in roles_map.values():
                role_line.unlink()
                if role_id == ce_admin_value and not user.company_id.check_ce_has_admin():
                    raise wrapJsonException(
                        BadRequest(
                            _("CE must have and admin!")),
                        include_description=True,
                    )
        new_role_vals = {'user_id': user.id, 'role_id': roles_map[new_role]}
        RoleLineSudo.create(new_role_vals)
        #user.update_user_data_to_keyckoack(['groups'])
        return self._to_dict(user, partner, new_role, email)

    def _prepare_create(self, params):
        """Prepare a writable dictionary of values"""
        return {
            "company_id": params["company_id"],
            "user_id": params["user_id"],
            "target_company_id": params["target_company_id"],
            "target_user_id": params["target_user_id"],
            "new_role": params["new_role"],
        }

    def _validator_update(self):
        return schemas.S_MEMBER_PROFILE_PUT

    def _validator_return_update(self):
        return schemas.S_MEMBER_PROFILE_RETURN_PUT
    

    def _get_member_profile_objs(self, _keycloak_id):
        user = self.env["res.users"].sudo().search([('oauth_uid','=',_keycloak_id)])
        if not user:
            raise wrapJsonException(
                BadRequest(),
                include_description=False,
                extra_info={'message': _("No Odoo User found for KeyCloak user id %s") % _keycloak_id}
            )

        partner = user.partner_id or None
        if not partner:
            raise wrapJsonException(
                BadRequest(),
                include_description=False,
                extra_info={'message': _("No Odoo Partner found for Odoo user with login username %s") % user.login}
            )

        role = partner.user_ids.role_ids
        email = partner.email or False
        return user, partner, role, email


    @staticmethod
    def _to_dict(user, partner, role, email):
        return {
            'member':{
                "keycloak_id": user.oauth_uid,
                "name": user.display_name,
                "role": user.ce_role or "",
                "email": email,
            }
        }