import numpy as np
from matplotlib import pyplot as plt

from UncertainSCI.model_examples import ishigami_function
from UncertainSCI.distributions import BetaDistribution, TensorialDistribution
from UncertainSCI.pce import PolynomialChaosExpansion
from UncertainSCI.indexing import TotalDegreeSet

"""
This script demonstrates basic construction and usage of PCE emulators, and
shows performance of computed statistics as a function of the number of model
samples (computational effort).

Boxplots are generated by querying the PCE emulator and generating a large
ensemble.
"""

## Set up model
# Parameters for function
a = 1
b = 0.05

f = ishigami_function(a,b)

## Set up parameter distributions
bounds = np.reshape(np.array([-np.pi, np.pi]), [2, 1])
p1 = BetaDistribution(alpha=1, beta=1, domain=bounds)
p2 = BetaDistribution(alpha=1, beta=1, domain=bounds)
p3 = BetaDistribution(alpha=1, beta=1, domain=bounds)

p = TensorialDistribution(distributions=[p1, p2, p3])

## Build PCE's for various polynomial orders
ensemble_size = int(1e6)
orders = [3, 4, 5, 6]
pces = []
ensembles = []

for order in orders:
    index_set = TotalDegreeSet(dim=3, order=order)
    pce = PolynomialChaosExpansion(distribution=p, index_set=index_set)
    pce.build(model=f)
    pces.append(pce)

    # Post-processing: sample the PCE emulator
    pvals = p.MC_samples(M=ensemble_size)
    ensembles.append(pce.pce_eval(pvals))

## Compute MC statistics (for comparison)
pvals = p.MC_samples(M=ensemble_size)
oracle_ensemble = np.zeros(ensemble_size)
for i in range(ensemble_size):
    oracle_ensemble[i] = f(pvals[i,:])
ensembles.append(oracle_ensemble)

# Box plots require 1D arrays as input
for i in range(len(ensembles)):
    ensembles[i] = ensembles[i].flatten()

## Construct boxplots
plt.figure()
plt.boxplot(ensembles)
tick_locations = [1 + i for i in range(len(ensembles))]
tick_labels = ['UncertainSCI\n{1:d} samples\norder {0:d}'.format(orders[i], pces[i].samples.shape[0]) for i in range(len(pces))]
tick_labels.append('Monte Carlo\n{0:1.1e} samples'.format(ensemble_size))
plt.xticks(ticks=tick_locations, labels=tick_labels)
plt.title('Ishigami function boxplots')
plt.show()
