"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log('Create new ActionSecret with props ' + JSON.stringify(props));
    await createOrUpdateRepoSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log(`Update ActionSecret ${secretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateRepoSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onDelete = async (event, octokit) => {
    const secretName = event.ResourceProperties.repositorySecretName;
    console.log('Delete ActionSecret ' + secretName);
    await deleteRepoSecret(event, octokit);
    return { PhysicalResourceId: secretName };
};
const createOrUpdateRepoSecret = async (event, octokit, smClient) => {
    const { repositoryOwner: owner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const secretId = event.ResourceProperties.sourceSecretArn;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const secretResponse = await octokit.request('PUT /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteRepoSecret = async (event, octokit) => {
    const { repositoryOwner: owner, repositoryName: repo, repositorySecretName: secret_name } = event.ResourceProperties;
    const deleteSecretResponse = await octokit.request('DELETE /repos/{owner}/{repo}/actions/secrets/{secret_name}', {
        owner,
        repo,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,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