import * as appscaling from 'aws-cdk-lib/aws-applicationautoscaling';
import * as cloudwatch from 'aws-cdk-lib/aws-cloudwatch';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IEndpointConfig } from './endpoint-config';
import { ScalableInstanceCount } from './scalable-instance-count';
/**
 * The interface for a SageMaker Endpoint resource.
 */
export interface IEndpoint extends cdk.IResource {
    /**
     * The ARN of the endpoint.
     *
     * @attribute
     */
    readonly endpointArn: string;
    /**
     * The name of the endpoint.
     *
     * @attribute
     */
    readonly endpointName: string;
    /**
     * Permits an IAM principal to invoke this endpoint
     * @param grantee The principal to grant access to
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Represents the features common to all production variant types (e.g., instance, serverless) that
 * have been associated with an endpoint.
 */
interface IEndpointProductionVariant {
    /**
     * The name of the production variant.
     */
    readonly variantName: string;
    /**
     * Return the given named metric for Endpoint
     *
     * @default - sum over 5 minutes
     */
    metric(namespace: string, metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Represents an instance production variant that has been associated with an endpoint.
 */
export interface IEndpointInstanceProductionVariant extends IEndpointProductionVariant {
    /**
     * Metric for the number of invocations
     *
     * @default - sum over 5 minutes
     */
    metricInvocations(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations per instance
     *
     * @default - sum over 5 minutes
     */
    metricInvocationsPerInstance(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for model latency
     *
     * @default - average over 5 minutes
     */
    metricModelLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for overhead latency
     *
     * @default - average over 5 minutes
     */
    metricOverheadLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of invocations by HTTP response code
     *
     * @default - sum over 5 minutes
     */
    metricInvocationResponseCode(responseCode: InvocationHttpResponseCode, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for disk utilization
     *
     * @default - average over 5 minutes
     */
    metricDiskUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for CPU utilization
     *
     * @default - average over 5 minutes
     */
    metricCpuUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for memory utilization
     *
     * @default - average over 5 minutes
     */
    metricMemoryUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for GPU utilization
     *
     * @default - average over 5 minutes
     */
    metricGpuUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for GPU memory utilization
     *
     * @default - average over 5 minutes
     */
    metricGpuMemoryUtilization(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Enable autoscaling for SageMaker Endpoint production variant
     *
     * @param scalingProps EnableScalingProps
     */
    autoScaleInstanceCount(scalingProps: appscaling.EnableScalingProps): ScalableInstanceCount;
}
/**
 * Represents an Endpoint resource defined outside this stack.
 */
export interface EndpointAttributes {
    /**
     * The ARN of this endpoint.
     */
    readonly endpointArn: string;
}
declare abstract class EndpointBase extends cdk.Resource implements IEndpoint {
    /**
     * The ARN of the endpoint.
     *
     * @attribute
     */
    abstract readonly endpointArn: string;
    /**
     * The name of the endpoint.
     *
     * @attribute
     */
    abstract readonly endpointName: string;
    /**
     * Permits an IAM principal to invoke this endpoint
     * @param grantee The principal to grant access to
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Construction properties for a SageMaker Endpoint.
 */
export interface EndpointProps {
    /**
     * Name of the endpoint.
     *
     * @default - AWS CloudFormation generates a unique physical ID and uses that ID for the
     * endpoint's name.
     */
    readonly endpointName?: string;
    /**
     * The endpoint configuration to use for this endpoint.
     */
    readonly endpointConfig: IEndpointConfig;
}
/**
 * HTTP response codes for Endpoint invocations
 */
export declare enum InvocationHttpResponseCode {
    /**
     * 4xx response codes from Endpoint invocations
     */
    INVOCATION_4XX_ERRORS = "Invocation4XXErrors",
    /**
     * 5xx response codes from Endpoint invocations
     */
    INVOCATION_5XX_ERRORS = "Invocation5XXErrors"
}
/**
 * Defines a SageMaker endpoint.
 */
export declare class Endpoint extends EndpointBase {
    /**
     * Imports an Endpoint defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointArn the ARN of the endpoint.
     */
    static fromEndpointArn(scope: Construct, id: string, endpointArn: string): IEndpoint;
    /**
     * Imports an Endpoint defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param endpointName the name of the endpoint.
     */
    static fromEndpointName(scope: Construct, id: string, endpointName: string): IEndpoint;
    /**
     * Imports an Endpoint defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param attrs the attributes of the endpoint to import.
     */
    static fromEndpointAttributes(scope: Construct, id: string, attrs: EndpointAttributes): IEndpoint;
    /**
     * The ARN of the endpoint.
     *
     * @attribute
     */
    readonly endpointArn: string;
    /**
     * The name of the endpoint.
     *
     * @attribute
     */
    readonly endpointName: string;
    private readonly endpointConfig;
    constructor(scope: Construct, id: string, props: EndpointProps);
    private validateEnvironmentCompatibility;
    /**
     * Get instance production variants associated with endpoint.
     */
    get instanceProductionVariants(): IEndpointInstanceProductionVariant[];
    /**
     * Find instance production variant based on variant name
     * @param name Variant name from production variant
     */
    findInstanceProductionVariant(name: string): IEndpointInstanceProductionVariant;
}
export {};
