"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BillingAlarm = void 0;
const cdk = require("@aws-cdk/core");
const cw = require("@aws-cdk/aws-cloudwatch");
const cwa = require("@aws-cdk/aws-cloudwatch-actions");
const sns = require("@aws-cdk/aws-sns");
const sub = require("@aws-cdk/aws-sns-subscriptions");
/**
 * A CDK construct that sets up email notification for when you exceed a given AWS estimated charges amount.
 *
 * Note: The email addresses used as SNS Topic endpoint must be manually confirmed
 * once the stack is deployed.
 *
 * @stability stable
 */
class BillingAlarm extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.emails.length === 0) {
            throw new Error('Cannot supply an empty array of email subscriptions');
        }
        const billingAlarmTopic = new sns.Topic(this, 'BillingAlarmNotificationTopic', {
            topicName: 'BillingAlarmNotificationTopic',
        });
        props.emails.forEach((email) => {
            billingAlarmTopic.addSubscription(new sub.EmailSubscription(email, {
                json: true,
            }));
        });
        const billingAlarmMetric = new cw.Metric({
            metricName: 'EstimatedCharges',
            namespace: 'AWS/Billing',
            statistic: 'Maximum',
            dimensions: {
                Currency: 'USD',
            },
        }).with({
            // Evaluates the metric every 9 hours. This is needed because
            // `EstimatedCharges` metrics is updated every 6 hours
            // See: https://forums.aws.amazon.com/thread.jspa?threadID=135955
            period: cdk.Duration.hours(9),
        });
        const billingAlarm = new cw.Alarm(this, 'BillingCloudWatchAlarm', {
            alarmDescription: 'Upper monthly billing cost limit',
            comparisonOperator: cw.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            metric: billingAlarmMetric,
            threshold: props.monthlyThreshold,
        });
        const alarmAction = new cwa.SnsAction(billingAlarmTopic);
        billingAlarm.addAlarmAction(alarmAction);
    }
}
exports.BillingAlarm = BillingAlarm;
//# sourceMappingURL=data:application/json;base64,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