# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['snakeboost']

package_data = \
{'': ['*']}

install_requires = \
['Pygments>=2.10.0,<3.0.0', 'attrs>=21.2.0,<22.0.0', 'colorama>=0.4.4,<0.5.0']

setup_kwargs = {
    'name': 'snakeboost',
    'version': '0.2.0',
    'description': 'Utility functions to turbocharge your snakemake workflows. Virtualenvs, tarfiles, and more.',
    'long_description': '# Snakeboost\n\nSnakeboost provides enhancers and helpers to turbocharge your [snakemake](https://snakemake.readthedocs.io/en/stable/) workflows.\nThe project is currently in it\'s alpha stage.\n\n# Script Enhancers\n\n## Overview\n\nScript enhancer functions wrap around bash scripts given under the `script` key in Snakemake Rules.\nAll enhancers have a common interface designed for easy use in your workflow.\nTo illustrate, we\'ll take `PipEnv` as an example (it lets you use pip virtual environments!).\n\n1. Initiate the enhancer\n\nImport the enhancer at the top of your `Snakefile` and instantiate it.\nMost enhancers take a few arguments defining their global settings.\n\n```python\nfrom snakeboost import PipEnv\n\nmy_env = PipEnv(packages=["numpy", "flake8"], root=("/tmp"))\n```\n\n2. Use the enhancer in a rule\n\nWhen instantiated, enhancers can be called using the bash command as an argument.\n\n```python\nrule lint_python:\n    inputs: "some-script.py"\n    shell:\n        my_env.script("flake8 {input}")\n```\n\nSome enhancers, such as `PipEnv`, provide multiple functions (e.g. `.script`, `.python`, etc) that provide slightly different functionality.\nOthers, such as `Tar`, have methods that return a modified instance.\n\n```python\nrule inspect_tarball:\n    inputs: "some_archive.tar.gz"\n    shell:\n        tar.using(inputs=["{input}"])("ls {input}/")\n```\n\nSnakeboost uses this slightly convoluted way of setting arguments to allow easy chaining of multiple enhancers.\nThis leads us to step 3:\n\n3. Use `pipe()` to chain multiple enhancers\n\nSnakeboost packages a `pipe()` arg very similar to that provided by `PyToolz` (in fact, that\'s what we adapted it from).\nThis lets us use a very clean syntax to specify multiple enhancers:\n\n```python\nfrom snakeboost import pipe\n\nrule lint_tarred_scripts:\n    inputs: "script_archive.tar.gz"\n    shell:\n        pipe(\n            tar.using(inputs=["{input}"]),\n            my_env.script,\n            "flake8 {input}/script.py"\n        )\n```\n\n## Available Enhancers\n\n### `PipEnv`\n\nSnakemake prefers Conda for Python environment management.\nConda, however, is not appropriate for certain environments, such as compute clusters.\nPipEnv allows you to create traditional virtual environments on the fly, using `virtualenv` under the hood.\nYou can create as many environments as you want per workflow, and each environment can be instantiate with a combination of package listings and `requirements.txt` files.\nAll environments will be saved to the folder you specify under the `root` argument; typically, this would be a temporary directory.\nIndividual environments will be saved in a folder named with the hash of all package names and `requirements.txt` paths, and will be reused when possible.\nEnvironment creation is threadsafe, so having 32 rules using the same environment causes no issues.\n\n[Source](snakeboost/pipenv.py)\n\n### `Pyscript`\n\nBecause Pipenv modifies the shell script, it cannot be used with Snakemake\'s `script` directive.\n`Pyscript` fills this gap, providing similar features to the native `script` directive while remaining compatible with Pipenv.\nIt can also be combined with any of the other enhancers!\n\nPyscript works by calling your script as a command line progam, passing inputs, outputs, params, and every other snakemake entity as arguments.\nOn the Snakefile side, snakeboost provides the `Pyscript` enhancer, which composes the bash to call your python file.\nOn the script side, snakeboost has the `snakemake_parser()` function, which automatically parses the command line arguments sent by Snakemake and organizes them into the `SnakemakeArgs` class.\nThis object has the same interface as the `snakemake` object used in ordinary `snakemake` scripts; the only difference is that your script must retrieve it by importing and calling `snakemake_parser().\nThus, any scripts called by `Pyscript` must have `snakeboost` as a dependency in their Python environment.\n\n`Pyscript` will attempt to automatically pass on every entity (including inputs, outputs, params, etc) in your rule on to the script.\nThis works fantastic for single items or lists:\n\n```python\nrule some_rule:\n    input: "path/to/input"\n\nrule rule_with_list_input:\n    input:\n        "/path/1",\n        "/path/2",\n        "/path/3"\n```\n\nThese will be received in your script as:\n\n```python\n\nfrom snakeboost import snakemake_parser\n\nsnakemake = snakemake_parser()\n\n# some_rule\nassert snakemake.input == Path("path/to/input")\n\n# rule_with_list_input\nassert snakemake.input == [\n    Path("/path/1"),\n    Path("/path/2"),\n    Path("/path/3")\n]\n```\n\nUnfortunately, it is unable to infer the names of named entities:\n\n```python\nrule named_inputs:\n    input:\n        input_1="/path/1",\n        input_2="/path2\n```\n\nBy default, these will be treated as lists.\nYou can, however, annotate your rules to inform `Pyscript` of the entity names.\nIt can then pass them on to `snakemake_parser()` to form dicts.\n\nThere are two methods to achieve this.\nThe first is through inline annotation:\n\n```python\nmy_script = Pyscript(workflow.basedir)\nrule rule_with_script:\n    input:\n        **my_script.input(\n            input_1="path/1",\n            input_2="path/2\n        )\n    output: "path/to/output\n    shell: my_script("scripts/my_script.py")\n```\n\nHere, we start by defining the script before the rule.\nNotice how `workflow.basedir` is passed into `Pyscript` as an argument.\nThis is a variable provided by Snakemake set to the location of the Snakefile.\nWe use it here to inform `Pyscript` where the scripts directory will be.\nSecond, we wrap named entities using the appropriate method from our `Pyscript` instance (we would use `my_method.params()` for params, etc).\nNotice the double asterisk `**` before the method call.\nThis is essential to properly unpack the Dictionary returned by `.input()` so it can be read by Snakemake.\nFinally, we call our `Pyscript` instance with the path of our script relative to the Snakefile (just like in normal Snakemake scripts) and pass it to the **shell** directive.\nNotice how we only need to wrap named entities.\nIn the above example, `output` will still be passed on to the script.\n\nThe second method is explicately listing the entities we\'d like to include as parameters in our final call to the `Pyscript` instance:\n\n```python\nrule rule_with_script:\n    input:\n        input_1="path/1",\n        input_2="path/2",\n        input_3="path/3\n    output: "path/to/output\n    shell:\n        Pyscript(workflow.basedir)(\n            "scripts/my_script.py",\n            input=["input_1", "input_2"]\n        )\n```\n\nThis method has the advantage of only passing the explicitely named entities.\nIn the above example, `input_1` and `input_2` would be passed to the script, but not `input_3`.\nNote that other entity types, such as output in the above example, would be passed as normal.\nThe method also allows a completely inline creation of the `Pyscript`, which may be preferable.\n\nWhichever method you choose, named entities can be consumed in your script as follows:\n\n```python\n\nfrom snakeboost import snakemake_parser\n\nsnakemake = snakemake_parser()\n\nassert snakemake.input == {\n    "input_1": Path("path/1")\n    "input_2": Path("path/2")\n}\n```\n\nFinally, `Pyscript` is easily combined with a pipenv by supplying the env object as an argument to the `Pyscript` constructor:\n\n```python\nmy_env = Pipenv("/tmp", packages=["snakeboost"])\nmy_script = Pyscript(workflow.basedir, my_env)\n```\n\nNote that `Pyscript` and `Pipenv` must be combined in the above way.\n`Pyscript` should not be nested under a call of a `Pipenv` method:\n\n```python\n# Don\'t do this!! You\'ll just get an error.\nrule mistake:\n    shell:\n        my_env.python(\n            Pyscript(workflow.basedir)("scripts/my_script.py")\n        )\n```\n\n### `Tar`\n\nSometimes, rules produce a directory full of files as output.\nSnakemake can easily handle this using the `directory` function, but for large scale applications, it\'s easier on the filesystem to immediately package the directory into a tarball.\n`Tar` facilitates this exercise, effectively "mounting" tar files as directories in a scratch directory of your choice (e.g. your `tmp` dir).\nIt supports outputs (creating new tar files), inputs (reading existing tar files), and modification (reading and writing an existing tar file).\nIn all cases, you can treat the tar file in your script as if it were a regular directory.\n`Tar` will manage packing and unpacking.\n\nFor example, we could make a virtual env and package it in a tar file as follows:\n\n```python\nfrom snakeboost import Tar\n\ntar = Tar("/tmp/prepdwi_tarfolders")\n\nrule make_pipenv:\n    output:\n        "venv.tar.gz"\n    shell:\n        tar.using(outputs = ["{output}"])(\n            "virtualenv {output} && cat {output}/pyvenv.cfg",\n        )\n```\n\nWe can then read a file from this pipenv as follows:\n\n```python\nrule read_pipenv:\n    input:\n        pip=rules.make_pipenv.output,\n        out=rules.read_gitignore.output\n    shell:\n        tar.using(inputs=["{input.pip}"])(\n            "cat {input.pip}/pyvenv.cfg"\n        )\n```\n\nWe can add a file in a similar way:\n\n```python\nrule add_gitignore:\n    input:\n        rules.make_pipenv.output\n    output:\n        "gitignore.added"\n    shell:\n        tar.using(modify=["{input}"])(\n            "echo .pyenv >> {input}/.gitignore && touch {output}"\n        )\n```\n\nNote that we use `gitignore.added` as a marker file to indicate completion of the rule.\nThis is important, as here `rules.make_pipenv.output` is both the `input` and the `output` (since it\'s being modified), and this is inherently ambiguous to Snakemake.\n\nWhen using `"{input}"` or `"{output}"` wildcards, be sure to fully resolve them.\nIn other words, if you have multiple `inputs:`, use the full dot-syntax to access the precise input to be tarred (e.g. `"{input.pip}"`).\nOtherwise, you\'ll get errors.\n\nA few more features:\n* `inputs`, `ouputs`, and `modify` can be mixed and matched in one single rule as much as you please.\n* Unpacked tar files are stored in a directory under `root` using the hashed tar file path as the directory name.\n  These directories will not be deleted by snakeboost, meaning they will be seamlessly reused over the course of your workflow.\n\nNote that `Tar` is **NOT** necessarily threadsafe at this time, so if two or more jobs try to unpack the exact same file, you may get a strange failure.\nIf you require this behaviour, please leave an issue so it can be prioritized.\n\n### `XvfbRun`\n\n`xvfb-run` is a linux utility that starts a virtual X-server.\nThis is useful for performing rendering jobs on headless servers and cluster environments, as an easier alternative to X-11 forwarding with SSH.\nThe `XvfbRun` enhancer wraps commands with this call.\nIt\'s safer than manually prepending `xvfb-run -a` to your command, automatically handling quote escaping and preventing typos.\n\n# Contributing\n\nIf you have a small utility function for your Snakemake workflows, it would likely make a great addition to the Snakeboost ecosystem.\nScript enhancers should follow the basic interface of the other enhancers: a class initialized with global settings that exposes one or more functions that take a bash script as argument.\n\nSnakebids uses [Poetry](https://python-poetry.org/) for dependency management and [pre-commit](https://pre-commit.com/) for Quality Assurance tests.\nIf Poetry is not already installed on your system, follow the [instructions](https://python-poetry.org/docs/master/) on their website.\nThen clone the repo and initialize by running:\n\n```bash\npoetry install\npoetry run pre-commit install\n```\n',
    'author': 'Peter Van Dyken',
    'author_email': 'pvandyk2@uwo.ca',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<3.11',
}


setup(**setup_kwargs)
