import { ISignal, Signal } from '@phosphor/signaling';
import { ServerConnection } from '..';
import { TerminalSession } from './terminal';
/**
 * An implementation of a terminal interface.
 */
export declare class DefaultTerminalSession implements TerminalSession.ISession {
    /**
     * Construct a new terminal session.
     */
    constructor(name: string, options?: TerminalSession.IOptions);
    /**
     * A signal emitted when the session is shut down.
     */
    readonly terminated: Signal<this, void>;
    /**
     * A signal emitted when a message is received from the server.
     */
    readonly messageReceived: ISignal<this, TerminalSession.IMessage>;
    /**
     * Get the name of the terminal session.
     */
    readonly name: string;
    /**
     * Get the model for the terminal session.
     */
    readonly model: TerminalSession.IModel;
    /**
     * The server settings for the session.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the session is ready.
     */
    readonly isReady: boolean;
    /**
     * A promise that fulfills when the session is ready.
     */
    readonly ready: Promise<void>;
    /**
     * Test whether the session is disposed.
     */
    readonly isDisposed: boolean;
    /**
     * Dispose of the resources held by the session.
     */
    dispose(): void;
    /**
     * Send a message to the terminal session.
     */
    send(message: TerminalSession.IMessage): void;
    /**
     * Reconnect to the terminal.
     *
     * @returns A promise that resolves when the terminal has reconnected.
     */
    reconnect(): Promise<void>;
    /**
     * Shut down the terminal session.
     */
    shutdown(): Promise<void>;
    /**
     * Clone the current session object.
     */
    clone(): TerminalSession.ISession;
    /**
     * Connect to the websocket.
     */
    private _initializeSocket;
    private _reconnectSocket;
    private _isDisposed;
    private _isReady;
    private _messageReceived;
    private _terminated;
    private _name;
    private _readyPromise;
    private _url;
    private _ws;
    private _noOp;
    private _reconnectLimit;
    private _reconnectAttempt;
    private _disconnected;
}
/**
 * The static namespace for `DefaultTerminalSession`.
 */
export declare namespace DefaultTerminalSession {
    /**
     * Whether the terminal service is available.
     */
    function isAvailable(): boolean;
    /**
     * Start a new terminal session.
     *
     * @param options - The session options to use.
     *
     * @returns A promise that resolves with the session instance.
     */
    function startNew(options?: TerminalSession.IOptions): Promise<TerminalSession.ISession>;
    function connectTo(name: string, options?: TerminalSession.IOptions): Promise<TerminalSession.ISession>;
    /**
     * List the running terminal sessions.
     *
     * @param settings - The server settings to use.
     *
     * @returns A promise that resolves with the list of running session models.
     */
    function listRunning(settings?: ServerConnection.ISettings): Promise<TerminalSession.IModel[]>;
    /**
     * Shut down a terminal session by name.
     *
     * @param name - The name of the target session.
     *
     * @param settings - The server settings to use.
     *
     * @returns A promise that resolves when the session is shut down.
     */
    function shutdown(name: string, settings?: ServerConnection.ISettings): Promise<void>;
    /**
     * Shut down all terminal sessions.
     *
     * @param settings - The server settings to use.
     *
     * @returns A promise that resolves when all the sessions are shut down.
     */
    function shutdownAll(settings?: ServerConnection.ISettings): Promise<void>;
}
